"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi
 */
class RestApiBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
}
exports.RestApiBase = RestApiBase;
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 *
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, {}, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing RestApi that can be configured with additional Methods and Resources.
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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