import { Construct, Duration, IResource, Resource } from '@aws-cdk/core';
import { AccessLogFormat, IAccessLogDestination } from './access-log';
import { Deployment } from './deployment';
import { IRestApi } from './restapi';
/**
 * Represents an APIGateway Stage.
 */
export interface IStage extends IResource {
    /**
     * Name of this stage.
     * @attribute
     */
    readonly stageName: string;
    /**
     * RestApi to which this stage is associated.
     */
    readonly restApi: IRestApi;
}
export interface StageOptions extends MethodDeploymentOptions {
    /**
     * The name of the stage, which API Gateway uses as the first path segment
     * in the invoked Uniform Resource Identifier (URI).
     *
     * @default - "prod"
     */
    readonly stageName?: string;
    /**
     * The CloudWatch Logs log group.
     *
     * @default - No destination
     */
    readonly accessLogDestination?: IAccessLogDestination;
    /**
     * A single line format of access logs of data, as specified by selected $content variables.
     * The format must include at least `AccessLogFormat.contextRequestId()`.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference
     *
     * @default - Common Log Format
     */
    readonly accessLogFormat?: AccessLogFormat;
    /**
     * Specifies whether Amazon X-Ray tracing is enabled for this method.
     *
     * @default false
     */
    readonly tracingEnabled?: boolean;
    /**
     * Indicates whether cache clustering is enabled for the stage.
     *
     * @default - Disabled for the stage.
     */
    readonly cacheClusterEnabled?: boolean;
    /**
     * The stage's cache cluster size.
     * @default 0.5
     */
    readonly cacheClusterSize?: string;
    /**
     * The identifier of the client certificate that API Gateway uses to call
     * your integration endpoints in the stage.
     *
     * @default - None.
     */
    readonly clientCertificateId?: string;
    /**
     * A description of the purpose of the stage.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * The version identifier of the API documentation snapshot.
     *
     * @default - No documentation version.
     */
    readonly documentationVersion?: string;
    /**
     * A map that defines the stage variables. Variable names must consist of
     * alphanumeric characters, and the values must match the following regular
     * expression: [A-Za-z0-9-._~:/?#&amp;=,]+.
     *
     * @default - No stage variables.
     */
    readonly variables?: {
        [key: string]: string;
    };
    /**
     * Method deployment options for specific resources/methods. These will
     * override common options defined in `StageOptions#methodOptions`.
     *
     * @param path is {resource_path}/{http_method} (i.e. /api/toys/GET) for an
     * individual method override. You can use `*` for both {resource_path} and {http_method}
     * to define options for all methods/resources.
     *
     * @default - Common options will be used.
     */
    readonly methodOptions?: {
        [path: string]: MethodDeploymentOptions;
    };
}
export interface StageProps extends StageOptions {
    /**
     * The deployment that this stage points to [disable-awslint:ref-via-interface].
     */
    readonly deployment: Deployment;
}
export declare enum MethodLoggingLevel {
    OFF = "OFF",
    ERROR = "ERROR",
    INFO = "INFO"
}
export interface MethodDeploymentOptions {
    /**
     * Specifies whether Amazon CloudWatch metrics are enabled for this method.
     *
     * @default false
     */
    readonly metricsEnabled?: boolean;
    /**
     * Specifies the logging level for this method, which effects the log
     * entries pushed to Amazon CloudWatch Logs.
     *
     * @default - Off
     */
    readonly loggingLevel?: MethodLoggingLevel;
    /**
     * Specifies whether data trace logging is enabled for this method, which
     * effects the log entries pushed to Amazon CloudWatch Logs.
     *
     * @default false
     */
    readonly dataTraceEnabled?: boolean;
    /**
     * Specifies the throttling burst limit.
     * The total rate of all requests in your AWS account is limited to 5,000 requests.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html
     *
     * @default - No additional restriction.
     */
    readonly throttlingBurstLimit?: number;
    /**
     * Specifies the throttling rate limit.
     * The total rate of all requests in your AWS account is limited to 10,000 requests per second (rps).
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html
     *
     * @default - No additional restriction.
     */
    readonly throttlingRateLimit?: number;
    /**
     * Specifies whether responses should be cached and returned for requests. A
     * cache cluster must be enabled on the stage for responses to be cached.
     *
     * @default - Caching is Disabled.
     */
    readonly cachingEnabled?: boolean;
    /**
     * Specifies the time to live (TTL), in seconds, for cached responses. The
     * higher the TTL, the longer the response will be cached.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html
     *
     * @default Duration.minutes(5)
     */
    readonly cacheTtl?: Duration;
    /**
     * Indicates whether the cached responses are encrypted.
     *
     * @default false
     */
    readonly cacheDataEncrypted?: boolean;
}
export declare class Stage extends Resource implements IStage {
    readonly stageName: string;
    readonly restApi: IRestApi;
    private enableCacheCluster?;
    constructor(scope: Construct, id: string, props: StageProps);
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path?: string): string;
    private renderMethodSettings;
}
