"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonSchemaMapper = exports.validateInteger = exports.parseAwsApiCall = exports.parseMethodOptionsPath = exports.validateHttpMethod = exports.ALL_METHODS = void 0;
const url_1 = require("url");
const jsonSchema = require("./json-schema");
exports.ALL_METHODS = ['OPTIONS', 'GET', 'PUT', 'POST', 'DELETE', 'PATCH', 'HEAD'];
const ALLOWED_METHODS = ['ANY', ...exports.ALL_METHODS];
function validateHttpMethod(method, messagePrefix = '') {
    if (!ALLOWED_METHODS.includes(method)) {
        throw new Error(`${messagePrefix}Invalid HTTP method "${method}". Allowed methods: ${ALLOWED_METHODS.join(',')}`);
    }
}
exports.validateHttpMethod = validateHttpMethod;
function parseMethodOptionsPath(originalPath) {
    if (!originalPath.startsWith('/')) {
        throw new Error(`Method options path must start with '/': ${originalPath}`);
    }
    const path = originalPath.substr(1); // trim trailing '/'
    const components = path.split('/');
    if (components.length < 2) {
        throw new Error(`Method options path must include at least two components: /{resource}/{method} (i.e. /foo/bar/GET): ${path}`);
    }
    const httpMethod = components.pop().toUpperCase(); // last component is an HTTP method
    if (httpMethod !== '*') {
        validateHttpMethod(httpMethod, `${originalPath}: `);
    }
    let resourcePath = '/~1' + components.join('~1');
    if (components.length === 1 && components[0] === '*') {
        resourcePath = '/*';
    }
    else if (components.length === 1 && components[0] === '') {
        resourcePath = '/';
    }
    return {
        httpMethod,
        resourcePath,
    };
}
exports.parseMethodOptionsPath = parseMethodOptionsPath;
function parseAwsApiCall(path, action, actionParams) {
    if (actionParams && !action) {
        throw new Error('"actionParams" requires that "action" will be set');
    }
    if (path && action) {
        throw new Error(`"path" and "action" are mutually exclusive (path="${path}", action="${action}")`);
    }
    if (path) {
        return {
            apiType: 'path',
            apiValue: path,
        };
    }
    if (action) {
        if (actionParams) {
            action += '&' + url_1.format({ query: actionParams }).substr(1);
        }
        return {
            apiType: 'action',
            apiValue: action,
        };
    }
    throw new Error('Either "path" or "action" are required');
}
exports.parseAwsApiCall = parseAwsApiCall;
function validateInteger(property, messagePrefix) {
    if (property && !Number.isInteger(property)) {
        throw new Error(`${messagePrefix} should be an integer`);
    }
}
exports.validateInteger = validateInteger;
class JsonSchemaMapper {
    /**
     * Transforms naming of some properties to prefix with a $, where needed
     * according to the JSON schema spec
     * @param schema The JsonSchema object to transform for CloudFormation output
     */
    static toCfnJsonSchema(schema) {
        const result = JsonSchemaMapper._toCfnJsonSchema(schema);
        if (!('$schema' in result)) {
            result.$schema = jsonSchema.JsonSchemaVersion.DRAFT4;
        }
        return result;
    }
    static _toCfnJsonSchema(schema, preserveKeys = false) {
        if (schema == null || typeof schema !== 'object') {
            return schema;
        }
        if (Array.isArray(schema)) {
            return schema.map(entry => JsonSchemaMapper._toCfnJsonSchema(entry));
        }
        return Object.assign({}, ...Object.entries(schema).map(([key, value]) => {
            const mapKey = !preserveKeys && (key in JsonSchemaMapper.SchemaPropsWithPrefix);
            const newKey = mapKey ? JsonSchemaMapper.SchemaPropsWithPrefix[key] : key;
            // If keys were preserved, don't consider SchemaPropsWithUserDefinedChildren for those keys (they are user-defined!)
            const newValue = JsonSchemaMapper._toCfnJsonSchema(value, !preserveKeys && JsonSchemaMapper.SchemaPropsWithUserDefinedChildren[key]);
            return { [newKey]: newValue };
        }));
    }
}
exports.JsonSchemaMapper = JsonSchemaMapper;
JsonSchemaMapper.SchemaPropsWithPrefix = {
    schema: '$schema',
    ref: '$ref',
    id: '$id',
};
// The value indicates whether direct children should be key-mapped.
JsonSchemaMapper.SchemaPropsWithUserDefinedChildren = {
    definitions: true,
    properties: true,
    patternProperties: true,
    dependencies: true,
};
//# sourceMappingURL=data:application/json;base64,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