"use strict";
const assert_1 = require("@aws-cdk/assert");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const apigateway = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', { deployment });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                testapiD6451F70: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'test-api',
                    },
                },
                testapiGETD8DE4ED1: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: {
                            'Fn::GetAtt': [
                                'testapiD6451F70',
                                'RootResourceId',
                            ],
                        },
                        RestApiId: {
                            Ref: 'testapiD6451F70',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                mydeployment71ED3B4B5ce82e617e0729f75657ddcca51e3b91: {
                    Type: 'AWS::ApiGateway::Deployment',
                    Properties: {
                        RestApiId: {
                            Ref: 'testapiD6451F70',
                        },
                    },
                    DependsOn: [
                        'testapiGETD8DE4ED1',
                    ],
                },
                mystage7483BE9A: {
                    Type: 'AWS::ApiGateway::Stage',
                    Properties: {
                        RestApiId: {
                            Ref: 'testapiD6451F70',
                        },
                        DeploymentId: {
                            Ref: 'mydeployment71ED3B4B5ce82e617e0729f75657ddcca51e3b91',
                        },
                        StageName: 'prod',
                    },
                },
            },
        });
        test.done();
    },
    'common method settings can be set at the stage level'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            loggingLevel: apigateway.MethodLoggingLevel.INFO,
            throttlingRateLimit: 12,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            MethodSettings: [
                {
                    HttpMethod: '*',
                    LoggingLevel: 'INFO',
                    ResourcePath: '/*',
                    ThrottlingRateLimit: 12,
                },
            ],
        }));
        test.done();
    },
    'custom method settings can be set by their path'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            loggingLevel: apigateway.MethodLoggingLevel.INFO,
            throttlingRateLimit: 12,
            methodOptions: {
                '/goo/bar/GET': {
                    loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            MethodSettings: [
                {
                    HttpMethod: '*',
                    LoggingLevel: 'INFO',
                    ResourcePath: '/*',
                    ThrottlingRateLimit: 12,
                },
                {
                    HttpMethod: 'GET',
                    LoggingLevel: 'ERROR',
                    ResourcePath: '/~1goo~1bar',
                },
            ],
        }));
        test.done();
    },
    'default "cacheClusterSize" is 0.5 (if cache cluster is enabled)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cacheClusterEnabled: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            CacheClusterEnabled: true,
            CacheClusterSize: '0.5',
        }));
        test.done();
    },
    'setting "cacheClusterSize" implies "cacheClusterEnabled"'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cacheClusterSize: '0.5',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            CacheClusterEnabled: true,
            CacheClusterSize: '0.5',
        }));
        test.done();
    },
    'fails when "cacheClusterEnabled" is "false" and "cacheClusterSize" is set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cacheClusterSize: '0.5',
            cacheClusterEnabled: false,
        }), /Cannot set "cacheClusterSize" to 0.5 and "cacheClusterEnabled" to "false"/);
        test.done();
    },
    'if "cachingEnabled" in method settings, implicitly enable cache cluster'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cachingEnabled: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            CacheClusterEnabled: true,
            CacheClusterSize: '0.5',
            MethodSettings: [
                {
                    CachingEnabled: true,
                    HttpMethod: '*',
                    ResourcePath: '/*',
                },
            ],
            StageName: 'prod',
        }));
        test.done();
    },
    'if caching cluster is explicitly disabled, do not auto-enable cache cluster when "cachingEnabled" is set in method options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => new apigateway.Stage(stack, 'my-stage', {
            cacheClusterEnabled: false,
            deployment,
            cachingEnabled: true,
        }), /Cannot enable caching for method \/\*\/\* since cache cluster is disabled on stage/);
        test.done();
    },
    'if only the custom log destination log group is set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        const testLogGroup = new logs.LogGroup(stack, 'LogGroup');
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            accessLogDestination: new apigateway.LogGroupLogDestination(testLogGroup),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            AccessLogSetting: {
                DestinationArn: {
                    'Fn::GetAtt': [
                        'LogGroupF5B46931',
                        'Arn',
                    ],
                },
                Format: '$context.identity.sourceIp $context.identity.caller $context.identity.user [$context.requestTime] "$context.httpMethod $context.resourcePath $context.protocol" $context.status $context.responseLength $context.requestId',
            },
            StageName: 'prod',
        }));
        test.done();
    },
    'if the custom log destination log group and format is set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        const testLogGroup = new logs.LogGroup(stack, 'LogGroup');
        const testFormat = apigateway.AccessLogFormat.jsonWithStandardFields();
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            accessLogDestination: new apigateway.LogGroupLogDestination(testLogGroup),
            accessLogFormat: testFormat,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            AccessLogSetting: {
                DestinationArn: {
                    'Fn::GetAtt': [
                        'LogGroupF5B46931',
                        'Arn',
                    ],
                },
                Format: '{"requestId":"$context.requestId","ip":"$context.identity.sourceIp","user":"$context.identity.user","caller":"$context.identity.caller","requestTime":"$context.requestTime","httpMethod":"$context.httpMethod","resourcePath":"$context.resourcePath","status":"$context.status","protocol":"$context.protocol","responseLength":"$context.responseLength"}',
            },
            StageName: 'prod',
        }));
        test.done();
    },
    'fails when access log format does not contain `AccessLogFormat.contextRequestId()`'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        const testLogGroup = new logs.LogGroup(stack, 'LogGroup');
        const testFormat = apigateway.AccessLogFormat.custom('');
        // THEN
        test.throws(() => new apigateway.Stage(stack, 'my-stage', {
            deployment,
            accessLogDestination: new apigateway.LogGroupLogDestination(testLogGroup),
            accessLogFormat: testFormat,
        }), /Access log must include at least `AccessLogFormat.contextRequestId\(\)`/);
        test.done();
    },
    'fails when access log destination is empty'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        const testFormat = apigateway.AccessLogFormat.jsonWithStandardFields();
        // THEN
        test.throws(() => new apigateway.Stage(stack, 'my-stage', {
            deployment,
            accessLogFormat: testFormat,
        }), /Access log format is specified without a destination/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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