"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
// import { Construct } from 'constructs';
const codebuild = require("@aws-cdk/aws-codebuild");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
// eslint-disable-next-line import/no-extraneous-dependencies
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const statement = require("cdk-iam-floyd");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket.
 *
 * This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 *
 * @stability stable
 */
class ProwlerAudit extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(parent, id, props) {
        var _b, _c;
        super(parent, id);
        // defaults
        this.serviceName = (props === null || props === void 0 ? void 0 : props.serviceName) ? props.serviceName : 'prowler';
        this.logsRetentionInDays = (props === null || props === void 0 ? void 0 : props.logsRetentionInDays) ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = (props === null || props === void 0 ? void 0 : props.enableScheduler) ? props.enableScheduler : false;
        this.prowlerScheduler = (props === null || props === void 0 ? void 0 : props.prowlerScheduler) ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = (props === null || props === void 0 ? void 0 : props.prowlerOptions) ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        const reportBucket = new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const rerunProwler = Boolean(this.node.tryGetContext('reRunProwler'));
        if (rerunProwler) {
            core_1.Tags.of(this).add('reRunProwler', Date.now().toString());
        }
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: core_1.RemovalPolicy.DESTROY });
        reportGroup;
        const prowlerBuild = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: core_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            'git clone https://github.com/toniblyx/prowler',
                        ],
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/ --recursive',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        (_b = prowlerBuild.role) === null || _b === void 0 ? void 0 : _b.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        (_c = prowlerBuild.role) === null || _c === void 0 ? void 0 : _c.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'Lambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: core_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new core_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: { Build: prowlerBuild.projectName },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: core_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Rule', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "1.117.8" };
//# sourceMappingURL=data:application/json;base64,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