"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroupRule = void 0;
/**
 * A single security group rule, either egress or ingress
 */
class SecurityGroupRule {
    constructor(ruleObject, groupRef) {
        this.ipProtocol = ruleObject.IpProtocol || '*unknown*';
        this.fromPort = ruleObject.FromPort;
        this.toPort = ruleObject.ToPort;
        this.groupId = ruleObject.GroupId || groupRef || '*unknown*'; // In case of an inline rule
        this.peer =
            findFirst(ruleObject, ['CidrIp', 'CidrIpv6'], (ip) => ({ kind: 'cidr-ip', ip }))
                ||
                    findFirst(ruleObject, ['DestinationSecurityGroupId', 'SourceSecurityGroupId'], (securityGroupId) => ({ kind: 'security-group', securityGroupId }))
                ||
                    findFirst(ruleObject, ['DestinationPrefixListId', 'SourcePrefixListId'], (prefixListId) => ({ kind: 'prefix-list', prefixListId }));
    }
    equal(other) {
        return this.ipProtocol === other.ipProtocol
            && this.fromPort === other.fromPort
            && this.toPort === other.toPort
            && peerEqual(this.peer, other.peer);
    }
    describeProtocol() {
        if (this.ipProtocol === '-1') {
            return 'Everything';
        }
        const ipProtocol = this.ipProtocol.toUpperCase();
        if (this.fromPort === -1) {
            return `All ${ipProtocol}`;
        }
        if (this.fromPort === this.toPort) {
            return `${ipProtocol} ${this.fromPort}`;
        }
        return `${ipProtocol} ${this.fromPort}-${this.toPort}`;
    }
    describePeer() {
        if (this.peer) {
            switch (this.peer.kind) {
                case 'cidr-ip':
                    if (this.peer.ip === '0.0.0.0/0') {
                        return 'Everyone (IPv4)';
                    }
                    if (this.peer.ip === '::/0') {
                        return 'Everyone (IPv6)';
                    }
                    return `${this.peer.ip}`;
                case 'prefix-list': return `${this.peer.prefixListId}`;
                case 'security-group': return `${this.peer.securityGroupId}`;
            }
        }
        return '?';
    }
    toJson() {
        return {
            groupId: this.groupId,
            ipProtocol: this.ipProtocol,
            fromPort: this.fromPort,
            toPort: this.toPort,
            peer: this.peer,
        };
    }
}
exports.SecurityGroupRule = SecurityGroupRule;
function peerEqual(a, b) {
    if ((a === undefined) !== (b === undefined)) {
        return false;
    }
    if (a === undefined) {
        return true;
    }
    if (a.kind !== b.kind) {
        return false;
    }
    switch (a.kind) {
        case 'cidr-ip': return a.ip === b.ip;
        case 'security-group': return a.securityGroupId === b.securityGroupId;
        case 'prefix-list': return a.prefixListId === b.prefixListId;
    }
}
function findFirst(obj, keys, fn) {
    for (const key of keys) {
        if (key in obj) {
            return fn(obj[key]);
        }
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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