/**
 * Bundling options.
 *
 * @stability stable
 */
export interface BundlingOptions {
    /**
     * The Docker image where the command will run.
     *
     * @stability stable
     */
    readonly image: DockerImage;
    /**
     * The entrypoint to run in the Docker container.
     *
     * @default - run the entrypoint defined in the image
     * @see https://docs.docker.com/engine/reference/builder/#entrypoint
     * @stability stable
     * @example
     *
     * ['/bin/sh', '-c']
     */
    readonly entrypoint?: string[];
    /**
     * The command to run in the Docker container.
     *
     * @default - run the command defined in the image
     * @see https://docs.docker.com/engine/reference/run/
     * @stability stable
     * @example
     *
     * ['npm', 'install']
     */
    readonly command?: string[];
    /**
     * Additional Docker volumes to mount.
     *
     * @default - no additional volumes are mounted
     * @stability stable
     */
    readonly volumes?: DockerVolume[];
    /**
     * The environment variables to pass to the Docker container.
     *
     * @default - no environment variables.
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * Working directory inside the Docker container.
     *
     * @default /asset-input
     * @stability stable
     */
    readonly workingDirectory?: string;
    /**
     * The user to use when running the Docker container.
     *
     * user | user:group | uid | uid:gid | user:gid | uid:group
     *
     * @default - uid:gid of the current user or 1000:1000 on Windows
     * @see https://docs.docker.com/engine/reference/run/#user
     * @stability stable
     */
    readonly user?: string;
    /**
     * Local bundling provider.
     *
     * The provider implements a method `tryBundle()` which should return `true`
     * if local bundling was performed. If `false` is returned, docker bundling
     * will be done.
     *
     * @default - bundling will only be performed in a Docker container
     * @stability stable
     */
    readonly local?: ILocalBundling;
    /**
     * The type of output that this bundling operation is producing.
     *
     * @default BundlingOutput.AUTO_DISCOVER
     * @stability stable
     */
    readonly outputType?: BundlingOutput;
    /**
     * [Security configuration](https://docs.docker.com/engine/reference/run/#security-configuration) when running the docker container.
     *
     * @default - no security options
     * @stability stable
     */
    readonly securityOpt?: string;
}
/**
 * The type of output that a bundling operation is producing.
 *
 * @stability stable
 */
export declare enum BundlingOutput {
    /**
     * The bundling output directory includes a single .zip or .jar file which will be used as the final bundle. If the output directory does not include exactly a single archive, bundling will fail.
     *
     * @stability stable
     */
    ARCHIVED = "archived",
    /**
     * The bundling output directory contains one or more files which will be archived and uploaded as a .zip file to S3.
     *
     * @stability stable
     */
    NOT_ARCHIVED = "not-archived",
    /**
     * If the bundling output directory contains a single archive file (zip or jar) it will be used as the bundle output as-is.
     *
     * Otherwise all the files in the bundling output directory will be zipped.
     *
     * @stability stable
     */
    AUTO_DISCOVER = "auto-discover"
}
/**
 * Local bundling.
 *
 * @stability stable
 */
export interface ILocalBundling {
    /**
     * This method is called before attempting docker bundling to allow the bundler to be executed locally.
     *
     * If the local bundler exists, and bundling
     * was performed locally, return `true`. Otherwise, return `false`.
     *
     * @param outputDir the directory where the bundled asset should be output.
     * @param options bundling options for this asset.
     * @stability stable
     */
    tryBundle(outputDir: string, options: BundlingOptions): boolean;
}
/**
 * (deprecated) A Docker image used for asset bundling.
 *
 * @deprecated use DockerImage
 */
export declare class BundlingDockerImage {
    readonly image: string;
    private readonly _imageHash?;
    /**
     * (deprecated) Reference an image on DockerHub or another online registry.
     *
     * @param image the image name.
     * @deprecated
     */
    static fromRegistry(image: string): DockerImage;
    /**
     * (deprecated) Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     * @deprecated use DockerImage.fromBuild()
     */
    static fromAsset(path: string, options?: DockerBuildOptions): BundlingDockerImage;
    /**
     * @param image The Docker image.
     * @deprecated
     */
    protected constructor(image: string, _imageHash?: string | undefined);
    /**
     * (deprecated) Provides a stable representation of this image for JSON serialization.
     *
     * @returns The overridden image name if set or image hash name in that order
     * @deprecated
     */
    toJSON(): string;
    /**
     * (deprecated) Runs a Docker image.
     *
     * @deprecated
     */
    run(options?: DockerRunOptions): void;
    /**
     * (deprecated) Copies a file or directory out of the Docker image to the local filesystem.
     *
     * If `outputPath` is omitted the destination path is a temporary directory.
     *
     * @param imagePath the path in the Docker image.
     * @param outputPath the destination path for the copy operation.
     * @returns the destination path
     * @deprecated
     */
    cp(imagePath: string, outputPath?: string): string;
}
/**
 * A Docker image.
 *
 * @stability stable
 */
export declare class DockerImage extends BundlingDockerImage {
    readonly image: string;
    /**
     * Builds a Docker image.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     * @stability stable
     */
    static fromBuild(path: string, options?: DockerBuildOptions): DockerImage;
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name.
     * @stability stable
     */
    static fromRegistry(image: string): DockerImage;
    /**
     * @param image The Docker image.
     * @stability stable
     */
    constructor(image: string, _imageHash?: string);
    /**
     * Provides a stable representation of this image for JSON serialization.
     *
     * @returns The overridden image name if set or image hash name in that order
     * @stability stable
     */
    toJSON(): string;
    /**
     * Runs a Docker image.
     *
     * @stability stable
     */
    run(options?: DockerRunOptions): void;
    /**
     * Copies a file or directory out of the Docker image to the local filesystem.
     *
     * If `outputPath` is omitted the destination path is a temporary directory.
     *
     * @param imagePath the path in the Docker image.
     * @param outputPath the destination path for the copy operation.
     * @returns the destination path
     * @stability stable
     */
    cp(imagePath: string, outputPath?: string): string;
}
/**
 * A Docker volume.
 *
 * @stability stable
 */
export interface DockerVolume {
    /**
     * The path to the file or directory on the host machine.
     *
     * @stability stable
     */
    readonly hostPath: string;
    /**
     * The path where the file or directory is mounted in the container.
     *
     * @stability stable
     */
    readonly containerPath: string;
    /**
     * Mount consistency.
     *
     * Only applicable for macOS
     *
     * @default DockerConsistency.DELEGATED
     * @see https://docs.docker.com/storage/bind-mounts/#configure-mount-consistency-for-macos
     * @stability stable
     */
    readonly consistency?: DockerVolumeConsistency;
}
/**
 * Supported Docker volume consistency types.
 *
 * Only valid on macOS due to the way file storage works on Mac
 *
 * @stability stable
 */
export declare enum DockerVolumeConsistency {
    /**
     * Read/write operations inside the Docker container are applied immediately on the mounted host machine volumes.
     *
     * @stability stable
     */
    CONSISTENT = "consistent",
    /**
     * Read/write operations on mounted Docker volumes are first written inside the container and then synchronized to the host machine.
     *
     * @stability stable
     */
    DELEGATED = "delegated",
    /**
     * Read/write operations on mounted Docker volumes are first applied on the host machine and then synchronized to the container.
     *
     * @stability stable
     */
    CACHED = "cached"
}
/**
 * Docker run options.
 *
 * @stability stable
 */
export interface DockerRunOptions {
    /**
     * The entrypoint to run in the container.
     *
     * @default - run the entrypoint defined in the image
     * @stability stable
     */
    readonly entrypoint?: string[];
    /**
     * The command to run in the container.
     *
     * @default - run the command defined in the image
     * @stability stable
     */
    readonly command?: string[];
    /**
     * Docker volumes to mount.
     *
     * @default - no volumes are mounted
     * @stability stable
     */
    readonly volumes?: DockerVolume[];
    /**
     * The environment variables to pass to the container.
     *
     * @default - no environment variables.
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * Working directory inside the container.
     *
     * @default - image default
     * @stability stable
     */
    readonly workingDirectory?: string;
    /**
     * The user to use when running the container.
     *
     * @default - root or image default
     * @stability stable
     */
    readonly user?: string;
    /**
     * [Security configuration](https://docs.docker.com/engine/reference/run/#security-configuration) when running the docker container.
     *
     * @default - no security options
     * @stability stable
     */
    readonly securityOpt?: string;
}
/**
 * Docker build options.
 *
 * @stability stable
 */
export interface DockerBuildOptions {
    /**
     * Build args.
     *
     * @default - no build args
     * @stability stable
     */
    readonly buildArgs?: {
        [key: string]: string;
    };
    /**
     * Name of the Dockerfile, must relative to the docker build path.
     *
     * @default `Dockerfile`
     * @stability stable
     */
    readonly file?: string;
    /**
     * Set platform if server is multi-platform capable.
     *
     * _Requires Docker Engine API v1.38+_.
     *
     * @default - no platform specified
     * @stability stable
     * @example
     *
     * 'linux/amd64'
     */
    readonly platform?: string;
}
