"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.assertBound = exports.contentHash = exports.addStackArtifactToAssembly = void 0;
const crypto = require("crypto");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
/**
 * Shared logic of writing stack artifact to the Cloud Assembly
 *
 * This logic is shared between StackSyntheses.
 *
 * It could have been a protected method on a base class, but it
 * uses `Partial<cxapi.AwsCloudFormationStackProperties>` in the
 * parameters (which is convenient so I can remain typesafe without
 * copy/pasting), and jsii will choke on this type.
 */
function addStackArtifactToAssembly(session, stack, stackProps, additionalStackDependencies) {
    // nested stack tags are applied at the AWS::CloudFormation::Stack resource
    // level and are not needed in the cloud assembly.
    if (stack.tags.hasTags()) {
        stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.STACK_TAGS, stack.tags.renderTags());
    }
    const deps = [
        ...stack.dependencies.map(s => s.artifactId),
        ...additionalStackDependencies,
    ];
    const meta = collectStackMetadata(stack);
    // backwards compatibility since originally artifact ID was always equal to
    // stack name the stackName attribute is optional and if it is not specified
    // the CLI will use the artifact ID as the stack name. we *could have*
    // always put the stack name here but wanted to minimize the risk around
    // changes to the assembly manifest. so this means that as long as stack
    // name and artifact ID are the same, the cloud assembly manifest will not
    // change.
    const stackNameProperty = stack.stackName === stack.artifactId
        ? {}
        : { stackName: stack.stackName };
    const properties = {
        templateFile: stack.templateFile,
        terminationProtection: stack.terminationProtection,
        tags: nonEmptyDict(stack.tags.tagValues()),
        validateOnSynth: session.validateOnSynth,
        ...stackProps,
        ...stackNameProperty,
    };
    // add an artifact that represents this stack
    session.assembly.addArtifact(stack.artifactId, {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: stack.environment,
        properties,
        dependencies: deps.length > 0 ? deps : undefined,
        metadata: Object.keys(meta).length > 0 ? meta : undefined,
        displayName: stack.node.path,
    });
}
exports.addStackArtifactToAssembly = addStackArtifactToAssembly;
/**
 * Collect the metadata from a stack
 */
function collectStackMetadata(stack) {
    const output = {};
    visit(stack);
    return output;
    function visit(node) {
        // break off if we reached a node that is not a child of this stack
        const parent = findParentStack(node);
        if (parent !== stack) {
            return;
        }
        if (node.node.metadata.length > 0) {
            // Make the path absolute
            output[construct_compat_1.ConstructNode.PATH_SEP + node.node.path] = node.node.metadata.map(md => stack.resolve(md));
        }
        for (const child of node.node.children) {
            visit(child);
        }
    }
    function findParentStack(node) {
        if (node instanceof stack_1.Stack && node.nestedStackParent === undefined) {
            return node;
        }
        if (!node.node.scope) {
            return undefined;
        }
        return findParentStack(node.node.scope);
    }
}
/**
 * Hash a string
 */
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
exports.contentHash = contentHash;
/**
 * Throw an error message about binding() if we don't have a value for x.
 *
 * This replaces the ! assertions we would need everywhere otherwise.
 */
function assertBound(x) {
    if (x === null && x === undefined) {
        throw new Error('You must call bindStack() first');
    }
}
exports.assertBound = assertBound;
function nonEmptyDict(xs) {
    return Object.keys(xs).length > 0 ? xs : undefined;
}
//# sourceMappingURL=data:application/json;base64,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