#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("source-map-support/register");
const cxapi = require("@aws-cdk/cx-api");
const colors = require("colors/safe");
const yargs = require("yargs");
const aws_auth_1 = require("../lib/api/aws-auth");
const bootstrap_1 = require("../lib/api/bootstrap");
const cloudformation_deployments_1 = require("../lib/api/cloudformation-deployments");
const cloud_executable_1 = require("../lib/api/cxapp/cloud-executable");
const exec_1 = require("../lib/api/cxapp/exec");
const toolkit_info_1 = require("../lib/api/toolkit-info");
const stack_activity_monitor_1 = require("../lib/api/util/cloudformation/stack-activity-monitor");
const cdk_toolkit_1 = require("../lib/cdk-toolkit");
const diff_1 = require("../lib/diff");
const init_1 = require("../lib/init");
const logging_1 = require("../lib/logging");
const plugin_1 = require("../lib/plugin");
const serialize_1 = require("../lib/serialize");
const settings_1 = require("../lib/settings");
const version = require("../lib/version");
/* eslint-disable max-len */
/* eslint-disable @typescript-eslint/no-shadow */ // yargs
async function parseCommandLineArguments() {
    // Use the following configuration for array arguments:
    //
    //     { type: 'array', default: [], nargs: 1, requiresArg: true }
    //
    // The default behavior of yargs is to eat all strings following an array argument:
    //
    //   ./prog --arg one two positional  => will parse to { arg: ['one', 'two', 'positional'], _: [] } (so no positional arguments)
    //   ./prog --arg one two -- positional  => does not help, for reasons that I can't understand. Still gets parsed incorrectly.
    //
    // By using the config above, every --arg will only consume one argument, so you can do the following:
    //
    //   ./prog --arg one --arg two position  =>  will parse to  { arg: ['one', 'two'], _: ['positional'] }.
    const initTemplateLanguages = await init_1.availableInitLanguages();
    return yargs
        .env('CDK')
        .usage('Usage: cdk -a <cdk-app> COMMAND')
        .option('app', { type: 'string', alias: 'a', desc: 'REQUIRED: command-line for executing your app or a cloud assembly directory (e.g. "node bin/my-app.js")', requiresArg: true })
        .option('context', { type: 'array', alias: 'c', desc: 'Add contextual string parameter (KEY=VALUE)', nargs: 1, requiresArg: true })
        .option('plugin', { type: 'array', alias: 'p', desc: 'Name or path of a node package that extend the CDK features. Can be specified multiple times', nargs: 1 })
        .option('trace', { type: 'boolean', desc: 'Print trace for stack warnings' })
        .option('strict', { type: 'boolean', desc: 'Do not construct stacks with warnings' })
        .option('lookups', { type: 'boolean', desc: 'Perform context lookups (synthesis fails if this is disabled and context lookups need to be performed)', default: true })
        .option('ignore-errors', { type: 'boolean', default: false, desc: 'Ignores synthesis errors, which will likely produce an invalid output' })
        .option('json', { type: 'boolean', alias: 'j', desc: 'Use JSON output instead of YAML when templates are printed to STDOUT', default: false })
        .option('verbose', { type: 'boolean', alias: 'v', desc: 'Show debug logs (specify multiple times to increase verbosity)', default: false })
        .count('verbose')
        .option('debug', { type: 'boolean', desc: 'Enable emission of additional debugging information, such as creation stack traces of tokens', default: false })
        .option('profile', { type: 'string', desc: 'Use the indicated AWS profile as the default environment', requiresArg: true })
        .option('proxy', { type: 'string', desc: 'Use the indicated proxy. Will read from HTTPS_PROXY environment variable if not specified', requiresArg: true })
        .option('ca-bundle-path', { type: 'string', desc: 'Path to CA certificate to use when validating HTTPS requests. Will read from AWS_CA_BUNDLE environment variable if not specified', requiresArg: true })
        .option('ec2creds', { type: 'boolean', alias: 'i', default: undefined, desc: 'Force trying to fetch EC2 instance credentials. Default: guess EC2 instance status' })
        .option('version-reporting', { type: 'boolean', desc: 'Include the "AWS::CDK::Metadata" resource in synthesized templates (enabled by default)', default: undefined })
        .option('path-metadata', { type: 'boolean', desc: 'Include "aws:cdk:path" CloudFormation metadata for each resource (enabled by default)', default: true })
        .option('asset-metadata', { type: 'boolean', desc: 'Include "aws:asset:*" CloudFormation metadata for resources that uses assets (enabled by default)', default: true })
        .option('role-arn', { type: 'string', alias: 'r', desc: 'ARN of Role to use when invoking CloudFormation', default: undefined, requiresArg: true })
        .option('toolkit-stack-name', { type: 'string', desc: 'The name of the CDK toolkit stack', requiresArg: true })
        .option('staging', { type: 'boolean', desc: 'Copy assets to the output directory (use --no-staging to disable, needed for local debugging the source files with SAM CLI)', default: true })
        .option('output', { type: 'string', alias: 'o', desc: 'Emits the synthesized cloud assembly into a directory (default: cdk.out)', requiresArg: true })
        .option('no-color', { type: 'boolean', desc: 'Removes colors and other style from console output', default: false })
        .command(['list [STACKS..]', 'ls [STACKS..]'], 'Lists all stacks in the app', yargs => yargs
        .option('long', { type: 'boolean', default: false, alias: 'l', desc: 'Display environment information for each stack' }))
        .command(['synthesize [STACKS..]', 'synth [STACKS..]'], 'Synthesizes and prints the CloudFormation template for this stack', yargs => yargs
        .option('exclusively', { type: 'boolean', alias: 'e', desc: 'Only synthesize requested stacks, don\'t include dependencies' })
        .option('validation', { type: 'boolean', desc: 'After synthesis, validate stacks with the "validateOnSynth" attribute set (can also be controlled with CDK_VALIDATION)', default: true })
        .option('quiet', { type: 'boolean', alias: 'q', desc: 'Do not output CloudFormation Template to stdout', default: false }))
        .command('bootstrap [ENVIRONMENTS..]', 'Deploys the CDK toolkit stack into an AWS environment', yargs => yargs
        .option('bootstrap-bucket-name', { type: 'string', alias: ['b', 'toolkit-bucket-name'], desc: 'The name of the CDK toolkit bucket; bucket will be created and must not exist', default: undefined })
        .option('bootstrap-kms-key-id', { type: 'string', desc: 'AWS KMS master key ID used for the SSE-KMS encryption', default: undefined, conflicts: 'bootstrap-customer-key' })
        .option('bootstrap-customer-key', { type: 'boolean', desc: 'Create a Customer Master Key (CMK) for the bootstrap bucket (you will be charged but can customize permissions, modern bootstrapping only)', default: undefined, conflicts: 'bootstrap-kms-key-id' })
        .option('qualifier', { type: 'string', desc: 'Unique string to distinguish multiple bootstrap stacks', default: undefined })
        .option('public-access-block-configuration', { type: 'boolean', desc: 'Block public access configuration on CDK toolkit bucket (enabled by default) ', default: undefined })
        .option('tags', { type: 'array', alias: 't', desc: 'Tags to add for the stack (KEY=VALUE)', nargs: 1, requiresArg: true, default: [] })
        .option('execute', { type: 'boolean', desc: 'Whether to execute ChangeSet (--no-execute will NOT execute the ChangeSet)', default: true })
        .option('trust', { type: 'array', desc: 'The AWS account IDs that should be trusted to perform deployments into this environment (may be repeated, modern bootstrapping only)', default: [], nargs: 1, requiresArg: true })
        .option('trust-for-lookup', { type: 'array', desc: 'The AWS account IDs that should be trusted to look up values in this environment (may be repeated, modern bootstrapping only)', default: [], nargs: 1, requiresArg: true })
        .option('cloudformation-execution-policies', { type: 'array', desc: 'The Managed Policy ARNs that should be attached to the role performing deployments into this environment (may be repeated, modern bootstrapping only)', default: [], nargs: 1, requiresArg: true })
        .option('force', { alias: 'f', type: 'boolean', desc: 'Always bootstrap even if it would downgrade template version', default: false })
        .option('termination-protection', { type: 'boolean', default: undefined, desc: 'Toggle CloudFormation termination protection on the bootstrap stacks' })
        .option('show-template', { type: 'boolean', desc: 'Instead of actual bootstrapping, print the current CLI\'s bootstrapping template to stdout for customization', default: false })
        .option('template', { type: 'string', requiresArg: true, desc: 'Use the template from the given file instead of the built-in one (use --show-template to obtain an example)' }))
        .command('deploy [STACKS..]', 'Deploys the stack(s) named STACKS into your AWS account', yargs => yargs
        .option('all', { type: 'boolean', default: false, desc: 'Deploy all available stacks' })
        .option('build-exclude', { type: 'array', alias: 'E', nargs: 1, desc: 'Do not rebuild asset with the given ID. Can be specified multiple times', default: [] })
        .option('exclusively', { type: 'boolean', alias: 'e', desc: 'Only deploy requested stacks, don\'t include dependencies' })
        .option('require-approval', { type: 'string', choices: [diff_1.RequireApproval.Never, diff_1.RequireApproval.AnyChange, diff_1.RequireApproval.Broadening], desc: 'What security-sensitive changes need manual approval' })
        .option('ci', { type: 'boolean', desc: 'Force CI detection', default: process.env.CI !== undefined })
        .option('notification-arns', { type: 'array', desc: 'ARNs of SNS topics that CloudFormation will notify with stack related events', nargs: 1, requiresArg: true })
        // @deprecated(v2) -- tags are part of the Cloud Assembly and tags specified here will be overwritten on the next deployment
        .option('tags', { type: 'array', alias: 't', desc: 'Tags to add to the stack (KEY=VALUE), overrides tags from Cloud Assembly (deprecated)', nargs: 1, requiresArg: true })
        .option('execute', { type: 'boolean', desc: 'Whether to execute ChangeSet (--no-execute will NOT execute the ChangeSet)', default: true })
        .option('change-set-name', { type: 'string', desc: 'Name of the CloudFormation change set to create' })
        .option('force', { alias: 'f', type: 'boolean', desc: 'Always deploy stack even if templates are identical', default: false })
        .option('parameters', { type: 'array', desc: 'Additional parameters passed to CloudFormation at deploy time (STACK:KEY=VALUE)', nargs: 1, requiresArg: true, default: {} })
        .option('outputs-file', { type: 'string', alias: 'O', desc: 'Path to file where stack outputs will be written as JSON', requiresArg: true })
        .option('previous-parameters', { type: 'boolean', default: true, desc: 'Use previous values for existing parameters (you must specify all parameters on every deployment if this is disabled)' })
        .option('progress', { type: 'string', choices: [stack_activity_monitor_1.StackActivityProgress.BAR, stack_activity_monitor_1.StackActivityProgress.EVENTS], desc: 'Display mode for stack activity events' }))
        .command('destroy [STACKS..]', 'Destroy the stack(s) named STACKS', yargs => yargs
        .option('all', { type: 'boolean', default: false, desc: 'Destroy all available stacks' })
        .option('exclusively', { type: 'boolean', alias: 'e', desc: 'Only destroy requested stacks, don\'t include dependees' })
        .option('force', { type: 'boolean', alias: 'f', desc: 'Do not ask for confirmation before destroying the stacks' }))
        .command('diff [STACKS..]', 'Compares the specified stack with the deployed stack or a local template file, and returns with status 1 if any difference is found', yargs => yargs
        .option('exclusively', { type: 'boolean', alias: 'e', desc: 'Only diff requested stacks, don\'t include dependencies' })
        .option('context-lines', { type: 'number', desc: 'Number of context lines to include in arbitrary JSON diff rendering', default: 3, requiresArg: true })
        .option('template', { type: 'string', desc: 'The path to the CloudFormation template to compare with', requiresArg: true })
        .option('strict', { type: 'boolean', desc: 'Do not filter out AWS::CDK::Metadata resources', default: false })
        .option('security-only', { type: 'boolean', desc: 'Only diff for broadened security changes', default: false })
        .option('fail', { type: 'boolean', desc: 'Fail with exit code 1 in case of diff', default: false }))
        .command('metadata [STACK]', 'Returns all metadata associated with this stack')
        .command('init [TEMPLATE]', 'Create a new, empty CDK project from a template.', yargs => yargs
        .option('language', { type: 'string', alias: 'l', desc: 'The language to be used for the new project (default can be configured in ~/.cdk.json)', choices: initTemplateLanguages })
        .option('list', { type: 'boolean', desc: 'List the available templates' })
        .option('generate-only', { type: 'boolean', default: false, desc: 'If true, only generates project files, without executing additional operations such as setting up a git repo, installing dependencies or compiling the project' }))
        .commandDir('../lib/commands', { exclude: /^_.*/ })
        .version(version.DISPLAY_VERSION)
        .demandCommand(1, '') // just print help
        .recommendCommands()
        .help()
        .alias('h', 'help')
        .epilogue([
        'If your app has a single stack, there is no need to specify the stack name',
        'If one of cdk.json or ~/.cdk.json exists, options specified there will be used as defaults. Settings in cdk.json take precedence.',
    ].join('\n\n'))
        .argv;
}
if (!process.stdout.isTTY) {
    colors.disable();
}
async function initCommandLine() {
    const argv = await parseCommandLineArguments();
    if (argv.verbose) {
        logging_1.setLogLevel(argv.verbose);
    }
    logging_1.debug('CDK toolkit version:', version.DISPLAY_VERSION);
    logging_1.debug('Command line arguments:', argv);
    const configuration = new settings_1.Configuration({
        commandLineArguments: {
            ...argv,
            _: argv._,
        },
    });
    await configuration.load();
    const sdkProvider = await aws_auth_1.SdkProvider.withAwsCliCompatibleDefaults({
        profile: configuration.settings.get(['profile']),
        ec2creds: argv.ec2creds,
        httpOptions: {
            proxyAddress: argv.proxy,
            caBundlePath: argv['ca-bundle-path'],
        },
    });
    const cloudFormation = new cloudformation_deployments_1.CloudFormationDeployments({ sdkProvider });
    const cloudExecutable = new cloud_executable_1.CloudExecutable({
        configuration,
        sdkProvider,
        synthesizer: exec_1.execProgram,
    });
    /** Function to load plug-ins, using configurations additively. */
    function loadPlugins(...settings) {
        const loaded = new Set();
        for (const source of settings) {
            const plugins = source.get(['plugin']) || [];
            for (const plugin of plugins) {
                const resolved = tryResolve(plugin);
                if (loaded.has(resolved)) {
                    continue;
                }
                logging_1.debug(`Loading plug-in: ${colors.green(plugin)} from ${colors.blue(resolved)}`);
                plugin_1.PluginHost.instance.load(plugin);
                loaded.add(resolved);
            }
        }
        function tryResolve(plugin) {
            try {
                return require.resolve(plugin);
            }
            catch (e) {
                logging_1.error(`Unable to resolve plugin ${colors.green(plugin)}: ${e.stack}`);
                throw new Error(`Unable to resolve plug-in: ${plugin}`);
            }
        }
    }
    loadPlugins(configuration.settings);
    const cmd = argv._[0];
    if (typeof (cmd) !== 'string') {
        throw new Error(`First argument should be a string. Got: ${cmd} (${typeof (cmd)})`);
    }
    // Bundle up global objects so the commands have access to them
    const commandOptions = { args: argv, configuration, aws: sdkProvider };
    try {
        const returnValue = argv.commandHandler
            ? await argv.commandHandler(commandOptions)
            : await main(cmd, argv);
        if (typeof returnValue === 'object') {
            return toJsonOrYaml(returnValue);
        }
        else if (typeof returnValue === 'string') {
            return returnValue;
        }
        else {
            return returnValue;
        }
    }
    finally {
        await version.displayVersionMessage();
    }
    async function main(command, args) {
        const toolkitStackName = toolkit_info_1.ToolkitInfo.determineName(configuration.settings.get(['toolkitStackName']));
        logging_1.debug(`Toolkit stack: ${colors.bold(toolkitStackName)}`);
        if (args.all && args.STACKS) {
            throw new Error('You must either specify a list of Stacks or the `--all` argument');
        }
        args.STACKS = args.STACKS || [];
        args.ENVIRONMENTS = args.ENVIRONMENTS || [];
        const selector = {
            allTopLevel: args.all,
            patterns: args.STACKS,
        };
        const cli = new cdk_toolkit_1.CdkToolkit({
            cloudExecutable,
            cloudFormation,
            verbose: argv.trace || argv.verbose > 0,
            ignoreErrors: argv['ignore-errors'],
            strict: argv.strict,
            configuration,
            sdkProvider,
        });
        switch (command) {
            case 'ls':
            case 'list':
                return cli.list(args.STACKS, { long: args.long });
            case 'diff':
                const enableDiffNoFail = isFeatureEnabled(configuration, cxapi.ENABLE_DIFF_NO_FAIL);
                return cli.diff({
                    stackNames: args.STACKS,
                    exclusively: args.exclusively,
                    templatePath: args.template,
                    strict: args.strict,
                    contextLines: args.contextLines,
                    securityOnly: args.securityOnly,
                    fail: args.fail || !enableDiffNoFail,
                });
            case 'bootstrap':
                const source = determineBootsrapVersion(args, configuration);
                const bootstrapper = new bootstrap_1.Bootstrapper(source);
                if (args.showTemplate) {
                    return bootstrapper.showTemplate();
                }
                return cli.bootstrap(args.ENVIRONMENTS, bootstrapper, {
                    roleArn: args.roleArn,
                    force: argv.force,
                    toolkitStackName: toolkitStackName,
                    execute: args.execute,
                    tags: configuration.settings.get(['tags']),
                    terminationProtection: args.terminationProtection,
                    parameters: {
                        bucketName: configuration.settings.get(['toolkitBucket', 'bucketName']),
                        kmsKeyId: configuration.settings.get(['toolkitBucket', 'kmsKeyId']),
                        createCustomerMasterKey: args.bootstrapCustomerKey,
                        qualifier: args.qualifier,
                        publicAccessBlockConfiguration: args.publicAccessBlockConfiguration,
                        trustedAccounts: arrayFromYargs(args.trust),
                        trustedAccountsForLookup: arrayFromYargs(args.trustForLookup),
                        cloudFormationExecutionPolicies: arrayFromYargs(args.cloudformationExecutionPolicies),
                    },
                });
            case 'deploy':
                const parameterMap = {};
                for (const parameter of args.parameters) {
                    if (typeof parameter === 'string') {
                        const keyValue = parameter.split('=');
                        parameterMap[keyValue[0]] = keyValue.slice(1).join('=');
                    }
                }
                return cli.deploy({
                    selector,
                    exclusively: args.exclusively,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    notificationArns: args.notificationArns,
                    requireApproval: configuration.settings.get(['requireApproval']),
                    reuseAssets: args['build-exclude'],
                    tags: configuration.settings.get(['tags']),
                    execute: args.execute,
                    changeSetName: args.changeSetName,
                    force: args.force,
                    parameters: parameterMap,
                    usePreviousParameters: args['previous-parameters'],
                    outputsFile: configuration.settings.get(['outputsFile']),
                    progress: configuration.settings.get(['progress']),
                    ci: args.ci,
                });
            case 'destroy':
                return cli.destroy({
                    selector,
                    exclusively: args.exclusively,
                    force: args.force,
                    roleArn: args.roleArn,
                });
            case 'synthesize':
            case 'synth':
                if (args.exclusively) {
                    return cli.synth(args.STACKS, args.exclusively, args.quiet, args.validation);
                }
                else {
                    return cli.synth(args.STACKS, true, args.quiet, args.validation);
                }
            case 'metadata':
                return cli.metadata(args.STACK);
            case 'init':
                const language = configuration.settings.get(['language']);
                if (args.list) {
                    return init_1.printAvailableTemplates(language);
                }
                else {
                    return init_1.cliInit(args.TEMPLATE, language, undefined, args.generateOnly);
                }
            case 'version':
                return logging_1.data(version.DISPLAY_VERSION);
            default:
                throw new Error('Unknown command: ' + command);
        }
    }
    function toJsonOrYaml(object) {
        return serialize_1.serializeStructure(object, argv.json);
    }
}
/**
 * Determine which version of bootstrapping
 * (legacy, or "new") should be used.
 */
function determineBootsrapVersion(args, configuration) {
    const isV1 = version.DISPLAY_VERSION.startsWith('1.');
    return isV1 ? determineV1BootstrapSource(args, configuration) : determineV2BootstrapSource(args);
}
function determineV1BootstrapSource(args, configuration) {
    let source;
    if (args.template) {
        logging_1.print(`Using bootstrapping template from ${args.template}`);
        source = { source: 'custom', templateFile: args.template };
    }
    else if (process.env.CDK_NEW_BOOTSTRAP) {
        logging_1.print('CDK_NEW_BOOTSTRAP set, using new-style bootstrapping');
        source = { source: 'default' };
    }
    else if (isFeatureEnabled(configuration, cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT)) {
        logging_1.print(`'${cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT}' context set, using new-style bootstrapping`);
        source = { source: 'default' };
    }
    else {
        // in V1, the "legacy" bootstrapping is the default
        source = { source: 'legacy' };
    }
    return source;
}
function determineV2BootstrapSource(args) {
    let source;
    if (args.template) {
        logging_1.print(`Using bootstrapping template from ${args.template}`);
        source = { source: 'custom', templateFile: args.template };
    }
    else if (process.env.CDK_LEGACY_BOOTSTRAP) {
        logging_1.print('CDK_LEGACY_BOOTSTRAP set, using legacy-style bootstrapping');
        source = { source: 'legacy' };
    }
    else {
        // in V2, the "new" bootstrapping is the default
        source = { source: 'default' };
    }
    return source;
}
function isFeatureEnabled(configuration, featureFlag) {
    var _a;
    return (_a = configuration.context.get(featureFlag)) !== null && _a !== void 0 ? _a : cxapi.futureFlagDefault(featureFlag);
}
/**
 * Translate a Yargs input array to something that makes more sense in a programming language
 * model (telling the difference between absence and an empty array)
 *
 * - An empty array is the default case, meaning the user didn't pass any arguments. We return
 *   undefined.
 * - If the user passed a single empty string, they did something like `--array=`, which we'll
 *   take to mean they passed an empty array.
 */
function arrayFromYargs(xs) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.filter(x => x !== '');
}
initCommandLine()
    .then(value => {
    if (value == null) {
        return;
    }
    if (typeof value === 'string') {
        logging_1.data(value);
    }
    else if (typeof value === 'number') {
        process.exitCode = value;
    }
})
    .catch(err => {
    logging_1.error(err.message);
    if (err.stack) {
        logging_1.debug(err.stack);
    }
    process.exitCode = 1;
});
//# sourceMappingURL=data:application/json;base64,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