"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bootstrapper = void 0;
const console_1 = require("console");
const path = require("path");
const logging_1 = require("../../logging");
const serialize_1 = require("../../serialize");
const deploy_bootstrap_1 = require("./deploy-bootstrap");
const legacy_template_1 = require("./legacy-template");
class Bootstrapper {
    constructor(source) {
        this.source = source;
    }
    bootstrapEnvironment(environment, sdkProvider, options = {}) {
        switch (this.source.source) {
            case 'legacy':
                return this.legacyBootstrap(environment, sdkProvider, options);
            case 'default':
                return this.modernBootstrap(environment, sdkProvider, options);
            case 'custom':
                return this.customBootstrap(environment, sdkProvider, options);
        }
    }
    async showTemplate() {
        const template = await this.loadTemplate();
        process.stdout.write(`${serialize_1.toYAML(template)}\n`);
    }
    /**
     * Deploy legacy bootstrap stack
     *
     */
    async legacyBootstrap(environment, sdkProvider, options = {}) {
        var _a, _b, _c, _d;
        const params = (_a = options.parameters) !== null && _a !== void 0 ? _a : {};
        if ((_b = params.trustedAccounts) === null || _b === void 0 ? void 0 : _b.length) {
            throw new Error('--trust can only be passed for the modern bootstrap experience.');
        }
        if ((_c = params.cloudFormationExecutionPolicies) === null || _c === void 0 ? void 0 : _c.length) {
            throw new Error('--cloudformation-execution-policies can only be passed for the modern bootstrap experience.');
        }
        if (params.createCustomerMasterKey !== undefined) {
            throw new Error('--bootstrap-customer-key can only be passed for the modern bootstrap experience.');
        }
        if (params.qualifier) {
            throw new Error('--qualifier can only be passed for the modern bootstrap experience.');
        }
        const current = await deploy_bootstrap_1.BootstrapStack.lookup(sdkProvider, environment, options.toolkitStackName);
        return current.update(await this.loadTemplate(params), {}, {
            ...options,
            terminationProtection: (_d = options.terminationProtection) !== null && _d !== void 0 ? _d : current.terminationProtection,
        });
    }
    /**
     * Deploy CI/CD-ready bootstrap stack from template
     *
     */
    async modernBootstrap(environment, sdkProvider, options = {}) {
        var _a, _b, _c, _d, _e, _f;
        const params = (_a = options.parameters) !== null && _a !== void 0 ? _a : {};
        const bootstrapTemplate = await this.loadTemplate();
        const current = await deploy_bootstrap_1.BootstrapStack.lookup(sdkProvider, environment, options.toolkitStackName);
        if (params.createCustomerMasterKey !== undefined && params.kmsKeyId) {
            throw new Error('You cannot pass \'--bootstrap-kms-key-id\' and \'--bootstrap-customer-key\' together. Specify one or the other');
        }
        // If people re-bootstrap, existing parameter values are reused so that people don't accidentally change the configuration
        // on their bootstrap stack (this happens automatically in deployStack). However, to do proper validation on the
        // combined arguments (such that if --trust has been given, --cloudformation-execution-policies is necessary as well)
        // we need to take this parameter reuse into account.
        //
        // Ideally we'd do this inside the template, but the `Rules` section of CFN
        // templates doesn't seem to be able to express the conditions that we need
        // (can't use Fn::Join or reference Conditions) so we do it here instead.
        const trustedAccounts = (_b = params.trustedAccounts) !== null && _b !== void 0 ? _b : splitCfnArray(current.parameters.TrustedAccounts);
        console_1.info(`Trusted accounts for deployment: ${trustedAccounts.length > 0 ? trustedAccounts.join(', ') : '(none)'}`);
        const trustedAccountsForLookup = (_c = params.trustedAccountsForLookup) !== null && _c !== void 0 ? _c : splitCfnArray(current.parameters.TrustedAccountsForLookup);
        console_1.info(`Trusted accounts for lookup: ${trustedAccountsForLookup.length > 0 ? trustedAccountsForLookup.join(', ') : '(none)'}`);
        const cloudFormationExecutionPolicies = (_d = params.cloudFormationExecutionPolicies) !== null && _d !== void 0 ? _d : splitCfnArray(current.parameters.CloudFormationExecutionPolicies);
        if (trustedAccounts.length === 0 && cloudFormationExecutionPolicies.length === 0) {
            // For self-trust it's okay to default to AdministratorAccess, and it improves the usability of bootstrapping a lot.
            //
            // We don't actually make the implicity policy a physical parameter. The template will infer it instead,
            // we simply do the UI advertising that behavior here.
            //
            // If we DID make it an explicit parameter, we wouldn't be able to tell the difference between whether
            // we inferred it or whether the user told us, and the sequence:
            //
            // $ cdk bootstrap
            // $ cdk bootstrap --trust 1234
            //
            // Would leave AdministratorAccess policies with a trust relationship, without the user explicitly
            // approving the trust policy.
            const implicitPolicy = `arn:${await current.partition()}:iam::aws:policy/AdministratorAccess`;
            logging_1.warning(`Using default execution policy of '${implicitPolicy}'. Pass '--cloudformation-execution-policies' to customize.`);
        }
        else if (cloudFormationExecutionPolicies.length === 0) {
            throw new Error('Please pass \'--cloudformation-execution-policies\' when using \'--trust\' to specify deployment permissions. Try a managed policy of the form \'arn:aws:iam::aws:policy/<PolicyName>\'.');
        }
        else {
            // Remind people what the current settings are
            console_1.info(`Execution policies: ${cloudFormationExecutionPolicies.join(', ')}`);
        }
        // * If an ARN is given, that ARN. Otherwise:
        //   * '-' if customerKey = false
        //   * '' if customerKey = true
        //   * if customerKey is also not given
        //     * undefined if we already had a value in place (reusing what we had)
        //     * '-' if this is the first time we're deploying this stack (or upgrading from old to new bootstrap)
        const currentKmsKeyId = current.parameters.FileAssetsBucketKmsKeyId;
        const kmsKeyId = (_e = params.kmsKeyId) !== null && _e !== void 0 ? _e : (params.createCustomerMasterKey === true ? CREATE_NEW_KEY :
            params.createCustomerMasterKey === false || currentKmsKeyId === undefined ? USE_AWS_MANAGED_KEY :
                undefined);
        return current.update(bootstrapTemplate, {
            FileAssetsBucketName: params.bucketName,
            FileAssetsBucketKmsKeyId: kmsKeyId,
            // Empty array becomes empty string
            TrustedAccounts: trustedAccounts.join(','),
            TrustedAccountsForLookup: trustedAccountsForLookup.join(','),
            CloudFormationExecutionPolicies: cloudFormationExecutionPolicies.join(','),
            Qualifier: params.qualifier,
            PublicAccessBlockConfiguration: params.publicAccessBlockConfiguration || params.publicAccessBlockConfiguration === undefined ? 'true' : 'false',
        }, {
            ...options,
            terminationProtection: (_f = options.terminationProtection) !== null && _f !== void 0 ? _f : current.terminationProtection,
        });
    }
    async customBootstrap(environment, sdkProvider, options = {}) {
        // Look at the template, decide whether it's most likely a legacy or modern bootstrap
        // template, and use the right bootstrapper for that.
        const version = deploy_bootstrap_1.bootstrapVersionFromTemplate(await this.loadTemplate());
        if (version === 0) {
            return this.legacyBootstrap(environment, sdkProvider, options);
        }
        else {
            return this.modernBootstrap(environment, sdkProvider, options);
        }
    }
    async loadTemplate(params = {}) {
        switch (this.source.source) {
            case 'custom':
                return serialize_1.loadStructuredFile(this.source.templateFile);
            case 'default':
                return serialize_1.loadStructuredFile(path.join(__dirname, 'bootstrap-template.yaml'));
            case 'legacy':
                return legacy_template_1.legacyBootstrapTemplate(params);
        }
    }
}
exports.Bootstrapper = Bootstrapper;
/**
 * Magic parameter value that will cause the bootstrap-template.yml to NOT create a CMK but use the default keyo
 */
const USE_AWS_MANAGED_KEY = 'AWS_MANAGED_KEY';
/**
 * Magic parameter value that will cause the bootstrap-template.yml to create a CMK
 */
const CREATE_NEW_KEY = '';
/**
 * Split an array-like CloudFormation parameter on ,
 *
 * An empty string is the empty array (instead of `['']`).
 */
function splitCfnArray(xs) {
    if (xs === '' || xs === undefined) {
        return [];
    }
    return xs.split(',');
}
//# sourceMappingURL=data:application/json;base64,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