"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.destroyStack = exports.deployStack = void 0;
const cxapi = require("@aws-cdk/cx-api");
const colors = require("colors/safe");
const uuid = require("uuid");
const assets_1 = require("../assets");
const logging_1 = require("../logging");
const serialize_1 = require("../serialize");
const asset_manifest_builder_1 = require("../util/asset-manifest-builder");
const asset_publishing_1 = require("../util/asset-publishing");
const content_hash_1 = require("../util/content-hash");
const cloudformation_1 = require("./util/cloudformation");
const stack_activity_monitor_1 = require("./util/cloudformation/stack-activity-monitor");
// We need to map regions to domain suffixes, and the SDK already has a function to do this.
// It's not part of the public API, but it's also unlikely to go away.
//
// Reuse that function, and add a safety check so we don't accidentally break if they ever
// refactor that away.
/* eslint-disable @typescript-eslint/no-require-imports */
const regionUtil = require('aws-sdk/lib/region_config');
/* eslint-enable @typescript-eslint/no-require-imports */
if (!regionUtil.getEndpointSuffix) {
    throw new Error('This version of AWS SDK for JS does not have the \'getEndpointSuffix\' function!');
}
const LARGE_TEMPLATE_SIZE_KB = 50;
async function deployStack(options) {
    var _a, _b;
    const stackArtifact = options.stack;
    const stackEnv = options.resolvedEnvironment;
    const cfn = options.sdk.cloudFormation();
    const deployName = options.deployName || stackArtifact.stackName;
    let cloudFormationStack = await cloudformation_1.CloudFormationStack.lookup(cfn, deployName);
    if (cloudFormationStack.stackStatus.isCreationFailure) {
        logging_1.debug(`Found existing stack ${deployName} that had previously failed creation. Deleting it before attempting to re-create it.`);
        await cfn.deleteStack({ StackName: deployName }).promise();
        const deletedStack = await cloudformation_1.waitForStackDelete(cfn, deployName);
        if (deletedStack && deletedStack.stackStatus.name !== 'DELETE_COMPLETE') {
            throw new Error(`Failed deleting stack ${deployName} that had previously failed creation (current state: ${deletedStack.stackStatus})`);
        }
        // Update variable to mark that the stack does not exist anymore, but avoid
        // doing an actual lookup in CloudFormation (which would be silly to do if
        // we just deleted it).
        cloudFormationStack = cloudformation_1.CloudFormationStack.doesNotExist(cfn, deployName);
    }
    // Detect "legacy" assets (which remain in the metadata) and publish them via
    // an ad-hoc asset manifest, while passing their locations via template
    // parameters.
    const legacyAssets = new asset_manifest_builder_1.AssetManifestBuilder();
    const assetParams = await assets_1.addMetadataAssetsToManifest(stackArtifact, legacyAssets, options.toolkitInfo, options.reuseAssets);
    const finalParameterValues = { ...options.parameters, ...assetParams };
    const templateParams = cloudformation_1.TemplateParameters.fromTemplate(stackArtifact.template);
    const stackParams = options.usePreviousParameters
        ? templateParams.updateExisting(finalParameterValues, cloudFormationStack.parameters)
        : templateParams.supplyAll(finalParameterValues);
    if (await canSkipDeploy(options, cloudFormationStack, stackParams.hasChanges(cloudFormationStack.parameters))) {
        logging_1.debug(`${deployName}: skipping deployment (use --force to override)`);
        return {
            noOp: true,
            outputs: cloudFormationStack.outputs,
            stackArn: cloudFormationStack.stackId,
            stackArtifact,
        };
    }
    else {
        logging_1.debug(`${deployName}: deploying...`);
    }
    const executionId = uuid.v4();
    const bodyParameter = await makeBodyParameter(stackArtifact, options.resolvedEnvironment, legacyAssets, options.toolkitInfo);
    await asset_publishing_1.publishAssets(legacyAssets.toManifest(stackArtifact.assembly.directory), options.sdkProvider, stackEnv);
    const changeSetName = options.changeSetName || 'cdk-deploy-change-set';
    if (cloudFormationStack.exists) {
        //Delete any existing change sets generated by CDK since change set names must be unique.
        //The delete request is successful as long as the stack exists (even if the change set does not exist).
        logging_1.debug(`Removing existing change set with name ${changeSetName} if it exists`);
        await cfn.deleteChangeSet({ StackName: deployName, ChangeSetName: changeSetName }).promise();
    }
    const update = cloudFormationStack.exists && cloudFormationStack.stackStatus.name !== 'REVIEW_IN_PROGRESS';
    logging_1.debug(`Attempting to create ChangeSet with name ${changeSetName} to ${update ? 'update' : 'create'} stack ${deployName}`);
    logging_1.print('%s: creating CloudFormation changeset...', colors.bold(deployName));
    const changeSet = await cfn.createChangeSet({
        StackName: deployName,
        ChangeSetName: changeSetName,
        ChangeSetType: update ? 'UPDATE' : 'CREATE',
        Description: `CDK Changeset for execution ${executionId}`,
        TemplateBody: bodyParameter.TemplateBody,
        TemplateURL: bodyParameter.TemplateURL,
        Parameters: stackParams.apiParameters,
        RoleARN: options.roleArn,
        NotificationARNs: options.notificationArns,
        Capabilities: ['CAPABILITY_IAM', 'CAPABILITY_NAMED_IAM', 'CAPABILITY_AUTO_EXPAND'],
        Tags: options.tags,
    }).promise();
    logging_1.debug('Initiated creation of changeset: %s; waiting for it to finish creating...', changeSet.Id);
    const changeSetDescription = await cloudformation_1.waitForChangeSet(cfn, deployName, changeSetName);
    // Update termination protection only if it has changed.
    const terminationProtection = (_a = stackArtifact.terminationProtection) !== null && _a !== void 0 ? _a : false;
    if (!!cloudFormationStack.terminationProtection !== terminationProtection) {
        logging_1.debug('Updating termination protection from %s to %s for stack %s', cloudFormationStack.terminationProtection, terminationProtection, deployName);
        await cfn.updateTerminationProtection({
            StackName: deployName,
            EnableTerminationProtection: terminationProtection,
        }).promise();
        logging_1.debug('Termination protection updated to %s for stack %s', terminationProtection, deployName);
    }
    if (cloudformation_1.changeSetHasNoChanges(changeSetDescription)) {
        logging_1.debug('No changes are to be performed on %s.', deployName);
        if (options.execute) {
            logging_1.debug('Deleting empty change set %s', changeSet.Id);
            await cfn.deleteChangeSet({ StackName: deployName, ChangeSetName: changeSetName }).promise();
        }
        return { noOp: true, outputs: cloudFormationStack.outputs, stackArn: changeSet.StackId, stackArtifact };
    }
    const execute = options.execute === undefined ? true : options.execute;
    if (execute) {
        logging_1.debug('Initiating execution of changeset %s on stack %s', changeSet.Id, deployName);
        await cfn.executeChangeSet({ StackName: deployName, ChangeSetName: changeSetName }).promise();
        // eslint-disable-next-line max-len
        const monitor = options.quiet ? undefined : stack_activity_monitor_1.StackActivityMonitor.withDefaultPrinter(cfn, deployName, stackArtifact, {
            resourcesTotal: ((_b = changeSetDescription.Changes) !== null && _b !== void 0 ? _b : []).length,
            progress: options.progress,
            changeSetCreationTime: changeSetDescription.CreationTime,
        }).start();
        logging_1.debug('Execution of changeset %s on stack %s has started; waiting for the update to complete...', changeSet.Id, deployName);
        try {
            const finalStack = await cloudformation_1.waitForStackDeploy(cfn, deployName);
            // This shouldn't really happen, but catch it anyway. You never know.
            if (!finalStack) {
                throw new Error('Stack deploy failed (the stack disappeared while we were deploying it)');
            }
            cloudFormationStack = finalStack;
        }
        finally {
            await (monitor === null || monitor === void 0 ? void 0 : monitor.stop());
        }
        logging_1.debug('Stack %s has completed updating', deployName);
    }
    else {
        logging_1.print('Changeset %s created and waiting in review for manual execution (--no-execute)', changeSet.Id);
    }
    return { noOp: false, outputs: cloudFormationStack.outputs, stackArn: changeSet.StackId, stackArtifact };
}
exports.deployStack = deployStack;
/**
 * Prepares the body parameter for +CreateChangeSet+.
 *
 * If the template is small enough to be inlined into the API call, just return
 * it immediately.
 *
 * Otherwise, add it to the asset manifest to get uploaded to the staging
 * bucket and return its coordinates. If there is no staging bucket, an error
 * is thrown.
 *
 * @param stack     the synthesized stack that provides the CloudFormation template
 * @param toolkitInfo information about the toolkit stack
 */
async function makeBodyParameter(stack, resolvedEnvironment, assetManifest, toolkitInfo) {
    // If the template has already been uploaded to S3, just use it from there.
    if (stack.stackTemplateAssetObjectUrl) {
        return { TemplateURL: restUrlFromManifest(stack.stackTemplateAssetObjectUrl, resolvedEnvironment) };
    }
    // Otherwise, pass via API call (if small) or upload here (if large)
    const templateJson = serialize_1.toYAML(stack.template);
    if (templateJson.length <= LARGE_TEMPLATE_SIZE_KB * 1024) {
        return { TemplateBody: templateJson };
    }
    if (!toolkitInfo.found) {
        logging_1.error(`The template for stack "${stack.displayName}" is ${Math.round(templateJson.length / 1024)}KiB. ` +
            `Templates larger than ${LARGE_TEMPLATE_SIZE_KB}KiB must be uploaded to S3.\n` +
            'Run the following command in order to setup an S3 bucket in this environment, and then re-deploy:\n\n', colors.blue(`\t$ cdk bootstrap ${resolvedEnvironment.name}\n`));
        throw new Error('Template too large to deploy ("cdk bootstrap" is required)');
    }
    const templateHash = content_hash_1.contentHash(templateJson);
    const key = `cdk/${stack.id}/${templateHash}.yml`;
    assetManifest.addFileAsset(templateHash, {
        path: stack.templateFile,
    }, {
        bucketName: toolkitInfo.bucketName,
        objectKey: key,
    });
    const templateURL = `${toolkitInfo.bucketUrl}/${key}`;
    logging_1.debug('Storing template in S3 at:', templateURL);
    return { TemplateURL: templateURL };
}
async function destroyStack(options) {
    const deployName = options.deployName || options.stack.stackName;
    const cfn = options.sdk.cloudFormation();
    const currentStack = await cloudformation_1.CloudFormationStack.lookup(cfn, deployName);
    if (!currentStack.exists) {
        return;
    }
    const monitor = options.quiet ? undefined : stack_activity_monitor_1.StackActivityMonitor.withDefaultPrinter(cfn, deployName, options.stack).start();
    try {
        await cfn.deleteStack({ StackName: deployName, RoleARN: options.roleArn }).promise();
        const destroyedStack = await cloudformation_1.waitForStackDelete(cfn, deployName);
        if (destroyedStack && destroyedStack.stackStatus.name !== 'DELETE_COMPLETE') {
            throw new Error(`Failed to destroy ${deployName}: ${destroyedStack.stackStatus}`);
        }
    }
    finally {
        if (monitor) {
            await monitor.stop();
        }
    }
}
exports.destroyStack = destroyStack;
/**
 * Checks whether we can skip deployment
 *
 * We do this in a complicated way by preprocessing (instead of just
 * looking at the changeset), because if there are nested stacks involved
 * the changeset will always show the nested stacks as needing to be
 * updated, and the deployment will take a long time to in effect not
 * do anything.
 */
async function canSkipDeploy(deployStackOptions, cloudFormationStack, parameterChanges) {
    var _a;
    const deployName = deployStackOptions.deployName || deployStackOptions.stack.stackName;
    logging_1.debug(`${deployName}: checking if we can skip deploy`);
    // Forced deploy
    if (deployStackOptions.force) {
        logging_1.debug(`${deployName}: forced deployment`);
        return false;
    }
    // Creating changeset only (default true), never skip
    if (deployStackOptions.execute === false) {
        logging_1.debug(`${deployName}: --no-execute, always creating change set`);
        return false;
    }
    // No existing stack
    if (!cloudFormationStack.exists) {
        logging_1.debug(`${deployName}: no existing stack`);
        return false;
    }
    // Template has changed (assets taken into account here)
    if (JSON.stringify(deployStackOptions.stack.template) !== JSON.stringify(await cloudFormationStack.template())) {
        logging_1.debug(`${deployName}: template has changed`);
        return false;
    }
    // Tags have changed
    if (!compareTags(cloudFormationStack.tags, (_a = deployStackOptions.tags) !== null && _a !== void 0 ? _a : [])) {
        logging_1.debug(`${deployName}: tags have changed`);
        return false;
    }
    // Termination protection has been updated
    if (!!deployStackOptions.stack.terminationProtection !== !!cloudFormationStack.terminationProtection) {
        logging_1.debug(`${deployName}: termination protection has been updated`);
        return false;
    }
    // Parameters have changed
    if (parameterChanges) {
        logging_1.debug(`${deployName}: parameters have changed`);
        return false;
    }
    // Existing stack is in a failed state
    if (cloudFormationStack.stackStatus.isFailure) {
        logging_1.debug(`${deployName}: stack is in a failure state`);
        return false;
    }
    // We can skip deploy
    return true;
}
/**
 * Compares two list of tags, returns true if identical.
 */
function compareTags(a, b) {
    if (a.length !== b.length) {
        return false;
    }
    for (const aTag of a) {
        const bTag = b.find(tag => tag.Key === aTag.Key);
        if (!bTag || bTag.Value !== aTag.Value) {
            return false;
        }
    }
    return true;
}
/**
 * Format an S3 URL in the manifest for use with CloudFormation
 *
 * Replaces environment placeholders (which this field may contain),
 * and reformats s3://.../... urls into S3 REST URLs (which CloudFormation
 * expects)
 */
function restUrlFromManifest(url, environment) {
    const doNotUseMarker = '**DONOTUSE**';
    // This URL may contain placeholders, so still substitute those.
    url = cxapi.EnvironmentPlaceholders.replace(url, {
        accountId: environment.account,
        region: environment.region,
        partition: doNotUseMarker,
    });
    // Yes, this is extremely crude, but we don't actually need this so I'm not inclined to spend
    // a lot of effort trying to thread the right value to this location.
    if (url.indexOf(doNotUseMarker) > -1) {
        throw new Error('Cannot use \'${AWS::Partition}\' in the \'stackTemplateAssetObjectUrl\' field');
    }
    const s3Url = url.match(/s3:\/\/([^/]+)\/(.*)$/);
    if (!s3Url) {
        return url;
    }
    // We need to pass an 'https://s3.REGION.amazonaws.com[.cn]/bucket/object' URL to CloudFormation, but we
    // got an 's3://bucket/object' URL instead. Construct the rest API URL here.
    const bucketName = s3Url[1];
    const objectKey = s3Url[2];
    const urlSuffix = regionUtil.getEndpointSuffix(environment.region);
    return `https://s3.${environment.region}.${urlSuffix}/${bucketName}/${objectKey}`;
}
//# sourceMappingURL=data:application/json;base64,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