"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CurrentActivityPrinter = exports.HistoryActivityPrinter = exports.StackActivityMonitor = exports.StackActivityProgress = void 0;
const util = require("util");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const colors = require("colors/safe");
const logging_1 = require("../../../logging");
const display_1 = require("../display");
/**
 * Supported display modes for stack deployment activity
 */
var StackActivityProgress;
(function (StackActivityProgress) {
    /**
     * Displays a progress bar with only the events for the resource currently being deployed
     */
    StackActivityProgress["BAR"] = "bar";
    /**
     * Displays complete history with all CloudFormation stack events
     */
    StackActivityProgress["EVENTS"] = "events";
})(StackActivityProgress = exports.StackActivityProgress || (exports.StackActivityProgress = {}));
class StackActivityMonitor {
    constructor(cfn, stackName, printer, stack, changeSetCreationTime) {
        var _a;
        this.cfn = cfn;
        this.stackName = stackName;
        this.printer = printer;
        this.stack = stack;
        this.active = false;
        this.activity = {};
        this.startTime = (_a = changeSetCreationTime === null || changeSetCreationTime === void 0 ? void 0 : changeSetCreationTime.getTime()) !== null && _a !== void 0 ? _a : Date.now();
    }
    /**
     * Create a Stack Activity Monitor using a default printer, based on context clues
     */
    static withDefaultPrinter(cfn, stackName, stackArtifact, options = {}) {
        var _a, _b;
        const stream = process.stderr;
        const props = {
            resourceTypeColumnWidth: calcMaxResourceTypeLength(stackArtifact.template),
            resourcesTotal: options.resourcesTotal,
            stream,
        };
        const isWindows = process.platform === 'win32';
        const verbose = (_a = options.logLevel) !== null && _a !== void 0 ? _a : logging_1.logLevel;
        // On some CI systems (such as CircleCI) output still reports as a TTY so we also
        // need an individual check for whether we're running on CI.
        // see: https://discuss.circleci.com/t/circleci-terminal-is-a-tty-but-term-is-not-set/9965
        const fancyOutputAvailable = !isWindows && stream.isTTY && !options.ci;
        const progress = (_b = options.progress) !== null && _b !== void 0 ? _b : StackActivityProgress.BAR;
        const printer = fancyOutputAvailable && !verbose && (progress === StackActivityProgress.BAR)
            ? new CurrentActivityPrinter(props)
            : new HistoryActivityPrinter(props);
        return new StackActivityMonitor(cfn, stackName, printer, stackArtifact, options.changeSetCreationTime);
    }
    start() {
        this.active = true;
        this.printer.start();
        this.scheduleNextTick();
        return this;
    }
    async stop() {
        this.active = false;
        if (this.tickTimer) {
            clearTimeout(this.tickTimer);
        }
        // Do a final poll for all events. This is to handle the situation where DescribeStackStatus
        // already returned an error, but the monitor hasn't seen all the events yet and we'd end
        // up not printing the failure reason to users.
        await this.finalPollToEnd();
        this.printer.stop();
    }
    scheduleNextTick() {
        if (!this.active) {
            return;
        }
        this.tickTimer = setTimeout(() => void (this.tick()), this.printer.updateSleep);
    }
    async tick() {
        if (!this.active) {
            return;
        }
        try {
            this.readPromise = this.readNewEvents();
            await this.readPromise;
            this.readPromise = undefined;
            // We might have been stop()ped while the network call was in progress.
            if (!this.active) {
                return;
            }
            this.printer.print();
        }
        catch (e) {
            logging_1.error('Error occurred while monitoring stack: %s', e);
        }
        this.scheduleNextTick();
    }
    findMetadataFor(logicalId) {
        var _a, _b;
        const metadata = (_b = (_a = this.stack) === null || _a === void 0 ? void 0 : _a.manifest) === null || _b === void 0 ? void 0 : _b.metadata;
        if (!logicalId || !metadata) {
            return undefined;
        }
        for (const path of Object.keys(metadata)) {
            const entry = metadata[path]
                .filter(e => e.type === cxschema.ArtifactMetadataEntryType.LOGICAL_ID)
                .find(e => e.data === logicalId);
            if (entry) {
                return {
                    entry,
                    constructPath: this.simplifyConstructPath(path),
                };
            }
        }
        return undefined;
    }
    /**
     * Reads all new events from the stack history
     *
     * The events are returned in reverse chronological order; we continue to the next page if we
     * see a next page and the last event in the page is new to us (and within the time window).
     * haven't seen the final event
     */
    async readNewEvents() {
        var _a;
        const events = [];
        try {
            let nextToken;
            let finished = false;
            while (!finished) {
                const response = await this.cfn.describeStackEvents({ StackName: this.stackName, NextToken: nextToken }).promise();
                const eventPage = (_a = response === null || response === void 0 ? void 0 : response.StackEvents) !== null && _a !== void 0 ? _a : [];
                for (const event of eventPage) {
                    // Event from before we were interested in 'em
                    if (event.Timestamp.valueOf() < this.startTime) {
                        finished = true;
                        break;
                    }
                    // Already seen this one
                    if (event.EventId in this.activity) {
                        finished = true;
                        break;
                    }
                    // Fresh event
                    events.push(this.activity[event.EventId] = {
                        event: event,
                        metadata: this.findMetadataFor(event.LogicalResourceId),
                    });
                }
                // We're also done if there's nothing left to read
                nextToken = response === null || response === void 0 ? void 0 : response.NextToken;
                if (nextToken === undefined) {
                    finished = true;
                }
            }
        }
        catch (e) {
            if (e.code === 'ValidationError' && e.message === `Stack [${this.stackName}] does not exist`) {
                return;
            }
            throw e;
        }
        events.reverse();
        for (const event of events) {
            this.printer.addActivity(event);
        }
    }
    /**
     * Perform a final poll to the end and flush out all events to the printer
     *
     * Finish any poll currently in progress, then do a final one until we've
     * reached the last page.
     */
    async finalPollToEnd() {
        // If we were doing a poll, finish that first. It was started before
        // the moment we were sure we weren't going to get any new events anymore
        // so we need to do a new one anyway. Need to wait for this one though
        // because our state is single-threaded.
        if (this.readPromise) {
            await this.readPromise;
        }
        await this.readNewEvents();
    }
    simplifyConstructPath(path) {
        path = path.replace(/\/Resource$/, '');
        path = path.replace(/^\//, ''); // remove "/" prefix
        // remove "<stack-name>/" prefix
        if (path.startsWith(this.stackName + '/')) {
            path = path.substr(this.stackName.length + 1);
        }
        return path;
    }
}
exports.StackActivityMonitor = StackActivityMonitor;
function padRight(n, x) {
    return x + ' '.repeat(Math.max(0, n - x.length));
}
/**
 * Infamous padLeft()
 */
function padLeft(n, x) {
    return ' '.repeat(Math.max(0, n - x.length)) + x;
}
function calcMaxResourceTypeLength(template) {
    const resources = (template && template.Resources) || {};
    let maxWidth = 0;
    for (const id of Object.keys(resources)) {
        const type = resources[id].Type || '';
        if (type.length > maxWidth) {
            maxWidth = type.length;
        }
    }
    return maxWidth;
}
class ActivityPrinterBase {
    constructor(props) {
        this.props = props;
        /**
         * Fetch new activity every 5 seconds
         */
        this.updateSleep = 5000;
        /**
         * A list of resource IDs which are currently being processed
         */
        this.resourcesInProgress = {};
        /**
         * Previous completion state observed by logical ID
         *
         * We use this to detect that if we see a DELETE_COMPLETE after a
         * CREATE_COMPLETE, it's actually a rollback and we should DECREASE
         * resourcesDone instead of increase it
         */
        this.resourcesPrevCompleteState = {};
        /**
         * Count of resources that have reported a _COMPLETE status
         */
        this.resourcesDone = 0;
        /**
         * How many digits we need to represent the total count (for lining up the status reporting)
         */
        this.resourceDigits = 0;
        this.rollingBack = false;
        this.failures = new Array();
        // +1 because the stack also emits a "COMPLETE" event at the end, and that wasn't
        // counted yet. This makes it line up with the amount of events we expect.
        this.resourcesTotal = props.resourcesTotal ? props.resourcesTotal + 1 : undefined;
        // How many digits does this number take to represent?
        this.resourceDigits = this.resourcesTotal ? Math.ceil(Math.log10(this.resourcesTotal)) : 0;
        this.stream = props.stream;
    }
    addActivity(activity) {
        var _a;
        const status = activity.event.ResourceStatus;
        if (!status || !activity.event.LogicalResourceId) {
            return;
        }
        if (status === 'ROLLBACK_IN_PROGRESS' || status === 'UPDATE_ROLLBACK_IN_PROGRESS') {
            // Only triggered on the stack once we've started doing a rollback
            this.rollingBack = true;
        }
        if (status.endsWith('_IN_PROGRESS')) {
            this.resourcesInProgress[activity.event.LogicalResourceId] = activity;
        }
        if (hasErrorMessage(status)) {
            const isCancelled = ((_a = activity.event.ResourceStatusReason) !== null && _a !== void 0 ? _a : '').indexOf('cancelled') > -1;
            // Cancelled is not an interesting failure reason
            if (!isCancelled) {
                this.failures.push(activity);
            }
        }
        if (status.endsWith('_COMPLETE') || status.endsWith('_FAILED')) {
            delete this.resourcesInProgress[activity.event.LogicalResourceId];
        }
        if (status.endsWith('_COMPLETE_CLEANUP_IN_PROGRESS')) {
            this.resourcesDone++;
        }
        if (status.endsWith('_COMPLETE')) {
            const prevState = this.resourcesPrevCompleteState[activity.event.LogicalResourceId];
            if (!prevState) {
                this.resourcesDone++;
            }
            else {
                // If we completed this before and we're completing it AGAIN, means we're rolling back.
                // Protect against silly underflow.
                this.resourcesDone--;
                if (this.resourcesDone < 0) {
                    this.resourcesDone = 0;
                }
            }
            this.resourcesPrevCompleteState[activity.event.LogicalResourceId] = status;
        }
    }
    start() {
        // Empty on purpose
    }
    stop() {
        // Empty on purpose
    }
}
/**
 * Activity Printer which shows a full log of all CloudFormation events
 *
 * When there hasn't been activity for a while, it will print the resources
 * that are currently in progress, to show what's holding up the deployment.
 */
class HistoryActivityPrinter extends ActivityPrinterBase {
    constructor(props) {
        super(props);
        /**
         * Last time we printed something to the console.
         *
         * Used to measure timeout for progress reporting.
         */
        this.lastPrintTime = Date.now();
        /**
         * Number of ms of change absence before we tell the user about the resources that are currently in progress.
         */
        this.inProgressDelay = 30000;
        this.printable = new Array();
    }
    addActivity(activity) {
        super.addActivity(activity);
        this.printable.push(activity);
        this.print();
    }
    print() {
        for (const activity of this.printable) {
            this.printOne(activity);
        }
        this.printable.splice(0, this.printable.length);
        this.printInProgress();
    }
    stop() {
        // Print failures at the end
        if (this.failures.length > 0) {
            this.stream.write('\nFailed resources:\n');
            for (const failure of this.failures) {
                // Root stack failures are not interesting
                if (failure.event.StackName === failure.event.LogicalResourceId) {
                    continue;
                }
                this.printOne(failure, false);
            }
        }
    }
    printOne(activity, progress) {
        const e = activity.event;
        const color = colorFromStatusResult(e.ResourceStatus);
        let reasonColor = colors.cyan;
        let stackTrace = '';
        const md = activity.metadata;
        if (md && e.ResourceStatus && e.ResourceStatus.indexOf('FAILED') !== -1) {
            stackTrace = md.entry.trace ? `\n\t${md.entry.trace.join('\n\t\\_ ')}` : '';
            reasonColor = colors.red;
        }
        const resourceName = md ? md.constructPath : (e.LogicalResourceId || '');
        const logicalId = resourceName !== e.LogicalResourceId ? `(${e.LogicalResourceId}) ` : '';
        this.stream.write(util.format(' %s%s | %s | %s | %s %s%s%s\n', (progress !== false ? ` ${this.progress()} |` : ''), new Date(e.Timestamp).toLocaleTimeString(), color(padRight(STATUS_WIDTH, (e.ResourceStatus || '').substr(0, STATUS_WIDTH))), // pad left and trim
        padRight(this.props.resourceTypeColumnWidth, e.ResourceType || ''), color(colors.bold(resourceName)), logicalId, reasonColor(colors.bold(e.ResourceStatusReason ? e.ResourceStatusReason : '')), reasonColor(stackTrace)));
        this.lastPrintTime = Date.now();
    }
    /**
     * Report the current progress as a [34/42] string, or just [34] if the total is unknown
     */
    progress() {
        if (this.resourcesTotal == null) {
            // Don't have total, show simple count and hope the human knows
            return padLeft(3, util.format('%s', this.resourcesDone)); // max 500 resources
        }
        return util.format('%s/%s', padLeft(this.resourceDigits, this.resourcesDone.toString()), padLeft(this.resourceDigits, this.resourcesTotal != null ? this.resourcesTotal.toString() : '?'));
    }
    /**
     * If some resources are taking a while to create, notify the user about what's currently in progress
     */
    printInProgress() {
        if (Date.now() < this.lastPrintTime + this.inProgressDelay) {
            return;
        }
        if (Object.keys(this.resourcesInProgress).length > 0) {
            this.stream.write(util.format('%s Currently in progress: %s\n', this.progress(), colors.bold(Object.keys(this.resourcesInProgress).join(', '))));
        }
        // We cheat a bit here. To prevent printInProgress() from repeatedly triggering,
        // we set the timestamp into the future. It will be reset whenever a regular print
        // occurs, after which we can be triggered again.
        this.lastPrintTime = +Infinity;
    }
}
exports.HistoryActivityPrinter = HistoryActivityPrinter;
/**
 * Activity Printer which shows the resources currently being updated
 *
 * It will continuously reupdate the terminal and show only the resources
 * that are currently being updated, in addition to a progress bar which
 * shows how far along the deployment is.
 *
 * Resources that have failed will always be shown, and will be recapitulated
 * along with their stack trace when the monitoring ends.
 *
 * Resources that failed deployment because they have been cancelled are
 * not included.
 */
class CurrentActivityPrinter extends ActivityPrinterBase {
    constructor(props) {
        super(props);
        /**
         * This looks very disorienting sleeping for 5 seconds. Update quicker.
         */
        this.updateSleep = 2000;
        this.oldLogLevel = 0 /* DEFAULT */;
        this.block = new display_1.RewritableBlock(this.stream);
    }
    print() {
        var _a;
        const lines = [];
        // Add a progress bar at the top
        const progressWidth = Math.max(Math.min(((_a = this.block.width) !== null && _a !== void 0 ? _a : 80) - PROGRESSBAR_EXTRA_SPACE - 1, MAX_PROGRESSBAR_WIDTH), MIN_PROGRESSBAR_WIDTH);
        const prog = this.progressBar(progressWidth);
        if (prog) {
            lines.push('  ' + prog, '');
        }
        // Normally we'd only print "resources in progress", but it's also useful
        // to keep an eye on the failures and know about the specific errors asquickly
        // as possible (while the stack is still rolling back), so add those in.
        const toPrint = [...this.failures, ...Object.values(this.resourcesInProgress)];
        toPrint.sort((a, b) => a.event.Timestamp.getTime() - b.event.Timestamp.getTime());
        lines.push(...toPrint.map(res => {
            var _a, _b, _c;
            const color = colorFromStatusActivity(res.event.ResourceStatus);
            const resourceName = (_c = (_b = (_a = res.metadata) === null || _a === void 0 ? void 0 : _a.constructPath) !== null && _b !== void 0 ? _b : res.event.LogicalResourceId) !== null && _c !== void 0 ? _c : '';
            return util.format('%s | %s | %s | %s%s', padLeft(TIMESTAMP_WIDTH, new Date(res.event.Timestamp).toLocaleTimeString()), color(padRight(STATUS_WIDTH, (res.event.ResourceStatus || '').substr(0, STATUS_WIDTH))), padRight(this.props.resourceTypeColumnWidth, res.event.ResourceType || ''), color(colors.bold(shorten(40, resourceName))), this.failureReasonOnNextLine(res));
        }));
        this.block.displayLines(lines);
    }
    start() {
        // Need to prevent the waiter from printing 'stack not stable' every 5 seconds, it messes
        // with the output calculations.
        this.oldLogLevel = logging_1.logLevel;
        logging_1.setLogLevel(0 /* DEFAULT */);
    }
    stop() {
        var _a, _b, _c;
        logging_1.setLogLevel(this.oldLogLevel);
        // Print failures at the end
        const lines = new Array();
        for (const failure of this.failures) {
            // Root stack failures are not interesting
            if (failure.event.StackName === failure.event.LogicalResourceId) {
                continue;
            }
            lines.push(util.format(colors.red('%s | %s | %s | %s%s') + '\n', padLeft(TIMESTAMP_WIDTH, new Date(failure.event.Timestamp).toLocaleTimeString()), padRight(STATUS_WIDTH, (failure.event.ResourceStatus || '').substr(0, STATUS_WIDTH)), padRight(this.props.resourceTypeColumnWidth, failure.event.ResourceType || ''), shorten(40, (_a = failure.event.LogicalResourceId) !== null && _a !== void 0 ? _a : ''), this.failureReasonOnNextLine(failure)));
            const trace = (_c = (_b = failure.metadata) === null || _b === void 0 ? void 0 : _b.entry) === null || _c === void 0 ? void 0 : _c.trace;
            if (trace) {
                lines.push(colors.red(`\t${trace.join('\n\t\\_ ')}\n`));
            }
        }
        // Display in the same block space, otherwise we're going to have silly empty lines.
        this.block.displayLines(lines);
    }
    progressBar(width) {
        if (!this.resourcesTotal) {
            return '';
        }
        const fraction = Math.min(this.resourcesDone / this.resourcesTotal, 1);
        const innerWidth = Math.max(1, width - 2);
        const chars = innerWidth * fraction;
        const remainder = chars - Math.floor(chars);
        const fullChars = FULL_BLOCK.repeat(Math.floor(chars));
        const partialChar = PARTIAL_BLOCK[Math.floor(remainder * PARTIAL_BLOCK.length)];
        const filler = '·'.repeat(innerWidth - Math.floor(chars) - (partialChar ? 1 : 0));
        const color = this.rollingBack ? colors.yellow : colors.green;
        return '[' + color(fullChars + partialChar) + filler + `] (${this.resourcesDone}/${this.resourcesTotal})`;
    }
    failureReasonOnNextLine(activity) {
        var _a, _b;
        return hasErrorMessage((_a = activity.event.ResourceStatus) !== null && _a !== void 0 ? _a : '')
            ? `\n${' '.repeat(TIMESTAMP_WIDTH + STATUS_WIDTH + 6)}${colors.red((_b = activity.event.ResourceStatusReason) !== null && _b !== void 0 ? _b : '')}`
            : '';
    }
}
exports.CurrentActivityPrinter = CurrentActivityPrinter;
const FULL_BLOCK = '█';
const PARTIAL_BLOCK = ['', '▏', '▎', '▍', '▌', '▋', '▊', '▉'];
const MAX_PROGRESSBAR_WIDTH = 60;
const MIN_PROGRESSBAR_WIDTH = 10;
const PROGRESSBAR_EXTRA_SPACE = 2 /* leading spaces */ + 2 /* brackets */ + 4 /* progress number decoration */ + 6 /* 2 progress numbers up to 999 */;
function hasErrorMessage(status) {
    return status.endsWith('_FAILED') || status === 'ROLLBACK_IN_PROGRESS' || status === 'UPDATE_ROLLBACK_IN_PROGRESS';
}
function colorFromStatusResult(status) {
    if (!status) {
        return colors.reset;
    }
    if (status.indexOf('FAILED') !== -1) {
        return colors.red;
    }
    if (status.indexOf('ROLLBACK') !== -1) {
        return colors.yellow;
    }
    if (status.indexOf('COMPLETE') !== -1) {
        return colors.green;
    }
    return colors.reset;
}
function colorFromStatusActivity(status) {
    if (!status) {
        return colors.reset;
    }
    if (status.endsWith('_FAILED')) {
        return colors.red;
    }
    if (status.startsWith('CREATE_') || status.startsWith('UPDATE_')) {
        return colors.green;
    }
    // For stacks, it may also be 'UPDDATE_ROLLBACK_IN_PROGRESS'
    if (status.indexOf('ROLLBACK_') !== -1) {
        return colors.yellow;
    }
    if (status.startsWith('DELETE_')) {
        return colors.yellow;
    }
    return colors.reset;
}
function shorten(maxWidth, p) {
    if (p.length <= maxWidth) {
        return p;
    }
    const half = Math.floor((maxWidth - 3) / 2);
    return p.substr(0, half) + '...' + p.substr(p.length - half);
}
const TIMESTAMP_WIDTH = 12;
const STATUS_WIDTH = 20;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhY2stYWN0aXZpdHktbW9uaXRvci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInN0YWNrLWFjdGl2aXR5LW1vbml0b3IudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLDJEQUEyRDtBQUczRCxzQ0FBc0M7QUFDdEMsOENBQTBFO0FBQzFFLHdDQUE2QztBQVk3Qzs7R0FFRztBQUNILElBQVkscUJBVVg7QUFWRCxXQUFZLHFCQUFxQjtJQUMvQjs7T0FFRztJQUNILG9DQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILDBDQUFpQixDQUFBO0FBQ25CLENBQUMsRUFWVyxxQkFBcUIsR0FBckIsNkJBQXFCLEtBQXJCLDZCQUFxQixRQVVoQztBQXNERCxNQUFhLG9CQUFvQjtJQW1EL0IsWUFDbUIsR0FBdUIsRUFDdkIsU0FBaUIsRUFDakIsT0FBeUIsRUFDekIsS0FBeUMsRUFDMUQscUJBQTRCOztRQUpYLFFBQUcsR0FBSCxHQUFHLENBQW9CO1FBQ3ZCLGNBQVMsR0FBVCxTQUFTLENBQVE7UUFDakIsWUFBTyxHQUFQLE9BQU8sQ0FBa0I7UUFDekIsVUFBSyxHQUFMLEtBQUssQ0FBb0M7UUF0QnBELFdBQU0sR0FBRyxLQUFLLENBQUM7UUFDZixhQUFRLEdBQXlDLEVBQUcsQ0FBQztRQXdCM0QsSUFBSSxDQUFDLFNBQVMsU0FBRyxxQkFBcUIsYUFBckIscUJBQXFCLHVCQUFyQixxQkFBcUIsQ0FBRSxPQUFPLHFDQUFNLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQztJQUNsRSxDQUFDO0lBekREOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUM5QixHQUF1QixFQUN2QixTQUFpQixFQUNqQixhQUFnRCxFQUFFLFVBQW1DLEVBQUU7O1FBQ3ZGLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFFOUIsTUFBTSxLQUFLLEdBQWlCO1lBQzFCLHVCQUF1QixFQUFFLHlCQUF5QixDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUM7WUFDMUUsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjO1lBQ3RDLE1BQU07U0FDUCxDQUFDO1FBRUYsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLFFBQVEsS0FBSyxPQUFPLENBQUM7UUFDL0MsTUFBTSxPQUFPLFNBQUcsT0FBTyxDQUFDLFFBQVEsbUNBQUksa0JBQVEsQ0FBQztRQUM3QyxpRkFBaUY7UUFDakYsNERBQTREO1FBQzVELDBGQUEwRjtRQUMxRixNQUFNLG9CQUFvQixHQUFHLENBQUMsU0FBUyxJQUFJLE1BQU0sQ0FBQyxLQUFLLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1FBQ3ZFLE1BQU0sUUFBUSxTQUFHLE9BQU8sQ0FBQyxRQUFRLG1DQUFJLHFCQUFxQixDQUFDLEdBQUcsQ0FBQztRQUUvRCxNQUFNLE9BQU8sR0FBRyxvQkFBb0IsSUFBSSxDQUFDLE9BQU8sSUFBSSxDQUFDLFFBQVEsS0FBSyxxQkFBcUIsQ0FBQyxHQUFHLENBQUM7WUFDMUYsQ0FBQyxDQUFDLElBQUksc0JBQXNCLENBQUMsS0FBSyxDQUFDO1lBQ25DLENBQUMsQ0FBQyxJQUFJLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXRDLE9BQU8sSUFBSSxvQkFBb0IsQ0FBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxhQUFhLEVBQUUsT0FBTyxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDekcsQ0FBQztJQStCTSxLQUFLO1FBQ1YsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUM7UUFDbkIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUNyQixJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztRQUN4QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFTSxLQUFLLENBQUMsSUFBSTtRQUNmLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDO1FBQ3BCLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNsQixZQUFZLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQzlCO1FBRUQsNEZBQTRGO1FBQzVGLHlGQUF5RjtRQUN6RiwrQ0FBK0M7UUFDL0MsTUFBTSxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7UUFFNUIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUN0QixDQUFDO0lBRU8sZ0JBQWdCO1FBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ2hCLE9BQU87U0FDUjtRQUVELElBQUksQ0FBQyxTQUFTLEdBQUcsVUFBVSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFTyxLQUFLLENBQUMsSUFBSTtRQUNoQixJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNoQixPQUFPO1NBQ1I7UUFFRCxJQUFJO1lBQ0YsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDeEMsTUFBTSxJQUFJLENBQUMsV0FBVyxDQUFDO1lBQ3ZCLElBQUksQ0FBQyxXQUFXLEdBQUcsU0FBUyxDQUFDO1lBRTdCLHVFQUF1RTtZQUN2RSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRTtnQkFBRSxPQUFPO2FBQUU7WUFFN0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsQ0FBQztTQUN0QjtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsZUFBSyxDQUFDLDJDQUEyQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1NBQ3ZEO1FBQ0QsSUFBSSxDQUFDLGdCQUFnQixFQUFFLENBQUM7SUFDMUIsQ0FBQztJQUVPLGVBQWUsQ0FBQyxTQUE2Qjs7UUFDbkQsTUFBTSxRQUFRLGVBQUcsSUFBSSxDQUFDLEtBQUssMENBQUUsUUFBUSwwQ0FBRSxRQUFRLENBQUM7UUFDaEQsSUFBSSxDQUFDLFNBQVMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUFFLE9BQU8sU0FBUyxDQUFDO1NBQUU7UUFDbEQsS0FBSyxNQUFNLElBQUksSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ3hDLE1BQU0sS0FBSyxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUM7aUJBQ3pCLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLEtBQUssUUFBUSxDQUFDLHlCQUF5QixDQUFDLFVBQVUsQ0FBQztpQkFDckUsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxTQUFTLENBQUMsQ0FBQztZQUNuQyxJQUFJLEtBQUssRUFBRTtnQkFDVCxPQUFPO29CQUNMLEtBQUs7b0JBQ0wsYUFBYSxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUM7aUJBQ2hELENBQUM7YUFDSDtTQUNGO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNLLEtBQUssQ0FBQyxhQUFhOztRQUN6QixNQUFNLE1BQU0sR0FBb0IsRUFBRSxDQUFDO1FBRW5DLElBQUk7WUFDRixJQUFJLFNBQTZCLENBQUM7WUFDbEMsSUFBSSxRQUFRLEdBQUcsS0FBSyxDQUFDO1lBQ3JCLE9BQU8sQ0FBQyxRQUFRLEVBQUU7Z0JBQ2hCLE1BQU0sUUFBUSxHQUFHLE1BQU0sSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO2dCQUNuSCxNQUFNLFNBQVMsU0FBRyxRQUFRLGFBQVIsUUFBUSx1QkFBUixRQUFRLENBQUUsV0FBVyxtQ0FBSSxFQUFFLENBQUM7Z0JBRTlDLEtBQUssTUFBTSxLQUFLLElBQUksU0FBUyxFQUFFO29CQUM3Qiw4Q0FBOEM7b0JBQzlDLElBQUksS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsR0FBRyxJQUFJLENBQUMsU0FBUyxFQUFFO3dCQUM5QyxRQUFRLEdBQUcsSUFBSSxDQUFDO3dCQUNoQixNQUFNO3FCQUNQO29CQUVELHdCQUF3QjtvQkFDeEIsSUFBSSxLQUFLLENBQUMsT0FBTyxJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUU7d0JBQ2xDLFFBQVEsR0FBRyxJQUFJLENBQUM7d0JBQ2hCLE1BQU07cUJBQ1A7b0JBRUQsY0FBYztvQkFDZCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHO3dCQUN6QyxLQUFLLEVBQUUsS0FBSzt3QkFDWixRQUFRLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUM7cUJBQ3hELENBQUMsQ0FBQztpQkFDSjtnQkFFRCxrREFBa0Q7Z0JBQ2xELFNBQVMsR0FBRyxRQUFRLGFBQVIsUUFBUSx1QkFBUixRQUFRLENBQUUsU0FBUyxDQUFDO2dCQUNoQyxJQUFJLFNBQVMsS0FBSyxTQUFTLEVBQUU7b0JBQzNCLFFBQVEsR0FBRyxJQUFJLENBQUM7aUJBQ2pCO2FBQ0Y7U0FDRjtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsSUFBSSxDQUFDLENBQUMsSUFBSSxLQUFLLGlCQUFpQixJQUFJLENBQUMsQ0FBQyxPQUFPLEtBQUssVUFBVSxJQUFJLENBQUMsU0FBUyxrQkFBa0IsRUFBRTtnQkFDNUYsT0FBTzthQUNSO1lBQ0QsTUFBTSxDQUFDLENBQUM7U0FDVDtRQUVELE1BQU0sQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUNqQixLQUFLLE1BQU0sS0FBSyxJQUFJLE1BQU0sRUFBRTtZQUMxQixJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUNqQztJQUNILENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLEtBQUssQ0FBQyxjQUFjO1FBQzFCLG9FQUFvRTtRQUNwRSx5RUFBeUU7UUFDekUsc0VBQXNFO1FBQ3RFLHdDQUF3QztRQUN4QyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDcEIsTUFBTSxJQUFJLENBQUMsV0FBVyxDQUFDO1NBQ3hCO1FBRUQsTUFBTSxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDN0IsQ0FBQztJQUVPLHFCQUFxQixDQUFDLElBQVk7UUFDeEMsSUFBSSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZDLElBQUksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLG9CQUFvQjtRQUVwRCxnQ0FBZ0M7UUFDaEMsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLEdBQUcsR0FBRyxDQUFDLEVBQUU7WUFDekMsSUFBSSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDL0M7UUFDRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Q0FDRjtBQW5ORCxvREFtTkM7QUFFRCxTQUFTLFFBQVEsQ0FBQyxDQUFTLEVBQUUsQ0FBUztJQUNwQyxPQUFPLENBQUMsR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztBQUNuRCxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLE9BQU8sQ0FBQyxDQUFTLEVBQUUsQ0FBUztJQUNuQyxPQUFPLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUNuRCxDQUFDO0FBRUQsU0FBUyx5QkFBeUIsQ0FBQyxRQUFhO0lBQzlDLE1BQU0sU0FBUyxHQUFHLENBQUMsUUFBUSxJQUFJLFFBQVEsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDekQsSUFBSSxRQUFRLEdBQUcsQ0FBQyxDQUFDO0lBQ2pCLEtBQUssTUFBTSxFQUFFLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUN2QyxNQUFNLElBQUksR0FBRyxTQUFTLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxJQUFJLEVBQUUsQ0FBQztRQUN0QyxJQUFJLElBQUksQ0FBQyxNQUFNLEdBQUcsUUFBUSxFQUFFO1lBQzFCLFFBQVEsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO1NBQ3hCO0tBQ0Y7SUFDRCxPQUFPLFFBQVEsQ0FBQztBQUNsQixDQUFDO0FBNEJELE1BQWUsbUJBQW1CO0lBc0NoQyxZQUErQixLQUFtQjtRQUFuQixVQUFLLEdBQUwsS0FBSyxDQUFjO1FBckNsRDs7V0FFRztRQUNhLGdCQUFXLEdBQVcsSUFBSyxDQUFDO1FBRTVDOztXQUVHO1FBQ08sd0JBQW1CLEdBQWtDLEVBQUUsQ0FBQztRQUVsRTs7Ozs7O1dBTUc7UUFDTywrQkFBMEIsR0FBMkIsRUFBRSxDQUFDO1FBRWxFOztXQUVHO1FBQ08sa0JBQWEsR0FBVyxDQUFDLENBQUM7UUFFcEM7O1dBRUc7UUFDZ0IsbUJBQWMsR0FBVyxDQUFDLENBQUM7UUFJcEMsZ0JBQVcsR0FBRyxLQUFLLENBQUM7UUFFWCxhQUFRLEdBQUcsSUFBSSxLQUFLLEVBQWlCLENBQUM7UUFLdkQsaUZBQWlGO1FBQ2pGLDBFQUEwRTtRQUMxRSxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFFbEYsc0RBQXNEO1FBQ3RELElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFM0YsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO0lBQzdCLENBQUM7SUFFTSxXQUFXLENBQUMsUUFBdUI7O1FBQ3hDLE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1FBQzdDLElBQUksQ0FBQyxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQUUsT0FBTztTQUFFO1FBRTdELElBQUksTUFBTSxLQUFLLHNCQUFzQixJQUFJLE1BQU0sS0FBSyw2QkFBNkIsRUFBRTtZQUNqRixrRUFBa0U7WUFDbEUsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUM7U0FDekI7UUFFRCxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLEVBQUU7WUFDbkMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsR0FBRyxRQUFRLENBQUM7U0FDdkU7UUFFRCxJQUFJLGVBQWUsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUMzQixNQUFNLFdBQVcsR0FBRyxPQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsb0JBQW9CLG1DQUFJLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUUxRixpREFBaUQ7WUFDakQsSUFBSSxDQUFDLFdBQVcsRUFBRTtnQkFDaEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7YUFDOUI7U0FDRjtRQUVELElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQzlELE9BQU8sSUFBSSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQztTQUNuRTtRQUVELElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQywrQkFBK0IsQ0FBQyxFQUFFO1lBQ3BELElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztTQUN0QjtRQUVELElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUNoQyxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsMEJBQTBCLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1lBQ3BGLElBQUksQ0FBQyxTQUFTLEVBQUU7Z0JBQ2QsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDO2FBQ3RCO2lCQUFNO2dCQUNMLHVGQUF1RjtnQkFDdkYsbUNBQW1DO2dCQUNuQyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7Z0JBQ3JCLElBQUksSUFBSSxDQUFDLGFBQWEsR0FBRyxDQUFDLEVBQUU7b0JBQzFCLElBQUksQ0FBQyxhQUFhLEdBQUcsQ0FBQyxDQUFDO2lCQUN4QjthQUNGO1lBQ0QsSUFBSSxDQUFDLDBCQUEwQixDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsR0FBRyxNQUFNLENBQUM7U0FDNUU7SUFDSCxDQUFDO0lBSU0sS0FBSztRQUNWLG1CQUFtQjtJQUNyQixDQUFDO0lBRU0sSUFBSTtRQUNULG1CQUFtQjtJQUNyQixDQUFDO0NBQ0Y7QUFFRDs7Ozs7R0FLRztBQUNILE1BQWEsc0JBQXVCLFNBQVEsbUJBQW1CO0lBZTdELFlBQVksS0FBbUI7UUFDN0IsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBZmY7Ozs7V0FJRztRQUNLLGtCQUFhLEdBQUcsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBRW5DOztXQUVHO1FBQ2Msb0JBQWUsR0FBRyxLQUFNLENBQUM7UUFFekIsY0FBUyxHQUFHLElBQUksS0FBSyxFQUFpQixDQUFDO0lBSXhELENBQUM7SUFFTSxXQUFXLENBQUMsUUFBdUI7UUFDeEMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM1QixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM5QixJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDZixDQUFDO0lBRU0sS0FBSztRQUNWLEtBQUssTUFBTSxRQUFRLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNyQyxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3pCO1FBQ0QsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDaEQsSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO0lBQ3pCLENBQUM7SUFFTSxJQUFJO1FBQ1QsNEJBQTRCO1FBQzVCLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzVCLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUM7WUFDM0MsS0FBSyxNQUFNLE9BQU8sSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO2dCQUNuQywwQ0FBMEM7Z0JBQzFDLElBQUksT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTLEtBQUssT0FBTyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRTtvQkFDL0QsU0FBUztpQkFDVjtnQkFFRCxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQzthQUMvQjtTQUNGO0lBQ0gsQ0FBQztJQUVPLFFBQVEsQ0FBQyxRQUF1QixFQUFFLFFBQWtCO1FBQzFELE1BQU0sQ0FBQyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUM7UUFDekIsTUFBTSxLQUFLLEdBQUcscUJBQXFCLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3RELElBQUksV0FBVyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUM7UUFFOUIsSUFBSSxVQUFVLEdBQUcsRUFBRSxDQUFDO1FBQ3BCLE1BQU0sRUFBRSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUM7UUFDN0IsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDLGNBQWMsSUFBSSxDQUFDLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtZQUN2RSxVQUFVLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUM1RSxXQUFXLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQztTQUMxQjtRQUVELE1BQU0sWUFBWSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLElBQUksRUFBRSxDQUFDLENBQUM7UUFFekUsTUFBTSxTQUFTLEdBQUcsWUFBWSxLQUFLLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsaUJBQWlCLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBRTFGLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUNmLElBQUksQ0FBQyxNQUFNLENBQ1QsK0JBQStCLEVBQy9CLENBQUMsUUFBUSxLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQ25ELElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxrQkFBa0IsRUFBRSxFQUMxQyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUMsQ0FBQyxjQUFjLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDLEVBQUUsb0JBQW9CO1FBQ3JHLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDLEVBQ2xFLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLEVBQ2hDLFNBQVMsRUFDVCxXQUFXLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFDOUUsV0FBVyxDQUFDLFVBQVUsQ0FBQyxDQUN4QixDQUNGLENBQUM7UUFFRixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7O09BRUc7SUFDSyxRQUFRO1FBQ2QsSUFBSSxJQUFJLENBQUMsY0FBYyxJQUFJLElBQUksRUFBRTtZQUMvQiwrREFBK0Q7WUFDL0QsT0FBTyxPQUFPLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsb0JBQW9CO1NBQy9FO1FBRUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFDeEIsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxFQUMzRCxPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUN0RyxDQUFDO0lBRUQ7O09BRUc7SUFDSyxlQUFlO1FBQ3JCLElBQUksSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUMxRCxPQUFPO1NBQ1I7UUFFRCxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNwRCxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGdDQUFnQyxFQUM1RCxJQUFJLENBQUMsUUFBUSxFQUFFLEVBQ2YsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNuRTtRQUVELGdGQUFnRjtRQUNoRixrRkFBa0Y7UUFDbEYsaURBQWlEO1FBQ2pELElBQUksQ0FBQyxhQUFhLEdBQUcsQ0FBQyxRQUFRLENBQUM7SUFDakMsQ0FBQztDQUVGO0FBbkhELHdEQW1IQztBQUVEOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILE1BQWEsc0JBQXVCLFNBQVEsbUJBQW1CO0lBUzdELFlBQVksS0FBbUI7UUFDN0IsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBVGY7O1dBRUc7UUFDYSxnQkFBVyxHQUFXLElBQUssQ0FBQztRQUVwQyxnQkFBVyxtQkFBOEI7UUFDekMsVUFBSyxHQUFHLElBQUkseUJBQWUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFJakQsQ0FBQztJQUVNLEtBQUs7O1FBQ1YsTUFBTSxLQUFLLEdBQUcsRUFBRSxDQUFDO1FBRWpCLGdDQUFnQztRQUNoQyxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssbUNBQUksRUFBRSxDQUFDLEdBQUcsdUJBQXVCLEdBQUcsQ0FBQyxFQUFFLHFCQUFxQixDQUFDLEVBQUUscUJBQXFCLENBQUMsQ0FBQztRQUMvSSxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQzdDLElBQUksSUFBSSxFQUFFO1lBQ1IsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1NBQzdCO1FBRUQseUVBQXlFO1FBQ3pFLDhFQUE4RTtRQUM5RSx3RUFBd0U7UUFDeEUsTUFBTSxPQUFPLEdBQW9CLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxFQUFFLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDO1FBQ2hHLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBRWxGLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFOztZQUM5QixNQUFNLEtBQUssR0FBRyx1QkFBdUIsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQ2hFLE1BQU0sWUFBWSxxQkFBRyxHQUFHLENBQUMsUUFBUSwwQ0FBRSxhQUFhLG1DQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLG1DQUFJLEVBQUUsQ0FBQztZQUV0RixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMscUJBQXFCLEVBQ3RDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEVBQzVFLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDLEVBQ3ZGLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLHVCQUF1QixFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxJQUFJLEVBQUUsQ0FBQyxFQUMxRSxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRSxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUMsRUFDN0MsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDdkMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFTSxLQUFLO1FBQ1YseUZBQXlGO1FBQ3pGLGdDQUFnQztRQUNoQyxJQUFJLENBQUMsV0FBVyxHQUFHLGtCQUFRLENBQUM7UUFDNUIscUJBQVcsaUJBQWtCLENBQUM7SUFDaEMsQ0FBQztJQUVNLElBQUk7O1FBQ1QscUJBQVcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFOUIsNEJBQTRCO1FBQzVCLE1BQU0sS0FBSyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFDbEMsS0FBSyxNQUFNLE9BQU8sSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ25DLDBDQUEwQztZQUMxQyxJQUFJLE9BQU8sQ0FBQyxLQUFLLENBQUMsU0FBUyxLQUFLLE9BQU8sQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEVBQUU7Z0JBQy9ELFNBQVM7YUFDVjtZQUVELEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLEdBQUcsSUFBSSxFQUM3RCxPQUFPLENBQUMsZUFBZSxFQUFFLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsa0JBQWtCLEVBQUUsQ0FBQyxFQUNoRixRQUFRLENBQUMsWUFBWSxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxjQUFjLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQyxFQUNwRixRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsRUFBRSxPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksSUFBSSxFQUFFLENBQUMsRUFDOUUsT0FBTyxDQUFDLEVBQUUsUUFBRSxPQUFPLENBQUMsS0FBSyxDQUFDLGlCQUFpQixtQ0FBSSxFQUFFLENBQUMsRUFDbEQsSUFBSSxDQUFDLHVCQUF1QixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUUxQyxNQUFNLEtBQUssZUFBRyxPQUFPLENBQUMsUUFBUSwwQ0FBRSxLQUFLLDBDQUFFLEtBQUssQ0FBQztZQUM3QyxJQUFJLEtBQUssRUFBRTtnQkFDVCxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsS0FBSyxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO2FBQ3pEO1NBQ0Y7UUFFRCxvRkFBb0Y7UUFDcEYsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVPLFdBQVcsQ0FBQyxLQUFhO1FBQy9CLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQUUsT0FBTyxFQUFFLENBQUM7U0FBRTtRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN2RSxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxLQUFLLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDMUMsTUFBTSxLQUFLLEdBQUcsVUFBVSxHQUFHLFFBQVEsQ0FBQztRQUNwQyxNQUFNLFNBQVMsR0FBRyxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUU1QyxNQUFNLFNBQVMsR0FBRyxVQUFVLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUN2RCxNQUFNLFdBQVcsR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEdBQUcsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDaEYsTUFBTSxNQUFNLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRWxGLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUM7UUFFOUQsT0FBTyxHQUFHLEdBQUcsS0FBSyxDQUFDLFNBQVMsR0FBRyxXQUFXLENBQUMsR0FBRyxNQUFNLEdBQUcsTUFBTSxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxjQUFjLEdBQUcsQ0FBQztJQUM1RyxDQUFDO0lBRU8sdUJBQXVCLENBQUMsUUFBdUI7O1FBQ3JELE9BQU8sZUFBZSxPQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsY0FBYyxtQ0FBSSxFQUFFLENBQUM7WUFDekQsQ0FBQyxDQUFDLEtBQUssR0FBRyxDQUFDLE1BQU0sQ0FBQyxlQUFlLEdBQUcsWUFBWSxHQUFHLENBQUMsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxHQUFHLE9BQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsbUNBQUksRUFBRSxDQUFDLEVBQUU7WUFDL0csQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNULENBQUM7Q0FDRjtBQXBHRCx3REFvR0M7QUFFRCxNQUFNLFVBQVUsR0FBRyxHQUFHLENBQUM7QUFDdkIsTUFBTSxhQUFhLEdBQUcsQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7QUFDOUQsTUFBTSxxQkFBcUIsR0FBRyxFQUFFLENBQUM7QUFDakMsTUFBTSxxQkFBcUIsR0FBRyxFQUFFLENBQUM7QUFDakMsTUFBTSx1QkFBdUIsR0FBRyxDQUFDLENBQUMsb0JBQW9CLEdBQUcsQ0FBQyxDQUFDLGNBQWMsR0FBRyxDQUFDLENBQUMsZ0NBQWdDLEdBQUcsQ0FBQyxDQUFDLGtDQUFrQyxDQUFDO0FBRXRKLFNBQVMsZUFBZSxDQUFDLE1BQWM7SUFDckMsT0FBTyxNQUFNLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE1BQU0sS0FBSyxzQkFBc0IsSUFBSSxNQUFNLEtBQUssNkJBQTZCLENBQUM7QUFDckgsQ0FBQztBQUVELFNBQVMscUJBQXFCLENBQUMsTUFBZTtJQUM1QyxJQUFJLENBQUMsTUFBTSxFQUFFO1FBQ1gsT0FBTyxNQUFNLENBQUMsS0FBSyxDQUFDO0tBQ3JCO0lBRUQsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1FBQ25DLE9BQU8sTUFBTSxDQUFDLEdBQUcsQ0FBQztLQUNuQjtJQUNELElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtRQUNyQyxPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUM7S0FDdEI7SUFDRCxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUU7UUFDckMsT0FBTyxNQUFNLENBQUMsS0FBSyxDQUFDO0tBQ3JCO0lBRUQsT0FBTyxNQUFNLENBQUMsS0FBSyxDQUFDO0FBQ3RCLENBQUM7QUFFRCxTQUFTLHVCQUF1QixDQUFDLE1BQWU7SUFDOUMsSUFBSSxDQUFDLE1BQU0sRUFBRTtRQUNYLE9BQU8sTUFBTSxDQUFDLEtBQUssQ0FBQztLQUNyQjtJQUVELElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUM5QixPQUFPLE1BQU0sQ0FBQyxHQUFHLENBQUM7S0FDbkI7SUFFRCxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUNoRSxPQUFPLE1BQU0sQ0FBQyxLQUFLLENBQUM7S0FDckI7SUFDRCw0REFBNEQ7SUFDNUQsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1FBQ3RDLE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQztLQUN0QjtJQUNELElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUNoQyxPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUM7S0FDdEI7SUFFRCxPQUFPLE1BQU0sQ0FBQyxLQUFLLENBQUM7QUFDdEIsQ0FBQztBQUVELFNBQVMsT0FBTyxDQUFDLFFBQWdCLEVBQUUsQ0FBUztJQUMxQyxJQUFJLENBQUMsQ0FBQyxNQUFNLElBQUksUUFBUSxFQUFFO1FBQUUsT0FBTyxDQUFDLENBQUM7S0FBRTtJQUN2QyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQzVDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLEdBQUcsS0FBSyxHQUFHLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsQ0FBQztBQUMvRCxDQUFDO0FBRUQsTUFBTSxlQUFlLEdBQUcsRUFBRSxDQUFDO0FBQzNCLE1BQU0sWUFBWSxHQUFHLEVBQUUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHV0aWwgZnJvbSAndXRpbCc7XG5pbXBvcnQgKiBhcyBjeHNjaGVtYSBmcm9tICdAYXdzLWNkay9jbG91ZC1hc3NlbWJseS1zY2hlbWEnO1xuaW1wb3J0ICogYXMgY3hhcGkgZnJvbSAnQGF3cy1jZGsvY3gtYXBpJztcbmltcG9ydCAqIGFzIGF3cyBmcm9tICdhd3Mtc2RrJztcbmltcG9ydCAqIGFzIGNvbG9ycyBmcm9tICdjb2xvcnMvc2FmZSc7XG5pbXBvcnQgeyBlcnJvciwgbG9nTGV2ZWwsIExvZ0xldmVsLCBzZXRMb2dMZXZlbCB9IGZyb20gJy4uLy4uLy4uL2xvZ2dpbmcnO1xuaW1wb3J0IHsgUmV3cml0YWJsZUJsb2NrIH0gZnJvbSAnLi4vZGlzcGxheSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgU3RhY2tBY3Rpdml0eSB7XG4gIHJlYWRvbmx5IGV2ZW50OiBhd3MuQ2xvdWRGb3JtYXRpb24uU3RhY2tFdmVudDtcbiAgcmVhZG9ubHkgbWV0YWRhdGE/OiBSZXNvdXJjZU1ldGFkYXRhO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlc291cmNlTWV0YWRhdGEge1xuICBlbnRyeTogY3hzY2hlbWEuTWV0YWRhdGFFbnRyeTtcbiAgY29uc3RydWN0UGF0aDogc3RyaW5nO1xufVxuXG4vKipcbiAqIFN1cHBvcnRlZCBkaXNwbGF5IG1vZGVzIGZvciBzdGFjayBkZXBsb3ltZW50IGFjdGl2aXR5XG4gKi9cbmV4cG9ydCBlbnVtIFN0YWNrQWN0aXZpdHlQcm9ncmVzcyB7XG4gIC8qKlxuICAgKiBEaXNwbGF5cyBhIHByb2dyZXNzIGJhciB3aXRoIG9ubHkgdGhlIGV2ZW50cyBmb3IgdGhlIHJlc291cmNlIGN1cnJlbnRseSBiZWluZyBkZXBsb3llZFxuICAgKi9cbiAgQkFSID0gJ2JhcicsXG5cbiAgLyoqXG4gICAqIERpc3BsYXlzIGNvbXBsZXRlIGhpc3Rvcnkgd2l0aCBhbGwgQ2xvdWRGb3JtYXRpb24gc3RhY2sgZXZlbnRzXG4gICAqL1xuICBFVkVOVFMgPSAnZXZlbnRzJyxcbn1cblxuZXhwb3J0IGludGVyZmFjZSBXaXRoRGVmYXVsdFByaW50ZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBUb3RhbCBudW1iZXIgb2YgcmVzb3VyY2VzIHRvIHVwZGF0ZVxuICAgKlxuICAgKiBVc2VkIHRvIGNhbGN1bGF0ZSBhIHByb2dyZXNzIGJhci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwcm9ncmVzcyByZXBvcnRpbmcuXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZXNUb3RhbD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGxvZyBsZXZlbCB0aGF0IHdhcyByZXF1ZXN0ZWQgaW4gdGhlIENMSVxuICAgKlxuICAgKiBJZiB2ZXJib3NlIG9yIHRyYWNlIGlzIHJlcXVlc3RlZCwgd2UnbGwgYWx3YXlzIHVzZSB0aGUgZnVsbCBoaXN0b3J5IHByaW50ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVXNlIHZhbHVlIGZyb20gbG9nZ2luZy5sb2dMZXZlbFxuICAgKi9cbiAgcmVhZG9ubHkgbG9nTGV2ZWw/OiBMb2dMZXZlbDtcblxuICAvKipcbiAgICogV2hldGhlciB0byBkaXNwbGF5IGFsbCBzdGFjayBldmVudHMgb3IgdG8gZGlzcGxheSBvbmx5IHRoZSBldmVudHMgZm9yIHRoZVxuICAgKiByZXNvdXJjZSBjdXJyZW50bHkgYmVpbmcgZGVwbG95ZWRcbiAgICpcbiAgICogSWYgbm90IHNldCwgdGhlIHN0YWNrIGhpc3Rvcnkgd2l0aCBhbGwgc3RhY2sgZXZlbnRzIHdpbGwgYmUgZGlzcGxheWVkXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICBwcm9ncmVzcz86IFN0YWNrQWN0aXZpdHlQcm9ncmVzcztcblxuICAvKipcbiAgICogV2hldGhlciB3ZSBhcmUgb24gYSBDSSBzeXN0ZW1cbiAgICpcbiAgICogSWYgc28sIGRpc2FibGUgdGhlIFwib3B0aW1pemVkXCIgc3RhY2sgbW9uaXRvci5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNpPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ3JlYXRpb24gdGltZSBvZiB0aGUgY2hhbmdlIHNldFxuICAgKlxuICAgKiBUaGlzIHdpbGwgYmUgdXNlZCB0byBmaWx0ZXIgZXZlbnRzLCBvbmx5IHNob3dpbmcgdGhvc2UgZnJvbSBhZnRlciB0aGUgY2hhbmdlXG4gICAqIHNldCBjcmVhdGlvbiB0aW1lLlxuICAgKlxuICAgKiBJdCBpcyByZWNvbW1lbmRlZCB0byB1c2UgdGhpcywgb3RoZXJ3aXNlIHRoZSBmaWx0ZXJpbmcgd2lsbCBiZSBzdWJqZWN0XG4gICAqIHRvIGNsb2NrIGRyaWZ0IGJldHdlZW4gbG9jYWwgYW5kIGNsb3VkIG1hY2hpbmVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGxvY2FsIG1hY2hpbmUncyBjdXJyZW50IHRpbWVcbiAgICovXG4gIHJlYWRvbmx5IGNoYW5nZVNldENyZWF0aW9uVGltZT86IERhdGU7XG59XG5cbmV4cG9ydCBjbGFzcyBTdGFja0FjdGl2aXR5TW9uaXRvciB7XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIFN0YWNrIEFjdGl2aXR5IE1vbml0b3IgdXNpbmcgYSBkZWZhdWx0IHByaW50ZXIsIGJhc2VkIG9uIGNvbnRleHQgY2x1ZXNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgd2l0aERlZmF1bHRQcmludGVyKFxuICAgIGNmbjogYXdzLkNsb3VkRm9ybWF0aW9uLFxuICAgIHN0YWNrTmFtZTogc3RyaW5nLFxuICAgIHN0YWNrQXJ0aWZhY3Q6IGN4YXBpLkNsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdCwgb3B0aW9uczogV2l0aERlZmF1bHRQcmludGVyUHJvcHMgPSB7fSkge1xuICAgIGNvbnN0IHN0cmVhbSA9IHByb2Nlc3Muc3RkZXJyO1xuXG4gICAgY29uc3QgcHJvcHM6IFByaW50ZXJQcm9wcyA9IHtcbiAgICAgIHJlc291cmNlVHlwZUNvbHVtbldpZHRoOiBjYWxjTWF4UmVzb3VyY2VUeXBlTGVuZ3RoKHN0YWNrQXJ0aWZhY3QudGVtcGxhdGUpLFxuICAgICAgcmVzb3VyY2VzVG90YWw6IG9wdGlvbnMucmVzb3VyY2VzVG90YWwsXG4gICAgICBzdHJlYW0sXG4gICAgfTtcblxuICAgIGNvbnN0IGlzV2luZG93cyA9IHByb2Nlc3MucGxhdGZvcm0gPT09ICd3aW4zMic7XG4gICAgY29uc3QgdmVyYm9zZSA9IG9wdGlvbnMubG9nTGV2ZWwgPz8gbG9nTGV2ZWw7XG4gICAgLy8gT24gc29tZSBDSSBzeXN0ZW1zIChzdWNoIGFzIENpcmNsZUNJKSBvdXRwdXQgc3RpbGwgcmVwb3J0cyBhcyBhIFRUWSBzbyB3ZSBhbHNvXG4gICAgLy8gbmVlZCBhbiBpbmRpdmlkdWFsIGNoZWNrIGZvciB3aGV0aGVyIHdlJ3JlIHJ1bm5pbmcgb24gQ0kuXG4gICAgLy8gc2VlOiBodHRwczovL2Rpc2N1c3MuY2lyY2xlY2kuY29tL3QvY2lyY2xlY2ktdGVybWluYWwtaXMtYS10dHktYnV0LXRlcm0taXMtbm90LXNldC85OTY1XG4gICAgY29uc3QgZmFuY3lPdXRwdXRBdmFpbGFibGUgPSAhaXNXaW5kb3dzICYmIHN0cmVhbS5pc1RUWSAmJiAhb3B0aW9ucy5jaTtcbiAgICBjb25zdCBwcm9ncmVzcyA9IG9wdGlvbnMucHJvZ3Jlc3MgPz8gU3RhY2tBY3Rpdml0eVByb2dyZXNzLkJBUjtcblxuICAgIGNvbnN0IHByaW50ZXIgPSBmYW5jeU91dHB1dEF2YWlsYWJsZSAmJiAhdmVyYm9zZSAmJiAocHJvZ3Jlc3MgPT09IFN0YWNrQWN0aXZpdHlQcm9ncmVzcy5CQVIpXG4gICAgICA/IG5ldyBDdXJyZW50QWN0aXZpdHlQcmludGVyKHByb3BzKVxuICAgICAgOiBuZXcgSGlzdG9yeUFjdGl2aXR5UHJpbnRlcihwcm9wcyk7XG5cbiAgICByZXR1cm4gbmV3IFN0YWNrQWN0aXZpdHlNb25pdG9yKGNmbiwgc3RhY2tOYW1lLCBwcmludGVyLCBzdGFja0FydGlmYWN0LCBvcHRpb25zLmNoYW5nZVNldENyZWF0aW9uVGltZSk7XG4gIH1cblxuXG4gIHByaXZhdGUgYWN0aXZlID0gZmFsc2U7XG4gIHByaXZhdGUgYWN0aXZpdHk6IHsgW2V2ZW50SWQ6IHN0cmluZ106IFN0YWNrQWN0aXZpdHkgfSA9IHsgfTtcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyB3aGljaCBldmVudHMgbm90IHRvIGRpc3BsYXlcbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgc3RhcnRUaW1lOiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEN1cnJlbnQgdGljayB0aW1lclxuICAgKi9cbiAgcHJpdmF0ZSB0aWNrVGltZXI/OiBOb2RlSlMuVGltZXI7XG5cbiAgLyoqXG4gICAqIFNldCB0byB0aGUgYWN0aXZpdHkgb2YgcmVhZGluZyB0aGUgY3VycmVudCBldmVudHNcbiAgICovXG4gIHByaXZhdGUgcmVhZFByb21pc2U/OiBQcm9taXNlPGFueT47XG5cbiAgY29uc3RydWN0b3IoXG4gICAgcHJpdmF0ZSByZWFkb25seSBjZm46IGF3cy5DbG91ZEZvcm1hdGlvbixcbiAgICBwcml2YXRlIHJlYWRvbmx5IHN0YWNrTmFtZTogc3RyaW5nLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgcHJpbnRlcjogSUFjdGl2aXR5UHJpbnRlcixcbiAgICBwcml2YXRlIHJlYWRvbmx5IHN0YWNrPzogY3hhcGkuQ2xvdWRGb3JtYXRpb25TdGFja0FydGlmYWN0LFxuICAgIGNoYW5nZVNldENyZWF0aW9uVGltZT86IERhdGUsXG4gICkge1xuICAgIHRoaXMuc3RhcnRUaW1lID0gY2hhbmdlU2V0Q3JlYXRpb25UaW1lPy5nZXRUaW1lKCkgPz8gRGF0ZS5ub3coKTtcbiAgfVxuXG4gIHB1YmxpYyBzdGFydCgpIHtcbiAgICB0aGlzLmFjdGl2ZSA9IHRydWU7XG4gICAgdGhpcy5wcmludGVyLnN0YXJ0KCk7XG4gICAgdGhpcy5zY2hlZHVsZU5leHRUaWNrKCk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwdWJsaWMgYXN5bmMgc3RvcCgpIHtcbiAgICB0aGlzLmFjdGl2ZSA9IGZhbHNlO1xuICAgIGlmICh0aGlzLnRpY2tUaW1lcikge1xuICAgICAgY2xlYXJUaW1lb3V0KHRoaXMudGlja1RpbWVyKTtcbiAgICB9XG5cbiAgICAvLyBEbyBhIGZpbmFsIHBvbGwgZm9yIGFsbCBldmVudHMuIFRoaXMgaXMgdG8gaGFuZGxlIHRoZSBzaXR1YXRpb24gd2hlcmUgRGVzY3JpYmVTdGFja1N0YXR1c1xuICAgIC8vIGFscmVhZHkgcmV0dXJuZWQgYW4gZXJyb3IsIGJ1dCB0aGUgbW9uaXRvciBoYXNuJ3Qgc2VlbiBhbGwgdGhlIGV2ZW50cyB5ZXQgYW5kIHdlJ2QgZW5kXG4gICAgLy8gdXAgbm90IHByaW50aW5nIHRoZSBmYWlsdXJlIHJlYXNvbiB0byB1c2Vycy5cbiAgICBhd2FpdCB0aGlzLmZpbmFsUG9sbFRvRW5kKCk7XG5cbiAgICB0aGlzLnByaW50ZXIuc3RvcCgpO1xuICB9XG5cbiAgcHJpdmF0ZSBzY2hlZHVsZU5leHRUaWNrKCkge1xuICAgIGlmICghdGhpcy5hY3RpdmUpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLnRpY2tUaW1lciA9IHNldFRpbWVvdXQoKCkgPT4gdm9pZCh0aGlzLnRpY2soKSksIHRoaXMucHJpbnRlci51cGRhdGVTbGVlcCk7XG4gIH1cblxuICBwcml2YXRlIGFzeW5jIHRpY2soKSB7XG4gICAgaWYgKCF0aGlzLmFjdGl2ZSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRyeSB7XG4gICAgICB0aGlzLnJlYWRQcm9taXNlID0gdGhpcy5yZWFkTmV3RXZlbnRzKCk7XG4gICAgICBhd2FpdCB0aGlzLnJlYWRQcm9taXNlO1xuICAgICAgdGhpcy5yZWFkUHJvbWlzZSA9IHVuZGVmaW5lZDtcblxuICAgICAgLy8gV2UgbWlnaHQgaGF2ZSBiZWVuIHN0b3AoKXBlZCB3aGlsZSB0aGUgbmV0d29yayBjYWxsIHdhcyBpbiBwcm9ncmVzcy5cbiAgICAgIGlmICghdGhpcy5hY3RpdmUpIHsgcmV0dXJuOyB9XG5cbiAgICAgIHRoaXMucHJpbnRlci5wcmludCgpO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIGVycm9yKCdFcnJvciBvY2N1cnJlZCB3aGlsZSBtb25pdG9yaW5nIHN0YWNrOiAlcycsIGUpO1xuICAgIH1cbiAgICB0aGlzLnNjaGVkdWxlTmV4dFRpY2soKTtcbiAgfVxuXG4gIHByaXZhdGUgZmluZE1ldGFkYXRhRm9yKGxvZ2ljYWxJZDogc3RyaW5nIHwgdW5kZWZpbmVkKTogUmVzb3VyY2VNZXRhZGF0YSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgbWV0YWRhdGEgPSB0aGlzLnN0YWNrPy5tYW5pZmVzdD8ubWV0YWRhdGE7XG4gICAgaWYgKCFsb2dpY2FsSWQgfHwgIW1ldGFkYXRhKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgICBmb3IgKGNvbnN0IHBhdGggb2YgT2JqZWN0LmtleXMobWV0YWRhdGEpKSB7XG4gICAgICBjb25zdCBlbnRyeSA9IG1ldGFkYXRhW3BhdGhdXG4gICAgICAgIC5maWx0ZXIoZSA9PiBlLnR5cGUgPT09IGN4c2NoZW1hLkFydGlmYWN0TWV0YWRhdGFFbnRyeVR5cGUuTE9HSUNBTF9JRClcbiAgICAgICAgLmZpbmQoZSA9PiBlLmRhdGEgPT09IGxvZ2ljYWxJZCk7XG4gICAgICBpZiAoZW50cnkpIHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBlbnRyeSxcbiAgICAgICAgICBjb25zdHJ1Y3RQYXRoOiB0aGlzLnNpbXBsaWZ5Q29uc3RydWN0UGF0aChwYXRoKSxcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWFkcyBhbGwgbmV3IGV2ZW50cyBmcm9tIHRoZSBzdGFjayBoaXN0b3J5XG4gICAqXG4gICAqIFRoZSBldmVudHMgYXJlIHJldHVybmVkIGluIHJldmVyc2UgY2hyb25vbG9naWNhbCBvcmRlcjsgd2UgY29udGludWUgdG8gdGhlIG5leHQgcGFnZSBpZiB3ZVxuICAgKiBzZWUgYSBuZXh0IHBhZ2UgYW5kIHRoZSBsYXN0IGV2ZW50IGluIHRoZSBwYWdlIGlzIG5ldyB0byB1cyAoYW5kIHdpdGhpbiB0aGUgdGltZSB3aW5kb3cpLlxuICAgKiBoYXZlbid0IHNlZW4gdGhlIGZpbmFsIGV2ZW50XG4gICAqL1xuICBwcml2YXRlIGFzeW5jIHJlYWROZXdFdmVudHMoKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgY29uc3QgZXZlbnRzOiBTdGFja0FjdGl2aXR5W10gPSBbXTtcblxuICAgIHRyeSB7XG4gICAgICBsZXQgbmV4dFRva2VuOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gICAgICBsZXQgZmluaXNoZWQgPSBmYWxzZTtcbiAgICAgIHdoaWxlICghZmluaXNoZWQpIHtcbiAgICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCB0aGlzLmNmbi5kZXNjcmliZVN0YWNrRXZlbnRzKHsgU3RhY2tOYW1lOiB0aGlzLnN0YWNrTmFtZSwgTmV4dFRva2VuOiBuZXh0VG9rZW4gfSkucHJvbWlzZSgpO1xuICAgICAgICBjb25zdCBldmVudFBhZ2UgPSByZXNwb25zZT8uU3RhY2tFdmVudHMgPz8gW107XG5cbiAgICAgICAgZm9yIChjb25zdCBldmVudCBvZiBldmVudFBhZ2UpIHtcbiAgICAgICAgICAvLyBFdmVudCBmcm9tIGJlZm9yZSB3ZSB3ZXJlIGludGVyZXN0ZWQgaW4gJ2VtXG4gICAgICAgICAgaWYgKGV2ZW50LlRpbWVzdGFtcC52YWx1ZU9mKCkgPCB0aGlzLnN0YXJ0VGltZSkge1xuICAgICAgICAgICAgZmluaXNoZWQgPSB0cnVlO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgLy8gQWxyZWFkeSBzZWVuIHRoaXMgb25lXG4gICAgICAgICAgaWYgKGV2ZW50LkV2ZW50SWQgaW4gdGhpcy5hY3Rpdml0eSkge1xuICAgICAgICAgICAgZmluaXNoZWQgPSB0cnVlO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgLy8gRnJlc2ggZXZlbnRcbiAgICAgICAgICBldmVudHMucHVzaCh0aGlzLmFjdGl2aXR5W2V2ZW50LkV2ZW50SWRdID0ge1xuICAgICAgICAgICAgZXZlbnQ6IGV2ZW50LFxuICAgICAgICAgICAgbWV0YWRhdGE6IHRoaXMuZmluZE1ldGFkYXRhRm9yKGV2ZW50LkxvZ2ljYWxSZXNvdXJjZUlkKSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFdlJ3JlIGFsc28gZG9uZSBpZiB0aGVyZSdzIG5vdGhpbmcgbGVmdCB0byByZWFkXG4gICAgICAgIG5leHRUb2tlbiA9IHJlc3BvbnNlPy5OZXh0VG9rZW47XG4gICAgICAgIGlmIChuZXh0VG9rZW4gPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIGZpbmlzaGVkID0gdHJ1ZTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIGlmIChlLmNvZGUgPT09ICdWYWxpZGF0aW9uRXJyb3InICYmIGUubWVzc2FnZSA9PT0gYFN0YWNrIFske3RoaXMuc3RhY2tOYW1lfV0gZG9lcyBub3QgZXhpc3RgKSB7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cbiAgICAgIHRocm93IGU7XG4gICAgfVxuXG4gICAgZXZlbnRzLnJldmVyc2UoKTtcbiAgICBmb3IgKGNvbnN0IGV2ZW50IG9mIGV2ZW50cykge1xuICAgICAgdGhpcy5wcmludGVyLmFkZEFjdGl2aXR5KGV2ZW50KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUGVyZm9ybSBhIGZpbmFsIHBvbGwgdG8gdGhlIGVuZCBhbmQgZmx1c2ggb3V0IGFsbCBldmVudHMgdG8gdGhlIHByaW50ZXJcbiAgICpcbiAgICogRmluaXNoIGFueSBwb2xsIGN1cnJlbnRseSBpbiBwcm9ncmVzcywgdGhlbiBkbyBhIGZpbmFsIG9uZSB1bnRpbCB3ZSd2ZVxuICAgKiByZWFjaGVkIHRoZSBsYXN0IHBhZ2UuXG4gICAqL1xuICBwcml2YXRlIGFzeW5jIGZpbmFsUG9sbFRvRW5kKCkge1xuICAgIC8vIElmIHdlIHdlcmUgZG9pbmcgYSBwb2xsLCBmaW5pc2ggdGhhdCBmaXJzdC4gSXQgd2FzIHN0YXJ0ZWQgYmVmb3JlXG4gICAgLy8gdGhlIG1vbWVudCB3ZSB3ZXJlIHN1cmUgd2Ugd2VyZW4ndCBnb2luZyB0byBnZXQgYW55IG5ldyBldmVudHMgYW55bW9yZVxuICAgIC8vIHNvIHdlIG5lZWQgdG8gZG8gYSBuZXcgb25lIGFueXdheS4gTmVlZCB0byB3YWl0IGZvciB0aGlzIG9uZSB0aG91Z2hcbiAgICAvLyBiZWNhdXNlIG91ciBzdGF0ZSBpcyBzaW5nbGUtdGhyZWFkZWQuXG4gICAgaWYgKHRoaXMucmVhZFByb21pc2UpIHtcbiAgICAgIGF3YWl0IHRoaXMucmVhZFByb21pc2U7XG4gICAgfVxuXG4gICAgYXdhaXQgdGhpcy5yZWFkTmV3RXZlbnRzKCk7XG4gIH1cblxuICBwcml2YXRlIHNpbXBsaWZ5Q29uc3RydWN0UGF0aChwYXRoOiBzdHJpbmcpIHtcbiAgICBwYXRoID0gcGF0aC5yZXBsYWNlKC9cXC9SZXNvdXJjZSQvLCAnJyk7XG4gICAgcGF0aCA9IHBhdGgucmVwbGFjZSgvXlxcLy8sICcnKTsgLy8gcmVtb3ZlIFwiL1wiIHByZWZpeFxuXG4gICAgLy8gcmVtb3ZlIFwiPHN0YWNrLW5hbWU+L1wiIHByZWZpeFxuICAgIGlmIChwYXRoLnN0YXJ0c1dpdGgodGhpcy5zdGFja05hbWUgKyAnLycpKSB7XG4gICAgICBwYXRoID0gcGF0aC5zdWJzdHIodGhpcy5zdGFja05hbWUubGVuZ3RoICsgMSk7XG4gICAgfVxuICAgIHJldHVybiBwYXRoO1xuICB9XG59XG5cbmZ1bmN0aW9uIHBhZFJpZ2h0KG46IG51bWJlciwgeDogc3RyaW5nKTogc3RyaW5nIHtcbiAgcmV0dXJuIHggKyAnICcucmVwZWF0KE1hdGgubWF4KDAsIG4gLSB4Lmxlbmd0aCkpO1xufVxuXG4vKipcbiAqIEluZmFtb3VzIHBhZExlZnQoKVxuICovXG5mdW5jdGlvbiBwYWRMZWZ0KG46IG51bWJlciwgeDogc3RyaW5nKTogc3RyaW5nIHtcbiAgcmV0dXJuICcgJy5yZXBlYXQoTWF0aC5tYXgoMCwgbiAtIHgubGVuZ3RoKSkgKyB4O1xufVxuXG5mdW5jdGlvbiBjYWxjTWF4UmVzb3VyY2VUeXBlTGVuZ3RoKHRlbXBsYXRlOiBhbnkpIHtcbiAgY29uc3QgcmVzb3VyY2VzID0gKHRlbXBsYXRlICYmIHRlbXBsYXRlLlJlc291cmNlcykgfHwge307XG4gIGxldCBtYXhXaWR0aCA9IDA7XG4gIGZvciAoY29uc3QgaWQgb2YgT2JqZWN0LmtleXMocmVzb3VyY2VzKSkge1xuICAgIGNvbnN0IHR5cGUgPSByZXNvdXJjZXNbaWRdLlR5cGUgfHwgJyc7XG4gICAgaWYgKHR5cGUubGVuZ3RoID4gbWF4V2lkdGgpIHtcbiAgICAgIG1heFdpZHRoID0gdHlwZS5sZW5ndGg7XG4gICAgfVxuICB9XG4gIHJldHVybiBtYXhXaWR0aDtcbn1cblxuaW50ZXJmYWNlIFByaW50ZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBUb3RhbCByZXNvdXJjZXMgdG8gZGVwbG95XG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZXNUb3RhbD86IG51bWJlclxuXG4gIC8qKlxuICAgKiBUaGUgd2l0aCBvZiB0aGUgXCJyZXNvdXJjZSB0eXBlXCIgY29sdW1uLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVzb3VyY2VUeXBlQ29sdW1uV2lkdGg6IG51bWJlcjtcblxuICAvKipcbiAgICogU3RyZWFtIHRvIHdyaXRlIHRvXG4gICAqL1xuICByZWFkb25seSBzdHJlYW06IE5vZGVKUy5Xcml0ZVN0cmVhbTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJQWN0aXZpdHlQcmludGVyIHtcbiAgcmVhZG9ubHkgdXBkYXRlU2xlZXA6IG51bWJlcjtcblxuICBhZGRBY3Rpdml0eShhY3Rpdml0eTogU3RhY2tBY3Rpdml0eSk6IHZvaWQ7XG4gIHByaW50KCk6IHZvaWQ7XG4gIHN0YXJ0KCk6IHZvaWQ7XG4gIHN0b3AoKTogdm9pZDtcbn1cblxuYWJzdHJhY3QgY2xhc3MgQWN0aXZpdHlQcmludGVyQmFzZSBpbXBsZW1lbnRzIElBY3Rpdml0eVByaW50ZXIge1xuICAvKipcbiAgICogRmV0Y2ggbmV3IGFjdGl2aXR5IGV2ZXJ5IDUgc2Vjb25kc1xuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHVwZGF0ZVNsZWVwOiBudW1iZXIgPSA1XzAwMDtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIHJlc291cmNlIElEcyB3aGljaCBhcmUgY3VycmVudGx5IGJlaW5nIHByb2Nlc3NlZFxuICAgKi9cbiAgcHJvdGVjdGVkIHJlc291cmNlc0luUHJvZ3Jlc3M6IFJlY29yZDxzdHJpbmcsIFN0YWNrQWN0aXZpdHk+ID0ge307XG5cbiAgLyoqXG4gICAqIFByZXZpb3VzIGNvbXBsZXRpb24gc3RhdGUgb2JzZXJ2ZWQgYnkgbG9naWNhbCBJRFxuICAgKlxuICAgKiBXZSB1c2UgdGhpcyB0byBkZXRlY3QgdGhhdCBpZiB3ZSBzZWUgYSBERUxFVEVfQ09NUExFVEUgYWZ0ZXIgYVxuICAgKiBDUkVBVEVfQ09NUExFVEUsIGl0J3MgYWN0dWFsbHkgYSByb2xsYmFjayBhbmQgd2Ugc2hvdWxkIERFQ1JFQVNFXG4gICAqIHJlc291cmNlc0RvbmUgaW5zdGVhZCBvZiBpbmNyZWFzZSBpdFxuICAgKi9cbiAgcHJvdGVjdGVkIHJlc291cmNlc1ByZXZDb21wbGV0ZVN0YXRlOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0ge307XG5cbiAgLyoqXG4gICAqIENvdW50IG9mIHJlc291cmNlcyB0aGF0IGhhdmUgcmVwb3J0ZWQgYSBfQ09NUExFVEUgc3RhdHVzXG4gICAqL1xuICBwcm90ZWN0ZWQgcmVzb3VyY2VzRG9uZTogbnVtYmVyID0gMDtcblxuICAvKipcbiAgICogSG93IG1hbnkgZGlnaXRzIHdlIG5lZWQgdG8gcmVwcmVzZW50IHRoZSB0b3RhbCBjb3VudCAoZm9yIGxpbmluZyB1cCB0aGUgc3RhdHVzIHJlcG9ydGluZylcbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSByZXNvdXJjZURpZ2l0czogbnVtYmVyID0gMDtcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgcmVzb3VyY2VzVG90YWw/OiBudW1iZXI7XG5cbiAgcHJvdGVjdGVkIHJvbGxpbmdCYWNrID0gZmFsc2U7XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGZhaWx1cmVzID0gbmV3IEFycmF5PFN0YWNrQWN0aXZpdHk+KCk7XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHN0cmVhbTogTm9kZUpTLldyaXRlU3RyZWFtO1xuXG4gIGNvbnN0cnVjdG9yKHByb3RlY3RlZCByZWFkb25seSBwcm9wczogUHJpbnRlclByb3BzKSB7XG4gICAgLy8gKzEgYmVjYXVzZSB0aGUgc3RhY2sgYWxzbyBlbWl0cyBhIFwiQ09NUExFVEVcIiBldmVudCBhdCB0aGUgZW5kLCBhbmQgdGhhdCB3YXNuJ3RcbiAgICAvLyBjb3VudGVkIHlldC4gVGhpcyBtYWtlcyBpdCBsaW5lIHVwIHdpdGggdGhlIGFtb3VudCBvZiBldmVudHMgd2UgZXhwZWN0LlxuICAgIHRoaXMucmVzb3VyY2VzVG90YWwgPSBwcm9wcy5yZXNvdXJjZXNUb3RhbCA/IHByb3BzLnJlc291cmNlc1RvdGFsICsgMSA6IHVuZGVmaW5lZDtcblxuICAgIC8vIEhvdyBtYW55IGRpZ2l0cyBkb2VzIHRoaXMgbnVtYmVyIHRha2UgdG8gcmVwcmVzZW50P1xuICAgIHRoaXMucmVzb3VyY2VEaWdpdHMgPSB0aGlzLnJlc291cmNlc1RvdGFsID8gTWF0aC5jZWlsKE1hdGgubG9nMTAodGhpcy5yZXNvdXJjZXNUb3RhbCkpIDogMDtcblxuICAgIHRoaXMuc3RyZWFtID0gcHJvcHMuc3RyZWFtO1xuICB9XG5cbiAgcHVibGljIGFkZEFjdGl2aXR5KGFjdGl2aXR5OiBTdGFja0FjdGl2aXR5KSB7XG4gICAgY29uc3Qgc3RhdHVzID0gYWN0aXZpdHkuZXZlbnQuUmVzb3VyY2VTdGF0dXM7XG4gICAgaWYgKCFzdGF0dXMgfHwgIWFjdGl2aXR5LmV2ZW50LkxvZ2ljYWxSZXNvdXJjZUlkKSB7IHJldHVybjsgfVxuXG4gICAgaWYgKHN0YXR1cyA9PT0gJ1JPTExCQUNLX0lOX1BST0dSRVNTJyB8fCBzdGF0dXMgPT09ICdVUERBVEVfUk9MTEJBQ0tfSU5fUFJPR1JFU1MnKSB7XG4gICAgICAvLyBPbmx5IHRyaWdnZXJlZCBvbiB0aGUgc3RhY2sgb25jZSB3ZSd2ZSBzdGFydGVkIGRvaW5nIGEgcm9sbGJhY2tcbiAgICAgIHRoaXMucm9sbGluZ0JhY2sgPSB0cnVlO1xuICAgIH1cblxuICAgIGlmIChzdGF0dXMuZW5kc1dpdGgoJ19JTl9QUk9HUkVTUycpKSB7XG4gICAgICB0aGlzLnJlc291cmNlc0luUHJvZ3Jlc3NbYWN0aXZpdHkuZXZlbnQuTG9naWNhbFJlc291cmNlSWRdID0gYWN0aXZpdHk7XG4gICAgfVxuXG4gICAgaWYgKGhhc0Vycm9yTWVzc2FnZShzdGF0dXMpKSB7XG4gICAgICBjb25zdCBpc0NhbmNlbGxlZCA9IChhY3Rpdml0eS5ldmVudC5SZXNvdXJjZVN0YXR1c1JlYXNvbiA/PyAnJykuaW5kZXhPZignY2FuY2VsbGVkJykgPiAtMTtcblxuICAgICAgLy8gQ2FuY2VsbGVkIGlzIG5vdCBhbiBpbnRlcmVzdGluZyBmYWlsdXJlIHJlYXNvblxuICAgICAgaWYgKCFpc0NhbmNlbGxlZCkge1xuICAgICAgICB0aGlzLmZhaWx1cmVzLnB1c2goYWN0aXZpdHkpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChzdGF0dXMuZW5kc1dpdGgoJ19DT01QTEVURScpIHx8IHN0YXR1cy5lbmRzV2l0aCgnX0ZBSUxFRCcpKSB7XG4gICAgICBkZWxldGUgdGhpcy5yZXNvdXJjZXNJblByb2dyZXNzW2FjdGl2aXR5LmV2ZW50LkxvZ2ljYWxSZXNvdXJjZUlkXTtcbiAgICB9XG5cbiAgICBpZiAoc3RhdHVzLmVuZHNXaXRoKCdfQ09NUExFVEVfQ0xFQU5VUF9JTl9QUk9HUkVTUycpKSB7XG4gICAgICB0aGlzLnJlc291cmNlc0RvbmUrKztcbiAgICB9XG5cbiAgICBpZiAoc3RhdHVzLmVuZHNXaXRoKCdfQ09NUExFVEUnKSkge1xuICAgICAgY29uc3QgcHJldlN0YXRlID0gdGhpcy5yZXNvdXJjZXNQcmV2Q29tcGxldGVTdGF0ZVthY3Rpdml0eS5ldmVudC5Mb2dpY2FsUmVzb3VyY2VJZF07XG4gICAgICBpZiAoIXByZXZTdGF0ZSkge1xuICAgICAgICB0aGlzLnJlc291cmNlc0RvbmUrKztcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIElmIHdlIGNvbXBsZXRlZCB0aGlzIGJlZm9yZSBhbmQgd2UncmUgY29tcGxldGluZyBpdCBBR0FJTiwgbWVhbnMgd2UncmUgcm9sbGluZyBiYWNrLlxuICAgICAgICAvLyBQcm90ZWN0IGFnYWluc3Qgc2lsbHkgdW5kZXJmbG93LlxuICAgICAgICB0aGlzLnJlc291cmNlc0RvbmUtLTtcbiAgICAgICAgaWYgKHRoaXMucmVzb3VyY2VzRG9uZSA8IDApIHtcbiAgICAgICAgICB0aGlzLnJlc291cmNlc0RvbmUgPSAwO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICB0aGlzLnJlc291cmNlc1ByZXZDb21wbGV0ZVN0YXRlW2FjdGl2aXR5LmV2ZW50LkxvZ2ljYWxSZXNvdXJjZUlkXSA9IHN0YXR1cztcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYWJzdHJhY3QgcHJpbnQoKTogdm9pZDtcblxuICBwdWJsaWMgc3RhcnQoKSB7XG4gICAgLy8gRW1wdHkgb24gcHVycG9zZVxuICB9XG5cbiAgcHVibGljIHN0b3AoKSB7XG4gICAgLy8gRW1wdHkgb24gcHVycG9zZVxuICB9XG59XG5cbi8qKlxuICogQWN0aXZpdHkgUHJpbnRlciB3aGljaCBzaG93cyBhIGZ1bGwgbG9nIG9mIGFsbCBDbG91ZEZvcm1hdGlvbiBldmVudHNcbiAqXG4gKiBXaGVuIHRoZXJlIGhhc24ndCBiZWVuIGFjdGl2aXR5IGZvciBhIHdoaWxlLCBpdCB3aWxsIHByaW50IHRoZSByZXNvdXJjZXNcbiAqIHRoYXQgYXJlIGN1cnJlbnRseSBpbiBwcm9ncmVzcywgdG8gc2hvdyB3aGF0J3MgaG9sZGluZyB1cCB0aGUgZGVwbG95bWVudC5cbiAqL1xuZXhwb3J0IGNsYXNzIEhpc3RvcnlBY3Rpdml0eVByaW50ZXIgZXh0ZW5kcyBBY3Rpdml0eVByaW50ZXJCYXNlIHtcbiAgLyoqXG4gICAqIExhc3QgdGltZSB3ZSBwcmludGVkIHNvbWV0aGluZyB0byB0aGUgY29uc29sZS5cbiAgICpcbiAgICogVXNlZCB0byBtZWFzdXJlIHRpbWVvdXQgZm9yIHByb2dyZXNzIHJlcG9ydGluZy5cbiAgICovXG4gIHByaXZhdGUgbGFzdFByaW50VGltZSA9IERhdGUubm93KCk7XG5cbiAgLyoqXG4gICAqIE51bWJlciBvZiBtcyBvZiBjaGFuZ2UgYWJzZW5jZSBiZWZvcmUgd2UgdGVsbCB0aGUgdXNlciBhYm91dCB0aGUgcmVzb3VyY2VzIHRoYXQgYXJlIGN1cnJlbnRseSBpbiBwcm9ncmVzcy5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgaW5Qcm9ncmVzc0RlbGF5ID0gMzBfMDAwO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcHJpbnRhYmxlID0gbmV3IEFycmF5PFN0YWNrQWN0aXZpdHk+KCk7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFByaW50ZXJQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRBY3Rpdml0eShhY3Rpdml0eTogU3RhY2tBY3Rpdml0eSkge1xuICAgIHN1cGVyLmFkZEFjdGl2aXR5KGFjdGl2aXR5KTtcbiAgICB0aGlzLnByaW50YWJsZS5wdXNoKGFjdGl2aXR5KTtcbiAgICB0aGlzLnByaW50KCk7XG4gIH1cblxuICBwdWJsaWMgcHJpbnQoKSB7XG4gICAgZm9yIChjb25zdCBhY3Rpdml0eSBvZiB0aGlzLnByaW50YWJsZSkge1xuICAgICAgdGhpcy5wcmludE9uZShhY3Rpdml0eSk7XG4gICAgfVxuICAgIHRoaXMucHJpbnRhYmxlLnNwbGljZSgwLCB0aGlzLnByaW50YWJsZS5sZW5ndGgpO1xuICAgIHRoaXMucHJpbnRJblByb2dyZXNzKCk7XG4gIH1cblxuICBwdWJsaWMgc3RvcCgpIHtcbiAgICAvLyBQcmludCBmYWlsdXJlcyBhdCB0aGUgZW5kXG4gICAgaWYgKHRoaXMuZmFpbHVyZXMubGVuZ3RoID4gMCkge1xuICAgICAgdGhpcy5zdHJlYW0ud3JpdGUoJ1xcbkZhaWxlZCByZXNvdXJjZXM6XFxuJyk7XG4gICAgICBmb3IgKGNvbnN0IGZhaWx1cmUgb2YgdGhpcy5mYWlsdXJlcykge1xuICAgICAgICAvLyBSb290IHN0YWNrIGZhaWx1cmVzIGFyZSBub3QgaW50ZXJlc3RpbmdcbiAgICAgICAgaWYgKGZhaWx1cmUuZXZlbnQuU3RhY2tOYW1lID09PSBmYWlsdXJlLmV2ZW50LkxvZ2ljYWxSZXNvdXJjZUlkKSB7XG4gICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLnByaW50T25lKGZhaWx1cmUsIGZhbHNlKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHByaW50T25lKGFjdGl2aXR5OiBTdGFja0FjdGl2aXR5LCBwcm9ncmVzcz86IGJvb2xlYW4pIHtcbiAgICBjb25zdCBlID0gYWN0aXZpdHkuZXZlbnQ7XG4gICAgY29uc3QgY29sb3IgPSBjb2xvckZyb21TdGF0dXNSZXN1bHQoZS5SZXNvdXJjZVN0YXR1cyk7XG4gICAgbGV0IHJlYXNvbkNvbG9yID0gY29sb3JzLmN5YW47XG5cbiAgICBsZXQgc3RhY2tUcmFjZSA9ICcnO1xuICAgIGNvbnN0IG1kID0gYWN0aXZpdHkubWV0YWRhdGE7XG4gICAgaWYgKG1kICYmIGUuUmVzb3VyY2VTdGF0dXMgJiYgZS5SZXNvdXJjZVN0YXR1cy5pbmRleE9mKCdGQUlMRUQnKSAhPT0gLTEpIHtcbiAgICAgIHN0YWNrVHJhY2UgPSBtZC5lbnRyeS50cmFjZSA/IGBcXG5cXHQke21kLmVudHJ5LnRyYWNlLmpvaW4oJ1xcblxcdFxcXFxfICcpfWAgOiAnJztcbiAgICAgIHJlYXNvbkNvbG9yID0gY29sb3JzLnJlZDtcbiAgICB9XG5cbiAgICBjb25zdCByZXNvdXJjZU5hbWUgPSBtZCA/IG1kLmNvbnN0cnVjdFBhdGggOiAoZS5Mb2dpY2FsUmVzb3VyY2VJZCB8fCAnJyk7XG5cbiAgICBjb25zdCBsb2dpY2FsSWQgPSByZXNvdXJjZU5hbWUgIT09IGUuTG9naWNhbFJlc291cmNlSWQgPyBgKCR7ZS5Mb2dpY2FsUmVzb3VyY2VJZH0pIGAgOiAnJztcblxuICAgIHRoaXMuc3RyZWFtLndyaXRlKFxuICAgICAgdXRpbC5mb3JtYXQoXG4gICAgICAgICcgJXMlcyB8ICVzIHwgJXMgfCAlcyAlcyVzJXNcXG4nLFxuICAgICAgICAocHJvZ3Jlc3MgIT09IGZhbHNlID8gYCAke3RoaXMucHJvZ3Jlc3MoKX0gfGAgOiAnJyksXG4gICAgICAgIG5ldyBEYXRlKGUuVGltZXN0YW1wKS50b0xvY2FsZVRpbWVTdHJpbmcoKSxcbiAgICAgICAgY29sb3IocGFkUmlnaHQoU1RBVFVTX1dJRFRILCAoZS5SZXNvdXJjZVN0YXR1cyB8fCAnJykuc3Vic3RyKDAsIFNUQVRVU19XSURUSCkpKSwgLy8gcGFkIGxlZnQgYW5kIHRyaW1cbiAgICAgICAgcGFkUmlnaHQodGhpcy5wcm9wcy5yZXNvdXJjZVR5cGVDb2x1bW5XaWR0aCwgZS5SZXNvdXJjZVR5cGUgfHwgJycpLFxuICAgICAgICBjb2xvcihjb2xvcnMuYm9sZChyZXNvdXJjZU5hbWUpKSxcbiAgICAgICAgbG9naWNhbElkLFxuICAgICAgICByZWFzb25Db2xvcihjb2xvcnMuYm9sZChlLlJlc291cmNlU3RhdHVzUmVhc29uID8gZS5SZXNvdXJjZVN0YXR1c1JlYXNvbiA6ICcnKSksXG4gICAgICAgIHJlYXNvbkNvbG9yKHN0YWNrVHJhY2UpLFxuICAgICAgKSxcbiAgICApO1xuXG4gICAgdGhpcy5sYXN0UHJpbnRUaW1lID0gRGF0ZS5ub3coKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXBvcnQgdGhlIGN1cnJlbnQgcHJvZ3Jlc3MgYXMgYSBbMzQvNDJdIHN0cmluZywgb3IganVzdCBbMzRdIGlmIHRoZSB0b3RhbCBpcyB1bmtub3duXG4gICAqL1xuICBwcml2YXRlIHByb2dyZXNzKCk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMucmVzb3VyY2VzVG90YWwgPT0gbnVsbCkge1xuICAgICAgLy8gRG9uJ3QgaGF2ZSB0b3RhbCwgc2hvdyBzaW1wbGUgY291bnQgYW5kIGhvcGUgdGhlIGh1bWFuIGtub3dzXG4gICAgICByZXR1cm4gcGFkTGVmdCgzLCB1dGlsLmZvcm1hdCgnJXMnLCB0aGlzLnJlc291cmNlc0RvbmUpKTsgLy8gbWF4IDUwMCByZXNvdXJjZXNcbiAgICB9XG5cbiAgICByZXR1cm4gdXRpbC5mb3JtYXQoJyVzLyVzJyxcbiAgICAgIHBhZExlZnQodGhpcy5yZXNvdXJjZURpZ2l0cywgdGhpcy5yZXNvdXJjZXNEb25lLnRvU3RyaW5nKCkpLFxuICAgICAgcGFkTGVmdCh0aGlzLnJlc291cmNlRGlnaXRzLCB0aGlzLnJlc291cmNlc1RvdGFsICE9IG51bGwgPyB0aGlzLnJlc291cmNlc1RvdGFsLnRvU3RyaW5nKCkgOiAnPycpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJZiBzb21lIHJlc291cmNlcyBhcmUgdGFraW5nIGEgd2hpbGUgdG8gY3JlYXRlLCBub3RpZnkgdGhlIHVzZXIgYWJvdXQgd2hhdCdzIGN1cnJlbnRseSBpbiBwcm9ncmVzc1xuICAgKi9cbiAgcHJpdmF0ZSBwcmludEluUHJvZ3Jlc3MoKSB7XG4gICAgaWYgKERhdGUubm93KCkgPCB0aGlzLmxhc3RQcmludFRpbWUgKyB0aGlzLmluUHJvZ3Jlc3NEZWxheSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmIChPYmplY3Qua2V5cyh0aGlzLnJlc291cmNlc0luUHJvZ3Jlc3MpLmxlbmd0aCA+IDApIHtcbiAgICAgIHRoaXMuc3RyZWFtLndyaXRlKHV0aWwuZm9ybWF0KCclcyBDdXJyZW50bHkgaW4gcHJvZ3Jlc3M6ICVzXFxuJyxcbiAgICAgICAgdGhpcy5wcm9ncmVzcygpLFxuICAgICAgICBjb2xvcnMuYm9sZChPYmplY3Qua2V5cyh0aGlzLnJlc291cmNlc0luUHJvZ3Jlc3MpLmpvaW4oJywgJykpKSk7XG4gICAgfVxuXG4gICAgLy8gV2UgY2hlYXQgYSBiaXQgaGVyZS4gVG8gcHJldmVudCBwcmludEluUHJvZ3Jlc3MoKSBmcm9tIHJlcGVhdGVkbHkgdHJpZ2dlcmluZyxcbiAgICAvLyB3ZSBzZXQgdGhlIHRpbWVzdGFtcCBpbnRvIHRoZSBmdXR1cmUuIEl0IHdpbGwgYmUgcmVzZXQgd2hlbmV2ZXIgYSByZWd1bGFyIHByaW50XG4gICAgLy8gb2NjdXJzLCBhZnRlciB3aGljaCB3ZSBjYW4gYmUgdHJpZ2dlcmVkIGFnYWluLlxuICAgIHRoaXMubGFzdFByaW50VGltZSA9ICtJbmZpbml0eTtcbiAgfVxuXG59XG5cbi8qKlxuICogQWN0aXZpdHkgUHJpbnRlciB3aGljaCBzaG93cyB0aGUgcmVzb3VyY2VzIGN1cnJlbnRseSBiZWluZyB1cGRhdGVkXG4gKlxuICogSXQgd2lsbCBjb250aW51b3VzbHkgcmV1cGRhdGUgdGhlIHRlcm1pbmFsIGFuZCBzaG93IG9ubHkgdGhlIHJlc291cmNlc1xuICogdGhhdCBhcmUgY3VycmVudGx5IGJlaW5nIHVwZGF0ZWQsIGluIGFkZGl0aW9uIHRvIGEgcHJvZ3Jlc3MgYmFyIHdoaWNoXG4gKiBzaG93cyBob3cgZmFyIGFsb25nIHRoZSBkZXBsb3ltZW50IGlzLlxuICpcbiAqIFJlc291cmNlcyB0aGF0IGhhdmUgZmFpbGVkIHdpbGwgYWx3YXlzIGJlIHNob3duLCBhbmQgd2lsbCBiZSByZWNhcGl0dWxhdGVkXG4gKiBhbG9uZyB3aXRoIHRoZWlyIHN0YWNrIHRyYWNlIHdoZW4gdGhlIG1vbml0b3JpbmcgZW5kcy5cbiAqXG4gKiBSZXNvdXJjZXMgdGhhdCBmYWlsZWQgZGVwbG95bWVudCBiZWNhdXNlIHRoZXkgaGF2ZSBiZWVuIGNhbmNlbGxlZCBhcmVcbiAqIG5vdCBpbmNsdWRlZC5cbiAqL1xuZXhwb3J0IGNsYXNzIEN1cnJlbnRBY3Rpdml0eVByaW50ZXIgZXh0ZW5kcyBBY3Rpdml0eVByaW50ZXJCYXNlIHtcbiAgLyoqXG4gICAqIFRoaXMgbG9va3MgdmVyeSBkaXNvcmllbnRpbmcgc2xlZXBpbmcgZm9yIDUgc2Vjb25kcy4gVXBkYXRlIHF1aWNrZXIuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdXBkYXRlU2xlZXA6IG51bWJlciA9IDJfMDAwO1xuXG4gIHByaXZhdGUgb2xkTG9nTGV2ZWw6IExvZ0xldmVsID0gTG9nTGV2ZWwuREVGQVVMVDtcbiAgcHJpdmF0ZSBibG9jayA9IG5ldyBSZXdyaXRhYmxlQmxvY2sodGhpcy5zdHJlYW0pO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBQcmludGVyUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cblxuICBwdWJsaWMgcHJpbnQoKTogdm9pZCB7XG4gICAgY29uc3QgbGluZXMgPSBbXTtcblxuICAgIC8vIEFkZCBhIHByb2dyZXNzIGJhciBhdCB0aGUgdG9wXG4gICAgY29uc3QgcHJvZ3Jlc3NXaWR0aCA9IE1hdGgubWF4KE1hdGgubWluKCh0aGlzLmJsb2NrLndpZHRoID8/IDgwKSAtIFBST0dSRVNTQkFSX0VYVFJBX1NQQUNFIC0gMSwgTUFYX1BST0dSRVNTQkFSX1dJRFRIKSwgTUlOX1BST0dSRVNTQkFSX1dJRFRIKTtcbiAgICBjb25zdCBwcm9nID0gdGhpcy5wcm9ncmVzc0Jhcihwcm9ncmVzc1dpZHRoKTtcbiAgICBpZiAocHJvZykge1xuICAgICAgbGluZXMucHVzaCgnICAnICsgcHJvZywgJycpO1xuICAgIH1cblxuICAgIC8vIE5vcm1hbGx5IHdlJ2Qgb25seSBwcmludCBcInJlc291cmNlcyBpbiBwcm9ncmVzc1wiLCBidXQgaXQncyBhbHNvIHVzZWZ1bFxuICAgIC8vIHRvIGtlZXAgYW4gZXllIG9uIHRoZSBmYWlsdXJlcyBhbmQga25vdyBhYm91dCB0aGUgc3BlY2lmaWMgZXJyb3JzIGFzcXVpY2tseVxuICAgIC8vIGFzIHBvc3NpYmxlICh3aGlsZSB0aGUgc3RhY2sgaXMgc3RpbGwgcm9sbGluZyBiYWNrKSwgc28gYWRkIHRob3NlIGluLlxuICAgIGNvbnN0IHRvUHJpbnQ6IFN0YWNrQWN0aXZpdHlbXSA9IFsuLi50aGlzLmZhaWx1cmVzLCAuLi5PYmplY3QudmFsdWVzKHRoaXMucmVzb3VyY2VzSW5Qcm9ncmVzcyldO1xuICAgIHRvUHJpbnQuc29ydCgoYSwgYikgPT4gYS5ldmVudC5UaW1lc3RhbXAuZ2V0VGltZSgpIC0gYi5ldmVudC5UaW1lc3RhbXAuZ2V0VGltZSgpKTtcblxuICAgIGxpbmVzLnB1c2goLi4udG9QcmludC5tYXAocmVzID0+IHtcbiAgICAgIGNvbnN0IGNvbG9yID0gY29sb3JGcm9tU3RhdHVzQWN0aXZpdHkocmVzLmV2ZW50LlJlc291cmNlU3RhdHVzKTtcbiAgICAgIGNvbnN0IHJlc291cmNlTmFtZSA9IHJlcy5tZXRhZGF0YT8uY29uc3RydWN0UGF0aCA/PyByZXMuZXZlbnQuTG9naWNhbFJlc291cmNlSWQgPz8gJyc7XG5cbiAgICAgIHJldHVybiB1dGlsLmZvcm1hdCgnJXMgfCAlcyB8ICVzIHwgJXMlcycsXG4gICAgICAgIHBhZExlZnQoVElNRVNUQU1QX1dJRFRILCBuZXcgRGF0ZShyZXMuZXZlbnQuVGltZXN0YW1wKS50b0xvY2FsZVRpbWVTdHJpbmcoKSksXG4gICAgICAgIGNvbG9yKHBhZFJpZ2h0KFNUQVRVU19XSURUSCwgKHJlcy5ldmVudC5SZXNvdXJjZVN0YXR1cyB8fCAnJykuc3Vic3RyKDAsIFNUQVRVU19XSURUSCkpKSxcbiAgICAgICAgcGFkUmlnaHQodGhpcy5wcm9wcy5yZXNvdXJjZVR5cGVDb2x1bW5XaWR0aCwgcmVzLmV2ZW50LlJlc291cmNlVHlwZSB8fCAnJyksXG4gICAgICAgIGNvbG9yKGNvbG9ycy5ib2xkKHNob3J0ZW4oNDAsIHJlc291cmNlTmFtZSkpKSxcbiAgICAgICAgdGhpcy5mYWlsdXJlUmVhc29uT25OZXh0TGluZShyZXMpKTtcbiAgICB9KSk7XG5cbiAgICB0aGlzLmJsb2NrLmRpc3BsYXlMaW5lcyhsaW5lcyk7XG4gIH1cblxuICBwdWJsaWMgc3RhcnQoKSB7XG4gICAgLy8gTmVlZCB0byBwcmV2ZW50IHRoZSB3YWl0ZXIgZnJvbSBwcmludGluZyAnc3RhY2sgbm90IHN0YWJsZScgZXZlcnkgNSBzZWNvbmRzLCBpdCBtZXNzZXNcbiAgICAvLyB3aXRoIHRoZSBvdXRwdXQgY2FsY3VsYXRpb25zLlxuICAgIHRoaXMub2xkTG9nTGV2ZWwgPSBsb2dMZXZlbDtcbiAgICBzZXRMb2dMZXZlbChMb2dMZXZlbC5ERUZBVUxUKTtcbiAgfVxuXG4gIHB1YmxpYyBzdG9wKCkge1xuICAgIHNldExvZ0xldmVsKHRoaXMub2xkTG9nTGV2ZWwpO1xuXG4gICAgLy8gUHJpbnQgZmFpbHVyZXMgYXQgdGhlIGVuZFxuICAgIGNvbnN0IGxpbmVzID0gbmV3IEFycmF5PHN0cmluZz4oKTtcbiAgICBmb3IgKGNvbnN0IGZhaWx1cmUgb2YgdGhpcy5mYWlsdXJlcykge1xuICAgICAgLy8gUm9vdCBzdGFjayBmYWlsdXJlcyBhcmUgbm90IGludGVyZXN0aW5nXG4gICAgICBpZiAoZmFpbHVyZS5ldmVudC5TdGFja05hbWUgPT09IGZhaWx1cmUuZXZlbnQuTG9naWNhbFJlc291cmNlSWQpIHtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG5cbiAgICAgIGxpbmVzLnB1c2godXRpbC5mb3JtYXQoY29sb3JzLnJlZCgnJXMgfCAlcyB8ICVzIHwgJXMlcycpICsgJ1xcbicsXG4gICAgICAgIHBhZExlZnQoVElNRVNUQU1QX1dJRFRILCBuZXcgRGF0ZShmYWlsdXJlLmV2ZW50LlRpbWVzdGFtcCkudG9Mb2NhbGVUaW1lU3RyaW5nKCkpLFxuICAgICAgICBwYWRSaWdodChTVEFUVVNfV0lEVEgsIChmYWlsdXJlLmV2ZW50LlJlc291cmNlU3RhdHVzIHx8ICcnKS5zdWJzdHIoMCwgU1RBVFVTX1dJRFRIKSksXG4gICAgICAgIHBhZFJpZ2h0KHRoaXMucHJvcHMucmVzb3VyY2VUeXBlQ29sdW1uV2lkdGgsIGZhaWx1cmUuZXZlbnQuUmVzb3VyY2VUeXBlIHx8ICcnKSxcbiAgICAgICAgc2hvcnRlbig0MCwgZmFpbHVyZS5ldmVudC5Mb2dpY2FsUmVzb3VyY2VJZCA/PyAnJyksXG4gICAgICAgIHRoaXMuZmFpbHVyZVJlYXNvbk9uTmV4dExpbmUoZmFpbHVyZSkpKTtcblxuICAgICAgY29uc3QgdHJhY2UgPSBmYWlsdXJlLm1ldGFkYXRhPy5lbnRyeT8udHJhY2U7XG4gICAgICBpZiAodHJhY2UpIHtcbiAgICAgICAgbGluZXMucHVzaChjb2xvcnMucmVkKGBcXHQke3RyYWNlLmpvaW4oJ1xcblxcdFxcXFxfICcpfVxcbmApKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBEaXNwbGF5IGluIHRoZSBzYW1lIGJsb2NrIHNwYWNlLCBvdGhlcndpc2Ugd2UncmUgZ29pbmcgdG8gaGF2ZSBzaWxseSBlbXB0eSBsaW5lcy5cbiAgICB0aGlzLmJsb2NrLmRpc3BsYXlMaW5lcyhsaW5lcyk7XG4gIH1cblxuICBwcml2YXRlIHByb2dyZXNzQmFyKHdpZHRoOiBudW1iZXIpIHtcbiAgICBpZiAoIXRoaXMucmVzb3VyY2VzVG90YWwpIHsgcmV0dXJuICcnOyB9XG4gICAgY29uc3QgZnJhY3Rpb24gPSBNYXRoLm1pbih0aGlzLnJlc291cmNlc0RvbmUgLyB0aGlzLnJlc291cmNlc1RvdGFsLCAxKTtcbiAgICBjb25zdCBpbm5lcldpZHRoID0gTWF0aC5tYXgoMSwgd2lkdGggLSAyKTtcbiAgICBjb25zdCBjaGFycyA9IGlubmVyV2lkdGggKiBmcmFjdGlvbjtcbiAgICBjb25zdCByZW1haW5kZXIgPSBjaGFycyAtIE1hdGguZmxvb3IoY2hhcnMpO1xuXG4gICAgY29uc3QgZnVsbENoYXJzID0gRlVMTF9CTE9DSy5yZXBlYXQoTWF0aC5mbG9vcihjaGFycykpO1xuICAgIGNvbnN0IHBhcnRpYWxDaGFyID0gUEFSVElBTF9CTE9DS1tNYXRoLmZsb29yKHJlbWFpbmRlciAqIFBBUlRJQUxfQkxPQ0subGVuZ3RoKV07XG4gICAgY29uc3QgZmlsbGVyID0gJ8K3Jy5yZXBlYXQoaW5uZXJXaWR0aCAtIE1hdGguZmxvb3IoY2hhcnMpIC0gKHBhcnRpYWxDaGFyID8gMSA6IDApKTtcblxuICAgIGNvbnN0IGNvbG9yID0gdGhpcy5yb2xsaW5nQmFjayA/IGNvbG9ycy55ZWxsb3cgOiBjb2xvcnMuZ3JlZW47XG5cbiAgICByZXR1cm4gJ1snICsgY29sb3IoZnVsbENoYXJzICsgcGFydGlhbENoYXIpICsgZmlsbGVyICsgYF0gKCR7dGhpcy5yZXNvdXJjZXNEb25lfS8ke3RoaXMucmVzb3VyY2VzVG90YWx9KWA7XG4gIH1cblxuICBwcml2YXRlIGZhaWx1cmVSZWFzb25Pbk5leHRMaW5lKGFjdGl2aXR5OiBTdGFja0FjdGl2aXR5KSB7XG4gICAgcmV0dXJuIGhhc0Vycm9yTWVzc2FnZShhY3Rpdml0eS5ldmVudC5SZXNvdXJjZVN0YXR1cyA/PyAnJylcbiAgICAgID8gYFxcbiR7JyAnLnJlcGVhdChUSU1FU1RBTVBfV0lEVEggKyBTVEFUVVNfV0lEVEggKyA2KX0ke2NvbG9ycy5yZWQoYWN0aXZpdHkuZXZlbnQuUmVzb3VyY2VTdGF0dXNSZWFzb24gPz8gJycpfWBcbiAgICAgIDogJyc7XG4gIH1cbn1cblxuY29uc3QgRlVMTF9CTE9DSyA9ICfilognO1xuY29uc3QgUEFSVElBTF9CTE9DSyA9IFsnJywgJ+KWjycsICfilo4nLCAn4paNJywgJ+KWjCcsICfilosnLCAn4paKJywgJ+KWiSddO1xuY29uc3QgTUFYX1BST0dSRVNTQkFSX1dJRFRIID0gNjA7XG5jb25zdCBNSU5fUFJPR1JFU1NCQVJfV0lEVEggPSAxMDtcbmNvbnN0IFBST0dSRVNTQkFSX0VYVFJBX1NQQUNFID0gMiAvKiBsZWFkaW5nIHNwYWNlcyAqLyArIDIgLyogYnJhY2tldHMgKi8gKyA0IC8qIHByb2dyZXNzIG51bWJlciBkZWNvcmF0aW9uICovICsgNiAvKiAyIHByb2dyZXNzIG51bWJlcnMgdXAgdG8gOTk5ICovO1xuXG5mdW5jdGlvbiBoYXNFcnJvck1lc3NhZ2Uoc3RhdHVzOiBzdHJpbmcpIHtcbiAgcmV0dXJuIHN0YXR1cy5lbmRzV2l0aCgnX0ZBSUxFRCcpIHx8IHN0YXR1cyA9PT0gJ1JPTExCQUNLX0lOX1BST0dSRVNTJyB8fCBzdGF0dXMgPT09ICdVUERBVEVfUk9MTEJBQ0tfSU5fUFJPR1JFU1MnO1xufVxuXG5mdW5jdGlvbiBjb2xvckZyb21TdGF0dXNSZXN1bHQoc3RhdHVzPzogc3RyaW5nKSB7XG4gIGlmICghc3RhdHVzKSB7XG4gICAgcmV0dXJuIGNvbG9ycy5yZXNldDtcbiAgfVxuXG4gIGlmIChzdGF0dXMuaW5kZXhPZignRkFJTEVEJykgIT09IC0xKSB7XG4gICAgcmV0dXJuIGNvbG9ycy5yZWQ7XG4gIH1cbiAgaWYgKHN0YXR1cy5pbmRleE9mKCdST0xMQkFDSycpICE9PSAtMSkge1xuICAgIHJldHVybiBjb2xvcnMueWVsbG93O1xuICB9XG4gIGlmIChzdGF0dXMuaW5kZXhPZignQ09NUExFVEUnKSAhPT0gLTEpIHtcbiAgICByZXR1cm4gY29sb3JzLmdyZWVuO1xuICB9XG5cbiAgcmV0dXJuIGNvbG9ycy5yZXNldDtcbn1cblxuZnVuY3Rpb24gY29sb3JGcm9tU3RhdHVzQWN0aXZpdHkoc3RhdHVzPzogc3RyaW5nKSB7XG4gIGlmICghc3RhdHVzKSB7XG4gICAgcmV0dXJuIGNvbG9ycy5yZXNldDtcbiAgfVxuXG4gIGlmIChzdGF0dXMuZW5kc1dpdGgoJ19GQUlMRUQnKSkge1xuICAgIHJldHVybiBjb2xvcnMucmVkO1xuICB9XG5cbiAgaWYgKHN0YXR1cy5zdGFydHNXaXRoKCdDUkVBVEVfJykgfHwgc3RhdHVzLnN0YXJ0c1dpdGgoJ1VQREFURV8nKSkge1xuICAgIHJldHVybiBjb2xvcnMuZ3JlZW47XG4gIH1cbiAgLy8gRm9yIHN0YWNrcywgaXQgbWF5IGFsc28gYmUgJ1VQRERBVEVfUk9MTEJBQ0tfSU5fUFJPR1JFU1MnXG4gIGlmIChzdGF0dXMuaW5kZXhPZignUk9MTEJBQ0tfJykgIT09IC0xKSB7XG4gICAgcmV0dXJuIGNvbG9ycy55ZWxsb3c7XG4gIH1cbiAgaWYgKHN0YXR1cy5zdGFydHNXaXRoKCdERUxFVEVfJykpIHtcbiAgICByZXR1cm4gY29sb3JzLnllbGxvdztcbiAgfVxuXG4gIHJldHVybiBjb2xvcnMucmVzZXQ7XG59XG5cbmZ1bmN0aW9uIHNob3J0ZW4obWF4V2lkdGg6IG51bWJlciwgcDogc3RyaW5nKSB7XG4gIGlmIChwLmxlbmd0aCA8PSBtYXhXaWR0aCkgeyByZXR1cm4gcDsgfVxuICBjb25zdCBoYWxmID0gTWF0aC5mbG9vcigobWF4V2lkdGggLSAzKSAvIDIpO1xuICByZXR1cm4gcC5zdWJzdHIoMCwgaGFsZikgKyAnLi4uJyArIHAuc3Vic3RyKHAubGVuZ3RoIC0gaGFsZik7XG59XG5cbmNvbnN0IFRJTUVTVEFNUF9XSURUSCA9IDEyO1xuY29uc3QgU1RBVFVTX1dJRFRIID0gMjA7XG4iXX0=