"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkToolkit = void 0;
const path = require("path");
const util_1 = require("util");
const colors = require("colors/safe");
const fs = require("fs-extra");
const promptly = require("promptly");
const environments_1 = require("../lib/api/cxapp/environments");
const cloud_assembly_1 = require("./api/cxapp/cloud-assembly");
const diff_1 = require("./diff");
const logging_1 = require("./logging");
const serialize_1 = require("./serialize");
const util_2 = require("./util");
/**
 * Toolkit logic
 *
 * The toolkit runs the `cloudExecutable` to obtain a cloud assembly and
 * deploys applies them to `cloudFormation`.
 */
class CdkToolkit {
    constructor(props) {
        this.props = props;
    }
    async metadata(stackName) {
        var _a;
        const stacks = await this.selectSingleStackByName(stackName);
        return (_a = stacks.firstStack.manifest.metadata) !== null && _a !== void 0 ? _a : {};
    }
    async diff(options) {
        const stacks = await this.selectStacksForDiff(options.stackNames, options.exclusively);
        const strict = !!options.strict;
        const contextLines = options.contextLines || 3;
        const stream = options.stream || process.stderr;
        let diffs = 0;
        if (options.templatePath !== undefined) {
            // Compare single stack against fixed template
            if (stacks.stackCount !== 1) {
                throw new Error('Can only select one stack when comparing to fixed template. Use --exclusively to avoid selecting multiple stacks.');
            }
            if (!await fs.pathExists(options.templatePath)) {
                throw new Error(`There is no file at ${options.templatePath}`);
            }
            const template = serialize_1.deserializeStructure(await fs.readFile(options.templatePath, { encoding: 'UTF-8' }));
            diffs = options.securityOnly
                ? util_2.numberFromBool(diff_1.printSecurityDiff(template, stacks.firstStack, diff_1.RequireApproval.Broadening))
                : diff_1.printStackDiff(template, stacks.firstStack, strict, contextLines, stream);
        }
        else {
            // Compare N stacks against deployed templates
            for (const stack of stacks.stackArtifacts) {
                stream.write(util_1.format('Stack %s\n', colors.bold(stack.displayName)));
                const currentTemplate = await this.props.cloudFormation.readCurrentTemplate(stack);
                diffs += options.securityOnly
                    ? util_2.numberFromBool(diff_1.printSecurityDiff(currentTemplate, stack, diff_1.RequireApproval.Broadening))
                    : diff_1.printStackDiff(currentTemplate, stack, strict, contextLines, stream);
            }
        }
        return diffs && options.fail ? 1 : 0;
    }
    async deploy(options) {
        var _a;
        const stacks = await this.selectStacksForDeploy(options.selector, options.exclusively);
        const requireApproval = (_a = options.requireApproval) !== null && _a !== void 0 ? _a : diff_1.RequireApproval.Broadening;
        const parameterMap = { '*': {} };
        for (const key in options.parameters) {
            if (options.parameters.hasOwnProperty(key)) {
                const [stack, parameter] = key.split(':', 2);
                if (!parameter) {
                    parameterMap['*'][stack] = options.parameters[key];
                }
                else {
                    if (!parameterMap[stack]) {
                        parameterMap[stack] = {};
                    }
                    parameterMap[stack][parameter] = options.parameters[key];
                }
            }
        }
        const stackOutputs = {};
        const outputsFile = options.outputsFile;
        for (const stack of stacks.stackArtifacts) {
            if (stacks.stackCount !== 1) {
                logging_1.highlight(stack.displayName);
            }
            if (!stack.environment) {
                // eslint-disable-next-line max-len
                throw new Error(`Stack ${stack.displayName} does not define an environment, and AWS credentials could not be obtained from standard locations or no region was configured.`);
            }
            if (Object.keys(stack.template.Resources || {}).length === 0) { // The generated stack has no resources
                if (!await this.props.cloudFormation.stackExists({ stack })) {
                    logging_1.warning('%s: stack has no resources, skipping deployment.', colors.bold(stack.displayName));
                }
                else {
                    logging_1.warning('%s: stack has no resources, deleting existing stack.', colors.bold(stack.displayName));
                    await this.destroy({
                        selector: { patterns: [stack.stackName] },
                        exclusively: true,
                        force: true,
                        roleArn: options.roleArn,
                        fromDeploy: true,
                    });
                }
                continue;
            }
            if (requireApproval !== diff_1.RequireApproval.Never) {
                const currentTemplate = await this.props.cloudFormation.readCurrentTemplate(stack);
                if (diff_1.printSecurityDiff(currentTemplate, stack, requireApproval)) {
                    // only talk to user if STDIN is a terminal (otherwise, fail)
                    if (!process.stdin.isTTY) {
                        throw new Error('"--require-approval" is enabled and stack includes security-sensitive updates, ' +
                            'but terminal (TTY) is not attached so we are unable to get a confirmation from the user');
                    }
                    const confirmed = await promptly.confirm('Do you wish to deploy these changes (y/n)?');
                    if (!confirmed) {
                        throw new Error('Aborted by user');
                    }
                }
            }
            logging_1.print('%s: deploying...', colors.bold(stack.displayName));
            let tags = options.tags;
            if (!tags || tags.length === 0) {
                tags = tagsForStack(stack);
            }
            try {
                const result = await this.props.cloudFormation.deployStack({
                    stack,
                    deployName: stack.stackName,
                    roleArn: options.roleArn,
                    toolkitStackName: options.toolkitStackName,
                    reuseAssets: options.reuseAssets,
                    notificationArns: options.notificationArns,
                    tags,
                    execute: options.execute,
                    changeSetName: options.changeSetName,
                    force: options.force,
                    parameters: Object.assign({}, parameterMap['*'], parameterMap[stack.stackName]),
                    usePreviousParameters: options.usePreviousParameters,
                    progress: options.progress,
                    ci: options.ci,
                });
                const message = result.noOp
                    ? ' ✅  %s (no changes)'
                    : ' ✅  %s';
                logging_1.success('\n' + message, stack.displayName);
                if (Object.keys(result.outputs).length > 0) {
                    logging_1.print('\nOutputs:');
                    stackOutputs[stack.stackName] = result.outputs;
                }
                for (const name of Object.keys(result.outputs).sort()) {
                    const value = result.outputs[name];
                    logging_1.print('%s.%s = %s', colors.cyan(stack.id), colors.cyan(name), colors.underline(colors.cyan(value)));
                }
                logging_1.print('\nStack ARN:');
                logging_1.data(result.stackArn);
            }
            catch (e) {
                logging_1.error('\n ❌  %s failed: %s', colors.bold(stack.displayName), e);
                throw e;
            }
            finally {
                // If an outputs file has been specified, create the file path and write stack outputs to it once.
                // Outputs are written after all stacks have been deployed. If a stack deployment fails,
                // all of the outputs from successfully deployed stacks before the failure will still be written.
                if (outputsFile) {
                    fs.ensureFileSync(outputsFile);
                    await fs.writeJson(outputsFile, stackOutputs, {
                        spaces: 2,
                        encoding: 'utf8',
                    });
                }
            }
        }
    }
    async destroy(options) {
        let stacks = await this.selectStacksForDestroy(options.selector, options.exclusively);
        // The stacks will have been ordered for deployment, so reverse them for deletion.
        stacks = stacks.reversed();
        if (!options.force) {
            // eslint-disable-next-line max-len
            const confirmed = await promptly.confirm(`Are you sure you want to delete: ${colors.blue(stacks.stackArtifacts.map(s => s.hierarchicalId).join(', '))} (y/n)?`);
            if (!confirmed) {
                return;
            }
        }
        const action = options.fromDeploy ? 'deploy' : 'destroy';
        for (const stack of stacks.stackArtifacts) {
            logging_1.success('%s: destroying...', colors.blue(stack.displayName));
            try {
                await this.props.cloudFormation.destroyStack({
                    stack,
                    deployName: stack.stackName,
                    roleArn: options.roleArn,
                });
                logging_1.success(`\n ✅  %s: ${action}ed`, colors.blue(stack.displayName));
            }
            catch (e) {
                logging_1.error(`\n ❌  %s: ${action} failed`, colors.blue(stack.displayName), e);
                throw e;
            }
        }
    }
    async list(selectors, options = {}) {
        const stacks = await this.selectStacksForList(selectors);
        // if we are in "long" mode, emit the array as-is (JSON/YAML)
        if (options.long) {
            const long = [];
            for (const stack of stacks.stackArtifacts) {
                long.push({
                    id: stack.id,
                    name: stack.stackName,
                    environment: stack.environment,
                });
            }
            return long; // will be YAML formatted output
        }
        // just print stack IDs
        for (const stack of stacks.stackArtifacts) {
            logging_1.data(stack.hierarchicalId);
        }
        return 0; // exit-code
    }
    /**
     * Synthesize the given set of stacks (called when the user runs 'cdk synth')
     *
     * INPUT: Stack names can be supplied using a glob filter. If no stacks are
     * given, all stacks from the application are implictly selected.
     *
     * OUTPUT: If more than one stack ends up being selected, an output directory
     * should be supplied, where the templates will be written.
     */
    async synth(stackNames, exclusively, quiet, autoValidate) {
        const stacks = await this.selectStacksForDiff(stackNames, exclusively, autoValidate);
        // if we have a single stack, print it to STDOUT
        if (stacks.stackCount === 1) {
            if (!quiet) {
                return stacks.firstStack.template;
            }
            return undefined;
        }
        // This is a slight hack; in integ mode we allow multiple stacks to be synthesized to stdout sequentially.
        // This is to make it so that we can support multi-stack integ test expectations, without so drastically
        // having to change the synthesis format that we have to rerun all integ tests.
        //
        // Because this feature is not useful to consumers (the output is missing
        // the stack names), it's not exposed as a CLI flag. Instead, it's hidden
        // behind an environment variable.
        const isIntegMode = process.env.CDK_INTEG_MODE === '1';
        if (isIntegMode) {
            return stacks.stackArtifacts.map(s => s.template);
        }
        // not outputting template to stdout, let's explain things to the user a little bit...
        logging_1.success(`Successfully synthesized to ${colors.blue(path.resolve(stacks.assembly.directory))}`);
        logging_1.print(`Supply a stack id (${stacks.stackArtifacts.map(s => colors.green(s.id)).join(', ')}) to display its template.`);
        return undefined;
    }
    /**
     * Bootstrap the CDK Toolkit stack in the accounts used by the specified stack(s).
     *
     * @param environmentSpecs environment names that need to have toolkit support
     *             provisioned, as a glob filter. If none is provided,
     *             all stacks are implicitly selected.
     * @param toolkitStackName the name to be used for the CDK Toolkit stack.
     */
    async bootstrap(userEnvironmentSpecs, bootstrapper, options) {
        // If there is an '--app' argument and an environment looks like a glob, we
        // select the environments from the app. Otherwise use what the user said.
        // By default glob for everything
        const environmentSpecs = userEnvironmentSpecs.length > 0 ? [...userEnvironmentSpecs] : ['**'];
        // Partition into globs and non-globs (this will mutate environmentSpecs).
        const globSpecs = util_2.partition(environmentSpecs, environments_1.looksLikeGlob);
        if (globSpecs.length > 0 && !this.props.cloudExecutable.hasApp) {
            if (userEnvironmentSpecs.length > 0) {
                // User did request this glob
                throw new Error(`'${globSpecs}' is not an environment name. Specify an environment name like 'aws://123456789012/us-east-1', or run in a directory with 'cdk.json' to use wildcards.`);
            }
            else {
                // User did not request anything
                throw new Error('Specify an environment name like \'aws://123456789012/us-east-1\', or run in a directory with \'cdk.json\'.');
            }
        }
        const environments = [
            ...environments_1.environmentsFromDescriptors(environmentSpecs),
        ];
        // If there is an '--app' argument, select the environments from the app.
        if (this.props.cloudExecutable.hasApp) {
            environments.push(...await environments_1.globEnvironmentsFromStacks(await this.selectStacksForList([]), globSpecs, this.props.sdkProvider));
        }
        await Promise.all(environments.map(async (environment) => {
            logging_1.success(' ⏳  Bootstrapping environment %s...', colors.blue(environment.name));
            try {
                const result = await bootstrapper.bootstrapEnvironment(environment, this.props.sdkProvider, options);
                const message = result.noOp
                    ? ' ✅  Environment %s bootstrapped (no changes).'
                    : ' ✅  Environment %s bootstrapped.';
                logging_1.success(message, colors.blue(environment.name));
            }
            catch (e) {
                logging_1.error(' ❌  Environment %s failed bootstrapping: %s', colors.blue(environment.name), e);
                throw e;
            }
        }));
    }
    async selectStacksForList(patterns) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks({ patterns }, { defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks });
        // No validation
        return stacks;
    }
    async selectStacksForDeploy(selector, exclusively) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks(selector, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Upstream,
            defaultBehavior: cloud_assembly_1.DefaultSelection.OnlySingle,
        });
        await this.validateStacks(stacks);
        return stacks;
    }
    async selectStacksForDiff(stackNames, exclusively, autoValidate) {
        const assembly = await this.assembly();
        const selectedForDiff = await assembly.selectStacks({ patterns: stackNames }, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Upstream,
            defaultBehavior: cloud_assembly_1.DefaultSelection.MainAssembly,
        });
        const allStacks = await this.selectStacksForList([]);
        const autoValidateStacks = autoValidate
            ? allStacks.filter(art => { var _a; return (_a = art.validateOnSynth) !== null && _a !== void 0 ? _a : false; })
            : new cloud_assembly_1.StackCollection(assembly, []);
        await this.validateStacks(selectedForDiff.concat(autoValidateStacks));
        return selectedForDiff;
    }
    async selectStacksForDestroy(selector, exclusively) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks(selector, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Downstream,
            defaultBehavior: cloud_assembly_1.DefaultSelection.OnlySingle,
        });
        // No validation
        return stacks;
    }
    /**
     * Validate the stacks for errors and warnings according to the CLI's current settings
     */
    async validateStacks(stacks) {
        stacks.processMetadataMessages({
            ignoreErrors: this.props.ignoreErrors,
            strict: this.props.strict,
            verbose: this.props.verbose,
        });
    }
    /**
     * Select a single stack by its name
     */
    async selectSingleStackByName(stackName) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks({ patterns: [stackName] }, {
            extend: cloud_assembly_1.ExtendedStackSelection.None,
            defaultBehavior: cloud_assembly_1.DefaultSelection.None,
        });
        // Could have been a glob so check that we evaluated to exactly one
        if (stacks.stackCount > 1) {
            throw new Error(`This command requires exactly one stack and we matched more than one: ${stacks.stackIds}`);
        }
        return assembly.stackById(stacks.firstStack.id);
    }
    assembly() {
        return this.props.cloudExecutable.synthesize();
    }
}
exports.CdkToolkit = CdkToolkit;
/**
 * @returns an array with the tags available in the stack metadata.
 */
function tagsForStack(stack) {
    return Object.entries(stack.tags).map(([Key, Value]) => ({ Key, Value }));
}
//# sourceMappingURL=data:application/json;base64,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