"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiContextProviderPlugin = void 0;
const cxapi = require("@aws-cdk/cx-api");
const api_1 = require("../api");
const logging_1 = require("../logging");
/**
 * Plugin to search AMIs for the current account
 */
class AmiContextProviderPlugin {
    constructor(aws) {
        this.aws = aws;
    }
    async getValue(args) {
        const region = args.region;
        const account = args.account;
        // Normally we'd do this only as 'debug', but searching AMIs typically takes dozens
        // of seconds, so be little more verbose about it so users know what is going on.
        logging_1.print(`Searching for AMI in ${account}:${region}`);
        logging_1.debug(`AMI search parameters: ${JSON.stringify(args)}`);
        const options = { assumeRoleArn: args.lookupRoleArn };
        const ec2 = (await this.aws.forEnvironment(cxapi.EnvironmentUtils.make(account, region), api_1.Mode.ForReading, options)).ec2();
        const response = await ec2.describeImages({
            Owners: args.owners,
            Filters: Object.entries(args.filters).map(([key, values]) => ({
                Name: key,
                Values: values,
            })),
        }).promise();
        const images = [...response.Images || []].filter(i => i.ImageId !== undefined);
        if (images.length === 0) {
            throw new Error('No AMI found that matched the search criteria');
        }
        // Return the most recent one
        // Note: Date.parse() is not going to respect the timezone of the string,
        // but since we only care about the relative values that is okay.
        images.sort(descending(i => Date.parse(i.CreationDate || '1970')));
        logging_1.debug(`Selected image '${images[0].ImageId}' created at '${images[0].CreationDate}'`);
        return images[0].ImageId;
    }
}
exports.AmiContextProviderPlugin = AmiContextProviderPlugin;
/**
 * Make a comparator that sorts in descending order given a sort key extractor
 */
function descending(valueOf) {
    return (a, b) => {
        return valueOf(b) - valueOf(a);
    };
}
//# sourceMappingURL=data:application/json;base64,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