"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PluginHost = void 0;
const safe_1 = require("colors/safe");
const logging_1 = require("./logging");
/**
 * A utility to manage plug-ins.
 *
 */
class PluginHost {
    constructor() {
        /**
         * Access the currently registered CredentialProviderSources. New sources can
         * be registered using the +registerCredentialProviderSource+ method.
         */
        this.credentialProviderSources = new Array();
        if (PluginHost.instance && PluginHost.instance !== this) {
            throw new Error('New instances of PluginHost must not be built. Use PluginHost.instance instead!');
        }
    }
    /**
     * Loads a plug-in into this PluginHost.
     *
     * @param moduleSpec the specification (path or name) of the plug-in module to be loaded.
     */
    load(moduleSpec) {
        try {
            /* eslint-disable @typescript-eslint/no-require-imports */
            const plugin = require(moduleSpec);
            /* eslint-enable */
            if (!isPlugin(plugin)) {
                logging_1.error(`Module ${safe_1.green(moduleSpec)} is not a valid plug-in, or has an unsupported version.`);
                throw new Error(`Module ${moduleSpec} does not define a valid plug-in.`);
            }
            if (plugin.init) {
                plugin.init(PluginHost.instance);
            }
        }
        catch (e) {
            logging_1.error(`Unable to load ${safe_1.green(moduleSpec)}: ${e.stack}`);
            throw new Error(`Unable to load plug-in: ${moduleSpec}`);
        }
        function isPlugin(x) {
            return x != null && x.version === '1';
        }
    }
    /**
     * Allows plug-ins to register new CredentialProviderSources.
     *
     * @param source a new CredentialProviderSource to register.
     */
    registerCredentialProviderSource(source) {
        this.credentialProviderSources.push(source);
    }
}
exports.PluginHost = PluginHost;
PluginHost.instance = new PluginHost();
//# sourceMappingURL=data:application/json;base64,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