"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const account_cache_1 = require("../lib/api/aws-auth/account-cache");
const util_1 = require("./util");
async function makeCache() {
    const dir = await fs.mkdtemp('/tmp/account-cache-test');
    const file = path.join(dir, 'cache.json');
    return {
        cacheDir: dir,
        cacheFile: file,
        cache: new account_cache_1.AccountAccessKeyCache(file),
    };
}
async function nukeCache(cacheDir) {
    await fs.remove(cacheDir);
}
test('default account cache uses CDK_HOME', () => {
    process.env.CDK_HOME = '/banana';
    const cache = new account_cache_1.AccountAccessKeyCache();
    expect(cache.cacheFile).toContain('/banana/');
});
test('account cache does not fail when given a nonwritable directory', async () => {
    const accessError = new Error('Oh no');
    accessError.code = 'EACCES';
    return util_1.withMocked(fs, 'mkdirs', async (mkdirs) => {
        // Have to do this because mkdirs has 2 overloads and it confuses TypeScript
        mkdirs.mockRejectedValue(accessError);
        const cache = new account_cache_1.AccountAccessKeyCache('/abc/xyz');
        await cache.fetch('xyz', () => Promise.resolve({ accountId: 'asdf', partition: 'swa' }));
        // No exception
    });
});
test('get(k) when cache is empty', async () => {
    const { cacheDir, cacheFile, cache } = await makeCache();
    try {
        expect(await cache.get('foo')).toBeUndefined();
        expect(await fs.pathExists(cacheFile)).toBeFalsy();
    }
    finally {
        await nukeCache(cacheDir);
    }
});
test('put(k,v) and then get(k)', async () => {
    const { cacheDir, cacheFile, cache } = await makeCache();
    try {
        await cache.put('key', { accountId: 'value', partition: 'aws' });
        await cache.put('boo', { accountId: 'bar', partition: 'aws' });
        expect(await cache.get('key')).toEqual({ accountId: 'value', partition: 'aws' });
        // create another cache instance on the same file, should still work
        const cache2 = new account_cache_1.AccountAccessKeyCache(cacheFile);
        expect(await cache2.get('boo')).toEqual({ accountId: 'bar', partition: 'aws' });
        // whitebox: read the file
        expect(await fs.readJson(cacheFile)).toEqual({
            key: { accountId: 'value', partition: 'aws' },
            boo: { accountId: 'bar', partition: 'aws' },
        });
    }
    finally {
        await nukeCache(cacheDir);
    }
});
test('fetch(k, resolver) can be used to "atomically" get + resolve + put', async () => {
    const { cacheDir, cache } = await makeCache();
    try {
        expect(await cache.get('foo')).toBeUndefined();
        expect(await cache.fetch('foo', async () => ({ accountId: 'bar', partition: 'aws' }))).toEqual({ accountId: 'bar', partition: 'aws' });
        expect(await cache.get('foo')).toEqual({ accountId: 'bar', partition: 'aws' });
    }
    finally {
        await nukeCache(cacheDir);
    }
});
test(`cache is nuked if it exceeds ${account_cache_1.AccountAccessKeyCache.MAX_ENTRIES} entries`, async () => {
    // This makes a lot of promises, so it can queue for a while...
    jest.setTimeout(30000);
    const { cacheDir, cacheFile, cache } = await makeCache();
    try {
        for (let i = 0; i < account_cache_1.AccountAccessKeyCache.MAX_ENTRIES; ++i) {
            await cache.put(`key${i}`, { accountId: `value${i}`, partition: 'aws' });
        }
        // verify all values are on disk
        const otherCache = new account_cache_1.AccountAccessKeyCache(cacheFile);
        for (let i = 0; i < account_cache_1.AccountAccessKeyCache.MAX_ENTRIES; ++i) {
            expect(await otherCache.get(`key${i}`)).toEqual({ accountId: `value${i}`, partition: 'aws' });
        }
        // add another value
        await cache.put('nuke-me', { accountId: 'genesis', partition: 'aws' });
        // now, we expect only `nuke-me` to exist on disk
        expect(await otherCache.get('nuke-me')).toEqual({ accountId: 'genesis', partition: 'aws' });
        for (let i = 0; i < account_cache_1.AccountAccessKeyCache.MAX_ENTRIES; ++i) {
            expect(await otherCache.get(`key${i}`)).toBeUndefined();
        }
    }
    finally {
        await nukeCache(cacheDir);
    }
});
test('cache pretends to be empty if cache file does not contain JSON', async () => {
    const { cacheDir, cacheFile, cache } = await makeCache();
    try {
        await fs.writeFile(cacheFile, '');
        await expect(cache.get('abc')).resolves.toEqual(undefined);
    }
    finally {
        await nukeCache(cacheDir);
    }
});
//# sourceMappingURL=data:application/json;base64,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