"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const cdk_1 = require("../helpers/cdk");
const test_helpers_1 = require("../helpers/test-helpers");
const timeout = process.env.CODEBUILD_BUILD_ID ? // if the process is running in CodeBuild
    3600000 : // 1 hour
    600000; // 10 minutes
jest.setTimeout(timeout);
process.stdout.write(`bootstrapping.integtest.ts: Setting jest time out to ${timeout} ms`);
test_helpers_1.integTest('can bootstrap without execution', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapLegacy({
        toolkitStackName: bootstrapStackName,
        noExecute: true,
    });
    const resp = await fixture.aws.cloudFormation('describeStacks', {
        StackName: bootstrapStackName,
    });
    expect((_a = resp.Stacks) === null || _a === void 0 ? void 0 : _a[0].StackStatus).toEqual('REVIEW_IN_PROGRESS');
}));
test_helpers_1.integTest('upgrade legacy bootstrap stack to new bootstrap stack while in use', cdk_1.withDefaultFixture(async (fixture) => {
    const bootstrapStackName = fixture.bootstrapStackName;
    const legacyBootstrapBucketName = `aws-cdk-bootstrap-integ-test-legacy-bckt-${cdk_1.randomString()}`;
    const newBootstrapBucketName = `aws-cdk-bootstrap-integ-test-v2-bckt-${cdk_1.randomString()}`;
    fixture.rememberToDeleteBucket(legacyBootstrapBucketName); // This one will leak
    fixture.rememberToDeleteBucket(newBootstrapBucketName); // This one shouldn't leak if the test succeeds, but let's be safe in case it doesn't
    // Legacy bootstrap
    await fixture.cdkBootstrapLegacy({
        toolkitStackName: bootstrapStackName,
        bootstrapBucketName: legacyBootstrapBucketName,
    });
    // Deploy stack that uses file assets
    await fixture.cdkDeploy('lambda', {
        options: ['--toolkit-stack-name', bootstrapStackName],
    });
    // Upgrade bootstrap stack to "new" style
    await fixture.cdkBootstrapModern({
        toolkitStackName: bootstrapStackName,
        bootstrapBucketName: newBootstrapBucketName,
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
    // (Force) deploy stack again
    // --force to bypass the check which says that the template hasn't changed.
    await fixture.cdkDeploy('lambda', {
        options: [
            '--toolkit-stack-name', bootstrapStackName,
            '--force',
        ],
    });
}));
test_helpers_1.integTest('can and deploy if omitting execution policies', cdk_1.withDefaultFixture(async (fixture) => {
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapModern({
        toolkitStackName: bootstrapStackName,
    });
    // Deploy stack that uses file assets
    await fixture.cdkDeploy('lambda', {
        options: [
            '--toolkit-stack-name', bootstrapStackName,
            '--context', `@aws-cdk/core:bootstrapQualifier=${fixture.qualifier}`,
            '--context', '@aws-cdk/core:newStyleStackSynthesis=1',
        ],
    });
}));
test_helpers_1.integTest('deploy new style synthesis to new style bootstrap', cdk_1.withDefaultFixture(async (fixture) => {
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapModern({
        toolkitStackName: bootstrapStackName,
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
    // Deploy stack that uses file assets
    await fixture.cdkDeploy('lambda', {
        options: [
            '--toolkit-stack-name', bootstrapStackName,
            '--context', `@aws-cdk/core:bootstrapQualifier=${fixture.qualifier}`,
            '--context', '@aws-cdk/core:newStyleStackSynthesis=1',
        ],
    });
}));
test_helpers_1.integTest('deploy new style synthesis to new style bootstrap (with docker image)', cdk_1.withDefaultFixture(async (fixture) => {
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapModern({
        toolkitStackName: bootstrapStackName,
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
    // Deploy stack that uses file assets
    await fixture.cdkDeploy('docker', {
        options: [
            '--toolkit-stack-name', bootstrapStackName,
            '--context', `@aws-cdk/core:bootstrapQualifier=${fixture.qualifier}`,
            '--context', '@aws-cdk/core:newStyleStackSynthesis=1',
        ],
    });
}));
test_helpers_1.integTest('deploy old style synthesis to new style bootstrap', cdk_1.withDefaultFixture(async (fixture) => {
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapModern({
        toolkitStackName: bootstrapStackName,
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
    // Deploy stack that uses file assets
    await fixture.cdkDeploy('lambda', {
        options: [
            '--toolkit-stack-name', bootstrapStackName,
        ],
    });
}));
test_helpers_1.integTest('deploying new style synthesis to old style bootstrap fails', cdk_1.withDefaultFixture(async (fixture) => {
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapLegacy({
        toolkitStackName: bootstrapStackName,
    });
    // Deploy stack that uses file assets, this fails because the bootstrap stack
    // is version checked.
    await expect(fixture.cdkDeploy('lambda', {
        options: [
            '--toolkit-stack-name', bootstrapStackName,
            '--context', '@aws-cdk/core:newStyleStackSynthesis=1',
        ],
    })).rejects.toThrow('exited with error');
}));
test_helpers_1.integTest('can create a legacy bootstrap stack with --public-access-block-configuration=false', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapLegacy({
        verbose: true,
        toolkitStackName: bootstrapStackName,
        publicAccessBlockConfiguration: false,
        tags: 'Foo=Bar',
    });
    const response = await fixture.aws.cloudFormation('describeStacks', { StackName: bootstrapStackName });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].Tags).toEqual([
        { Key: 'Foo', Value: 'Bar' },
    ]);
}));
test_helpers_1.integTest('can create multiple legacy bootstrap stacks', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const bootstrapStackName1 = `${fixture.bootstrapStackName}-1`;
    const bootstrapStackName2 = `${fixture.bootstrapStackName}-2`;
    // deploy two toolkit stacks into the same environment (see #1416)
    // one with tags
    await fixture.cdkBootstrapLegacy({
        verbose: true,
        toolkitStackName: bootstrapStackName1,
        tags: 'Foo=Bar',
    });
    await fixture.cdkBootstrapLegacy({
        verbose: true,
        toolkitStackName: bootstrapStackName2,
    });
    const response = await fixture.aws.cloudFormation('describeStacks', { StackName: bootstrapStackName1 });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].Tags).toEqual([
        { Key: 'Foo', Value: 'Bar' },
    ]);
}));
test_helpers_1.integTest('can dump the template, modify and use it to deploy a custom bootstrap stack', cdk_1.withDefaultFixture(async (fixture) => {
    let template = await fixture.cdkBootstrapModern({
        // toolkitStackName doesn't matter for this particular invocation
        toolkitStackName: fixture.bootstrapStackName,
        showTemplate: true,
        cliOptions: {
            captureStderr: false,
        },
    });
    expect(template).toContain('BootstrapVersion:');
    template += '\n' + [
        '  TwiddleDee:',
        '    Value: Template got twiddled',
    ].join('\n');
    const filename = path.join(fixture.integTestDir, `${fixture.qualifier}-template.yaml`);
    fs.writeFileSync(filename, template, { encoding: 'utf-8' });
    await fixture.cdkBootstrapModern({
        toolkitStackName: fixture.bootstrapStackName,
        template: filename,
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
}));
test_helpers_1.integTest('switch on termination protection, switch is left alone on re-bootstrap', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapModern({
        verbose: true,
        toolkitStackName: bootstrapStackName,
        terminationProtection: true,
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
    await fixture.cdkBootstrapModern({
        verbose: true,
        toolkitStackName: bootstrapStackName,
        force: true,
    });
    const response = await fixture.aws.cloudFormation('describeStacks', { StackName: bootstrapStackName });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].EnableTerminationProtection).toEqual(true);
}));
test_helpers_1.integTest('add tags, left alone on re-bootstrap', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapModern({
        verbose: true,
        toolkitStackName: bootstrapStackName,
        tags: 'Foo=Bar',
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
    await fixture.cdkBootstrapModern({
        verbose: true,
        toolkitStackName: bootstrapStackName,
        force: true,
    });
    const response = await fixture.aws.cloudFormation('describeStacks', { StackName: bootstrapStackName });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].Tags).toEqual([
        { Key: 'Foo', Value: 'Bar' },
    ]);
}));
test_helpers_1.integTest('can deploy modern-synthesized stack even if bootstrap stack name is unknown', cdk_1.withDefaultFixture(async (fixture) => {
    const bootstrapStackName = fixture.bootstrapStackName;
    await fixture.cdkBootstrapModern({
        toolkitStackName: bootstrapStackName,
        cfnExecutionPolicy: 'arn:aws:iam::aws:policy/AdministratorAccess',
    });
    // Deploy stack that uses file assets
    await fixture.cdkDeploy('lambda', {
        options: [
            // Explicity pass a name that's sure to not exist, otherwise the CLI might accidentally find a
            // default bootstracp stack if that happens to be in the account already.
            '--toolkit-stack-name', 'DefinitelyDoesNotExist',
            '--context', `@aws-cdk/core:bootstrapQualifier=${fixture.qualifier}`,
            '--context', '@aws-cdk/core:newStyleStackSynthesis=1',
        ],
    });
}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYm9vdHN0cmFwcGluZy5pbnRlZ3Rlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJib290c3RyYXBwaW5nLmludGVndGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFDN0Isd0NBQWtFO0FBQ2xFLDBEQUFvRDtBQUVwRCxNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyx5Q0FBeUM7SUFDeEYsT0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTO0lBQ3JCLE1BQU8sQ0FBQyxDQUFDLGFBQWE7QUFDeEIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztBQUN6QixPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyx3REFBd0QsT0FBTyxLQUFLLENBQUMsQ0FBQztBQUUzRix3QkFBUyxDQUFDLGlDQUFpQyxFQUFFLHdCQUFrQixDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTs7SUFDaEYsTUFBTSxrQkFBa0IsR0FBRyxPQUFPLENBQUMsa0JBQWtCLENBQUM7SUFFdEQsTUFBTSxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDL0IsZ0JBQWdCLEVBQUUsa0JBQWtCO1FBQ3BDLFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUMsQ0FBQztJQUVILE1BQU0sSUFBSSxHQUFHLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUU7UUFDOUQsU0FBUyxFQUFFLGtCQUFrQjtLQUM5QixDQUFDLENBQUM7SUFFSCxNQUFNLE9BQUMsSUFBSSxDQUFDLE1BQU0sMENBQUcsQ0FBQyxFQUFFLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0FBQ3JFLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFSix3QkFBUyxDQUFDLG9FQUFvRSxFQUFFLHdCQUFrQixDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtJQUNuSCxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLHlCQUF5QixHQUFHLDRDQUE0QyxrQkFBWSxFQUFFLEVBQUUsQ0FBQztJQUMvRixNQUFNLHNCQUFzQixHQUFHLHdDQUF3QyxrQkFBWSxFQUFFLEVBQUUsQ0FBQztJQUN4RixPQUFPLENBQUMsc0JBQXNCLENBQUMseUJBQXlCLENBQUMsQ0FBQyxDQUFDLHFCQUFxQjtJQUNoRixPQUFPLENBQUMsc0JBQXNCLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDLHFGQUFxRjtJQUU3SSxtQkFBbUI7SUFDbkIsTUFBTSxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDL0IsZ0JBQWdCLEVBQUUsa0JBQWtCO1FBQ3BDLG1CQUFtQixFQUFFLHlCQUF5QjtLQUMvQyxDQUFDLENBQUM7SUFFSCxxQ0FBcUM7SUFDckMsTUFBTSxPQUFPLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRTtRQUNoQyxPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxrQkFBa0IsQ0FBQztLQUN0RCxDQUFDLENBQUM7SUFFSCx5Q0FBeUM7SUFDekMsTUFBTSxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDL0IsZ0JBQWdCLEVBQUUsa0JBQWtCO1FBQ3BDLG1CQUFtQixFQUFFLHNCQUFzQjtRQUMzQyxrQkFBa0IsRUFBRSw2Q0FBNkM7S0FDbEUsQ0FBQyxDQUFDO0lBRUgsNkJBQTZCO0lBQzdCLDJFQUEyRTtJQUMzRSxNQUFNLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFO1FBQ2hDLE9BQU8sRUFBRTtZQUNQLHNCQUFzQixFQUFFLGtCQUFrQjtZQUMxQyxTQUFTO1NBQ1Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBRUosd0JBQVMsQ0FBQywrQ0FBK0MsRUFBRSx3QkFBa0IsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUU7SUFDOUYsTUFBTSxrQkFBa0IsR0FBRyxPQUFPLENBQUMsa0JBQWtCLENBQUM7SUFFdEQsTUFBTSxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDL0IsZ0JBQWdCLEVBQUUsa0JBQWtCO0tBQ3JDLENBQUMsQ0FBQztJQUVILHFDQUFxQztJQUNyQyxNQUFNLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFO1FBQ2hDLE9BQU8sRUFBRTtZQUNQLHNCQUFzQixFQUFFLGtCQUFrQjtZQUMxQyxXQUFXLEVBQUUsb0NBQW9DLE9BQU8sQ0FBQyxTQUFTLEVBQUU7WUFDcEUsV0FBVyxFQUFFLHdDQUF3QztTQUN0RDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFSix3QkFBUyxDQUFDLG1EQUFtRCxFQUFFLHdCQUFrQixDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtJQUNsRyxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixnQkFBZ0IsRUFBRSxrQkFBa0I7UUFDcEMsa0JBQWtCLEVBQUUsNkNBQTZDO0tBQ2xFLENBQUMsQ0FBQztJQUVILHFDQUFxQztJQUNyQyxNQUFNLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFO1FBQ2hDLE9BQU8sRUFBRTtZQUNQLHNCQUFzQixFQUFFLGtCQUFrQjtZQUMxQyxXQUFXLEVBQUUsb0NBQW9DLE9BQU8sQ0FBQyxTQUFTLEVBQUU7WUFDcEUsV0FBVyxFQUFFLHdDQUF3QztTQUN0RDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFSix3QkFBUyxDQUFDLHVFQUF1RSxFQUFFLHdCQUFrQixDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtJQUN0SCxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixnQkFBZ0IsRUFBRSxrQkFBa0I7UUFDcEMsa0JBQWtCLEVBQUUsNkNBQTZDO0tBQ2xFLENBQUMsQ0FBQztJQUVILHFDQUFxQztJQUNyQyxNQUFNLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFO1FBQ2hDLE9BQU8sRUFBRTtZQUNQLHNCQUFzQixFQUFFLGtCQUFrQjtZQUMxQyxXQUFXLEVBQUUsb0NBQW9DLE9BQU8sQ0FBQyxTQUFTLEVBQUU7WUFDcEUsV0FBVyxFQUFFLHdDQUF3QztTQUN0RDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFSix3QkFBUyxDQUFDLG1EQUFtRCxFQUFFLHdCQUFrQixDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtJQUNsRyxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixnQkFBZ0IsRUFBRSxrQkFBa0I7UUFDcEMsa0JBQWtCLEVBQUUsNkNBQTZDO0tBQ2xFLENBQUMsQ0FBQztJQUVILHFDQUFxQztJQUNyQyxNQUFNLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFO1FBQ2hDLE9BQU8sRUFBRTtZQUNQLHNCQUFzQixFQUFFLGtCQUFrQjtTQUMzQztLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFSix3QkFBUyxDQUFDLDREQUE0RCxFQUFFLHdCQUFrQixDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtJQUMzRyxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixnQkFBZ0IsRUFBRSxrQkFBa0I7S0FDckMsQ0FBQyxDQUFDO0lBRUgsNkVBQTZFO0lBQzdFLHNCQUFzQjtJQUN0QixNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRTtRQUN2QyxPQUFPLEVBQUU7WUFDUCxzQkFBc0IsRUFBRSxrQkFBa0I7WUFDMUMsV0FBVyxFQUFFLHdDQUF3QztTQUN0RDtLQUNGLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsQ0FBQztBQUMzQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBRUosd0JBQVMsQ0FBQyxvRkFBb0YsRUFBRSx3QkFBa0IsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUU7O0lBQ25JLE1BQU0sa0JBQWtCLEdBQUcsT0FBTyxDQUFDLGtCQUFrQixDQUFDO0lBRXRELE1BQU0sT0FBTyxDQUFDLGtCQUFrQixDQUFDO1FBQy9CLE9BQU8sRUFBRSxJQUFJO1FBQ2IsZ0JBQWdCLEVBQUUsa0JBQWtCO1FBQ3BDLDhCQUE4QixFQUFFLEtBQUs7UUFDckMsSUFBSSxFQUFFLFNBQVM7S0FDaEIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLFNBQVMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7SUFDdkcsTUFBTSxPQUFDLFFBQVEsQ0FBQyxNQUFNLDBDQUFHLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDeEMsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUU7S0FDN0IsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUVKLHdCQUFTLENBQUMsNkNBQTZDLEVBQUUsd0JBQWtCLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFOztJQUM1RixNQUFNLG1CQUFtQixHQUFHLEdBQUcsT0FBTyxDQUFDLGtCQUFrQixJQUFJLENBQUM7SUFDOUQsTUFBTSxtQkFBbUIsR0FBRyxHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsSUFBSSxDQUFDO0lBRTlELGtFQUFrRTtJQUNsRSxnQkFBZ0I7SUFDaEIsTUFBTSxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDL0IsT0FBTyxFQUFFLElBQUk7UUFDYixnQkFBZ0IsRUFBRSxtQkFBbUI7UUFDckMsSUFBSSxFQUFFLFNBQVM7S0FDaEIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDL0IsT0FBTyxFQUFFLElBQUk7UUFDYixnQkFBZ0IsRUFBRSxtQkFBbUI7S0FDdEMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLFNBQVMsRUFBRSxtQkFBbUIsRUFBRSxDQUFDLENBQUM7SUFDeEcsTUFBTSxPQUFDLFFBQVEsQ0FBQyxNQUFNLDBDQUFHLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDeEMsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUU7S0FDN0IsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUVKLHdCQUFTLENBQUMsNkVBQTZFLEVBQUUsd0JBQWtCLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFO0lBQzVILElBQUksUUFBUSxHQUFHLE1BQU0sT0FBTyxDQUFDLGtCQUFrQixDQUFDO1FBQzlDLGlFQUFpRTtRQUNqRSxnQkFBZ0IsRUFBRSxPQUFPLENBQUMsa0JBQWtCO1FBQzVDLFlBQVksRUFBRSxJQUFJO1FBQ2xCLFVBQVUsRUFBRTtZQUNWLGFBQWEsRUFBRSxLQUFLO1NBQ3JCO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBRWhELFFBQVEsSUFBSSxJQUFJLEdBQUc7UUFDakIsZUFBZTtRQUNmLGtDQUFrQztLQUNuQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUViLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxHQUFHLE9BQU8sQ0FBQyxTQUFTLGdCQUFnQixDQUFDLENBQUM7SUFDdkYsRUFBRSxDQUFDLGFBQWEsQ0FBQyxRQUFRLEVBQUUsUUFBUSxFQUFFLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDNUQsTUFBTSxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDL0IsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLGtCQUFrQjtRQUM1QyxRQUFRLEVBQUUsUUFBUTtRQUNsQixrQkFBa0IsRUFBRSw2Q0FBNkM7S0FDbEUsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUVKLHdCQUFTLENBQUMsd0VBQXdFLEVBQUUsd0JBQWtCLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFOztJQUN2SCxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixPQUFPLEVBQUUsSUFBSTtRQUNiLGdCQUFnQixFQUFFLGtCQUFrQjtRQUNwQyxxQkFBcUIsRUFBRSxJQUFJO1FBQzNCLGtCQUFrQixFQUFFLDZDQUE2QztLQUNsRSxDQUFDLENBQUM7SUFDSCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixPQUFPLEVBQUUsSUFBSTtRQUNiLGdCQUFnQixFQUFFLGtCQUFrQjtRQUNwQyxLQUFLLEVBQUUsSUFBSTtLQUNaLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxTQUFTLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZHLE1BQU0sT0FBQyxRQUFRLENBQUMsTUFBTSwwQ0FBRyxDQUFDLEVBQUUsMkJBQTJCLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDekUsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUVKLHdCQUFTLENBQUMsc0NBQXNDLEVBQUUsd0JBQWtCLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFOztJQUNyRixNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixPQUFPLEVBQUUsSUFBSTtRQUNiLGdCQUFnQixFQUFFLGtCQUFrQjtRQUNwQyxJQUFJLEVBQUUsU0FBUztRQUNmLGtCQUFrQixFQUFFLDZDQUE2QztLQUNsRSxDQUFDLENBQUM7SUFDSCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixPQUFPLEVBQUUsSUFBSTtRQUNiLGdCQUFnQixFQUFFLGtCQUFrQjtRQUNwQyxLQUFLLEVBQUUsSUFBSTtLQUNaLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxTQUFTLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZHLE1BQU0sT0FBQyxRQUFRLENBQUMsTUFBTSwwQ0FBRyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ3hDLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFO0tBQzdCLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFSix3QkFBUyxDQUFDLDZFQUE2RSxFQUFFLHdCQUFrQixDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtJQUM1SCxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztJQUV0RCxNQUFNLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztRQUMvQixnQkFBZ0IsRUFBRSxrQkFBa0I7UUFDcEMsa0JBQWtCLEVBQUUsNkNBQTZDO0tBQ2xFLENBQUMsQ0FBQztJQUVILHFDQUFxQztJQUNyQyxNQUFNLE9BQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFO1FBQ2hDLE9BQU8sRUFBRTtZQUNQLDhGQUE4RjtZQUM5Rix5RUFBeUU7WUFDekUsc0JBQXNCLEVBQUUsd0JBQXdCO1lBQ2hELFdBQVcsRUFBRSxvQ0FBb0MsT0FBTyxDQUFDLFNBQVMsRUFBRTtZQUNwRSxXQUFXLEVBQUUsd0NBQXdDO1NBQ3REO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyByYW5kb21TdHJpbmcsIHdpdGhEZWZhdWx0Rml4dHVyZSB9IGZyb20gJy4uL2hlbHBlcnMvY2RrJztcbmltcG9ydCB7IGludGVnVGVzdCB9IGZyb20gJy4uL2hlbHBlcnMvdGVzdC1oZWxwZXJzJztcblxuY29uc3QgdGltZW91dCA9IHByb2Nlc3MuZW52LkNPREVCVUlMRF9CVUlMRF9JRCA/IC8vIGlmIHRoZSBwcm9jZXNzIGlzIHJ1bm5pbmcgaW4gQ29kZUJ1aWxkXG4gIDNfNjAwXzAwMCA6IC8vIDEgaG91clxuICA2MDBfMDAwOyAvLyAxMCBtaW51dGVzXG5qZXN0LnNldFRpbWVvdXQodGltZW91dCk7XG5wcm9jZXNzLnN0ZG91dC53cml0ZShgYm9vdHN0cmFwcGluZy5pbnRlZ3Rlc3QudHM6IFNldHRpbmcgamVzdCB0aW1lIG91dCB0byAke3RpbWVvdXR9IG1zYCk7XG5cbmludGVnVGVzdCgnY2FuIGJvb3RzdHJhcCB3aXRob3V0IGV4ZWN1dGlvbicsIHdpdGhEZWZhdWx0Rml4dHVyZShhc3luYyAoZml4dHVyZSkgPT4ge1xuICBjb25zdCBib290c3RyYXBTdGFja05hbWUgPSBmaXh0dXJlLmJvb3RzdHJhcFN0YWNrTmFtZTtcblxuICBhd2FpdCBmaXh0dXJlLmNka0Jvb3RzdHJhcExlZ2FjeSh7XG4gICAgdG9vbGtpdFN0YWNrTmFtZTogYm9vdHN0cmFwU3RhY2tOYW1lLFxuICAgIG5vRXhlY3V0ZTogdHJ1ZSxcbiAgfSk7XG5cbiAgY29uc3QgcmVzcCA9IGF3YWl0IGZpeHR1cmUuYXdzLmNsb3VkRm9ybWF0aW9uKCdkZXNjcmliZVN0YWNrcycsIHtcbiAgICBTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgfSk7XG5cbiAgZXhwZWN0KHJlc3AuU3RhY2tzPy5bMF0uU3RhY2tTdGF0dXMpLnRvRXF1YWwoJ1JFVklFV19JTl9QUk9HUkVTUycpO1xufSkpO1xuXG5pbnRlZ1Rlc3QoJ3VwZ3JhZGUgbGVnYWN5IGJvb3RzdHJhcCBzdGFjayB0byBuZXcgYm9vdHN0cmFwIHN0YWNrIHdoaWxlIGluIHVzZScsIHdpdGhEZWZhdWx0Rml4dHVyZShhc3luYyAoZml4dHVyZSkgPT4ge1xuICBjb25zdCBib290c3RyYXBTdGFja05hbWUgPSBmaXh0dXJlLmJvb3RzdHJhcFN0YWNrTmFtZTtcblxuICBjb25zdCBsZWdhY3lCb290c3RyYXBCdWNrZXROYW1lID0gYGF3cy1jZGstYm9vdHN0cmFwLWludGVnLXRlc3QtbGVnYWN5LWJja3QtJHtyYW5kb21TdHJpbmcoKX1gO1xuICBjb25zdCBuZXdCb290c3RyYXBCdWNrZXROYW1lID0gYGF3cy1jZGstYm9vdHN0cmFwLWludGVnLXRlc3QtdjItYmNrdC0ke3JhbmRvbVN0cmluZygpfWA7XG4gIGZpeHR1cmUucmVtZW1iZXJUb0RlbGV0ZUJ1Y2tldChsZWdhY3lCb290c3RyYXBCdWNrZXROYW1lKTsgLy8gVGhpcyBvbmUgd2lsbCBsZWFrXG4gIGZpeHR1cmUucmVtZW1iZXJUb0RlbGV0ZUJ1Y2tldChuZXdCb290c3RyYXBCdWNrZXROYW1lKTsgLy8gVGhpcyBvbmUgc2hvdWxkbid0IGxlYWsgaWYgdGhlIHRlc3Qgc3VjY2VlZHMsIGJ1dCBsZXQncyBiZSBzYWZlIGluIGNhc2UgaXQgZG9lc24ndFxuXG4gIC8vIExlZ2FjeSBib290c3RyYXBcbiAgYXdhaXQgZml4dHVyZS5jZGtCb290c3RyYXBMZWdhY3koe1xuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBib290c3RyYXBCdWNrZXROYW1lOiBsZWdhY3lCb290c3RyYXBCdWNrZXROYW1lLFxuICB9KTtcblxuICAvLyBEZXBsb3kgc3RhY2sgdGhhdCB1c2VzIGZpbGUgYXNzZXRzXG4gIGF3YWl0IGZpeHR1cmUuY2RrRGVwbG95KCdsYW1iZGEnLCB7XG4gICAgb3B0aW9uczogWyctLXRvb2xraXQtc3RhY2stbmFtZScsIGJvb3RzdHJhcFN0YWNrTmFtZV0sXG4gIH0pO1xuXG4gIC8vIFVwZ3JhZGUgYm9vdHN0cmFwIHN0YWNrIHRvIFwibmV3XCIgc3R5bGVcbiAgYXdhaXQgZml4dHVyZS5jZGtCb290c3RyYXBNb2Rlcm4oe1xuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBib290c3RyYXBCdWNrZXROYW1lOiBuZXdCb290c3RyYXBCdWNrZXROYW1lLFxuICAgIGNmbkV4ZWN1dGlvblBvbGljeTogJ2Fybjphd3M6aWFtOjphd3M6cG9saWN5L0FkbWluaXN0cmF0b3JBY2Nlc3MnLFxuICB9KTtcblxuICAvLyAoRm9yY2UpIGRlcGxveSBzdGFjayBhZ2FpblxuICAvLyAtLWZvcmNlIHRvIGJ5cGFzcyB0aGUgY2hlY2sgd2hpY2ggc2F5cyB0aGF0IHRoZSB0ZW1wbGF0ZSBoYXNuJ3QgY2hhbmdlZC5cbiAgYXdhaXQgZml4dHVyZS5jZGtEZXBsb3koJ2xhbWJkYScsIHtcbiAgICBvcHRpb25zOiBbXG4gICAgICAnLS10b29sa2l0LXN0YWNrLW5hbWUnLCBib290c3RyYXBTdGFja05hbWUsXG4gICAgICAnLS1mb3JjZScsXG4gICAgXSxcbiAgfSk7XG59KSk7XG5cbmludGVnVGVzdCgnY2FuIGFuZCBkZXBsb3kgaWYgb21pdHRpbmcgZXhlY3V0aW9uIHBvbGljaWVzJywgd2l0aERlZmF1bHRGaXh0dXJlKGFzeW5jIChmaXh0dXJlKSA9PiB7XG4gIGNvbnN0IGJvb3RzdHJhcFN0YWNrTmFtZSA9IGZpeHR1cmUuYm9vdHN0cmFwU3RhY2tOYW1lO1xuXG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTW9kZXJuKHtcbiAgICB0b29sa2l0U3RhY2tOYW1lOiBib290c3RyYXBTdGFja05hbWUsXG4gIH0pO1xuXG4gIC8vIERlcGxveSBzdGFjayB0aGF0IHVzZXMgZmlsZSBhc3NldHNcbiAgYXdhaXQgZml4dHVyZS5jZGtEZXBsb3koJ2xhbWJkYScsIHtcbiAgICBvcHRpb25zOiBbXG4gICAgICAnLS10b29sa2l0LXN0YWNrLW5hbWUnLCBib290c3RyYXBTdGFja05hbWUsXG4gICAgICAnLS1jb250ZXh0JywgYEBhd3MtY2RrL2NvcmU6Ym9vdHN0cmFwUXVhbGlmaWVyPSR7Zml4dHVyZS5xdWFsaWZpZXJ9YCxcbiAgICAgICctLWNvbnRleHQnLCAnQGF3cy1jZGsvY29yZTpuZXdTdHlsZVN0YWNrU3ludGhlc2lzPTEnLFxuICAgIF0sXG4gIH0pO1xufSkpO1xuXG5pbnRlZ1Rlc3QoJ2RlcGxveSBuZXcgc3R5bGUgc3ludGhlc2lzIHRvIG5ldyBzdHlsZSBib290c3RyYXAnLCB3aXRoRGVmYXVsdEZpeHR1cmUoYXN5bmMgKGZpeHR1cmUpID0+IHtcbiAgY29uc3QgYm9vdHN0cmFwU3RhY2tOYW1lID0gZml4dHVyZS5ib290c3RyYXBTdGFja05hbWU7XG5cbiAgYXdhaXQgZml4dHVyZS5jZGtCb290c3RyYXBNb2Rlcm4oe1xuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBjZm5FeGVjdXRpb25Qb2xpY3k6ICdhcm46YXdzOmlhbTo6YXdzOnBvbGljeS9BZG1pbmlzdHJhdG9yQWNjZXNzJyxcbiAgfSk7XG5cbiAgLy8gRGVwbG95IHN0YWNrIHRoYXQgdXNlcyBmaWxlIGFzc2V0c1xuICBhd2FpdCBmaXh0dXJlLmNka0RlcGxveSgnbGFtYmRhJywge1xuICAgIG9wdGlvbnM6IFtcbiAgICAgICctLXRvb2xraXQtc3RhY2stbmFtZScsIGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICAgICctLWNvbnRleHQnLCBgQGF3cy1jZGsvY29yZTpib290c3RyYXBRdWFsaWZpZXI9JHtmaXh0dXJlLnF1YWxpZmllcn1gLFxuICAgICAgJy0tY29udGV4dCcsICdAYXdzLWNkay9jb3JlOm5ld1N0eWxlU3RhY2tTeW50aGVzaXM9MScsXG4gICAgXSxcbiAgfSk7XG59KSk7XG5cbmludGVnVGVzdCgnZGVwbG95IG5ldyBzdHlsZSBzeW50aGVzaXMgdG8gbmV3IHN0eWxlIGJvb3RzdHJhcCAod2l0aCBkb2NrZXIgaW1hZ2UpJywgd2l0aERlZmF1bHRGaXh0dXJlKGFzeW5jIChmaXh0dXJlKSA9PiB7XG4gIGNvbnN0IGJvb3RzdHJhcFN0YWNrTmFtZSA9IGZpeHR1cmUuYm9vdHN0cmFwU3RhY2tOYW1lO1xuXG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTW9kZXJuKHtcbiAgICB0b29sa2l0U3RhY2tOYW1lOiBib290c3RyYXBTdGFja05hbWUsXG4gICAgY2ZuRXhlY3V0aW9uUG9saWN5OiAnYXJuOmF3czppYW06OmF3czpwb2xpY3kvQWRtaW5pc3RyYXRvckFjY2VzcycsXG4gIH0pO1xuXG4gIC8vIERlcGxveSBzdGFjayB0aGF0IHVzZXMgZmlsZSBhc3NldHNcbiAgYXdhaXQgZml4dHVyZS5jZGtEZXBsb3koJ2RvY2tlcicsIHtcbiAgICBvcHRpb25zOiBbXG4gICAgICAnLS10b29sa2l0LXN0YWNrLW5hbWUnLCBib290c3RyYXBTdGFja05hbWUsXG4gICAgICAnLS1jb250ZXh0JywgYEBhd3MtY2RrL2NvcmU6Ym9vdHN0cmFwUXVhbGlmaWVyPSR7Zml4dHVyZS5xdWFsaWZpZXJ9YCxcbiAgICAgICctLWNvbnRleHQnLCAnQGF3cy1jZGsvY29yZTpuZXdTdHlsZVN0YWNrU3ludGhlc2lzPTEnLFxuICAgIF0sXG4gIH0pO1xufSkpO1xuXG5pbnRlZ1Rlc3QoJ2RlcGxveSBvbGQgc3R5bGUgc3ludGhlc2lzIHRvIG5ldyBzdHlsZSBib290c3RyYXAnLCB3aXRoRGVmYXVsdEZpeHR1cmUoYXN5bmMgKGZpeHR1cmUpID0+IHtcbiAgY29uc3QgYm9vdHN0cmFwU3RhY2tOYW1lID0gZml4dHVyZS5ib290c3RyYXBTdGFja05hbWU7XG5cbiAgYXdhaXQgZml4dHVyZS5jZGtCb290c3RyYXBNb2Rlcm4oe1xuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBjZm5FeGVjdXRpb25Qb2xpY3k6ICdhcm46YXdzOmlhbTo6YXdzOnBvbGljeS9BZG1pbmlzdHJhdG9yQWNjZXNzJyxcbiAgfSk7XG5cbiAgLy8gRGVwbG95IHN0YWNrIHRoYXQgdXNlcyBmaWxlIGFzc2V0c1xuICBhd2FpdCBmaXh0dXJlLmNka0RlcGxveSgnbGFtYmRhJywge1xuICAgIG9wdGlvbnM6IFtcbiAgICAgICctLXRvb2xraXQtc3RhY2stbmFtZScsIGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBdLFxuICB9KTtcbn0pKTtcblxuaW50ZWdUZXN0KCdkZXBsb3lpbmcgbmV3IHN0eWxlIHN5bnRoZXNpcyB0byBvbGQgc3R5bGUgYm9vdHN0cmFwIGZhaWxzJywgd2l0aERlZmF1bHRGaXh0dXJlKGFzeW5jIChmaXh0dXJlKSA9PiB7XG4gIGNvbnN0IGJvb3RzdHJhcFN0YWNrTmFtZSA9IGZpeHR1cmUuYm9vdHN0cmFwU3RhY2tOYW1lO1xuXG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTGVnYWN5KHtcbiAgICB0b29sa2l0U3RhY2tOYW1lOiBib290c3RyYXBTdGFja05hbWUsXG4gIH0pO1xuXG4gIC8vIERlcGxveSBzdGFjayB0aGF0IHVzZXMgZmlsZSBhc3NldHMsIHRoaXMgZmFpbHMgYmVjYXVzZSB0aGUgYm9vdHN0cmFwIHN0YWNrXG4gIC8vIGlzIHZlcnNpb24gY2hlY2tlZC5cbiAgYXdhaXQgZXhwZWN0KGZpeHR1cmUuY2RrRGVwbG95KCdsYW1iZGEnLCB7XG4gICAgb3B0aW9uczogW1xuICAgICAgJy0tdG9vbGtpdC1zdGFjay1uYW1lJywgYm9vdHN0cmFwU3RhY2tOYW1lLFxuICAgICAgJy0tY29udGV4dCcsICdAYXdzLWNkay9jb3JlOm5ld1N0eWxlU3RhY2tTeW50aGVzaXM9MScsXG4gICAgXSxcbiAgfSkpLnJlamVjdHMudG9UaHJvdygnZXhpdGVkIHdpdGggZXJyb3InKTtcbn0pKTtcblxuaW50ZWdUZXN0KCdjYW4gY3JlYXRlIGEgbGVnYWN5IGJvb3RzdHJhcCBzdGFjayB3aXRoIC0tcHVibGljLWFjY2Vzcy1ibG9jay1jb25maWd1cmF0aW9uPWZhbHNlJywgd2l0aERlZmF1bHRGaXh0dXJlKGFzeW5jIChmaXh0dXJlKSA9PiB7XG4gIGNvbnN0IGJvb3RzdHJhcFN0YWNrTmFtZSA9IGZpeHR1cmUuYm9vdHN0cmFwU3RhY2tOYW1lO1xuXG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTGVnYWN5KHtcbiAgICB2ZXJib3NlOiB0cnVlLFxuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBwdWJsaWNBY2Nlc3NCbG9ja0NvbmZpZ3VyYXRpb246IGZhbHNlLFxuICAgIHRhZ3M6ICdGb289QmFyJyxcbiAgfSk7XG5cbiAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBmaXh0dXJlLmF3cy5jbG91ZEZvcm1hdGlvbignZGVzY3JpYmVTdGFja3MnLCB7IFN0YWNrTmFtZTogYm9vdHN0cmFwU3RhY2tOYW1lIH0pO1xuICBleHBlY3QocmVzcG9uc2UuU3RhY2tzPy5bMF0uVGFncykudG9FcXVhbChbXG4gICAgeyBLZXk6ICdGb28nLCBWYWx1ZTogJ0JhcicgfSxcbiAgXSk7XG59KSk7XG5cbmludGVnVGVzdCgnY2FuIGNyZWF0ZSBtdWx0aXBsZSBsZWdhY3kgYm9vdHN0cmFwIHN0YWNrcycsIHdpdGhEZWZhdWx0Rml4dHVyZShhc3luYyAoZml4dHVyZSkgPT4ge1xuICBjb25zdCBib290c3RyYXBTdGFja05hbWUxID0gYCR7Zml4dHVyZS5ib290c3RyYXBTdGFja05hbWV9LTFgO1xuICBjb25zdCBib290c3RyYXBTdGFja05hbWUyID0gYCR7Zml4dHVyZS5ib290c3RyYXBTdGFja05hbWV9LTJgO1xuXG4gIC8vIGRlcGxveSB0d28gdG9vbGtpdCBzdGFja3MgaW50byB0aGUgc2FtZSBlbnZpcm9ubWVudCAoc2VlICMxNDE2KVxuICAvLyBvbmUgd2l0aCB0YWdzXG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTGVnYWN5KHtcbiAgICB2ZXJib3NlOiB0cnVlLFxuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZTEsXG4gICAgdGFnczogJ0Zvbz1CYXInLFxuICB9KTtcbiAgYXdhaXQgZml4dHVyZS5jZGtCb290c3RyYXBMZWdhY3koe1xuICAgIHZlcmJvc2U6IHRydWUsXG4gICAgdG9vbGtpdFN0YWNrTmFtZTogYm9vdHN0cmFwU3RhY2tOYW1lMixcbiAgfSk7XG5cbiAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBmaXh0dXJlLmF3cy5jbG91ZEZvcm1hdGlvbignZGVzY3JpYmVTdGFja3MnLCB7IFN0YWNrTmFtZTogYm9vdHN0cmFwU3RhY2tOYW1lMSB9KTtcbiAgZXhwZWN0KHJlc3BvbnNlLlN0YWNrcz8uWzBdLlRhZ3MpLnRvRXF1YWwoW1xuICAgIHsgS2V5OiAnRm9vJywgVmFsdWU6ICdCYXInIH0sXG4gIF0pO1xufSkpO1xuXG5pbnRlZ1Rlc3QoJ2NhbiBkdW1wIHRoZSB0ZW1wbGF0ZSwgbW9kaWZ5IGFuZCB1c2UgaXQgdG8gZGVwbG95IGEgY3VzdG9tIGJvb3RzdHJhcCBzdGFjaycsIHdpdGhEZWZhdWx0Rml4dHVyZShhc3luYyAoZml4dHVyZSkgPT4ge1xuICBsZXQgdGVtcGxhdGUgPSBhd2FpdCBmaXh0dXJlLmNka0Jvb3RzdHJhcE1vZGVybih7XG4gICAgLy8gdG9vbGtpdFN0YWNrTmFtZSBkb2Vzbid0IG1hdHRlciBmb3IgdGhpcyBwYXJ0aWN1bGFyIGludm9jYXRpb25cbiAgICB0b29sa2l0U3RhY2tOYW1lOiBmaXh0dXJlLmJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBzaG93VGVtcGxhdGU6IHRydWUsXG4gICAgY2xpT3B0aW9uczoge1xuICAgICAgY2FwdHVyZVN0ZGVycjogZmFsc2UsXG4gICAgfSxcbiAgfSk7XG5cbiAgZXhwZWN0KHRlbXBsYXRlKS50b0NvbnRhaW4oJ0Jvb3RzdHJhcFZlcnNpb246Jyk7XG5cbiAgdGVtcGxhdGUgKz0gJ1xcbicgKyBbXG4gICAgJyAgVHdpZGRsZURlZTonLFxuICAgICcgICAgVmFsdWU6IFRlbXBsYXRlIGdvdCB0d2lkZGxlZCcsXG4gIF0uam9pbignXFxuJyk7XG5cbiAgY29uc3QgZmlsZW5hbWUgPSBwYXRoLmpvaW4oZml4dHVyZS5pbnRlZ1Rlc3REaXIsIGAke2ZpeHR1cmUucXVhbGlmaWVyfS10ZW1wbGF0ZS55YW1sYCk7XG4gIGZzLndyaXRlRmlsZVN5bmMoZmlsZW5hbWUsIHRlbXBsYXRlLCB7IGVuY29kaW5nOiAndXRmLTgnIH0pO1xuICBhd2FpdCBmaXh0dXJlLmNka0Jvb3RzdHJhcE1vZGVybih7XG4gICAgdG9vbGtpdFN0YWNrTmFtZTogZml4dHVyZS5ib290c3RyYXBTdGFja05hbWUsXG4gICAgdGVtcGxhdGU6IGZpbGVuYW1lLFxuICAgIGNmbkV4ZWN1dGlvblBvbGljeTogJ2Fybjphd3M6aWFtOjphd3M6cG9saWN5L0FkbWluaXN0cmF0b3JBY2Nlc3MnLFxuICB9KTtcbn0pKTtcblxuaW50ZWdUZXN0KCdzd2l0Y2ggb24gdGVybWluYXRpb24gcHJvdGVjdGlvbiwgc3dpdGNoIGlzIGxlZnQgYWxvbmUgb24gcmUtYm9vdHN0cmFwJywgd2l0aERlZmF1bHRGaXh0dXJlKGFzeW5jIChmaXh0dXJlKSA9PiB7XG4gIGNvbnN0IGJvb3RzdHJhcFN0YWNrTmFtZSA9IGZpeHR1cmUuYm9vdHN0cmFwU3RhY2tOYW1lO1xuXG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTW9kZXJuKHtcbiAgICB2ZXJib3NlOiB0cnVlLFxuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICB0ZXJtaW5hdGlvblByb3RlY3Rpb246IHRydWUsXG4gICAgY2ZuRXhlY3V0aW9uUG9saWN5OiAnYXJuOmF3czppYW06OmF3czpwb2xpY3kvQWRtaW5pc3RyYXRvckFjY2VzcycsXG4gIH0pO1xuICBhd2FpdCBmaXh0dXJlLmNka0Jvb3RzdHJhcE1vZGVybih7XG4gICAgdmVyYm9zZTogdHJ1ZSxcbiAgICB0b29sa2l0U3RhY2tOYW1lOiBib290c3RyYXBTdGFja05hbWUsXG4gICAgZm9yY2U6IHRydWUsXG4gIH0pO1xuXG4gIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgZml4dHVyZS5hd3MuY2xvdWRGb3JtYXRpb24oJ2Rlc2NyaWJlU3RhY2tzJywgeyBTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSB9KTtcbiAgZXhwZWN0KHJlc3BvbnNlLlN0YWNrcz8uWzBdLkVuYWJsZVRlcm1pbmF0aW9uUHJvdGVjdGlvbikudG9FcXVhbCh0cnVlKTtcbn0pKTtcblxuaW50ZWdUZXN0KCdhZGQgdGFncywgbGVmdCBhbG9uZSBvbiByZS1ib290c3RyYXAnLCB3aXRoRGVmYXVsdEZpeHR1cmUoYXN5bmMgKGZpeHR1cmUpID0+IHtcbiAgY29uc3QgYm9vdHN0cmFwU3RhY2tOYW1lID0gZml4dHVyZS5ib290c3RyYXBTdGFja05hbWU7XG5cbiAgYXdhaXQgZml4dHVyZS5jZGtCb290c3RyYXBNb2Rlcm4oe1xuICAgIHZlcmJvc2U6IHRydWUsXG4gICAgdG9vbGtpdFN0YWNrTmFtZTogYm9vdHN0cmFwU3RhY2tOYW1lLFxuICAgIHRhZ3M6ICdGb289QmFyJyxcbiAgICBjZm5FeGVjdXRpb25Qb2xpY3k6ICdhcm46YXdzOmlhbTo6YXdzOnBvbGljeS9BZG1pbmlzdHJhdG9yQWNjZXNzJyxcbiAgfSk7XG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTW9kZXJuKHtcbiAgICB2ZXJib3NlOiB0cnVlLFxuICAgIHRvb2xraXRTdGFja05hbWU6IGJvb3RzdHJhcFN0YWNrTmFtZSxcbiAgICBmb3JjZTogdHJ1ZSxcbiAgfSk7XG5cbiAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBmaXh0dXJlLmF3cy5jbG91ZEZvcm1hdGlvbignZGVzY3JpYmVTdGFja3MnLCB7IFN0YWNrTmFtZTogYm9vdHN0cmFwU3RhY2tOYW1lIH0pO1xuICBleHBlY3QocmVzcG9uc2UuU3RhY2tzPy5bMF0uVGFncykudG9FcXVhbChbXG4gICAgeyBLZXk6ICdGb28nLCBWYWx1ZTogJ0JhcicgfSxcbiAgXSk7XG59KSk7XG5cbmludGVnVGVzdCgnY2FuIGRlcGxveSBtb2Rlcm4tc3ludGhlc2l6ZWQgc3RhY2sgZXZlbiBpZiBib290c3RyYXAgc3RhY2sgbmFtZSBpcyB1bmtub3duJywgd2l0aERlZmF1bHRGaXh0dXJlKGFzeW5jIChmaXh0dXJlKSA9PiB7XG4gIGNvbnN0IGJvb3RzdHJhcFN0YWNrTmFtZSA9IGZpeHR1cmUuYm9vdHN0cmFwU3RhY2tOYW1lO1xuXG4gIGF3YWl0IGZpeHR1cmUuY2RrQm9vdHN0cmFwTW9kZXJuKHtcbiAgICB0b29sa2l0U3RhY2tOYW1lOiBib290c3RyYXBTdGFja05hbWUsXG4gICAgY2ZuRXhlY3V0aW9uUG9saWN5OiAnYXJuOmF3czppYW06OmF3czpwb2xpY3kvQWRtaW5pc3RyYXRvckFjY2VzcycsXG4gIH0pO1xuXG4gIC8vIERlcGxveSBzdGFjayB0aGF0IHVzZXMgZmlsZSBhc3NldHNcbiAgYXdhaXQgZml4dHVyZS5jZGtEZXBsb3koJ2xhbWJkYScsIHtcbiAgICBvcHRpb25zOiBbXG4gICAgICAvLyBFeHBsaWNpdHkgcGFzcyBhIG5hbWUgdGhhdCdzIHN1cmUgdG8gbm90IGV4aXN0LCBvdGhlcndpc2UgdGhlIENMSSBtaWdodCBhY2NpZGVudGFsbHkgZmluZCBhXG4gICAgICAvLyBkZWZhdWx0IGJvb3RzdHJhY3Agc3RhY2sgaWYgdGhhdCBoYXBwZW5zIHRvIGJlIGluIHRoZSBhY2NvdW50IGFscmVhZHkuXG4gICAgICAnLS10b29sa2l0LXN0YWNrLW5hbWUnLCAnRGVmaW5pdGVseURvZXNOb3RFeGlzdCcsXG4gICAgICAnLS1jb250ZXh0JywgYEBhd3MtY2RrL2NvcmU6Ym9vdHN0cmFwUXVhbGlmaWVyPSR7Zml4dHVyZS5xdWFsaWZpZXJ9YCxcbiAgICAgICctLWNvbnRleHQnLCAnQGF3cy1jZGsvY29yZTpuZXdTdHlsZVN0YWNrU3ludGhlc2lzPTEnLFxuICAgIF0sXG4gIH0pO1xufSkpO1xuIl19