"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.obtainEcrCredentials = exports.fetchDockerLoginCredentials = exports.cdkCredentialsConfig = exports.cdkCredentialsConfigFile = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
/** Returns the presumed location of the CDK Docker credentials config file */
function cdkCredentialsConfigFile() {
    var _a, _b;
    return (_a = process.env.CDK_DOCKER_CREDS_FILE) !== null && _a !== void 0 ? _a : path.join(((_b = os.userInfo().homedir) !== null && _b !== void 0 ? _b : os.homedir()).trim() || '/', '.cdk', 'cdk-docker-creds.json');
}
exports.cdkCredentialsConfigFile = cdkCredentialsConfigFile;
let _cdkCredentials;
/** Loads and parses the CDK Docker credentials configuration, if it exists. */
function cdkCredentialsConfig() {
    if (!_cdkCredentials) {
        try {
            _cdkCredentials = JSON.parse(fs.readFileSync(cdkCredentialsConfigFile(), { encoding: 'utf-8' }));
        }
        catch (err) { }
    }
    return _cdkCredentials;
}
exports.cdkCredentialsConfig = cdkCredentialsConfig;
/** Fetches login credentials from the configured source (e.g., SecretsManager, ECR) */
async function fetchDockerLoginCredentials(aws, config, domain) {
    var _a, _b;
    if (!Object.keys(config.domainCredentials).includes(domain)) {
        throw new Error(`unknown domain ${domain}`);
    }
    const domainConfig = config.domainCredentials[domain];
    if (domainConfig.secretsManagerSecretId) {
        const sm = await aws.secretsManagerClient({ assumeRoleArn: domainConfig.assumeRoleArn });
        const secretValue = await sm.getSecretValue({ SecretId: domainConfig.secretsManagerSecretId }).promise();
        if (!secretValue.SecretString) {
            throw new Error(`unable to fetch SecretString from secret: ${domainConfig.secretsManagerSecretId}`);
        }
        ;
        const secret = JSON.parse(secretValue.SecretString);
        const usernameField = (_a = domainConfig.secretsUsernameField) !== null && _a !== void 0 ? _a : 'username';
        const secretField = (_b = domainConfig.secretsPasswordField) !== null && _b !== void 0 ? _b : 'secret';
        if (!secret[usernameField] || !secret[secretField]) {
            throw new Error(`malformed secret string ("${usernameField}" or "${secretField}" field missing)`);
        }
        return { Username: secret[usernameField], Secret: secret[secretField] };
    }
    else if (domainConfig.ecrRepository) {
        const ecr = await aws.ecrClient({ assumeRoleArn: domainConfig.assumeRoleArn });
        const ecrAuthData = await obtainEcrCredentials(ecr);
        return { Username: ecrAuthData.username, Secret: ecrAuthData.password };
    }
    else {
        throw new Error('unknown credential type: no secret ID or ECR repo');
    }
}
exports.fetchDockerLoginCredentials = fetchDockerLoginCredentials;
async function obtainEcrCredentials(ecr, logger) {
    if (logger) {
        logger('Fetching ECR authorization token');
    }
    const authData = (await ecr.getAuthorizationToken({}).promise()).authorizationData || [];
    if (authData.length === 0) {
        throw new Error('No authorization data received from ECR');
    }
    const token = Buffer.from(authData[0].authorizationToken, 'base64').toString('ascii');
    const [username, password] = token.split(':');
    if (!username || !password) {
        throw new Error('unexpected ECR authData format');
    }
    return {
        username,
        password,
        endpoint: authData[0].proxyEndpoint,
    };
}
exports.obtainEcrCredentials = obtainEcrCredentials;
//# sourceMappingURL=data:application/json;base64,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