"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImageAssetHandler = void 0;
const path = require("path");
const progress_1 = require("../../progress");
const docker_1 = require("../docker");
const placeholders_1 = require("../placeholders");
const shell_1 = require("../shell");
class ContainerImageAssetHandler {
    constructor(workDir, asset, host) {
        this.workDir = workDir;
        this.asset = asset;
        this.host = host;
        this.docker = new docker_1.Docker(m => this.host.emitMessage(progress_1.EventType.DEBUG, m));
    }
    async publish() {
        const destination = await placeholders_1.replaceAwsPlaceholders(this.asset.destination, this.host.aws);
        const ecr = await this.host.aws.ecrClient(destination);
        const account = async () => { var _a; return (_a = (await this.host.aws.discoverCurrentAccount())) === null || _a === void 0 ? void 0 : _a.accountId; };
        const repoUri = await repositoryUri(ecr, destination.repositoryName);
        if (!repoUri) {
            throw new Error(`No ECR repository named '${destination.repositoryName}' in account ${await account()}. Is this account bootstrapped?`);
        }
        const imageUri = `${repoUri}:${destination.imageTag}`;
        if (await this.destinationAlreadyExists(ecr, destination, imageUri)) {
            return;
        }
        if (this.host.aborted) {
            return;
        }
        // Default behavior is to login before build so that the Dockerfile can reference images in the ECR repo
        // However, if we're in a pipelines environment (for example),
        // we may have alternative credentials to the default ones to use for the build itself.
        // If the special config file is present, delay the login to the default credentials until the push.
        // If the config file is present, we will configure and use those credentials for the build.
        let cdkDockerCredentialsConfigured = this.docker.configureCdkCredentials();
        if (!cdkDockerCredentialsConfigured) {
            await this.docker.login(ecr);
        }
        const localTagName = this.asset.source.executable
            ? await this.buildExternalAsset(this.asset.source.executable)
            : await this.buildDirectoryAsset();
        if (localTagName === undefined || this.host.aborted) {
            return;
        }
        this.host.emitMessage(progress_1.EventType.UPLOAD, `Push ${imageUri}`);
        if (this.host.aborted) {
            return;
        }
        await this.docker.tag(localTagName, imageUri);
        if (cdkDockerCredentialsConfigured) {
            this.docker.resetAuthPlugins();
            await this.docker.login(ecr);
        }
        await this.docker.push(imageUri);
    }
    /**
     * Build a (local) Docker asset from a directory with a Dockerfile
     *
     * Tags under a deterministic, unique, local identifier wich will skip
     * the build if it already exists.
     */
    async buildDirectoryAsset() {
        const localTagName = `cdkasset-${this.asset.id.assetId.toLowerCase()}`;
        if (!(await this.isImageCached(localTagName))) {
            if (this.host.aborted) {
                return undefined;
            }
            await this.buildImage(localTagName);
        }
        return localTagName;
    }
    /**
     * Build a (local) Docker asset by running an external command
     *
     * External command is responsible for deduplicating the build if possible,
     * and is expected to return the generated image identifier on stdout.
     */
    async buildExternalAsset(executable) {
        this.host.emitMessage(progress_1.EventType.BUILD, `Building Docker image using command '${executable}'`);
        if (this.host.aborted) {
            return undefined;
        }
        return (await shell_1.shell(executable, { quiet: true })).trim();
    }
    /**
     * Check whether the image already exists in the ECR repo
     *
     * Use the fields from the destination to do the actual check. The imageUri
     * should correspond to that, but is only used to print Docker image location
     * for user benefit (the format is slightly different).
     */
    async destinationAlreadyExists(ecr, destination, imageUri) {
        this.host.emitMessage(progress_1.EventType.CHECK, `Check ${imageUri}`);
        if (await imageExists(ecr, destination.repositoryName, destination.imageTag)) {
            this.host.emitMessage(progress_1.EventType.FOUND, `Found ${imageUri}`);
            return true;
        }
        return false;
    }
    async buildImage(localTagName) {
        const source = this.asset.source;
        if (!source.directory) {
            throw new Error(`'directory' is expected in the DockerImage asset source, got: ${JSON.stringify(source)}`);
        }
        const fullPath = path.resolve(this.workDir, source.directory);
        this.host.emitMessage(progress_1.EventType.BUILD, `Building Docker image at ${fullPath}`);
        await this.docker.build({
            directory: fullPath,
            tag: localTagName,
            buildArgs: source.dockerBuildArgs,
            target: source.dockerBuildTarget,
            file: source.dockerFile,
        });
    }
    async isImageCached(localTagName) {
        if (await this.docker.exists(localTagName)) {
            this.host.emitMessage(progress_1.EventType.CACHED, `Cached ${localTagName}`);
            return true;
        }
        return false;
    }
}
exports.ContainerImageAssetHandler = ContainerImageAssetHandler;
async function imageExists(ecr, repositoryName, imageTag) {
    try {
        await ecr.describeImages({ repositoryName, imageIds: [{ imageTag }] }).promise();
        return true;
    }
    catch (e) {
        if (e.code !== 'ImageNotFoundException') {
            throw e;
        }
        return false;
    }
}
/**
 * Return the URI for the repository with the given name
 *
 * Returns undefined if the repository does not exist.
 */
async function repositoryUri(ecr, repositoryName) {
    var _a;
    try {
        const response = await ecr.describeRepositories({ repositoryNames: [repositoryName] }).promise();
        return (_a = (response.repositories || [])[0]) === null || _a === void 0 ? void 0 : _a.repositoryUri;
    }
    catch (e) {
        if (e.code !== 'RepositoryNotFoundException') {
            throw e;
        }
        return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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