"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.shell = void 0;
const child_process = require("child_process");
/**
 * OS helpers
 *
 * Shell function which both prints to stdout and collects the output into a
 * string.
 */
async function shell(command, options = {}) {
    if (options.logger) {
        options.logger(renderCommandLine(command));
    }
    const child = child_process.spawn(command[0], command.slice(1), {
        ...options,
        stdio: [options.input ? 'pipe' : 'ignore', 'pipe', 'pipe'],
    });
    return new Promise((resolve, reject) => {
        if (options.input) {
            child.stdin.write(options.input);
            child.stdin.end();
        }
        const stdout = new Array();
        const stderr = new Array();
        // Both write to stdout and collect
        child.stdout.on('data', chunk => {
            if (!options.quiet) {
                process.stdout.write(chunk);
            }
            stdout.push(chunk);
        });
        child.stderr.on('data', chunk => {
            if (!options.quiet) {
                process.stderr.write(chunk);
            }
            stderr.push(chunk);
        });
        child.once('error', reject);
        child.once('close', code => {
            if (code === 0) {
                resolve(Buffer.concat(stdout).toString('utf-8'));
            }
            else {
                const out = Buffer.concat(stderr).toString('utf-8').trim();
                reject(new ProcessFailed(code, `${renderCommandLine(command)} exited with error code ${code}: ${out}`));
            }
        });
    });
}
exports.shell = shell;
class ProcessFailed extends Error {
    constructor(exitCode, message) {
        super(message);
        this.exitCode = exitCode;
        this.code = 'PROCESS_FAILED';
    }
}
/**
 * Render the given command line as a string
 *
 * Probably missing some cases but giving it a good effort.
 */
function renderCommandLine(cmd) {
    if (process.platform !== 'win32') {
        return doRender(cmd, hasAnyChars(' ', '\\', '!', '"', "'", '&', '$'), posixEscape);
    }
    else {
        return doRender(cmd, hasAnyChars(' ', '"', '&', '^', '%'), windowsEscape);
    }
}
/**
 * Render a UNIX command line
 */
function doRender(cmd, needsEscaping, doEscape) {
    return cmd.map(x => needsEscaping(x) ? doEscape(x) : x).join(' ');
}
/**
 * Return a predicate that checks if a string has any of the indicated chars in it
 */
function hasAnyChars(...chars) {
    return (str) => {
        return chars.some(c => str.indexOf(c) !== -1);
    };
}
/**
 * Escape a shell argument for POSIX shells
 *
 * Wrapping in single quotes and escaping single quotes inside will do it for us.
 */
function posixEscape(x) {
    // Turn ' -> '"'"'
    x = x.replace("'", "'\"'\"'");
    return `'${x}'`;
}
/**
 * Escape a shell argument for cmd.exe
 *
 * This is how to do it right, but I'm not following everything:
 *
 * https://blogs.msdn.microsoft.com/twistylittlepassagesallalike/2011/04/23/everyone-quotes-command-line-arguments-the-wrong-way/
 */
function windowsEscape(x) {
    // First surround by double quotes, ignore the part about backslashes
    x = `"${x}"`;
    // Now escape all special characters
    const shellMeta = new Set(['"', '&', '^', '%']);
    return x.split('').map(c => shellMeta.has(x) ? '^' + c : c).join('');
}
//# sourceMappingURL=data:application/json;base64,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