"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpMethod = exports.Connection = exports.HttpParameter = exports.Authorization = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
/**
 * Authorization type for an API Destination Connection
 */
class Authorization {
    /**
     * Use API key authorization
     *
     * API key authorization has two components: an API key name and an API key value.
     * What these are depends on the target of your connection.
     */
    static apiKey(apiKeyName, apiKeyValue) {
        return new class extends Authorization {
            _bind() {
                return {
                    authorizationType: AuthorizationType.API_KEY,
                    authParameters: {
                        apiKeyAuthParameters: {
                            apiKeyName: apiKeyName,
                            apiKeyValue: apiKeyValue.unsafeUnwrap(),
                        },
                    },
                };
            }
        }();
    }
    /**
     * Use username and password authorization
     */
    static basic(username, password) {
        return new class extends Authorization {
            _bind() {
                return {
                    authorizationType: AuthorizationType.BASIC,
                    authParameters: {
                        basicAuthParameters: {
                            username: username,
                            password: password.unsafeUnwrap(),
                        },
                    },
                };
            }
        }();
    }
    /**
     * Use OAuth authorization
     */
    static oauth(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_events_OAuthAuthorizationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.oauth);
            }
            throw error;
        }
        if (![HttpMethod.POST, HttpMethod.GET, HttpMethod.PUT].includes(props.httpMethod)) {
            throw new Error('httpMethod must be one of GET, POST, PUT');
        }
        return new class extends Authorization {
            _bind() {
                return {
                    authorizationType: AuthorizationType.OAUTH_CLIENT_CREDENTIALS,
                    authParameters: {
                        oAuthParameters: {
                            authorizationEndpoint: props.authorizationEndpoint,
                            clientParameters: {
                                clientId: props.clientId,
                                clientSecret: props.clientSecret.unsafeUnwrap(),
                            },
                            httpMethod: props.httpMethod,
                            oAuthHttpParameters: {
                                bodyParameters: renderHttpParameters(props.bodyParameters),
                                headerParameters: renderHttpParameters(props.headerParameters),
                                queryStringParameters: renderHttpParameters(props.queryStringParameters),
                            },
                        },
                    },
                };
            }
        }();
    }
}
exports.Authorization = Authorization;
_a = JSII_RTTI_SYMBOL_1;
Authorization[_a] = { fqn: "@aws-cdk/aws-events.Authorization", version: "1.168.0" };
/**
 * An additional HTTP parameter to send along with the OAuth request
 */
class HttpParameter {
    /**
     * Make an OAuthParameter from a string value
     *
     * The value is not treated as a secret.
     */
    static fromString(value) {
        return new class extends HttpParameter {
            _render(name) {
                return {
                    key: name,
                    value,
                };
            }
        }();
    }
    /**
     * Make an OAuthParameter from a secret
     */
    static fromSecret(value) {
        return new class extends HttpParameter {
            _render(name) {
                return {
                    key: name,
                    value: value.unsafeUnwrap(),
                    isValueSecret: true,
                };
            }
        }();
    }
}
exports.HttpParameter = HttpParameter;
_b = JSII_RTTI_SYMBOL_1;
HttpParameter[_b] = { fqn: "@aws-cdk/aws-events.HttpParameter", version: "1.168.0" };
/**
 * Define an EventBridge Connection
 *
 * @resource AWS::Events::Connection
 */
class Connection extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.connectionName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_events_ConnectionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Connection);
            }
            throw error;
        }
        const authBind = props.authorization._bind();
        const invocationHttpParameters = !!props.headerParameters || !!props.queryStringParameters || !!props.bodyParameters ? {
            bodyParameters: renderHttpParameters(props.bodyParameters),
            headerParameters: renderHttpParameters(props.headerParameters),
            queryStringParameters: renderHttpParameters(props.queryStringParameters),
        } : undefined;
        let connection = new events_generated_1.CfnConnection(this, 'Connection', {
            authorizationType: authBind.authorizationType,
            authParameters: {
                ...authBind.authParameters,
                invocationHttpParameters: invocationHttpParameters,
            },
            description: props.description,
            name: this.physicalName,
        });
        this.connectionName = this.getResourceNameAttribute(connection.ref);
        this.connectionArn = connection.attrArn;
        this.connectionSecretArn = connection.attrSecretArn;
    }
    /**
     * Import an existing connection resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param connectionArn ARN of imported connection
     */
    static fromEventBusArn(scope, id, connectionArn, connectionSecretArn) {
        const parts = core_1.Stack.of(scope).parseArn(connectionArn);
        return new ImportedConnection(scope, id, {
            connectionArn: connectionArn,
            connectionName: parts.resourceName || '',
            connectionSecretArn: connectionSecretArn,
        });
    }
    /**
     * Import an existing connection resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param attrs Imported connection properties
     */
    static fromConnectionAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_events_ConnectionAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromConnectionAttributes);
            }
            throw error;
        }
        return new ImportedConnection(scope, id, attrs);
    }
}
exports.Connection = Connection;
_c = JSII_RTTI_SYMBOL_1;
Connection[_c] = { fqn: "@aws-cdk/aws-events.Connection", version: "1.168.0" };
class ImportedConnection extends core_1.Resource {
    constructor(scope, id, attrs) {
        const arnParts = core_1.Stack.of(scope).parseArn(attrs.connectionArn);
        super(scope, id, {
            account: arnParts.account,
            region: arnParts.region,
        });
        this.connectionArn = attrs.connectionArn;
        this.connectionName = attrs.connectionName;
        this.connectionSecretArn = attrs.connectionSecretArn;
    }
}
/**
 * Supported HTTP operations.
 */
var HttpMethod;
(function (HttpMethod) {
    /** POST */
    HttpMethod["POST"] = "POST";
    /** GET */
    HttpMethod["GET"] = "GET";
    /** HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** PUT */
    HttpMethod["PUT"] = "PUT";
    /** PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** DELETE */
    HttpMethod["DELETE"] = "DELETE";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * Supported Authorization Types.
 */
var AuthorizationType;
(function (AuthorizationType) {
    /** API_KEY */
    AuthorizationType["API_KEY"] = "API_KEY";
    /** BASIC */
    AuthorizationType["BASIC"] = "BASIC";
    /** OAUTH_CLIENT_CREDENTIALS */
    AuthorizationType["OAUTH_CLIENT_CREDENTIALS"] = "OAUTH_CLIENT_CREDENTIALS";
})(AuthorizationType || (AuthorizationType = {}));
function renderHttpParameters(ps) {
    if (!ps || Object.keys(ps).length === 0) {
        return undefined;
    }
    return Object.entries(ps).map(([name, p]) => p._render(name));
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29ubmVjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvbm5lY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsd0NBQXdFO0FBRXhFLHlEQUFtRDtBQWlEbkQ7O0dBRUc7QUFDSCxNQUFzQixhQUFhO0lBQ2pDOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxVQUFrQixFQUFFLFdBQXdCO1FBQy9ELE9BQU8sSUFBSSxLQUFNLFNBQVEsYUFBYTtZQUM3QixLQUFLO2dCQUNWLE9BQU87b0JBQ0wsaUJBQWlCLEVBQUUsaUJBQWlCLENBQUMsT0FBTztvQkFDNUMsY0FBYyxFQUFFO3dCQUNkLG9CQUFvQixFQUFFOzRCQUNwQixVQUFVLEVBQUUsVUFBVTs0QkFDdEIsV0FBVyxFQUFFLFdBQVcsQ0FBQyxZQUFZLEVBQUU7eUJBQ3hDO3FCQUNzQztpQkFDMUMsQ0FBQztZQUNKLENBQUM7U0FDRixFQUFFLENBQUM7S0FDTDtJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFnQixFQUFFLFFBQXFCO1FBQ3pELE9BQU8sSUFBSSxLQUFNLFNBQVEsYUFBYTtZQUM3QixLQUFLO2dCQUNWLE9BQU87b0JBQ0wsaUJBQWlCLEVBQUUsaUJBQWlCLENBQUMsS0FBSztvQkFDMUMsY0FBYyxFQUFFO3dCQUNkLG1CQUFtQixFQUFFOzRCQUNuQixRQUFRLEVBQUUsUUFBUTs0QkFDbEIsUUFBUSxFQUFFLFFBQVEsQ0FBQyxZQUFZLEVBQUU7eUJBQ2xDO3FCQUNzQztpQkFDMUMsQ0FBQztZQUNKLENBQUM7U0FDRixFQUFFLENBQUM7S0FDTDtJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEtBQUssQ0FBQyxLQUE4Qjs7Ozs7Ozs7OztRQUNoRCxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDakYsTUFBTSxJQUFJLEtBQUssQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1NBQzdEO1FBRUQsT0FBTyxJQUFJLEtBQU0sU0FBUSxhQUFhO1lBQzdCLEtBQUs7Z0JBQ1YsT0FBTztvQkFDTCxpQkFBaUIsRUFBRSxpQkFBaUIsQ0FBQyx3QkFBd0I7b0JBQzdELGNBQWMsRUFBRTt3QkFDZCxlQUFlLEVBQUU7NEJBQ2YscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjs0QkFDbEQsZ0JBQWdCLEVBQUU7Z0NBQ2hCLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtnQ0FDeEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsWUFBWSxFQUFFOzZCQUNoRDs0QkFDRCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7NEJBQzVCLG1CQUFtQixFQUFFO2dDQUNuQixjQUFjLEVBQUUsb0JBQW9CLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztnQ0FDMUQsZ0JBQWdCLEVBQUUsb0JBQW9CLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDO2dDQUM5RCxxQkFBcUIsRUFBRSxvQkFBb0IsQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUM7NkJBQ3pFO3lCQUNGO3FCQUNzQztpQkFDMUMsQ0FBQztZQUNKLENBQUM7U0FDRixFQUFFLENBQUM7S0FFTDs7QUF6RUgsc0NBaUZDOzs7QUFtREQ7O0dBRUc7QUFDSCxNQUFzQixhQUFhO0lBQ2pDOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQWE7UUFDcEMsT0FBTyxJQUFJLEtBQU0sU0FBUSxhQUFhO1lBQzdCLE9BQU8sQ0FBQyxJQUFZO2dCQUN6QixPQUFPO29CQUNMLEdBQUcsRUFBRSxJQUFJO29CQUNULEtBQUs7aUJBQzZCLENBQUM7WUFDdkMsQ0FBQztTQUNGLEVBQUUsQ0FBQztLQUNMO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQWtCO1FBQ3pDLE9BQU8sSUFBSSxLQUFNLFNBQVEsYUFBYTtZQUM3QixPQUFPLENBQUMsSUFBWTtnQkFDekIsT0FBTztvQkFDTCxHQUFHLEVBQUUsSUFBSTtvQkFDVCxLQUFLLEVBQUUsS0FBSyxDQUFDLFlBQVksRUFBRTtvQkFDM0IsYUFBYSxFQUFFLElBQUk7aUJBQ2UsQ0FBQztZQUN2QyxDQUFDO1NBQ0YsRUFBRSxDQUFDO0tBQ0w7O0FBOUJILHNDQXNDQzs7O0FBOEREOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSxlQUFRO0lBNkN0QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxjQUFjO1NBQ25DLENBQUMsQ0FBQzs7Ozs7OytDQWhETSxVQUFVOzs7O1FBa0RuQixNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTdDLE1BQU0sd0JBQXdCLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLHFCQUFxQixJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztZQUNySCxjQUFjLEVBQUUsb0JBQW9CLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUMxRCxnQkFBZ0IsRUFBRSxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7WUFDOUQscUJBQXFCLEVBQUUsb0JBQW9CLENBQUMsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1NBQ3pFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVkLElBQUksVUFBVSxHQUFHLElBQUksZ0NBQWEsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3JELGlCQUFpQixFQUFFLFFBQVEsQ0FBQyxpQkFBaUI7WUFDN0MsY0FBYyxFQUFFO2dCQUNkLEdBQUcsUUFBUSxDQUFDLGNBQWM7Z0JBQzFCLHdCQUF3QixFQUFFLHdCQUF3QjthQUNuRDtZQUNELFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixJQUFJLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDeEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxhQUFhLEdBQUcsVUFBVSxDQUFDLE9BQU8sQ0FBQztRQUN4QyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsVUFBVSxDQUFDLGFBQWEsQ0FBQztLQUNyRDtJQXRFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsYUFBcUIsRUFBRSxtQkFBMkI7UUFDNUcsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFdEQsT0FBTyxJQUFJLGtCQUFrQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDdkMsYUFBYSxFQUFFLGFBQWE7WUFDNUIsY0FBYyxFQUFFLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRTtZQUN4QyxtQkFBbUIsRUFBRSxtQkFBbUI7U0FDekMsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEyQjs7Ozs7Ozs7OztRQUM5RixPQUFPLElBQUksa0JBQWtCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUNqRDs7QUF6QkgsZ0NBd0VDOzs7QUFFRCxNQUFNLGtCQUFtQixTQUFRLGVBQVE7SUFJdkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEyQjtRQUNuRSxNQUFNLFFBQVEsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDL0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixPQUFPLEVBQUUsUUFBUSxDQUFDLE9BQU87WUFDekIsTUFBTSxFQUFFLFFBQVEsQ0FBQyxNQUFNO1NBQ3hCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUN6QyxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7UUFDM0MsSUFBSSxDQUFDLG1CQUFtQixHQUFHLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQztLQUN0RDtDQUNGO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLFVBZVg7QUFmRCxXQUFZLFVBQVU7SUFDcEIsV0FBVztJQUNYLDJCQUFhLENBQUE7SUFDYixVQUFVO0lBQ1YseUJBQVcsQ0FBQTtJQUNYLFdBQVc7SUFDWCwyQkFBYSxDQUFBO0lBQ2IsY0FBYztJQUNkLGlDQUFtQixDQUFBO0lBQ25CLFVBQVU7SUFDVix5QkFBVyxDQUFBO0lBQ1gsWUFBWTtJQUNaLDZCQUFlLENBQUE7SUFDZixhQUFhO0lBQ2IsK0JBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQWZXLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBZXJCO0FBRUQ7O0dBRUc7QUFDSCxJQUFLLGlCQU9KO0FBUEQsV0FBSyxpQkFBaUI7SUFDcEIsY0FBYztJQUNkLHdDQUFtQixDQUFBO0lBQ25CLFlBQVk7SUFDWixvQ0FBZSxDQUFBO0lBQ2YsK0JBQStCO0lBQy9CLDBFQUFxRCxDQUFBO0FBQ3ZELENBQUMsRUFQSSxpQkFBaUIsS0FBakIsaUJBQWlCLFFBT3JCO0FBRUQsU0FBUyxvQkFBb0IsQ0FBQyxFQUFrQztJQUM5RCxJQUFJLENBQUMsRUFBRSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFFOUQsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7QUFDaEUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElSZXNvdXJjZSwgUmVzb3VyY2UsIFN0YWNrLCBTZWNyZXRWYWx1ZSB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5Db25uZWN0aW9uIH0gZnJvbSAnLi9ldmVudHMuZ2VuZXJhdGVkJztcblxuLyoqXG4gKiBBbiBBUEkgRGVzdGluYXRpb24gQ29ubmVjdGlvblxuICpcbiAqIEEgY29ubmVjdGlvbiBkZWZpbmVzIHRoZSBhdXRob3JpemF0aW9uIHR5cGUgYW5kIGNyZWRlbnRpYWxzIHRvIHVzZSBmb3IgYXV0aG9yaXphdGlvbiB3aXRoIGFuIEFQSSBkZXN0aW5hdGlvbiBIVFRQIGVuZHBvaW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbm5lY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY29ubmVjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIG5hbWUgaXMgYXV0b21hdGljYWxseSBnZW5lcmF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25OYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY29ubmVjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGF1dGhvcml6YXRpb24gdHlwZSBmb3IgdGhlIGNvbm5lY3Rpb24uXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uOiBBdXRob3JpemF0aW9uO1xuXG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIHN0cmluZyBwYXJhbWV0ZXJzIHRvIGFkZCB0byB0aGUgaW52b2NhdGlvbiBib2RpZXNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBhZGRpdGlvbmFsIHBhcmFtZXRlcnNcbiAgICovXG4gIHJlYWRvbmx5IGJvZHlQYXJhbWV0ZXJzPzogUmVjb3JkPHN0cmluZywgSHR0cFBhcmFtZXRlcj47XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgc3RyaW5nIHBhcmFtZXRlcnMgdG8gYWRkIHRvIHRoZSBpbnZvY2F0aW9uIGhlYWRlcnNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBhZGRpdGlvbmFsIHBhcmFtZXRlcnNcbiAgICovXG4gIHJlYWRvbmx5IGhlYWRlclBhcmFtZXRlcnM/OiBSZWNvcmQ8c3RyaW5nLCBIdHRwUGFyYW1ldGVyPjtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBzdHJpbmcgcGFyYW1ldGVycyB0byBhZGQgdG8gdGhlIGludm9jYXRpb24gcXVlcnkgc3RyaW5nc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGFkZGl0aW9uYWwgcGFyYW1ldGVyc1xuICAgKi9cbiAgcmVhZG9ubHkgcXVlcnlTdHJpbmdQYXJhbWV0ZXJzPzogUmVjb3JkPHN0cmluZywgSHR0cFBhcmFtZXRlcj47XG59XG5cbi8qKlxuICogQXV0aG9yaXphdGlvbiB0eXBlIGZvciBhbiBBUEkgRGVzdGluYXRpb24gQ29ubmVjdGlvblxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQXV0aG9yaXphdGlvbiB7XG4gIC8qKlxuICAgKiBVc2UgQVBJIGtleSBhdXRob3JpemF0aW9uXG4gICAqXG4gICAqIEFQSSBrZXkgYXV0aG9yaXphdGlvbiBoYXMgdHdvIGNvbXBvbmVudHM6IGFuIEFQSSBrZXkgbmFtZSBhbmQgYW4gQVBJIGtleSB2YWx1ZS5cbiAgICogV2hhdCB0aGVzZSBhcmUgZGVwZW5kcyBvbiB0aGUgdGFyZ2V0IG9mIHlvdXIgY29ubmVjdGlvbi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYXBpS2V5KGFwaUtleU5hbWU6IHN0cmluZywgYXBpS2V5VmFsdWU6IFNlY3JldFZhbHVlKTogQXV0aG9yaXphdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIEF1dGhvcml6YXRpb24ge1xuICAgICAgcHVibGljIF9iaW5kKCkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZLFxuICAgICAgICAgIGF1dGhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBhcGlLZXlBdXRoUGFyYW1ldGVyczoge1xuICAgICAgICAgICAgICBhcGlLZXlOYW1lOiBhcGlLZXlOYW1lLFxuICAgICAgICAgICAgICBhcGlLZXlWYWx1ZTogYXBpS2V5VmFsdWUudW5zYWZlVW53cmFwKCksIC8vIFNhZmUgdXNhZ2VcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSBhcyBDZm5Db25uZWN0aW9uLkF1dGhQYXJhbWV0ZXJzUHJvcGVydHksXG4gICAgICAgIH07XG4gICAgICB9XG4gICAgfSgpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVzZSB1c2VybmFtZSBhbmQgcGFzc3dvcmQgYXV0aG9yaXphdGlvblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBiYXNpYyh1c2VybmFtZTogc3RyaW5nLCBwYXNzd29yZDogU2VjcmV0VmFsdWUpOiBBdXRob3JpemF0aW9uIHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgQXV0aG9yaXphdGlvbiB7XG4gICAgICBwdWJsaWMgX2JpbmQoKSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlLkJBU0lDLFxuICAgICAgICAgIGF1dGhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBiYXNpY0F1dGhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgIHVzZXJuYW1lOiB1c2VybmFtZSxcbiAgICAgICAgICAgICAgcGFzc3dvcmQ6IHBhc3N3b3JkLnVuc2FmZVVud3JhcCgpLCAvLyBTYWZlIHVzYWdlXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0gYXMgQ2ZuQ29ubmVjdGlvbi5BdXRoUGFyYW1ldGVyc1Byb3BlcnR5LFxuICAgICAgICB9O1xuICAgICAgfVxuICAgIH0oKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVc2UgT0F1dGggYXV0aG9yaXphdGlvblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvYXV0aChwcm9wczogT0F1dGhBdXRob3JpemF0aW9uUHJvcHMpOiBBdXRob3JpemF0aW9uIHtcbiAgICBpZiAoIVtIdHRwTWV0aG9kLlBPU1QsIEh0dHBNZXRob2QuR0VULCBIdHRwTWV0aG9kLlBVVF0uaW5jbHVkZXMocHJvcHMuaHR0cE1ldGhvZCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignaHR0cE1ldGhvZCBtdXN0IGJlIG9uZSBvZiBHRVQsIFBPU1QsIFBVVCcpO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBBdXRob3JpemF0aW9uIHtcbiAgICAgIHB1YmxpYyBfYmluZCgpIHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogQXV0aG9yaXphdGlvblR5cGUuT0FVVEhfQ0xJRU5UX0NSRURFTlRJQUxTLFxuICAgICAgICAgIGF1dGhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBvQXV0aFBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgYXV0aG9yaXphdGlvbkVuZHBvaW50OiBwcm9wcy5hdXRob3JpemF0aW9uRW5kcG9pbnQsXG4gICAgICAgICAgICAgIGNsaWVudFBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgICBjbGllbnRJZDogcHJvcHMuY2xpZW50SWQsXG4gICAgICAgICAgICAgICAgY2xpZW50U2VjcmV0OiBwcm9wcy5jbGllbnRTZWNyZXQudW5zYWZlVW53cmFwKCksIC8vIFNhZmUgdXNhZ2VcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgaHR0cE1ldGhvZDogcHJvcHMuaHR0cE1ldGhvZCxcbiAgICAgICAgICAgICAgb0F1dGhIdHRwUGFyYW1ldGVyczoge1xuICAgICAgICAgICAgICAgIGJvZHlQYXJhbWV0ZXJzOiByZW5kZXJIdHRwUGFyYW1ldGVycyhwcm9wcy5ib2R5UGFyYW1ldGVycyksXG4gICAgICAgICAgICAgICAgaGVhZGVyUGFyYW1ldGVyczogcmVuZGVySHR0cFBhcmFtZXRlcnMocHJvcHMuaGVhZGVyUGFyYW1ldGVycyksXG4gICAgICAgICAgICAgICAgcXVlcnlTdHJpbmdQYXJhbWV0ZXJzOiByZW5kZXJIdHRwUGFyYW1ldGVycyhwcm9wcy5xdWVyeVN0cmluZ1BhcmFtZXRlcnMpLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9IGFzIENmbkNvbm5lY3Rpb24uQXV0aFBhcmFtZXRlcnNQcm9wZXJ0eSxcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICB9KCk7XG5cbiAgfVxuXG4gIC8qKlxuICAgKiBCaW5kIHRoZSBhdXRob3JpemF0aW9uIHRvIHRoZSBjb25zdHJ1Y3QgYW5kIHJldHVybiB0aGUgYXV0aG9yaXphdGlvbiBwcm9wZXJ0aWVzXG4gICAqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IF9iaW5kKCk6IEF1dGhvcml6YXRpb25CaW5kUmVzdWx0O1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBBdXRob3JpemF0aW9uLm9hdXRoKClgXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT0F1dGhBdXRob3JpemF0aW9uUHJvcHMge1xuXG4gIC8qKlxuICAgKiBUaGUgVVJMIHRvIHRoZSBhdXRob3JpemF0aW9uIGVuZHBvaW50XG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uRW5kcG9pbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG1ldGhvZCB0byB1c2UgZm9yIHRoZSBhdXRob3JpemF0aW9uIHJlcXVlc3QuXG4gICAqXG4gICAqIChDYW4gb25seSBjaG9vc2UgUE9TVCwgR0VUIG9yIFBVVCkuXG4gICAqL1xuICByZWFkb25seSBodHRwTWV0aG9kOiBIdHRwTWV0aG9kO1xuXG4gIC8qKlxuICAgKiBUaGUgY2xpZW50IElEIHRvIHVzZSBmb3IgT0F1dGggYXV0aG9yaXphdGlvbiBmb3IgdGhlIGNvbm5lY3Rpb24uXG4gICAqL1xuICByZWFkb25seSBjbGllbnRJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY2xpZW50IHNlY3JldCBhc3NvY2lhdGVkIHdpdGggdGhlIGNsaWVudCBJRCB0byB1c2UgZm9yIE9BdXRoIGF1dGhvcml6YXRpb24gZm9yIHRoZSBjb25uZWN0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgY2xpZW50U2VjcmV0OiBTZWNyZXRWYWx1ZTtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBzdHJpbmcgcGFyYW1ldGVycyB0byBhZGQgdG8gdGhlIE9BdXRoIHJlcXVlc3QgYm9keVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGFkZGl0aW9uYWwgcGFyYW1ldGVyc1xuICAgKi9cbiAgcmVhZG9ubHkgYm9keVBhcmFtZXRlcnM/OiBSZWNvcmQ8c3RyaW5nLCBIdHRwUGFyYW1ldGVyPjtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBzdHJpbmcgcGFyYW1ldGVycyB0byBhZGQgdG8gdGhlIE9BdXRoIHJlcXVlc3QgaGVhZGVyXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBwYXJhbWV0ZXJzXG4gICAqL1xuICByZWFkb25seSBoZWFkZXJQYXJhbWV0ZXJzPzogUmVjb3JkPHN0cmluZywgSHR0cFBhcmFtZXRlcj47XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgc3RyaW5nIHBhcmFtZXRlcnMgdG8gYWRkIHRvIHRoZSBPQXV0aCByZXF1ZXN0IHF1ZXJ5IHN0cmluZ1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGFkZGl0aW9uYWwgcGFyYW1ldGVyc1xuICAgKi9cbiAgcmVhZG9ubHkgcXVlcnlTdHJpbmdQYXJhbWV0ZXJzPzogUmVjb3JkPHN0cmluZywgSHR0cFBhcmFtZXRlcj47XG59XG5cbi8qKlxuICogQW4gYWRkaXRpb25hbCBIVFRQIHBhcmFtZXRlciB0byBzZW5kIGFsb25nIHdpdGggdGhlIE9BdXRoIHJlcXVlc3RcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEh0dHBQYXJhbWV0ZXIge1xuICAvKipcbiAgICogTWFrZSBhbiBPQXV0aFBhcmFtZXRlciBmcm9tIGEgc3RyaW5nIHZhbHVlXG4gICAqXG4gICAqIFRoZSB2YWx1ZSBpcyBub3QgdHJlYXRlZCBhcyBhIHNlY3JldC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVN0cmluZyh2YWx1ZTogc3RyaW5nKTogSHR0cFBhcmFtZXRlciB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIEh0dHBQYXJhbWV0ZXIge1xuICAgICAgcHVibGljIF9yZW5kZXIobmFtZTogc3RyaW5nKSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAga2V5OiBuYW1lLFxuICAgICAgICAgIHZhbHVlLFxuICAgICAgICB9IGFzIENmbkNvbm5lY3Rpb24uUGFyYW1ldGVyUHJvcGVydHk7XG4gICAgICB9XG4gICAgfSgpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ha2UgYW4gT0F1dGhQYXJhbWV0ZXIgZnJvbSBhIHNlY3JldFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tU2VjcmV0KHZhbHVlOiBTZWNyZXRWYWx1ZSk6IEh0dHBQYXJhbWV0ZXIge1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBIdHRwUGFyYW1ldGVyIHtcbiAgICAgIHB1YmxpYyBfcmVuZGVyKG5hbWU6IHN0cmluZykge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGtleTogbmFtZSxcbiAgICAgICAgICB2YWx1ZTogdmFsdWUudW5zYWZlVW53cmFwKCksIC8vIFNhZmUgdXNhZ2VcbiAgICAgICAgICBpc1ZhbHVlU2VjcmV0OiB0cnVlLFxuICAgICAgICB9IGFzIENmbkNvbm5lY3Rpb24uUGFyYW1ldGVyUHJvcGVydHk7XG4gICAgICB9XG4gICAgfSgpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgcGFyYW10ZXIgdmFsdWVcbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgX3JlbmRlcihuYW1lOiBzdHJpbmcpOiBhbnk7XG59XG5cbi8qKlxuICogUmVzdWx0IG9mIHRoZSAnYmluZCcgb3BlcmF0aW9uIG9mIHRoZSAnQXV0aG9yaXphdGlvbicgY2xhc3NcbiAqXG4gKiBAaW50ZXJuYWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBdXRob3JpemF0aW9uQmluZFJlc3VsdCB7XG4gIC8qKlxuICAgKiBUaGUgYXV0aG9yaXphdGlvbiB0eXBlXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uVHlwZTogQXV0aG9yaXphdGlvblR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBhdXRob3JpemF0aW9uIHBhcmFtZXRlcnMgKGRlcGVuZHMgb24gdGhlIHR5cGUpXG4gICAqL1xuICByZWFkb25seSBhdXRoUGFyYW1ldGVyczogYW55O1xufVxuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgRXZlbnRCdXMgQ29ubmVjdGlvbnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQ29ubmVjdGlvbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgTmFtZSBmb3IgdGhlIGNvbm5lY3Rpb24uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25OYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIGNvbm5lY3Rpb24gY3JlYXRlZC5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29ubmVjdGlvbkFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIGZvciB0aGUgc2VjcmV0IGNyZWF0ZWQgZm9yIHRoZSBjb25uZWN0aW9uLlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBjb25uZWN0aW9uU2VjcmV0QXJuOiBzdHJpbmc7XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIHdpdGggcHJvcGVydGllcyBuZWNlc3NhcnkgdG8gaW1wb3J0IGEgcmV1c2FibGUgQ29ubmVjdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbm5lY3Rpb25BdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSBOYW1lIGZvciB0aGUgY29ubmVjdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25OYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIGNvbm5lY3Rpb24gY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFSTiBmb3IgdGhlIHNlY3JldCBjcmVhdGVkIGZvciB0aGUgY29ubmVjdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25TZWNyZXRBcm46IHN0cmluZztcbn1cblxuLyoqXG4gKiBEZWZpbmUgYW4gRXZlbnRCcmlkZ2UgQ29ubmVjdGlvblxuICpcbiAqIEByZXNvdXJjZSBBV1M6OkV2ZW50czo6Q29ubmVjdGlvblxuICovXG5leHBvcnQgY2xhc3MgQ29ubmVjdGlvbiBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUNvbm5lY3Rpb24ge1xuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIGNvbm5lY3Rpb24gcmVzb3VyY2VcbiAgICogQHBhcmFtIHNjb3BlIFBhcmVudCBjb25zdHJ1Y3RcbiAgICogQHBhcmFtIGlkIENvbnN0cnVjdCBJRFxuICAgKiBAcGFyYW0gY29ubmVjdGlvbkFybiBBUk4gb2YgaW1wb3J0ZWQgY29ubmVjdGlvblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tRXZlbnRCdXNBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgY29ubmVjdGlvbkFybjogc3RyaW5nLCBjb25uZWN0aW9uU2VjcmV0QXJuOiBzdHJpbmcpOiBJQ29ubmVjdGlvbiB7XG4gICAgY29uc3QgcGFydHMgPSBTdGFjay5vZihzY29wZSkucGFyc2VBcm4oY29ubmVjdGlvbkFybik7XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydGVkQ29ubmVjdGlvbihzY29wZSwgaWQsIHtcbiAgICAgIGNvbm5lY3Rpb25Bcm46IGNvbm5lY3Rpb25Bcm4sXG4gICAgICBjb25uZWN0aW9uTmFtZTogcGFydHMucmVzb3VyY2VOYW1lIHx8ICcnLFxuICAgICAgY29ubmVjdGlvblNlY3JldEFybjogY29ubmVjdGlvblNlY3JldEFybixcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgY29ubmVjdGlvbiByZXNvdXJjZVxuICAgKiBAcGFyYW0gc2NvcGUgUGFyZW50IGNvbnN0cnVjdFxuICAgKiBAcGFyYW0gaWQgQ29uc3RydWN0IElEXG4gICAqIEBwYXJhbSBhdHRycyBJbXBvcnRlZCBjb25uZWN0aW9uIHByb3BlcnRpZXNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNvbm5lY3Rpb25BdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBDb25uZWN0aW9uQXR0cmlidXRlcyk6IElDb25uZWN0aW9uIHtcbiAgICByZXR1cm4gbmV3IEltcG9ydGVkQ29ubmVjdGlvbihzY29wZSwgaWQsIGF0dHJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgTmFtZSBmb3IgdGhlIGNvbm5lY3Rpb24uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBjb25uZWN0aW9uIGNyZWF0ZWQuXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gZm9yIHRoZSBzZWNyZXQgY3JlYXRlZCBmb3IgdGhlIGNvbm5lY3Rpb24uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uU2VjcmV0QXJuOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENvbm5lY3Rpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5jb25uZWN0aW9uTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGF1dGhCaW5kID0gcHJvcHMuYXV0aG9yaXphdGlvbi5fYmluZCgpO1xuXG4gICAgY29uc3QgaW52b2NhdGlvbkh0dHBQYXJhbWV0ZXJzID0gISFwcm9wcy5oZWFkZXJQYXJhbWV0ZXJzIHx8ICEhcHJvcHMucXVlcnlTdHJpbmdQYXJhbWV0ZXJzIHx8ICEhcHJvcHMuYm9keVBhcmFtZXRlcnMgPyB7XG4gICAgICBib2R5UGFyYW1ldGVyczogcmVuZGVySHR0cFBhcmFtZXRlcnMocHJvcHMuYm9keVBhcmFtZXRlcnMpLFxuICAgICAgaGVhZGVyUGFyYW1ldGVyczogcmVuZGVySHR0cFBhcmFtZXRlcnMocHJvcHMuaGVhZGVyUGFyYW1ldGVycyksXG4gICAgICBxdWVyeVN0cmluZ1BhcmFtZXRlcnM6IHJlbmRlckh0dHBQYXJhbWV0ZXJzKHByb3BzLnF1ZXJ5U3RyaW5nUGFyYW1ldGVycyksXG4gICAgfSA6IHVuZGVmaW5lZDtcblxuICAgIGxldCBjb25uZWN0aW9uID0gbmV3IENmbkNvbm5lY3Rpb24odGhpcywgJ0Nvbm5lY3Rpb24nLCB7XG4gICAgICBhdXRob3JpemF0aW9uVHlwZTogYXV0aEJpbmQuYXV0aG9yaXphdGlvblR5cGUsXG4gICAgICBhdXRoUGFyYW1ldGVyczoge1xuICAgICAgICAuLi5hdXRoQmluZC5hdXRoUGFyYW1ldGVycyxcbiAgICAgICAgaW52b2NhdGlvbkh0dHBQYXJhbWV0ZXJzOiBpbnZvY2F0aW9uSHR0cFBhcmFtZXRlcnMsXG4gICAgICB9LFxuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgbmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmNvbm5lY3Rpb25OYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUoY29ubmVjdGlvbi5yZWYpO1xuICAgIHRoaXMuY29ubmVjdGlvbkFybiA9IGNvbm5lY3Rpb24uYXR0ckFybjtcbiAgICB0aGlzLmNvbm5lY3Rpb25TZWNyZXRBcm4gPSBjb25uZWN0aW9uLmF0dHJTZWNyZXRBcm47XG4gIH1cbn1cblxuY2xhc3MgSW1wb3J0ZWRDb25uZWN0aW9uIGV4dGVuZHMgUmVzb3VyY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbkFybjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbk5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25TZWNyZXRBcm46IHN0cmluZztcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IENvbm5lY3Rpb25BdHRyaWJ1dGVzKSB7XG4gICAgY29uc3QgYXJuUGFydHMgPSBTdGFjay5vZihzY29wZSkucGFyc2VBcm4oYXR0cnMuY29ubmVjdGlvbkFybik7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBhY2NvdW50OiBhcm5QYXJ0cy5hY2NvdW50LFxuICAgICAgcmVnaW9uOiBhcm5QYXJ0cy5yZWdpb24sXG4gICAgfSk7XG5cbiAgICB0aGlzLmNvbm5lY3Rpb25Bcm4gPSBhdHRycy5jb25uZWN0aW9uQXJuO1xuICAgIHRoaXMuY29ubmVjdGlvbk5hbWUgPSBhdHRycy5jb25uZWN0aW9uTmFtZTtcbiAgICB0aGlzLmNvbm5lY3Rpb25TZWNyZXRBcm4gPSBhdHRycy5jb25uZWN0aW9uU2VjcmV0QXJuO1xuICB9XG59XG5cbi8qKlxuICogU3VwcG9ydGVkIEhUVFAgb3BlcmF0aW9ucy5cbiAqL1xuZXhwb3J0IGVudW0gSHR0cE1ldGhvZCB7XG4gIC8qKiBQT1NUICovXG4gIFBPU1QgPSAnUE9TVCcsXG4gIC8qKiBHRVQgKi9cbiAgR0VUID0gJ0dFVCcsXG4gIC8qKiBIRUFEICovXG4gIEhFQUQgPSAnSEVBRCcsXG4gIC8qKiBPUFRJT05TICovXG4gIE9QVElPTlMgPSAnT1BUSU9OUycsXG4gIC8qKiBQVVQgKi9cbiAgUFVUID0gJ1BVVCcsXG4gIC8qKiBQQVRDSCAqL1xuICBQQVRDSCA9ICdQQVRDSCcsXG4gIC8qKiBERUxFVEUgKi9cbiAgREVMRVRFID0gJ0RFTEVURScsXG59XG5cbi8qKlxuICogU3VwcG9ydGVkIEF1dGhvcml6YXRpb24gVHlwZXMuXG4gKi9cbmVudW0gQXV0aG9yaXphdGlvblR5cGUge1xuICAvKiogQVBJX0tFWSAqL1xuICBBUElfS0VZID0gJ0FQSV9LRVknLFxuICAvKiogQkFTSUMgKi9cbiAgQkFTSUMgPSAnQkFTSUMnLFxuICAvKiogT0FVVEhfQ0xJRU5UX0NSRURFTlRJQUxTICovXG4gIE9BVVRIX0NMSUVOVF9DUkVERU5USUFMUyA9ICdPQVVUSF9DTElFTlRfQ1JFREVOVElBTFMnLFxufVxuXG5mdW5jdGlvbiByZW5kZXJIdHRwUGFyYW1ldGVycyhwcz86IFJlY29yZDxzdHJpbmcsIEh0dHBQYXJhbWV0ZXI+KTogQ2ZuQ29ubmVjdGlvbi5QYXJhbWV0ZXJQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgaWYgKCFwcyB8fCBPYmplY3Qua2V5cyhwcykubGVuZ3RoID09PSAwKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cblxuICByZXR1cm4gT2JqZWN0LmVudHJpZXMocHMpLm1hcCgoW25hbWUsIHBdKSA9PiBwLl9yZW5kZXIobmFtZSkpO1xufSJdfQ==