from __future__ import print_function

#   _   _   ___   ___   ___  _ __   ___   __ _  _ __ ___
#  | | | | / __| / _ \ / __|| '__| / _ \ / _` || '_ ` _ \
#   \__, || (__ |  __/| (__ | |   |  __/| (_| || | | | | |
#   |___/  \___| \___| \___||_|    \___| \__,_||_| |_| |_|
#                       sweeter debugging and benchmarking

__version__ = "1.3.14"

"""
See https://github.com/salabim/ycecream for details

(c)2023 Ruud van der Ham - rt.van.der.ham@gmail.com

Inspired by IceCream "Never use print() to debug again".
Also contains some of the original code.
IceCream was written by Ansgar Grunseid / grunseid.com / grunseid@gmail.com
"""


def copy_contents(package, prefer_installed, filecontents):
    import tempfile
    import shutil
    import sys
    from pathlib import Path
    import zlib
    import base64
    if package in sys.modules:
        return
    if prefer_installed:
        for dir in sys.path:
            dir = Path(dir)
            if (dir / package).is_dir() and (dir / package / '__init__.py').is_file():
                return
            if (dir / (package + '.py')).is_file():
                return
    target_dir = Path(tempfile.gettempdir()) / ('embedded_' + package) 
    if target_dir.is_dir():
        shutil.rmtree(target_dir, ignore_errors=True)
    for file, contents in filecontents:
        ((target_dir / file).parent).mkdir(parents=True, exist_ok=True)
        with open(target_dir / file, 'wb') as f:
            f.write(zlib.decompress(base64.b64decode(contents)))
    sys.path.insert(prefer_installed * len(sys.path), str(target_dir))
copy_contents(package='executing', prefer_installed=False, filecontents=(
    ('executing/executing.py',b'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'),
    ('executing/py.typed',b'eJwDAAAAAAE='),
    ('executing/version.py',b'eJyLjy9LLSrOzM+Lj1ewVVA31DPSM1AHAFGdBj0='),
    ('executing/_exceptions.py',b'eJylz8FOwzAMBuB7nsIal1bq+gCTxm1ICMEBIS4IVWnmtoEsjmJHbG9Pk6FuHDjhSyIl9v9ZGaeZ4cHTl79nTljtjgaDWPL1RsFcq9WqnM/aMu7BejCaEWgA7eEz90GI1Ds8tPBILO4EPRqdzn9MOMlEHvo0clvm7I5oklg/tp72CCMKA6OAEDyRxxwgk+WS0i6CcgkzVSl1Jr9itIPFeKetS/Fv98sUs3GgmMHJ4zGgkXmTgw5hZsAQ6TCnssRkcn+WaBbIvP+K9zhA11lvpesqRjc0IFYcNmV6cx37w851A3IKuIGK+o/Z2sDvs4b1bUleGvLkM257YS8P17ttryPV5V8KGKu6XaxFWS8SO3qK+Ga0c2sdx3f1DRnEs2M='),
    ('executing/_position_node_finder.py',b'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'),
    ('executing/__init__.py',b'eJyNUsFupDAMvecrrPQArCjSXpGq0R6m1V52D616qSrkCYZJFRIUnLb8fQMZOmhPGwkRPfvZz36RUooHYtC2c35A1s4Cnlxg+DgjA0LncSDQE6jgPVk2M7RO276Cp3nUCg2ECXuqhYB49DA6z0CfpALHrIRa1xLcXdHq0QWvqPoG8rVLUS2JK+MmcT60MXAiQAu/Hp8S5jz8cZaEjMpF590AyhlDapE+bQJsLNdyGA2J5pn8FGO/44RRxDWSZ/tQVkKeDfjmfLxlg7bbRXmXFUXqdFW8NUqTlHDcAiU4a+YyauS/lp7RBLp3wbYlKFTnmNkFDp6azmA/CfZzvQ6cyr8nPVvxprkATRM3wvNINeg+KqOXlPGalt6BbOldRhP3lFR4OR71RLBqOXrvvKBPRSNH0csvJteXpV+41TgvjmNgd9uTJY9MbTSDz8Bngl4zMPbxhZCFU9CmXZxeKuwF34E8HA7yH3x5Z2twv/v89mcJ6SsEmYnq/2L9GHDMteVy37eaRqM5z6poWiGEaBo0ZuW+yGSWfBVfV3/8ag=='),
))
copy_contents(package='asttokens', prefer_installed=False, filecontents=(
    ('asttokens/astroid_compat.py',b'eJx1kE0OwiAQhfec4qVuTQ/QxKVx1wsYQ7CMQkJLA9Sf20uhmrS2K8KbefO+meDeFQNuzrYQPjirJXTbWxfQWUk+il+djwJvjPCePIum3WoFXtnBSFwJT6UbRQ9yaK0cDEHFaUER6thfn5Aci/Qyx04MuZHRq6E+4JgebbsROWTybfYf0NYKS3M5c80JgD+G8QC9E/dWVDEPjY2LppmrVznESR2l+rT9pDDOhTGcx/+5WLMWexTZUlzYB8DNlgI='),
    ('asttokens/asttokens.py',b'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'),
    ('asttokens/line_numbers.py',b'eJydVk1v4zYQvetXDJxDZKyiOHsoirRewJukW6OpA8TeLhaGa9DyyGYrkSpJrWMU/e8dfkiWnRgFmoMUkcM3M+89kr6AO1ntFd9sDbwf3HwHnxTXBh7ZSicwFlkaXUQX8MgzFBrXUIs1KjBbhFHFMnqFmQR+Q6W5FPA+HUBsA3phqtf/gRD2soaS7UFIA7VGguAacl4g4EuGlQEuIJNlVXAmMoQdN1uXJoBQGfA1QMiVYRTNKL6ir7wbB8y4gulva0x1e3292+1S5mpNpdpcFz5OXz+O7x4m04crqtet+CwK1BoU/lVzRZ2u9sAqKidjKyqyYDuQCthGIc0ZacvdKW642CSgZW52TCGhrIk8xVe1OeKqKY5a7gYQW0xAbzSF8bQHH0fT8TQhjC/j2c9Pn2fwZfT8PJrMxg9TeHqGu6fJ/Xg2fprQ108wmnyFX8aT+wSQmKI0+FIpWz8VyS2LuLaUTRGPCsilL0hXmPGcZ9SX2NRsg7CR31AJagcqVCXXVktN5a0JpeAlN8y4kVdNpVFECaUysOIaM9N8ER25kiWYfWVRw+g9z0xCSzU9Z3VVYBQtCy5wqQ1TZqkQhrQytU4gb8Tq8vfLxA782o+irGDU4CNFT+pyRXaL5eoPyti/jQB6vR4971wIyZNJQf1QTWh2iGStLVMsM1S6zHONRoNzkMEXA1YQKyM1CxXjSkNsS0oIpKhL0bcWu7laMfI/pbBTLnTgh0IUCF8TwWhrcGcgI/9EzyKMpjPy/ho18QUws+73/ZBw1IMllmc07yTioqqNW6aNJFmhkpp7/qnsEJnCRwriVH9dWW414RrFhC6YtaVlwSI4EWqTf990nsBuy7MtkGHbQhlt+YrRBhabtCVzjTksl1xws1zGGos8cXw5ugEurLJ4CzHR14erDzCRAsPMiFgiROLtNe80iLQbHY+X9ghQFNmGpQ7BJku9L5plQ5iXqTNJPOg7kkrLxbF50pyLNSeY2BW66IA5pYeugZPRZUH+GAI9/arOtKUtVLDM3HE3hL//aVu3Zp5zEQxt/1ssaLJklfY2IU9YGbpkBB5cB46H1d6g7cQuiALrVjOfnMwVqPeWbAatL090cIXQw4lBbzfrlQT4xL+hNXzwcVseWfbIzM4oHt/uO1Mr69/DdDCpj0iPUhyUOkNeukHjdpanmOeHvaide3xDACjWYWEL1vXC3H4s7HrXxY9OutdhfcCCjrxjoUMGFxcscQiYn8kFV3CzgNtOXYsoAHXsyf3WTTKrJhK3qJjBuE3lXdvInaUoLJXxpeXpMni1BaQrq6J0sW3tgNAPQWcIDsQMGxBfpFexGZuX7CUeJFBy4cAbsq5ujs3VXzRm9OllyHTkxv9hxDt/MOv/dGI4Vmn7tCMnZ4nFlm5HHxvRSzaEm8ZmbuDD8JxPGtsFot4wDBYHsw1OwgchQuPJhGX4rHnfQdAhUJicpO037Ad9iX+7MLDvB9/g3VHu7tV5o8LibfrPkkqU2kuwcwd2L8Bz158DD1fgW8cCmbJjvONe2376rXx066zxhRb53xSpfy3dz9Q3SG0R7E7tmj7ugL2DmyaOws4o42MXRH/0Lwbcom0='),
    ('asttokens/mark_tokens.py',b'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'),
    ('asttokens/py.typed',b'eJwDAAAAAAE='),
    ('asttokens/util.py',b'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'),
    ('asttokens/version.py',b'eJyLjy9LLSrOzM+Lj1ewVVAy0jPSM1TiAgBXvAY/'),
    ('asttokens/__init__.py',b'eJxlU8FunDAQvfMVI3LYVqJsmkMPrXqgSdqirtgokEZRVa28MIAVsKk9lN2/79jQ1UZBK7H2PL957425gGs9HI1sWoKry/cf4JuRlmAj9jaCVJVxcBFcwEaWqCxWMKoKDVCLkAyi5NdSieAnGiu1gqv4Et44QLiUwrefmOGoR+jFEZQmGC0yhbRQyw4BDyUOBFJBqfuhk0KVCJOk1rdZSFgGPC0Uek+C0YLxA6/qcxwI8oL5aYmGj+v1NE2x8FpjbZp1N+PsepNe32b57TvW6088qA6tBYN/RmnY6f4IYmA5pdizyE5MoA2IxiDXSDu5k5EkVROB1TVNwiCzVByekfuRXmT1XxxbPgdwWkJBmOSQ5iF8SfI0j5jjMS2+bx8KeEzu75OsSG9z2N7D9Ta7SYt0m/HqKyTZE/xIs5sIkJPiNngYjNPPIqVLESsXWY74QkCtZ0F2wFLWsmRfqhlFg9Dov2gU24EBTS+tm6VleRWzdLKXJMjvvDIVB0EYhkHh5tnranQTVa0bovWwuyO1bDTJCyAOb5msfkZnvuLsRsPzLnXF8SjW1/tGHOpY164DMTiokLCkWfp8gPBAbvbIk/Xkihli8DL412I31GM3G9a6c1oE8fV5xvOWARmh7Kmrjb2XoDa6h7iTCndq7Pd8s12o2vB3wXvZvLWghCXv5gRhMQWLi/wfX3FmBlezOw91Fy0IdjvRdbsdfIZfq+XMKoLV6ZRbnLVzy9c0q9/BP/0mR0E='),
))
copy_contents(package='six', prefer_installed=False, filecontents=(
    ('six.py',b'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'),
))
del copy_contents

import inspect
import sys
import datetime
import time
import textwrap
import contextlib
import functools
import json
import logging
import collections
import numbers
import ast
import os
import copy
import traceback
import executing

nv = object()

PY2 = sys.version_info.major == 2
PY3 = sys.version_info.major == 3

if PY2:

    def ycecream_pformat(obj, width, indent, depth):
        return pformat(obj, width=width, indent=indent, depth=depth).replace(
            "\\n", "\n"
        )


if PY3:

    def perf_counter():
        return (
            time.perf_counter() if _fixed_perf_counter is None else _fixed_perf_counter
        )

    from pathlib import Path

    def ycecream_pformat(obj, width, compact, indent, depth, sort_dicts):
        return pformat(
            obj,
            width=width,
            compact=compact,
            indent=indent,
            depth=depth,
            sort_dicts=sort_dicts,
        ).replace("\\n", "\n")


class Source(executing.Source):
    def get_text_with_indentation(self, node):
        result = self.asttokens().get_text(node)
        if "\n" in result:
            result = " " * node.first_token.start[1] + result
            result = dedent(result)
        result = result.strip()
        return result


class Default(object):
    pass


default = Default()


def change_path(new_path):  # used in tests
    global Path
    Path = new_path


_fixed_perf_counter = None


def fix_perf_counter(val):  # for tests
    global _fixed_perf_counter
    _fixed_perf_counter = val


shortcut_to_name = {
    "p": "prefix",
    "o": "output",
    "sln": "show_line_number",
    "st": "show_time",
    "sd": "show_delta",
    "sdi": "sort_dicts",
    "se": "show_enter",
    "sx": "show_exit",
    "stb": "show_traceback",
    "e": "enabled",
    "ll": "line_length",
    "c": "compact",
    "i": "indent",
    "de": "depth",
    "wi": "wrap_indent",
    "cs": "context_separator",
    "sep": "separator",
    "es": "equals_separator",
    "vo": "values_only",
    "voff": "values_only_for_fstrings",
    "rn": "return_none",
    "ell": "enforce_line_length",
    "dl": "delta",
}


def set_defaults():
    default.prefix = "y| "
    default.output = "stderr"
    default.serialize = (
        ycecream_pformat  # can't use pformat directly as that is defined later
    )
    default.show_line_number = False
    default.show_time = False
    default.show_delta = False
    default.sort_dicts = False
    default.show_enter = True
    default.show_exit = True
    default.show_traceback = False
    default.enabled = True
    default.line_length = 80
    default.compact = False
    default.indent = 1
    default.depth = 1000000
    default.wrap_indent = "    "
    default.context_separator = " ==> "
    default.separator = ", "
    default.equals_separator = ": "
    default.values_only = False
    default.values_only_for_fstrings = False
    default.return_none = False
    default.enforce_line_length = False
    default.one_line_per_pairenforce_line_length = False
    default.start_time = perf_counter()


def apply_json():
    ycecream_name = "ycecream"

    config = {}
    for path in sys.path:
        json_file = os.path.join(path, ycecream_name + ".json")
        if os.path.isfile(json_file):
            with open(json_file, "r") as f:
                config = json.load(f)
            break
        json_dir = os.path.join(path, ycecream_name)
        json_file = os.path.join(json_dir, ycecream_name + ".json")
        if os.path.isfile(json_file):
            with open(json_file, "r") as f:
                config = json.load(f)
            break

    for k, v in config.items():

        if k in ("serialize", "start_time"):
            raise ValueError(
                "error in {json_file}: key {k} not allowed".format(
                    json_file=json_file, k=k
                )
            )

        if k in shortcut_to_name:
            k = shortcut_to_name[k]
        if hasattr(default, k):
            setattr(default, k, v)
        else:
            if k == "delta":
                setattr(default, "start_time", perf_counter() - v)
            else:
                raise ValueError(
                    "error in {json_file}: key {k} not recognized".format(
                        json_file=json_file, k=k
                    )
                )


def no_source_error(s=None):
    if s is not None:
        print(s)  # for debugging only
    raise NotImplementedError(
        """
Failed to access the underlying source code for analysis. Possible causes:
- decorated function/method definition spawns more than one line
- used from a frozen application (e.g. packaged with PyInstaller)
- underlying source code was changed during execution"""
    )


def return_args(args, return_none):
    if return_none:
        return None
    if len(args) == 0:
        return None
    if len(args) == 1:
        return args[0]
    return args


class _Y(object):
    def __init__(
        self,
        prefix=nv,
        output=nv,
        serialize=nv,
        show_line_number=nv,
        show_time=nv,
        show_delta=nv,
        show_enter=nv,
        show_exit=nv,
        show_traceback=nv,
        sort_dicts=nv,
        enabled=nv,
        line_length=nv,
        compact=nv,
        indent=nv,
        depth=nv,
        wrap_indent=nv,
        context_separator=nv,
        separator=nv,
        equals_separator=nv,
        values_only=nv,
        values_only_for_fstrings=nv,
        return_none=nv,
        enforce_line_length=nv,
        #     decorator=nv,
        #     context_manager=nv,
        delta=nv,
        _parent=nv,
        **kwargs
    ):
        self._attributes = {}
        if _parent is nv:
            self._parent = default
        else:
            self._parent = _parent
        for key in vars(default):
            setattr(self, key, None)

        if _parent == default:
            func = "y.new()"
        else:
            func = "y.fork()"
        self.assign(kwargs, locals(), func=func)

        self.check()

    def __repr__(self):
        pairs = []
        for key in vars(default):
            if not key.startswith("__"):
                value = getattr(self, key)
                if not callable(value):
                    pairs.append(str(key) + "=" + repr(value))
        return "y.new(" + ", ".join(pairs) + ")"

    def __getattr__(self, item):
        if item in shortcut_to_name:
            item = shortcut_to_name[item]
        if item == "delta":
            return perf_counter() - getattr(self, "start_time")

        if item in self._attributes:
            if self._attributes[item] is None:
                return getattr(self._parent, item)
            else:
                return self._attributes[item]
        raise AttributeError("{item} not found".format(item=item))

    def __setattr__(self, item, value):
        if item in shortcut_to_name:
            item = shortcut_to_name[item]
        if item == "delta":
            item = "start_time"
            if value is not None:
                value = perf_counter() - value

        if item in ["_attributes"]:
            super(_Y, self).__setattr__(item, value)
        else:
            self._attributes[item] = value

    def assign(self, shortcuts, source, func):
        for key, value in shortcuts.items():
            if key in shortcut_to_name:
                if value is not nv:
                    full_name = shortcut_to_name[key]
                    if source[full_name] is nv:
                        source[full_name] = value
                    else:
                        raise ValueError(
                            "can't use {key} and {full_name} in {func}".format(
                                key=key, full_name=full_name, func=func
                            )
                        )
            else:
                raise TypeError(
                    "{func} got an unexpected keyword argument {key}".format(
                        func=func, key=key
                    )
                )
        for key, value in source.items():
            if value is not nv:
                if key == "delta":
                    key = "start_time"
                    if value is not None:
                        value = perf_counter() - value
                setattr(self, key, value)

    def fork(self, **kwargs):
        kwargs["_parent"] = self
        return _Y(**kwargs)

    def __call__(self, *args, **kwargs):
        prefix = kwargs.pop("prefix", nv)
        output = kwargs.pop("output", nv)
        serialize = kwargs.pop("serialize", nv)
        show_line_number = kwargs.pop("show_line_number", nv)
        show_time = kwargs.pop("show_time", nv)
        show_delta = kwargs.pop("show_delta", nv)
        show_enter = kwargs.pop("show_enter", nv)
        show_exit = kwargs.pop("show_exit", nv)
        show_traceback = kwargs.pop("show_traceback", nv)
        sort_dicts = kwargs.pop("sort_dicts", nv)
        enabled = kwargs.pop("enabled", nv)
        line_length = kwargs.pop("line_length", nv)
        compact = kwargs.pop("compact", nv)
        indent = kwargs.pop("indent", nv)
        depth = kwargs.pop("depth", nv)
        wrap_indent = kwargs.pop("wrap_indent", nv)
        context_separator = kwargs.pop("context_separator", nv)
        separator = kwargs.pop("separator", nv)
        equals_separator = kwargs.pop("equals_separator", nv)
        values_only = kwargs.pop("values_only", nv)
        values_only_for_fstrings = kwargs.pop("values_only_for_fstrings", nv)
        return_none = kwargs.pop("return_none", nv)
        enforce_line_length = kwargs.pop("enforce_line_length", nv)
        decorator = kwargs.pop("decorator", nv)
        d = kwargs.pop("decorator", nv)
        context_manager = kwargs.pop("context_manager", nv)
        cm = kwargs.pop("cm", nv)
        delta = kwargs.pop("delta", nv)
        as_str = kwargs.pop("as_str", nv)
        provided = kwargs.pop("provided", nv)
        pr = kwargs.pop("pr", nv)

        if d is not nv and decorator is not nv:
            raise TypeError("can't use both d and decorator")
        if cm is not nv and context_manager is not nv:
            raise TypeError("can't use both cm and context_manager")
        if pr is not nv and provided is not nv:
            raise TypeError("can't use both pr and provided")

        as_str = False if as_str is nv else bool(as_str)
        provided = True if provided is nv else bool(provided)
        decorator = False if decorator is nv else bool(decorator)
        context_manager = False if context_manager is nv else bool(context_manager)

        if decorator and context_manager:
            raise TypeError("decorator and context_manager can't be specified both.")

        self.is_context_manager = False

        Pair = collections.namedtuple("Pair", "left right")

        this = self.fork()
        this.assign(kwargs, locals(), func="__call__")

        if this.enabled == [] and not (
            as_str or this.decorator or this.context_manager
        ):
            return return_args(args, this.return_none)

        if not provided:
            this.enabled = False

        this.check()

        call_frame = inspect.currentframe()
        filename0 = call_frame.f_code.co_filename

        call_frame = call_frame.f_back
        filename = call_frame.f_code.co_filename

        if filename == filename0:
            call_frame = call_frame.f_back
            filename = call_frame.f_code.co_filename

        if filename in ("<stdin>", "<string>"):
            filename_name = ""
            code = "\n\n"
            this_line = ""
            this_line_prev = ""
            line_number = 0
            parent_function = ""
        else:
            try:
                main_file = sys.modules["__main__"].__file__
                main_file_resolved = os.path.abspath(main_file)
            except AttributeError:
                main_file_resolved = None
            filename_resolved = os.path.abspath(filename)
            if (
                (filename.startswith("<") and filename.endswith(">"))
                or (main_file_resolved is None)
                or (filename_resolved == main_file_resolved)
            ):
                filename_name = ""
            else:
                filename_name = "[" + os.path.basename(filename) + "]"

            if filename not in codes:
                frame_info = inspect.getframeinfo(
                    call_frame, context=1000000
                )  # get the full source code
                if frame_info.code_context is None:
                    no_source_error()
                codes[filename] = frame_info.code_context
            code = codes[filename]
            frame_info = inspect.getframeinfo(call_frame, context=1)

            parent_function = frame_info.function  # changed in version 1.3.10 ****
            parent_function = Source.executing(call_frame).code_qualname()
            parent_function = parent_function.replace(".<locals>.", ".")
            if parent_function == "<module>" or str(this.show_line_number) in (
                "n",
                "no parent",
            ):
                parent_function = ""
            else:
                parent_function = " in {parent_function}()".format(
                    parent_function=parent_function
                )
            line_number = frame_info.lineno
            if 0 <= line_number - 1 < len(code):
                this_line = code[line_number - 1].strip()
            else:
                this_line = ""
            if 0 <= line_number - 2 < len(code):
                this_line_prev = code[line_number - 2].strip()
            else:
                this_line_prev = ""
        if (
            this_line.startswith("@")
            or this_line_prev.startswith("@")
            or this.decorator
        ):
            if as_str:
                raise TypeError("as_str may not be True when y used as decorator")

            for ln, line in enumerate(code[line_number - 1 :], line_number):
                if line.strip().startswith("def") or line.strip().startswith("class"):
                    line_number = ln
                    break
            else:
                line_number += 1
            this.line_number_with_filename_and_parent = (
                "#{line_number}{filename_name}{parent_function}".format(
                    line_number=line_number,
                    filename_name=filename_name,
                    parent_function=parent_function,
                )
            )

            def real_decorator(function):
                @functools.wraps(function)
                def wrapper(*args, **kwargs):
                    enter_time = perf_counter()
                    context = this.context()

                    args_kwargs = [repr(arg) for arg in args] + [
                        "{k}={repr_v}".format(k=k, repr_v=repr(v))
                        for k, v in kwargs.items()
                    ]
                    function_arguments = (
                        function.__name__ + "(" + (", ".join(args_kwargs)) + ")"
                    )

                    if this.show_enter:
                        this.do_output(
                            "{context}called {function_arguments}{traceback}".format(
                                context=context,
                                function_arguments=function_arguments,
                                traceback=this.traceback(),
                            )
                        )
                    result = function(*args, **kwargs)
                    duration = perf_counter() - enter_time

                    context = this.context()
                    if this.show_exit:
                        this.do_output(
                            "{context}returned {repr_result} from {function_arguments} in {duration:.6f} seconds{traceback}".format(
                                context=context,
                                repr_result=repr(result),
                                function_arguments=function_arguments,
                                duration=duration,
                                traceback=this.traceback(),
                            )
                        )

                    return result

                return wrapper

            if len(args) == 0:
                return real_decorator

            if len(args) == 1 and callable(args[0]):
                return real_decorator(args[0])
            raise TypeError("arguments are not allowed in y used as decorator")

        if filename in ("<stdin>", "<string>"):
            this.line_number_with_filename_and_parent = ""
        else:
            call_node = Source.executing(call_frame).node
            if call_node is None:
                no_source_error()
            line_number = call_node.lineno
            this_line = code[line_number - 1].strip()

            this.line_number_with_filename_and_parent = (
                "#{line_number}{filename_name}{parent_function}".format(
                    line_number=line_number,
                    filename_name=filename_name,
                    parent_function=parent_function,
                )
            )

        if (
            this_line.startswith("with ")
            or this_line.startswith("with\t")
            or this.context_manager
        ):
            if as_str:
                raise TypeError("as_str may not be True when y used as context manager")
            if args:
                raise TypeError(
                    "non-keyword arguments are not allowed when y used as context manager"
                )

            this.is_context_manager = True
            return this

        if not this.enabled and not as_str:
            return return_args(args, this.return_none)

        if args:
            context = this.context()
            pairs = []
            if filename in ("<stdin>", "<string>") or this.values_only:
                for right in args:
                    pairs.append(Pair(left="", right=right))
            else:
                source = Source.for_frame(call_frame)
                for node, right in zip(call_node.args, args):
                    left = source.asttokens().get_text(node)
                    if "\n" in left:
                        left = " " * node.first_token.start[1] + left
                        left = textwrap.dedent(left)
                    try:
                        ast.literal_eval(left)  # it's indeed a literal
                        left = ""
                    except Exception:
                        pass
                    if left:
                        try:
                            if isinstance(left, str):
                                s = ast.parse(left, mode="eval")
                            if isinstance(s, ast.Expression):
                                s = s.body
                            if s and isinstance(
                                s, ast.JoinedStr
                            ):  # it is indeed an f-string
                                if this.values_only_for_fstrings:
                                    left = ""
                        except Exception:
                            pass
                    if left:
                        left += this.equals_separator
                    pairs.append(Pair(left=left, right=right))

            just_one_line = False
            if not (len(pairs) > 1 and this.separator == ""):
                if not any("\n" in pair.left for pair in pairs):
                    as_one_line = context + this.separator.join(
                        pair.left + this.serialize_kwargs(obj=pair.right, width=10000)
                        for pair in pairs
                    )
                    if len(as_one_line) <= this.line_length and "\n" not in as_one_line:
                        out = as_one_line
                        just_one_line = True

            if not just_one_line:
                if isinstance(this.wrap_indent, numbers.Number):
                    wrap_indent = int(this.wrap_indent) * " "
                else:
                    wrap_indent = str(this.wrap_indent)

                if context.strip():
                    if len(context.rstrip()) >= len(wrap_indent):
                        indent1 = wrap_indent
                        indent1_rest = wrap_indent
                        lines = [context]
                    else:
                        indent1 = context.rstrip().ljust(len(wrap_indent))
                        indent1_rest = wrap_indent
                        lines = []
                else:
                    indent1 = ""
                    indent1_rest = ""
                    lines = []

                for pair in pairs:
                    do_right = False
                    if "\n" in pair.left:
                        for s in pair.left.splitlines():
                            lines.append(indent1 + s)
                            do_right = True
                    else:
                        start = indent1 + pair.left
                        line = start + this.serialize_kwargs(
                            obj=pair.right, width=this.line_length - len(start)
                        )
                        if "\n" in line:
                            lines.append(start)
                            do_right = True
                        else:
                            lines.append(line)
                    indent1 = indent1_rest
                    if do_right:
                        indent2 = indent1 + wrap_indent
                        line = this.serialize_kwargs(
                            obj=pair.right, width=this.line_length - len(indent2)
                        )
                        for s in line.splitlines():
                            lines.append(indent2 + s)

                out = "\n".join(line.rstrip() for line in lines)

        else:
            if not this.show_line_number:  # if "n" or "no parent", keep that info
                this.show_line_number = True
            out = this.context(omit_context_separator=True)

        if this.show_traceback:
            out += this.traceback()

        if as_str:
            if this.enabled:
                if this.enforce_line_length:
                    out = "\n".join(
                        line[: this.line_length] for line in out.splitlines()
                    )
                return out + "\n"
            else:
                return ""
        this.do_output(out)

        return return_args(args, this.return_none)

    def configure(
        self,
        prefix=nv,
        output=nv,
        serialize=nv,
        show_line_number=nv,
        show_time=nv,
        show_delta=nv,
        show_enter=nv,
        show_exit=nv,
        show_traceback=nv,
        sort_dicts=nv,
        enabled=nv,
        line_length=nv,
        compact=nv,
        indent=nv,
        depth=nv,
        wrap_indent=nv,
        context_separator=nv,
        separator=nv,
        equals_separator=nv,
        values_only=nv,
        values_only_for_fstrings=nv,
        return_none=nv,
        enforce_line_length=nv,
        #        decorator=nv,
        #        context_manager=nv,
        delta=nv,
        **kwargs
    ):
        self.assign(kwargs, locals(), "configure()")
        self.check()
        return self

    def new(self, ignore_json=False, **kwargs):
        if ignore_json:
            return _Y(_parent=default_pre_json, **kwargs)
        else:
            return _Y(**kwargs)

    def clone(
        self,
        prefix=nv,
        output=nv,
        serialize=nv,
        show_line_number=nv,
        show_time=nv,
        show_delta=nv,
        show_enter=nv,
        show_exit=nv,
        show_traceback=nv,
        sort_dicts=nv,
        enabled=nv,
        line_length=nv,
        compact=nv,
        indent=nv,
        depth=nv,
        wrap_indent=nv,
        context_separator=nv,
        separator=nv,
        equals_separator=nv,
        values_only=nv,
        values_only_for_fstrings=nv,
        return_none=nv,
        enforce_line_length=nv,
        #        decorator=nv,
        #        context_manager=nv,
        delta=nv,
        **kwargs
    ):
        this = _Y(_parent=self._parent)
        this.assign({}, self._attributes, func="clone()")
        this.assign(kwargs, locals(), func="clone()")

        return this

    def assert_(self, condition):
        if self.enabled:
            assert condition

    @contextlib.contextmanager
    def preserve(self):
        save = dict(self._attributes)
        yield
        self._attributes = save

    def __enter__(self):
        if not hasattr(self, "is_context_manager"):
            raise ValueError("not allowed as context_manager")
        self.save_traceback = self.traceback()
        self.enter_time = perf_counter()
        if self.show_enter:
            context = self.context()
            self.do_output(context + "enter" + self.save_traceback)
        return self

    def __exit__(self, *args):
        if self.show_exit:
            context = self.context()
            duration = perf_counter() - self.enter_time
            self.do_output(
                "{context}exit in {duration:.6f} seconds{traceback}".format(
                    context=context, duration=duration, traceback=self.save_traceback
                )
            )
        self.is_context_manager = False

    def context(self, omit_context_separator=False):
        if self.show_line_number and self.line_number_with_filename_and_parent != "":
            parts = [self.line_number_with_filename_and_parent]
        else:
            parts = []
        if self.show_time:
            parts.append("@ " + str(datetime.datetime.now().strftime("%H:%M:%S.%f")))

        if self.show_delta:
            t0 = perf_counter() - self.start_time
            parts.append("delta={t0:.3f}".format(t0=t0))

        context = " ".join(parts)
        if not omit_context_separator and context:
            context += self.context_separator

        return (self.prefix() if callable(self.prefix) else self.prefix) + context

    def do_output(self, s):
        if self.enforce_line_length:
            s = "\n".join(line[: self.line_length] for line in s.splitlines())
        if self.enabled:
            if callable(self.output):
                self.output(s)
            elif self.output == "stderr":
                print(s, file=sys.stderr)
            elif self.output == "stdout":
                print(s, file=sys.stdout)
            elif self.output == "logging.debug":
                logging.debug(s)
            elif self.output == "logging.info":
                logging.info(s)
            elif self.output == "logging.warning":
                logging.warning(s)
            elif self.output == "logging.error":
                logging.error(s)
            elif self.output == "logging.critical":
                logging.critical(s)
            elif self.output in ("", "null"):
                pass

            elif isinstance(self.output, str):
                if PY2:
                    with io.open(self.output, "a+", encoding="utf-8") as f:
                        print(s, file=f)
                if PY3:
                    with open(self.output, "a+", encoding="utf-8") as f:
                        print(s, file=f)
            elif isinstance(self.output, Path):
                with self.output.open("a+", encoding="utf-8") as f:
                    print(s, file=f)

            else:
                print(s, file=self.output)

    def traceback(self):
        if self.show_traceback:
            if isinstance(self.wrap_indent, numbers.Number):
                wrap_indent = int(self.wrap_indent) * " "
            else:
                wrap_indent = str(self.wrap_indent)

            result = "\n" + wrap_indent + "Traceback (most recent call last)\n"
            #  Python 2.7 does not allow entry.filename, entry.line, etc, so we have to index entry
            return result + "\n".join(
                wrap_indent
                + '  File "'
                + entry[0]
                + '", line '
                + str(entry[1])
                + ", in "
                + entry[2]
                + "\n"
                + wrap_indent
                + "    "
                + entry[3]
                for entry in traceback.extract_stack()[:-2]
            )
        else:
            return ""

    def check(self):

        if callable(self.output):
            return
        if isinstance(self.output, (str, Path)):
            return
        try:
            if PY2:
                self.output.write(unicode(""))
            if PY3:
                self.output.write("")
            return

        except Exception:
            pass
        raise TypeError("output should be a callable, str, Path or open text file.")

    if PY2:

        def serialize_kwargs(self, obj, width):

            kwargs = {
                key: getattr(self, key)
                for key in ("indent", "depth")
                if key in inspect.getargspec(self.serialize).args
            }
            kwargs["width"] = width

            return self.serialize(obj, **kwargs)

    if PY3:

        def serialize_kwargs(self, obj, width):
            kwargs = {
                key: getattr(self, key)
                for key in ("sort_dicts", "compact", "indent", "depth")
                if key in inspect.signature(self.serialize).parameters
            }
            if "width" in inspect.signature(self.serialize).parameters:
                kwargs["width"] = width
            return self.serialize(obj, **kwargs)


codes = {}

set_defaults()
default_pre_json = copy.copy(default)
apply_json()
y = _Y()
yc = y.fork(prefix="yc| ")


# source of pprint (3.8) module

#
#  Author:      Fred L. Drake, Jr.
#               fdrake@acm.org
#
#  This is a simple little module I wrote to make life easier.  I didn't
#  see anything quite like it in the library, though I may have overlooked
#  something.  I wrote this when I was trying to read some heavily nested
#  tuples with fairly non-descriptive content.  This is modeled very much
#  after Lisp/Scheme - style pretty-printing of lists.  If you find it
#  useful, thank small children who sleep at night.

"""Support to pretty-print lists, tuples, & dictionaries recursively.

Very simple, but useful, especially in debugging data structures.

Classes
-------

PrettyPrinter()
    Handle pretty-printing operations onto a stream using a configured
    set of formatting parameters.

Functions
---------

pformat()
    Format a Python object into a pretty-printed representation.

pprint()
    Pretty-print a Python object to a stream [default is sys.stdout].

saferepr()
    Generate a 'standard' repr()-like value, but protect against recursive
    data structures.

"""

import collections as _collections
import re
import sys as _sys
import types as _types
from io import StringIO as _StringIO


def pprint(
    object, stream=None, indent=1, width=80, depth=None, compact=False, sort_dicts=True
):
    """Pretty-print a Python object to a stream [default is sys.stdout]."""
    printer = PrettyPrinter(
        stream=stream,
        indent=indent,
        width=width,
        depth=depth,
        compact=compact,
        sort_dicts=sort_dicts,
    )
    printer.pprint(object)


def pformat(object, indent=1, width=80, depth=None, compact=False, sort_dicts=True):
    """Format a Python object into a pretty-printed representation."""
    return PrettyPrinter(
        indent=indent, width=width, depth=depth, compact=compact, sort_dicts=sort_dicts
    ).pformat(object)


def pp(object, *args, **kwargs):
    sort_dicts = kwargs.pop("sort_dicts", False)
    """Pretty-print a Python object"""
    pprint(object, *args, sort_dicts=sort_dicts, **kwargs)


def saferepr(object):
    """Version of repr() which can handle recursive data structures."""
    return _safe_repr(object, {}, None, 0, True)[0]


def isreadable(object):
    """Determine if saferepr(object) is readable by eval()."""
    return _safe_repr(object, {}, None, 0, True)[1]


def isrecursive(object):
    """Determine if object requires a recursive representation."""
    return _safe_repr(object, {}, None, 0, True)[2]


class _safe_key:
    """Helper function for key functions when sorting unorderable objects.

    The wrapped-object will fallback to a Py2.x style comparison for
    unorderable types (sorting first comparing the type name and then by
    the obj ids).  Does not work recursively, so dict.items() must have
    _safe_key applied to both the key and the value.

    """

    __slots__ = ["obj"]

    def __init__(self, obj):
        self.obj = obj

    def __lt__(self, other):
        try:
            return self.obj < other.obj
        except TypeError:
            return (str(type(self.obj)), id(self.obj)) < (
                str(type(other.obj)),
                id(other.obj),
            )


def _safe_tuple(t):
    "Helper function for comparing 2-tuples"
    return _safe_key(t[0]), _safe_key(t[1])


if PY3:

    class PrettyPrinter:
        def __init__(
            self,
            indent=1,
            width=80,
            depth=None,
            stream=None,
            compact=False,
            sort_dicts=True,
        ):
            """Handle pretty printing operations onto a stream using a set of
            configured parameters.

            indent
                Number of spaces to indent for each level of nesting.

            width
                Attempted maximum number of columns in the output.

            depth
                The maximum depth to print out nested structures.

            stream
                The desired output stream.  If omitted (or false), the standard
                output stream available at construction will be used.

            compact
                If true, several items will be combined in one line.

            sort_dicts
                If true, dict keys are sorted.

            """
            indent = int(indent)
            width = int(width)
            if indent < 0:
                raise ValueError("indent must be >= 0")
            if depth is not None and depth <= 0:
                raise ValueError("depth must be > 0")
            if not width:
                raise ValueError("width must be != 0")
            self._depth = depth
            self._indent_per_level = indent
            self._width = width
            if stream is not None:
                self._stream = stream
            else:
                self._stream = _sys.stdout
            self._compact = bool(compact)
            self._sort_dicts = sort_dicts

        def pprint(self, object):
            self._format(object, self._stream, 0, 0, {}, 0)
            self._stream.write("\n")

        def pformat(self, object):
            sio = _StringIO()
            self._format(object, sio, 0, 0, {}, 0)
            return sio.getvalue()

        def isrecursive(self, object):
            return self.format(object, {}, 0, 0)[2]

        def isreadable(self, object):
            s, readable, recursive = self.format(object, {}, 0, 0)
            return readable and not recursive

        def _format(self, object, stream, indent, allowance, context, level):
            objid = id(object)
            if objid in context:
                stream.write(_recursion(object))
                self._recursive = True
                self._readable = False
                return
            rep = self._repr(object, context, level)
            max_width = self._width - indent - allowance
            if len(rep) > max_width:
                p = self._dispatch.get(type(object).__repr__, None)
                if p is not None:
                    context[objid] = 1
                    p(self, object, stream, indent, allowance, context, level + 1)
                    del context[objid]
                    return
                elif isinstance(object, dict):
                    context[objid] = 1
                    self._pprint_dict(
                        object, stream, indent, allowance, context, level + 1
                    )
                    del context[objid]
                    return
            stream.write(rep)

        _dispatch = {}

        def _pprint_dict(self, object, stream, indent, allowance, context, level):
            write = stream.write
            write("{")
            if self._indent_per_level > 1:
                write((self._indent_per_level - 1) * " ")
            length = len(object)
            if length:
                if self._sort_dicts:
                    items = sorted(object.items(), key=_safe_tuple)
                else:
                    items = object.items()
                self._format_dict_items(
                    items, stream, indent, allowance + 1, context, level
                )
            write("}")

        _dispatch[dict.__repr__] = _pprint_dict

        def _pprint_ordered_dict(
            self, object, stream, indent, allowance, context, level
        ):
            if not len(object):
                stream.write(repr(object))
                return
            cls = object.__class__
            stream.write(cls.__name__ + "(")
            self._format(
                list(object.items()),
                stream,
                indent + len(cls.__name__) + 1,
                allowance + 1,
                context,
                level,
            )
            stream.write(")")

        _dispatch[_collections.OrderedDict.__repr__] = _pprint_ordered_dict

        def _pprint_list(self, object, stream, indent, allowance, context, level):
            stream.write("[")
            self._format_items(object, stream, indent, allowance + 1, context, level)
            stream.write("]")

        _dispatch[list.__repr__] = _pprint_list

        def _pprint_tuple(self, object, stream, indent, allowance, context, level):
            stream.write("(")
            endchar = ",)" if len(object) == 1 else ")"
            self._format_items(
                object, stream, indent, allowance + len(endchar), context, level
            )
            stream.write(endchar)

        _dispatch[tuple.__repr__] = _pprint_tuple

        def _pprint_set(self, object, stream, indent, allowance, context, level):
            if not len(object):
                stream.write(repr(object))
                return
            typ = object.__class__
            if typ is set:
                stream.write("{")
                endchar = "}"
            else:
                stream.write(typ.__name__ + "({")
                endchar = "})"
                indent += len(typ.__name__) + 1
            object = sorted(object, key=_safe_key)
            self._format_items(
                object, stream, indent, allowance + len(endchar), context, level
            )
            stream.write(endchar)

        _dispatch[set.__repr__] = _pprint_set
        _dispatch[frozenset.__repr__] = _pprint_set

        def _pprint_str(self, object, stream, indent, allowance, context, level):
            write = stream.write
            if not len(object):
                write(repr(object))
                return
            chunks = []
            lines = object.splitlines(True)
            if level == 1:
                indent += 1
                allowance += 1
            max_width1 = max_width = self._width - indent
            for i, line in enumerate(lines):
                rep = repr(line)
                if i == len(lines) - 1:
                    max_width1 -= allowance
                if len(rep) <= max_width1:
                    chunks.append(rep)
                else:
                    # A list of alternating (non-space, space) strings
                    parts = re.findall(r"\S*\s*", line)
                    assert parts
                    assert not parts[-1]
                    parts.pop()  # drop empty last part
                    max_width2 = max_width
                    current = ""
                    for j, part in enumerate(parts):
                        candidate = current + part
                        if j == len(parts) - 1 and i == len(lines) - 1:
                            max_width2 -= allowance
                        if len(repr(candidate)) > max_width2:
                            if current:
                                chunks.append(repr(current))
                            current = part
                        else:
                            current = candidate
                    if current:
                        chunks.append(repr(current))
            if len(chunks) == 1:
                write(rep)
                return
            if level == 1:
                write("(")
            for i, rep in enumerate(chunks):
                if i > 0:
                    write("\n" + " " * indent)
                write(rep)
            if level == 1:
                write(")")

        _dispatch[str.__repr__] = _pprint_str

        def _pprint_bytes(self, object, stream, indent, allowance, context, level):
            write = stream.write
            if len(object) <= 4:
                write(repr(object))
                return
            parens = level == 1
            if parens:
                indent += 1
                allowance += 1
                write("(")
            delim = ""
            for rep in _wrap_bytes_repr(object, self._width - indent, allowance):
                write(delim)
                write(rep)
                if not delim:
                    delim = "\n" + " " * indent
            if parens:
                write(")")

        _dispatch[bytes.__repr__] = _pprint_bytes

        def _pprint_bytearray(self, object, stream, indent, allowance, context, level):
            write = stream.write
            write("bytearray(")
            self._pprint_bytes(
                bytes(object), stream, indent + 10, allowance + 1, context, level + 1
            )
            write(")")

        _dispatch[bytearray.__repr__] = _pprint_bytearray

        def _pprint_mappingproxy(
            self, object, stream, indent, allowance, context, level
        ):
            stream.write("mappingproxy(")
            self._format(
                object.copy(), stream, indent + 13, allowance + 1, context, level
            )
            stream.write(")")

        _dispatch[_types.MappingProxyType.__repr__] = _pprint_mappingproxy

        def _format_dict_items(self, items, stream, indent, allowance, context, level):
            write = stream.write
            indent += self._indent_per_level
            delimnl = ",\n" + " " * indent
            last_index = len(items) - 1
            for i, (key, ent) in enumerate(items):
                last = i == last_index
                rep = self._repr(key, context, level)
                write(rep)
                write(": ")
                self._format(
                    ent,
                    stream,
                    indent + len(rep) + 2,
                    allowance if last else 1,
                    context,
                    level,
                )
                if not last:
                    write(delimnl)

        def _format_items(self, items, stream, indent, allowance, context, level):
            write = stream.write
            indent += self._indent_per_level
            if self._indent_per_level > 1:
                write((self._indent_per_level - 1) * " ")
            delimnl = ",\n" + " " * indent
            delim = ""
            width = max_width = self._width - indent + 1
            it = iter(items)
            try:
                next_ent = next(it)
            except StopIteration:
                return
            last = False
            while not last:
                ent = next_ent
                try:
                    next_ent = next(it)
                except StopIteration:
                    last = True
                    max_width -= allowance
                    width -= allowance
                if self._compact:
                    rep = self._repr(ent, context, level)
                    w = len(rep) + 2
                    if width < w:
                        width = max_width
                        if delim:
                            delim = delimnl
                    if width >= w:
                        width -= w
                        write(delim)
                        delim = ", "
                        write(rep)
                        continue
                write(delim)
                delim = delimnl
                self._format(
                    ent, stream, indent, allowance if last else 1, context, level
                )

        def _repr(self, object, context, level):
            repr, readable, recursive = self.format(
                object, context.copy(), self._depth, level
            )
            if not readable:
                self._readable = False
            if recursive:
                self._recursive = True
            return repr

        def format(self, object, context, maxlevels, level):
            """Format object for a specific context, returning a string
            and flags indicating whether the representation is 'readable'
            and whether the object represents a recursive construct.
            """
            return _safe_repr(object, context, maxlevels, level, self._sort_dicts)

        def _pprint_default_dict(
            self, object, stream, indent, allowance, context, level
        ):
            if not len(object):
                stream.write(repr(object))
                return
            rdf = self._repr(object.default_factory, context, level)
            cls = object.__class__
            indent += len(cls.__name__) + 1
            stream.write("%s(%s,\n%s" % (cls.__name__, rdf, " " * indent))
            self._pprint_dict(object, stream, indent, allowance + 1, context, level)
            stream.write(")")

        _dispatch[_collections.defaultdict.__repr__] = _pprint_default_dict

        def _pprint_counter(self, object, stream, indent, allowance, context, level):
            if not len(object):
                stream.write(repr(object))
                return
            cls = object.__class__
            stream.write(cls.__name__ + "({")
            if self._indent_per_level > 1:
                stream.write((self._indent_per_level - 1) * " ")
            items = object.most_common()
            self._format_dict_items(
                items,
                stream,
                indent + len(cls.__name__) + 1,
                allowance + 2,
                context,
                level,
            )
            stream.write("})")

        _dispatch[_collections.Counter.__repr__] = _pprint_counter

        def _pprint_chain_map(self, object, stream, indent, allowance, context, level):
            if not len(object.maps):
                stream.write(repr(object))
                return
            cls = object.__class__
            stream.write(cls.__name__ + "(")
            indent += len(cls.__name__) + 1
            for i, m in enumerate(object.maps):
                if i == len(object.maps) - 1:
                    self._format(m, stream, indent, allowance + 1, context, level)
                    stream.write(")")
                else:
                    self._format(m, stream, indent, 1, context, level)
                    stream.write(",\n" + " " * indent)

        _dispatch[_collections.ChainMap.__repr__] = _pprint_chain_map

        def _pprint_deque(self, object, stream, indent, allowance, context, level):
            if not len(object):
                stream.write(repr(object))
                return
            cls = object.__class__
            stream.write(cls.__name__ + "(")
            indent += len(cls.__name__) + 1
            stream.write("[")
            if object.maxlen is None:
                self._format_items(
                    object, stream, indent, allowance + 2, context, level
                )
                stream.write("])")
            else:
                self._format_items(object, stream, indent, 2, context, level)
                rml = self._repr(object.maxlen, context, level)
                stream.write("],\n%smaxlen=%s)" % (" " * indent, rml))

        _dispatch[_collections.deque.__repr__] = _pprint_deque

        def _pprint_user_dict(self, object, stream, indent, allowance, context, level):
            self._format(object.data, stream, indent, allowance, context, level - 1)

        _dispatch[_collections.UserDict.__repr__] = _pprint_user_dict

        def _pprint_user_list(self, object, stream, indent, allowance, context, level):
            self._format(object.data, stream, indent, allowance, context, level - 1)

        _dispatch[_collections.UserList.__repr__] = _pprint_user_list

        def _pprint_user_string(
            self, object, stream, indent, allowance, context, level
        ):
            self._format(object.data, stream, indent, allowance, context, level - 1)

        _dispatch[_collections.UserString.__repr__] = _pprint_user_string


# Return triple (repr_string, isreadable, isrecursive).


def _safe_repr(object, context, maxlevels, level, sort_dicts):
    typ = type(object)
    if typ in _builtin_scalars:
        return repr(object), True, False

    r = getattr(typ, "__repr__", None)
    if issubclass(typ, dict) and r is dict.__repr__:
        if not object:
            return "{}", True, False
        objid = id(object)
        if maxlevels and level >= maxlevels:
            return "{...}", False, objid in context
        if objid in context:
            return _recursion(object), False, True
        context[objid] = 1
        readable = True
        recursive = False
        components = []
        append = components.append
        level += 1
        if sort_dicts:
            items = sorted(object.items(), key=_safe_tuple)
        else:
            items = object.items()
        for k, v in items:
            krepr, kreadable, krecur = _safe_repr(
                k, context, maxlevels, level, sort_dicts
            )
            vrepr, vreadable, vrecur = _safe_repr(
                v, context, maxlevels, level, sort_dicts
            )
            append("%s: %s" % (krepr, vrepr))
            readable = readable and kreadable and vreadable
            if krecur or vrecur:
                recursive = True
        del context[objid]
        return "{%s}" % ", ".join(components), readable, recursive

    if (issubclass(typ, list) and r is list.__repr__) or (
        issubclass(typ, tuple) and r is tuple.__repr__
    ):
        if issubclass(typ, list):
            if not object:
                return "[]", True, False
            format = "[%s]"
        elif len(object) == 1:
            format = "(%s,)"
        else:
            if not object:
                return "()", True, False
            format = "(%s)"
        objid = id(object)
        if maxlevels and level >= maxlevels:
            return format % "...", False, objid in context
        if objid in context:
            return _recursion(object), False, True
        context[objid] = 1
        readable = True
        recursive = False
        components = []
        append = components.append
        level += 1
        for o in object:
            orepr, oreadable, orecur = _safe_repr(
                o, context, maxlevels, level, sort_dicts
            )
            append(orepr)
            if not oreadable:
                readable = False
            if orecur:
                recursive = True
        del context[objid]
        return format % ", ".join(components), readable, recursive

    rep = repr(object)
    return rep, (rep and not rep.startswith("<")), False


_builtin_scalars = frozenset(
    {str, bytes, bytearray, int, float, complex, bool, type(None)}
)


def _recursion(object):
    return "<Recursion on %s with id=%s>" % (type(object).__name__, id(object))


def _perfcheck(object=None):
    import time

    if object is None:
        object = [("string", (1, 2), [3, 4], {5: 6, 7: 8})] * 100000
    p = PrettyPrinter()
    t1 = time.perf_counter()
    _safe_repr(object, {}, None, 0, True)
    t2 = time.perf_counter()
    p.pformat(object)
    t3 = time.perf_counter()
    print("_safe_repr:", t2 - t1)
    print("pformat:", t3 - t2)


def _wrap_bytes_repr(object, width, allowance):
    current = b""
    last = len(object) // 4 * 4
    for i in range(0, len(object), 4):
        part = object[i : i + 4]
        candidate = current + part
        if i == last:
            width -= allowance
        if len(repr(candidate)) > width:
            if current:
                yield repr(current)
            current = part
        else:
            current = candidate
    if current:
        yield repr(current)


# end of source of pprint (3.8)

if PY2:
    import pprint

    pformat = pprint.pformat

