from psyneulink.core.compositions.composition import Composition
from psyneulink.core.components.functions.transferfunctions import Linear
from psyneulink.core.components.functions.learningfunctions import BackPropagation
from psyneulink.core.components.mechanisms.processing.processingmechanism import ProcessingMechanism
from psyneulink.core.components.mechanisms.processing.objectivemechanism import ObjectiveMechanism
from psyneulink.core.components.mechanisms.processing.transfermechanism import TransferMechanism
from psyneulink.core.components.mechanisms.modulatory.control.controlmechanism import ControlMechanism
from psyneulink.core.components.mechanisms.modulatory.control.optimizationcontrolmechanism import OptimizationControlMechanism
from psyneulink.core.components.ports.modulatorysignals.controlsignal import ControlSignal
from psyneulink.core.globals.keywords import ALL, INSET, INTERCEPT, NESTED, NOISE, SLOPE
from psyneulink.library.components.mechanisms.modulatory.control.agt.lccontrolmechanism import LCControlMechanism

class TestSimpleCompositions:
    def test_process(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")

        comp = Composition()
        comp.add_linear_processing_pathway([a, b])

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label

    def test_diverging_pathways(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")
        c = TransferMechanism(name="c", default_variable=[0, 0, 0, 0, 0])
        comp = Composition()
        comp.add_linear_processing_pathway([a, b])
        comp.add_linear_processing_pathway([a, c])

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)
        c_label = comp._show_graph._get_graph_node_label(comp, c, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label
        assert "out (5)" in c_label and "in (5)" in c_label

    def test_converging_pathways(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")
        c = TransferMechanism(name="c", default_variable=[0, 0, 0, 0, 0])
        comp = Composition()
        comp.add_linear_processing_pathway([a, c])
        comp.add_linear_processing_pathway([b, c])

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)
        c_label = comp._show_graph._get_graph_node_label(comp, c, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label
        assert "out (5)" in c_label and "in (5)" in c_label


class TestLearning:
    def test_process(self):
        a = TransferMechanism(name="a-sg", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b-sg")

        comp = Composition()
        comp.add_linear_learning_pathway([a, b], learning_function=BackPropagation)

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label

    def test_diverging_pathways(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")
        c = TransferMechanism(name="c", default_variable=[0, 0, 0, 0, 0])
        comp = Composition()
        comp.add_linear_learning_pathway(
            [a, b], learning_function=BackPropagation
        )
        comp.add_linear_learning_pathway(
            [a, c], learning_function=BackPropagation
        )

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)
        c_label = comp._show_graph._get_graph_node_label(comp, c, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label
        assert "out (5)" in c_label and "in (5)" in c_label

    def test_converging_pathways(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")
        c = TransferMechanism(name="c", default_variable=[0, 0, 0, 0, 0])
        comp = Composition()
        comp.add_linear_learning_pathway(
            [a, c], learning_function=BackPropagation
        )
        comp.add_linear_learning_pathway(
            [b, c], learning_function=BackPropagation
        )

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)
        c_label = comp._show_graph._get_graph_node_label(comp, c, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label
        assert "out (5)" in c_label and "in (5)" in c_label


class TestControl:
    def test_process(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")
        LC = LCControlMechanism(
            modulated_mechanisms=[a, b],
            objective_mechanism=ObjectiveMechanism(
                function=Linear, monitor=[b], name="lc_om"
            ),
            name="lc",
        )
        comp = Composition()
        comp.add_linear_processing_pathway([a, b])

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label

    def test_diverging_pathways(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")
        c = TransferMechanism(name="c", default_variable=[0, 0, 0, 0, 0])
        LC = LCControlMechanism(
            modulated_mechanisms=[a, b],
            objective_mechanism=ObjectiveMechanism(
                function=Linear, monitor=[b], name="lc_om"
            ),
            name="lc",
        )
        comp = Composition()
        comp.add_linear_processing_pathway([a, b])
        comp.add_linear_processing_pathway([a, c])

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)
        c_label = comp._show_graph._get_graph_node_label(comp, c, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label
        assert "out (5)" in c_label and "in (5)" in c_label

    def test_converging_pathways(self):
        a = TransferMechanism(name="a", default_variable=[0, 0, 0])
        b = TransferMechanism(name="b")
        c = TransferMechanism(name="c", default_variable=[0, 0, 0, 0, 0])
        LC = LCControlMechanism(
            modulated_mechanisms=[a, b],
            objective_mechanism=ObjectiveMechanism(
                function=Linear, monitor=[b], name="lc_om"
            ),
            name="lc",
        )
        comp = Composition()
        comp.add_linear_processing_pathway([a, c])
        comp.add_linear_processing_pathway([b, c])

        a_label = comp._show_graph._get_graph_node_label(comp, a, show_dimensions=ALL)
        b_label = comp._show_graph._get_graph_node_label(comp, b, show_dimensions=ALL)
        c_label = comp._show_graph._get_graph_node_label(comp, c, show_dimensions=ALL)

        assert "out (3)" in a_label and "in (3)" in a_label
        assert "out (1)" in b_label and "in (1)" in b_label
        assert "out (5)" in c_label and "in (5)" in c_label

    def test_no_nested_and_controler_name_with_space_in_it(self):

        ia = TransferMechanism(name='ia')
        ib = TransferMechanism(name='ib')
        ocm = OptimizationControlMechanism(name='my ocm',
                                           control_signals=[
                                               ControlSignal(projections=[(NOISE, ia)]),
                                               ControlSignal(projections=[(INTERCEPT, ia)]),
                                               ControlSignal(projections=[(SLOPE, ib)])
                                           ],
                                           search_space=[[1],[1],[1]])
        comp = Composition(name='ocomp', pathways=[ia, ib], controller=ocm)

        gv = comp.show_graph(show_controller=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tia [color=green penwidth=3 rank=source shape=oval]\n\tia -> ib [label="" arrowhead=normal color=black penwidth=1]\n\tib [color=red penwidth=3 rank=max shape=oval]\n}'
        gv = comp.show_graph(show_controller=True, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tia [color=green penwidth=3 rank=source shape=oval]\n\tia -> ib [label="" arrowhead=normal color=black penwidth=1]\n\t"my ocm" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t"my ocm" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t"my ocm" -> ib [label="" arrowhead=box color=purple penwidth=1]\n\tib [color=red penwidth=3 rank=max shape=oval]\n\t"my ocm" [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = comp.show_graph(show_controller=True, show_node_structure=True, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tia:"OutputPort-RESULT" -> ib:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"my ocm":"OutputPort-ia[noise] ControlSignal" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\t"my ocm":"OutputPort-ia[intercept] ControlSignal" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\t"my ocm":"OutputPort-ib[slope] ControlSignal" -> ib:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\tib [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ib" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ib</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t"my ocm" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-ib[slope] ControlSignal"><b>ib[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="my ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >my ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'

    def test_multiple_nesting_levels_with_control_mech_projection_one_level_deep(self):

        ia = TransferMechanism(name='ia')
        ib = TransferMechanism(name='ib')
        icomp = Composition(name='icomp', pathways=[ia,ib])

        # Middle Composition
        ma = TransferMechanism(name='ma')
        mb = TransferMechanism(name='mb')
        mcomp = Composition(name='mcomp', pathways=[ma, icomp, mb])

        # Outer Composition
        oa = TransferMechanism(name='oa')
        ob = TransferMechanism(name='ob')
        ctl_mech = ControlMechanism(name='ctl_mech',
                                    control_signals=[ControlSignal(projections=[(SLOPE, ma)])])
        ocomp = Composition(name='ocomp', pathways=[oa, mcomp, ob, ctl_mech])

        gv = ocomp.show_graph(show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\tmcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> mcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> mcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmcomp -> ob [label="" arrowhead=normal color=black penwidth=1]\n\tob -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tob [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n}'
        gv = ocomp.show_graph(show_nested=0, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\tmcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> mcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> mcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmcomp -> ob [label="" arrowhead=normal color=black penwidth=1]\n\tob -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tob [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n}'
        gv = ocomp.show_graph(show_nested=1, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> ma [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> ma [label="" arrowhead=box color=blue penwidth=1]\n\tmb -> ob [label="" arrowhead=normal color=black penwidth=1]\n\tob -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tob [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tsubgraph cluster_mcomp {\n\t\tgraph [label=mcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [color=green penwidth=3 rank=source shape=oval]\n\t\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\t\tma -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\t\ticomp -> mb [label="" arrowhead=normal color=black penwidth=1]\n\t\tmb [color=red penwidth=3 rank=max shape=oval]\n\t\tlabel=mcomp\n\t}\n}'
        gv = ocomp.show_graph(show_nested=2, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> ma [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> ma [label="" arrowhead=box color=blue penwidth=1]\n\tmb -> ob [label="" arrowhead=normal color=black penwidth=1]\n\tob -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tob [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tsubgraph cluster_mcomp {\n\t\tgraph [label=mcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [color=green penwidth=3 rank=source shape=oval]\n\t\tma -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\tib -> mb [label="" arrowhead=normal color=black penwidth=1]\n\t\tmb [color=red penwidth=3 rank=max shape=oval]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\tia [color=green penwidth=3 rank=source shape=oval]\n\t\t\tia -> ib [label="" arrowhead=normal color=black penwidth=1]\n\t\t\tib [color=red penwidth=3 rank=max shape=oval]\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=mcomp\n\t}\n}'
        gv = ocomp.show_graph(show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\tmcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> mcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> mcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmcomp -> ob [label="" arrowhead=normal color=black penwidth=1]\n\tob -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tob [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tsubgraph cluster_mcomp {\n\t\tgraph [label=mcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [color=green penwidth=3 rank=source shape=oval]\n\t\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\t\tma -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\t\ticomp -> mb [label="" arrowhead=normal color=black penwidth=1]\n\t\tmb [color=red penwidth=3 rank=max shape=oval]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\tia [color=green penwidth=3 rank=source shape=oval]\n\t\t\tia -> ib [label="" arrowhead=normal color=black penwidth=1]\n\t\t\tib [color=red penwidth=3 rank=max shape=oval]\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=mcomp\n\t}\n}'
        gv = ocomp.show_graph(show_nested=2, show_cim=True, show_node_structure=True, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\toa:"OutputPort-RESULT" -> "mcomp INPUT":"InputPort-INPUT_CIM_ma_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech:"OutputPort-ma[slope] ControlSignal" -> "mcomp CONTROL":"InputPort-PARAMETER_CIM_ma_slope" [label="" arrowhead=normal color=blue penwidth=1]\n\t"mcomp OUTPUT":"OutputPort-OUTPUT_CIM_mb_RESULT" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tob:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ma[slope] ControlSignal"><b>ma[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tsubgraph cluster_mcomp {\n\t\tgraph [label=mcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ma" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ma</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\tma:"OutputPort-RESULT" -> "icomp INPUT":"InputPort-INPUT_CIM_ia_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp OUTPUT":"OutputPort-OUTPUT_CIM_ib_RESULT" -> mb:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"mcomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ma_InputPort-0"><b>INPUT_CIM_ma_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="mcomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >mcomp Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_ma_InputPort-0"><b>INPUT_CIM_ma_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"mcomp INPUT":"OutputPort-INPUT_CIM_ma_InputPort-0" -> ma:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"mcomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ma_slope"><b>PARAMETER_CIM_ma_slope</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="mcomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >mcomp Parameter_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-PARAMETER_CIM_ma_slope"><b>PARAMETER_CIM_ma_slope</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=1 rank=same shape=plaintext]\n\t\t"mcomp CONTROL":"OutputPort-PARAMETER_CIM_ma_slope" -> ma:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\t\t"mcomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_mb_RESULT"><b>OUTPUT_CIM_mb_RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="mcomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >mcomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_mb_RESULT"><b>OUTPUT_CIM_mb_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\tmb:"OutputPort-RESULT" -> "mcomp OUTPUT":"InputPort-OUTPUT_CIM_mb_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tmb [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="mb" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >mb</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\t\tia:"OutputPort-RESULT" -> ib:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\t"icomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t\t"icomp INPUT":"OutputPort-INPUT_CIM_ia_InputPort-0" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\t"icomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_ib_RESULT"><b>OUTPUT_CIM_ib_RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ib_RESULT"><b>OUTPUT_CIM_ib_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\t\tib:"OutputPort-RESULT" -> "icomp OUTPUT":"InputPort-OUTPUT_CIM_ib_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\tib [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ib" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ib</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=mcomp\n\t}\n}'

    def test_nested_learning(self):
        ia = ProcessingMechanism(name='INNER INPUT')
        ib = ProcessingMechanism(name='INNER OUTPUT')
        input_mech = ProcessingMechanism(name='OUTER INPUT')
        internal_mech = ProcessingMechanism(name='INTERNAL')
        output_mech = ProcessingMechanism(name='OUTER OUTPUT')
        icomp = Composition(name="NESTED COMPOSITION", pathways=([ia, ib], BackPropagation))
        ocomp = Composition(name='COMPOSITION', pathways=[input_mech, internal_mech, icomp, output_mech])

        gv = ocomp.show_graph(show_nested=False, show_cim=False,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=False,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL -> "INNER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"INNER OUTPUT" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [color=orange penwidth=3 rank=min shape=oval]\n\t\t"INNER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\tComparator [color=orange penwidth=1 rank=min shape=oval]\n\t\t"INNER OUTPUT" -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=orange penwidth=1 rank=min shape=oval]\n\t\tComparator -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'
        gv = ocomp.show_graph(show_nested=False, show_cim=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"COMPOSITION INPUT" -> "OUTER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t"OUTER OUTPUT" -> "COMPOSITION OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL -> "NESTED COMPOSITION INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION OUTPUT" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"COMPOSITION INPUT" -> "OUTER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t"OUTER OUTPUT" -> "COMPOSITION OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [color=orange penwidth=3 rank=min shape=oval]\n\t\t"INNER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\t"NESTED COMPOSITION INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"NESTED COMPOSITION INPUT" -> "INNER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION INPUT" -> Target [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\t"INNER OUTPUT" -> "NESTED COMPOSITION OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tComparator [color=orange penwidth=1 rank=min shape=oval]\n\t\t"INNER OUTPUT" -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=orange penwidth=1 rank=min shape=oval]\n\t\tComparator -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'
        gv = ocomp.show_graph(show_nested=False, show_cim=False, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=False, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "INNER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Target" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Target</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=3 rank=min shape=plaintext]\n\t\t"INNER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT":"InputPort-InputPort-0" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"OutputPort-LearningSignal" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\tComparator [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td><td port="OutputPort-MSE"><b>MSE</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Comparator" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Comparator</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-SAMPLE"><b>SAMPLE</b></td><td port="InputPort-TARGET"><b>TARGET</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> Comparator:"InputPort-SAMPLE" [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget:"OutputPort-OutputPort-0" -> Comparator:"InputPort-TARGET" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-error_signal"><b>error_signal</b></td><td port="OutputPort-LearningSignal"><b>LearningSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-learning_rate"><b>learning_rate</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-activation_input"><b>activation_input</b></td><td port="InputPort-activation_output"><b>activation_output</b></td><td port="InputPort-error_signal"><b>error_signal</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\tComparator:"OutputPort-OUTCOME" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-error_signal" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_input" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_output" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'
        gv = ocomp.show_graph(show_nested=False, show_cim=True, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0"><b>INPUT_CIM_OUTER INPUT_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0" -> "OUTER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_OUTER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=True, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION INPUT":"InputPort-INPUT_CIM_INNER INPUT_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION OUTPUT":"OutputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0"><b>INPUT_CIM_OUTER INPUT_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0" -> "OUTER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_OUTER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Target" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Target</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=3 rank=min shape=plaintext]\n\t\t"INNER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT":"InputPort-InputPort-0" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"OutputPort-LearningSignal" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\t"NESTED COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_INNER INPUT_InputPort-0"><b>INPUT_CIM_INNER INPUT_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_Target_InputPort-0"><b>INPUT_CIM_Target_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="NESTED COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >NESTED COMPOSITION Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_INNER INPUT_InputPort-0"><b>INPUT_CIM_INNER INPUT_InputPort-0</b></td><td port="InputPort-INPUT_CIM_Target_InputPort-0"><b>INPUT_CIM_Target_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"NESTED COMPOSITION INPUT":"OutputPort-INPUT_CIM_INNER INPUT_InputPort-0" -> "INNER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION INPUT":"OutputPort-INPUT_CIM_Target_InputPort-0" -> Target:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_INNER OUTPUT_OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="NESTED COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >NESTED COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_INNER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "NESTED COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\tComparator [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td><td port="OutputPort-MSE"><b>MSE</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Comparator" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Comparator</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-SAMPLE"><b>SAMPLE</b></td><td port="InputPort-TARGET"><b>TARGET</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> Comparator:"InputPort-SAMPLE" [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget:"OutputPort-OutputPort-0" -> Comparator:"InputPort-TARGET" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-error_signal"><b>error_signal</b></td><td port="OutputPort-LearningSignal"><b>LearningSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-learning_rate"><b>learning_rate</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-activation_input"><b>activation_input</b></td><td port="InputPort-activation_output"><b>activation_output</b></td><td port="InputPort-error_signal"><b>error_signal</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\tComparator:"OutputPort-OUTCOME" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-error_signal" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_input" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_output" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'

    def test_nested_learning_test_with_user_specified_target_in_outer_composition(self):
        ia = ProcessingMechanism(name='INNER INPUT')
        ib = ProcessingMechanism(name='INNER OUTPUT')
        input_mech = ProcessingMechanism(name='OUTER INPUT')
        internal_mech = ProcessingMechanism(name='INTERNAL')
        output_mech = ProcessingMechanism(name='OUTER OUTPUT')
        target = ProcessingMechanism(name='TARGET')
        icomp = Composition(name="NESTED COMPOSITION")
        p = icomp.add_backpropagation_learning_pathway(pathway=[ia, ib])
        ocomp = Composition(name='COMPOSITION',
                            pathways=[input_mech, internal_mech, icomp, output_mech],
                            controller=OptimizationControlMechanism(
                                name='CONTROLLER',
                                objective_mechanism=ObjectiveMechanism(name='OBJECTIVE MECHANISM',
                                                                       monitor=[input_mech, output_mech]),
                                control=(SLOPE, internal_mech),
                                search_space=[1]
                            ))
        ocomp.add_node(target)
        ocomp.add_projection(sender=target, receiver=p.target)

        assert len(target.efferents) == 1
        assert target.efferents[0].receiver == icomp.input_CIM.input_ports['INPUT_CIM_Target_InputPort-0']
        assert icomp.input_CIM.output_ports['INPUT_CIM_Target_InputPort-0'].efferents[0].receiver.owner == p.target

        gv = ocomp.show_graph(show_nested=False, show_cim=False,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [color=green penwidth=3 rank=source shape=oval]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER -> INTERNAL [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [color=purple penwidth=1 rank=min shape=oval]\n\t"OBJECTIVE MECHANISM" -> CONTROLLER [label="" color=purple penwidth=1]\n\t"OUTER INPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\tCONTROLLER [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=False,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [color=green penwidth=3 rank=source shape=oval]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL -> "INNER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET -> Target [label="" arrowhead=normal color=orange penwidth=1]\n\t"INNER OUTPUT" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER -> INTERNAL [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [color=purple penwidth=1 rank=min shape=oval]\n\t"OBJECTIVE MECHANISM" -> CONTROLLER [label="" color=purple penwidth=1]\n\t"OUTER INPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\tCONTROLLER [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [color=orange penwidth=3 rank=min shape=oval]\n\t\t"INNER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\tComparator [color=orange penwidth=1 rank=min shape=oval]\n\t\t"INNER OUTPUT" -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=orange penwidth=1 rank=min shape=oval]\n\t\tComparator -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'
        gv = ocomp.show_graph(show_nested=False, show_cim=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [color=green penwidth=3 rank=source shape=oval]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"COMPOSITION INPUT" -> "OUTER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" -> TARGET [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t"OUTER OUTPUT" -> "COMPOSITION OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER -> INTERNAL [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [color=purple penwidth=1 rank=min shape=oval]\n\t"OBJECTIVE MECHANISM" -> CONTROLLER [label="" color=purple penwidth=1]\n\t"OUTER INPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\tCONTROLLER [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [color=green penwidth=3 rank=source shape=oval]\n\t"OUTER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\tINTERNAL [color=black penwidth=1 rank=same shape=oval]\n\t"OUTER INPUT" -> INTERNAL [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL -> "NESTED COMPOSITION INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET -> "NESTED COMPOSITION INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION OUTPUT" -> "OUTER OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"COMPOSITION INPUT" -> "OUTER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" -> TARGET [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t"OUTER OUTPUT" -> "COMPOSITION OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER -> INTERNAL [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [color=purple penwidth=1 rank=min shape=oval]\n\t"OBJECTIVE MECHANISM" -> CONTROLLER [label="" color=purple penwidth=1]\n\t"OUTER INPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" -> "OBJECTIVE MECHANISM" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\tCONTROLLER [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [color=orange penwidth=3 rank=min shape=oval]\n\t\t"INNER INPUT" [color=green penwidth=3 rank=source shape=oval]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\t"NESTED COMPOSITION INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"NESTED COMPOSITION INPUT" -> "INNER INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION INPUT" -> Target [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\t"INNER OUTPUT" -> "NESTED COMPOSITION OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tComparator [color=orange penwidth=1 rank=min shape=oval]\n\t\t"INNER OUTPUT" -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget -> Comparator [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=orange penwidth=1 rank=min shape=oval]\n\t\tComparator -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [color=red penwidth=3 rank=max shape=oval]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'
        gv = ocomp.show_graph(show_nested=False, show_cim=False, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="TARGET" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >TARGET</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER:"OutputPort-INTERNAL[slope] ControlSignal" -> INTERNAL:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OBJECTIVE MECHANISM" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OBJECTIVE MECHANISM</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-Value of OUTER INPUT [OutputPort-0]"><b>Value of OUTER INPUT [OutputPort-0]</b></td><td port="InputPort-Value of OUTER OUTPUT [OutputPort-0]"><b>Value of OUTER OUTPUT [OutputPort-0]</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\t"OBJECTIVE MECHANISM":"OutputPort-OUTCOME" -> CONTROLLER:"InputPort-OUTCOME" [label="" color=purple penwidth=1]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER INPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER OUTPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tCONTROLLER [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INTERNAL[slope] ControlSignal"><b>INTERNAL[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="CONTROLLER" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >CONTROLLER</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=False, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="TARGET" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >TARGET</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "INNER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET:"OutputPort-OutputPort-0" -> Target:"InputPort-InputPort-0" [label="" arrowhead=normal color=orange penwidth=1]\n\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER:"OutputPort-INTERNAL[slope] ControlSignal" -> INTERNAL:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OBJECTIVE MECHANISM" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OBJECTIVE MECHANISM</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-Value of OUTER INPUT [OutputPort-0]"><b>Value of OUTER INPUT [OutputPort-0]</b></td><td port="InputPort-Value of OUTER OUTPUT [OutputPort-0]"><b>Value of OUTER OUTPUT [OutputPort-0]</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\t"OBJECTIVE MECHANISM":"OutputPort-OUTCOME" -> CONTROLLER:"InputPort-OUTCOME" [label="" color=purple penwidth=1]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER INPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER OUTPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tCONTROLLER [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INTERNAL[slope] ControlSignal"><b>INTERNAL[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="CONTROLLER" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >CONTROLLER</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Target" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Target</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=3 rank=min shape=plaintext]\n\t\t"INNER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT":"InputPort-InputPort-0" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"OutputPort-LearningSignal" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\tComparator [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td><td port="OutputPort-MSE"><b>MSE</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Comparator" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Comparator</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-SAMPLE"><b>SAMPLE</b></td><td port="InputPort-TARGET"><b>TARGET</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> Comparator:"InputPort-SAMPLE" [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget:"OutputPort-OutputPort-0" -> Comparator:"InputPort-TARGET" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-error_signal"><b>error_signal</b></td><td port="OutputPort-LearningSignal"><b>LearningSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-learning_rate"><b>learning_rate</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-activation_input"><b>activation_input</b></td><td port="InputPort-activation_output"><b>activation_output</b></td><td port="InputPort-error_signal"><b>error_signal</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\tComparator:"OutputPort-OUTCOME" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-error_signal" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_input" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_output" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'
        gv = ocomp.show_graph(show_nested=False, show_cim=True, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="TARGET" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >TARGET</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" [color=pink penwidth=3 rank=same shape=rectangle]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0"><b>INPUT_CIM_OUTER INPUT_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_TARGET_InputPort-0"><b>INPUT_CIM_TARGET_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0" -> "OUTER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_TARGET_InputPort-0" -> TARGET:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_OUTER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER:"OutputPort-INTERNAL[slope] ControlSignal" -> INTERNAL:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OBJECTIVE MECHANISM" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OBJECTIVE MECHANISM</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-Value of OUTER INPUT [OutputPort-0]"><b>Value of OUTER INPUT [OutputPort-0]</b></td><td port="InputPort-Value of OUTER OUTPUT [OutputPort-0]"><b>Value of OUTER OUTPUT [OutputPort-0]</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\t"OBJECTIVE MECHANISM":"OutputPort-OUTCOME" -> CONTROLLER:"InputPort-OUTCOME" [label="" color=purple penwidth=1]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER INPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER OUTPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tCONTROLLER [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INTERNAL[slope] ControlSignal"><b>INTERNAL[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="CONTROLLER" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >CONTROLLER</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_nested=NESTED, show_cim=True, show_node_structure=True,
                              show_learning=True, output_fmt='source')
        assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="TARGET" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >TARGET</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION INPUT":"InputPort-INPUT_CIM_INNER INPUT_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tTARGET:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION INPUT":"InputPort-INPUT_CIM_Target_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"NESTED COMPOSITION OUTPUT":"OutputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0"><b>INPUT_CIM_OUTER INPUT_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_TARGET_InputPort-0"><b>INPUT_CIM_TARGET_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0" -> "OUTER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_TARGET_InputPort-0" -> TARGET:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_OUTER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tCONTROLLER:"OutputPort-INTERNAL[slope] ControlSignal" -> INTERNAL:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OBJECTIVE MECHANISM" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OBJECTIVE MECHANISM</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-Value of OUTER INPUT [OutputPort-0]"><b>Value of OUTER INPUT [OutputPort-0]</b></td><td port="InputPort-Value of OUTER OUTPUT [OutputPort-0]"><b>Value of OUTER OUTPUT [OutputPort-0]</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\t"OBJECTIVE MECHANISM":"OutputPort-OUTCOME" -> CONTROLLER:"InputPort-OUTCOME" [label="" color=purple penwidth=1]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER INPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER OUTPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tCONTROLLER [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INTERNAL[slope] ControlSignal"><b>INTERNAL[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="CONTROLLER" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >CONTROLLER</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Target" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Target</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=3 rank=min shape=plaintext]\n\t\t"INNER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT":"InputPort-InputPort-0" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"OutputPort-LearningSignal" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\t"NESTED COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_INNER INPUT_InputPort-0"><b>INPUT_CIM_INNER INPUT_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_Target_InputPort-0"><b>INPUT_CIM_Target_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="NESTED COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >NESTED COMPOSITION Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_INNER INPUT_InputPort-0"><b>INPUT_CIM_INNER INPUT_InputPort-0</b></td><td port="InputPort-INPUT_CIM_Target_InputPort-0"><b>INPUT_CIM_Target_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"NESTED COMPOSITION INPUT":"OutputPort-INPUT_CIM_INNER INPUT_InputPort-0" -> "INNER INPUT":"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION INPUT":"OutputPort-INPUT_CIM_Target_InputPort-0" -> Target:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"NESTED COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_INNER OUTPUT_OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="NESTED COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >NESTED COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_INNER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "NESTED COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\tComparator [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td><td port="OutputPort-MSE"><b>MSE</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Comparator" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Comparator</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-SAMPLE"><b>SAMPLE</b></td><td port="InputPort-TARGET"><b>TARGET</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> Comparator:"InputPort-SAMPLE" [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget:"OutputPort-OutputPort-0" -> Comparator:"InputPort-TARGET" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-error_signal"><b>error_signal</b></td><td port="OutputPort-LearningSignal"><b>LearningSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-learning_rate"><b>learning_rate</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-activation_input"><b>activation_input</b></td><td port="InputPort-activation_output"><b>activation_output</b></td><td port="InputPort-error_signal"><b>error_signal</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\tComparator:"OutputPort-OUTCOME" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-error_signal" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_input" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_output" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'

    # def test_nested_learning_test_with_user_specified_target_in_outer_composition_using_pathway_notation(self):
    #     ia = ProcessingMechanism(name='INNER INPUT')
    #     ib = ProcessingMechanism(name='INNER OUTPUT')
    #     input_mech = ProcessingMechanism(name='OUTER INPUT')
    #     internal_mech = ProcessingMechanism(name='INTERNAL')
    #     output_mech = ProcessingMechanism(name='OUTER OUTPUT')
    #     target = ProcessingMechanism(name='TARGET')
    #     icomp = Composition(name="NESTED COMPOSITION")
    #     p = icomp.add_backpropagation_learning_pathway(pathway=[ia, ib])
    #     ocomp = Composition(name='COMPOSITION',
    #                         pathways=[
    #                             [input_mech,
    #                             internal_mech,
    #                             icomp,
    #                             output_mech],
    #                             [target,p.target]
    #                         ],
    #                         controller=OptimizationControlMechanism(
    #                             name='CONTROLLER',
    #                             objective_mechanism=ObjectiveMechanism(name='OBJECTIVE MECHANISM',
    #                                                                    monitor=[input_mech,
    #                                                                             output_mech]),
    #                             control=(SLOPE, internal_mech))
    #                         )
    #     assert len(target.efferents) == 1
    #     assert target.efferents[0].receiver == icomp.input_CIM.input_ports['INPUT_CIM_Target_InputPort-0']
    #     assert icomp.input_CIM.output_ports['INPUT_CIM_Target_InputPort-0'].efferents[0].receiver.owner == p.target
    #
    #     gv = ocomp.show_graph(show_nested=NESTED, show_cim=True, show_node_structure=True,
    #                           show_learning=True, output_fmt='source')
    #     assert gv == 'digraph COMPOSITION {\n\tgraph [label=COMPOSITION overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\tTARGET [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="TARGET" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >TARGET</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t"OUTER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tINTERNAL [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INTERNAL" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INTERNAL</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=black penwidth=1 rank=same shape=plaintext]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> INTERNAL:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tINTERNAL:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION INPUT":"InputPort-INPUT_CIM_INNER INPUT_InputPort-0" [label="" color=black penwidth=1]\n\tTARGET:"OutputPort-OutputPort-0" -> "NESTED COMPOSITION INPUT":"InputPort-INPUT_CIM_Target_InputPort-0" [label="" color=black penwidth=1]\n\t"NESTED COMPOSITION OUTPUT":"OutputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0" -> "OUTER OUTPUT":"InputPort-InputPort-0" [label="" color=black penwidth=1]\n\t"COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0"><b>INPUT_CIM_OUTER INPUT_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_TARGET_InputPort-0"><b>INPUT_CIM_TARGET_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_OUTER INPUT_InputPort-0" -> "OUTER INPUT":"InputPort-InputPort-0" [label="" color=black penwidth=1]\n\t"COMPOSITION INPUT":"OutputPort-INPUT_CIM_TARGET_InputPort-0" -> TARGET:"InputPort-InputPort-0" [label="" color=black penwidth=1]\n\t"COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_OUTER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_OUTER OUTPUT_OutputPort-0" [label="" color=black penwidth=1]\n\tCONTROLLER:"OutputPort-INTERNAL[slope] ControlSignal" -> INTERNAL:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\t"OBJECTIVE MECHANISM" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OBJECTIVE MECHANISM" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OBJECTIVE MECHANISM</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-Value of OUTER INPUT [OutputPort-0]"><b>Value of OUTER INPUT [OutputPort-0]</b></td><td port="InputPort-Value of OUTER OUTPUT [OutputPort-0]"><b>Value of OUTER OUTPUT [OutputPort-0]</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\t"OBJECTIVE MECHANISM":"OutputPort-OUTCOME" -> CONTROLLER:"InputPort-OUTCOME" [label="" color=purple penwidth=1]\n\t"OUTER INPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER INPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT":"OutputPort-OutputPort-0" -> "OBJECTIVE MECHANISM":"InputPort-Value of OUTER OUTPUT [OutputPort-0]" [label="" color=purple penwidth=1]\n\t"OUTER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="OUTER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >OUTER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tCONTROLLER [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INTERNAL[slope] ControlSignal"><b>INTERNAL[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="CONTROLLER" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >CONTROLLER</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph "cluster_NESTED COMPOSITION" {\n\t\tgraph [label="NESTED COMPOSITION" overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tTarget [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Target" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Target</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=3 rank=min shape=plaintext]\n\t\t"INNER INPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER INPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER INPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [color=black penwidth=1 shape=diamond]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [arrowhead=none color=black penwidth=1]\n\t\t"MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" -> "INNER OUTPUT":"InputPort-InputPort-0" [color=black penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"OutputPort-LearningSignal" -> "MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label="" color=orange penwidth=1]\n\t\t"NESTED COMPOSITION INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_INNER INPUT_InputPort-0"><b>INPUT_CIM_INNER INPUT_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_Target_InputPort-0"><b>INPUT_CIM_Target_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="NESTED COMPOSITION Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >NESTED COMPOSITION Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_INNER INPUT_InputPort-0"><b>INPUT_CIM_INNER INPUT_InputPort-0</b></td><td port="InputPort-INPUT_CIM_Target_InputPort-0"><b>INPUT_CIM_Target_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"NESTED COMPOSITION INPUT":"OutputPort-INPUT_CIM_INNER INPUT_InputPort-0" -> "INNER INPUT":"InputPort-InputPort-0" [label="" color=black penwidth=1]\n\t\t"NESTED COMPOSITION INPUT":"OutputPort-INPUT_CIM_Target_InputPort-0" -> Target:"InputPort-InputPort-0" [label="" color=black penwidth=1]\n\t\t"NESTED COMPOSITION OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_INNER OUTPUT_OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="NESTED COMPOSITION Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >NESTED COMPOSITION Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0"><b>OUTPUT_CIM_INNER OUTPUT_OutputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "NESTED COMPOSITION OUTPUT":"InputPort-OUTPUT_CIM_INNER OUTPUT_OutputPort-0" [label="" color=black penwidth=1]\n\t\tComparator [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTCOME"><b>OUTCOME</b></td><td port="OutputPort-MSE"><b>MSE</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Comparator" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Comparator</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-scale"><b>scale</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-SAMPLE"><b>SAMPLE</b></td><td port="InputPort-TARGET"><b>TARGET</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> Comparator:"InputPort-SAMPLE" [label="" arrowhead=normal color=orange penwidth=1]\n\t\tTarget:"OutputPort-OutputPort-0" -> Comparator:"InputPort-TARGET" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-error_signal"><b>error_signal</b></td><td port="OutputPort-LearningSignal"><b>LearningSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-learning_rate"><b>learning_rate</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-activation_input"><b>activation_input</b></td><td port="InputPort-activation_output"><b>activation_output</b></td><td port="InputPort-error_signal"><b>error_signal</b></td></tr></table></td></tr></table></td></tr></table>> color=orange penwidth=1 rank=min shape=plaintext]\n\t\tComparator:"OutputPort-OUTCOME" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-error_signal" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER INPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_input" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT":"OutputPort-OutputPort-0" -> "Learning Mechanism for MappingProjection from INNER INPUT[OutputPort-0] to INNER OUTPUT[InputPort-0]":"InputPort-activation_output" [label="" arrowhead=normal color=orange penwidth=1]\n\t\t"INNER OUTPUT" [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OutputPort-0"><b>OutputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="INNER OUTPUT" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >INNER OUTPUT</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\t\tlabel="NESTED COMPOSITION"\n\t}\n}'

    def test_of_show_nested_show_cim_and_show_node_structure(self):

        # Inner Composition
        ia = TransferMechanism(name='ia')
        icomp = Composition(name='icomp', pathways=[ia])

        # Outer Composition
        oa = TransferMechanism(name='oa')
        ob = TransferMechanism(name='ob')
        oc = TransferMechanism(name='oc')
        ctl_mech = ControlMechanism(name='ctl_mech',
                                    control_signals=[ControlSignal(projections=[(SLOPE, ia)])])
        ocomp = Composition(name='ocomp', pathways=[[ob],[oa, icomp, oc, ctl_mech]])
        # ocomp.add_nodes(ob)
        ocm = OptimizationControlMechanism(name='ocm',
                                           agent_rep=ocomp,
                                           control_signals=[
                                               ControlSignal(projections=[(NOISE, ia)]),
                                               ControlSignal(projections=[(INTERCEPT, ia)]),
                                               ControlSignal(projections=[(SLOPE, oa)]),
                                           ],
                                           search_space=[[1],[1],[1]])
        ocomp.add_controller(ocm)

        # ocomp.show_graph(show_cim=True, show_nested=INSET)
        gv = ocomp.show_graph(show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> ia [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> ia [label="" arrowhead=box color=blue penwidth=1]\n\tia -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> ia [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> ia [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=normal color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"icomp INPUT" -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\tia -> "icomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> "icomp INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> "icomp CONTROL" [label="" arrowhead=normal color=blue penwidth=1]\n\t"icomp OUTPUT" -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> "icomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> "icomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"icomp INPUT" -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\tia -> "icomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\toa:"OutputPort-RESULT" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\tia:"OutputPort-RESULT" -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=normal color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"icomp INPUT":"OutputPort-INPUT_CIM_ia_InputPort-0" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Parameter_CIM</font></b></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="icomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\tia:"OutputPort-RESULT" -> "icomp OUTPUT":"InputPort-OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\toa:"OutputPort-RESULT" -> "icomp INPUT":"InputPort-INPUT_CIM_ia_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_slope" [label="" arrowhead=normal color=blue penwidth=1]\n\t"icomp OUTPUT":"OutputPort-OUTPUT_CIM_ia_RESULT" -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_noise" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_intercept" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"icomp INPUT":"OutputPort-INPUT_CIM_ia_InputPort-0" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Parameter_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="InputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="InputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\tia:"OutputPort-RESULT" -> "icomp OUTPUT":"InputPort-OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'

    def test_of_show_3_level_nested_show_cim_and_show_node_structure(self):

        # Inner Composition
        ia = TransferMechanism(name='ia')
        icomp = Composition(name='icomp', pathways=[ia])

        ma = TransferMechanism(name='ma')
        midcomp = Composition(name='midcomp', pathways=[ma, icomp])

        # Outer Composition
        oa = TransferMechanism(name='oa')
        ob = TransferMechanism(name='ob')
        oc = TransferMechanism(name='oc')
        ctl_mech = ControlMechanism(name='ctl_mech',
                                    control_signals=[ControlSignal(projections=[(SLOPE, ia)])])
        ocomp = Composition(name='ocomp', pathways=[[oa, midcomp, oc, ctl_mech], [ob]])
        ocm = OptimizationControlMechanism(name='ocm',
                                           agent_rep=ocomp,
                                           control_signals=[
                                               ControlSignal(projections=[(NOISE, ia)]),
                                               ControlSignal(projections=[(INTERCEPT, ia)]),
                                               ControlSignal(projections=[(SLOPE, oa)]),
                                           ],
                                           search_space=[[1],[1],[1]])
        ocomp.add_controller(ocm)

        gv = ocomp.show_graph(show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> midcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> midcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [color=green penwidth=3 rank=source shape=oval]\n\t\ticomp [color=red penwidth=3 rank=max shape=rectangle]\n\t\tma -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'
        gv = ocomp.show_graph(show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> ma [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> ia [label="" arrowhead=box color=blue penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> ia [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> ia [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [color=green penwidth=3 rank=source shape=oval]\n\t\tma -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> midcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> midcomp [label="" arrowhead=normal color=blue penwidth=1]\n\toa -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> midcomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> midcomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [color=green penwidth=3 rank=source shape=oval]\n\t\ticomp [color=red penwidth=3 rank=max shape=rectangle]\n\t\tma -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"midcomp INPUT" -> ma [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t"midcomp CONTROL" -> icomp [label="" arrowhead=normal color=blue penwidth=1]\n\t\t"midcomp CONTROL" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp CONTROL" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\ticomp -> "midcomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\t"icomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t\t"icomp INPUT" -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\t\t"icomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=blue penwidth=1]\n\t\t\t"icomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\t\tia -> "icomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> "midcomp INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> "midcomp CONTROL" [label="" arrowhead=normal color=blue penwidth=1]\n\t"midcomp OUTPUT" -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> "midcomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> "midcomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [color=green penwidth=3 rank=source shape=oval]\n\t\tma -> "icomp INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"midcomp INPUT" -> ma [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t"midcomp CONTROL" -> "icomp CONTROL" [label="" arrowhead=normal color=blue penwidth=1]\n\t\t"midcomp CONTROL" -> "icomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp CONTROL" -> "icomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\t"icomp OUTPUT" -> "midcomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\t"icomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t\t"icomp INPUT" -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\t\t"icomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=blue penwidth=1]\n\t\t\t"icomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\t\tia -> "icomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> midcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> midcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ma" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ma</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\ticomp [color=red penwidth=3 rank=max shape=rectangle]\n\t\tma:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\toa:"OutputPort-RESULT" -> ma:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ma" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ma</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\tma:"OutputPort-RESULT" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> midcomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> midcomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\tmidcomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> midcomp [label="" arrowhead=normal color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> midcomp [label="" arrowhead=normal color=black penwidth=1]\n\tmidcomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> midcomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> midcomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ma" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ma</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\ticomp [color=red penwidth=3 rank=max shape=rectangle]\n\t\tma:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ma_InputPort-0"><b>INPUT_CIM_ma_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="midcomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >midcomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"midcomp INPUT":"OutputPort-INPUT_CIM_ma_InputPort-0" -> ma:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="midcomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >midcomp Parameter_CIM</font></b></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t"midcomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> icomp [label="" arrowhead=normal color=blue penwidth=1]\n\t\t"midcomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="midcomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >midcomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\ticomp -> "midcomp OUTPUT":"InputPort-OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\t"icomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t\t"icomp INPUT":"OutputPort-INPUT_CIM_ia_InputPort-0" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\t"icomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Parameter_CIM</font></b></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\t\t\t"icomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="icomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\t\tia:"OutputPort-RESULT" -> "icomp OUTPUT":"InputPort-OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\toa:"OutputPort-RESULT" -> "midcomp INPUT":"InputPort-INPUT_CIM_ma_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> "midcomp CONTROL":"InputPort-PARAMETER_CIM_ia_slope" [label="" arrowhead=normal color=blue penwidth=1]\n\t"midcomp OUTPUT":"OutputPort-OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT" -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> "midcomp CONTROL":"InputPort-PARAMETER_CIM_ia_noise" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> "midcomp CONTROL":"InputPort-PARAMETER_CIM_ia_intercept" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_midcomp {\n\t\tgraph [label=midcomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tma [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ma" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ma</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\t\tma:"OutputPort-RESULT" -> "icomp INPUT":"InputPort-INPUT_CIM_ia_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ma_InputPort-0"><b>INPUT_CIM_ma_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="midcomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >midcomp Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_ma_InputPort-0"><b>INPUT_CIM_ma_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"midcomp INPUT":"OutputPort-INPUT_CIM_ma_InputPort-0" -> ma:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"midcomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="midcomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >midcomp Parameter_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td><td port="InputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="InputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t"midcomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_slope" [label="" arrowhead=normal color=blue penwidth=1]\n\t\t"midcomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_noise" [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_intercept" [label="" arrowhead=normal color=purple penwidth=1]\n\t\t"midcomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="midcomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >midcomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\t"icomp OUTPUT":"OutputPort-OUTPUT_CIM_ia_RESULT" -> "midcomp OUTPUT":"InputPort-OUTPUT_CIM_icomp_OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tsubgraph cluster_icomp {\n\t\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\t\tedge [fontname=arial fontsize=10]\n\t\t\t"icomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t\t"icomp INPUT":"OutputPort-INPUT_CIM_ia_InputPort-0" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\t"icomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Parameter_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="InputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="InputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\t\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\t\t\t"icomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\t\tia:"OutputPort-RESULT" -> "icomp OUTPUT":"InputPort-OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\t\tcolor=red\n\t\t\tlabel=icomp\n\t\t}\n\t\tlabel=midcomp\n\t}\n}'

    def test_of_show_nested_show_cim_and_show_node_structure_with_singleton_in_outer_comp_added_last(self):

        # Inner Composition
        ia = TransferMechanism(name='ia')
        icomp = Composition(name='icomp', pathways=[ia])

        # Outer Composition
        oa = TransferMechanism(name='oa')
        ob = TransferMechanism(name='ob')
        oc = TransferMechanism(name='oc')
        ctl_mech = ControlMechanism(name='ctl_mech',
                                    control_signals=[ControlSignal(projections=[(SLOPE, ia)])])
        ocomp = Composition(name='ocomp', pathways=[[oa, icomp, oc, ctl_mech],[ob]])
        ocm = OptimizationControlMechanism(name='ocm',
                                           agent_rep=ocomp,
                                           control_signals=[
                                               ControlSignal(projections=[(NOISE, ia)]),
                                               ControlSignal(projections=[(INTERCEPT, ia)]),
                                               ControlSignal(projections=[(SLOPE, oa)]),
                                           ],
                                           search_space=[[1],[1],[1]])
        ocomp.add_controller(ocm)

        ocomp.show_graph(show_cim=True, show_nested=INSET)
        gv = ocomp.show_graph(show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> ia [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> ia [label="" arrowhead=box color=blue penwidth=1]\n\tia -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> ia [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> ia [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech -> icomp [label="" arrowhead=normal color=blue penwidth=1]\n\toa -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"icomp INPUT" -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\tia -> "icomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_cim=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [color=green penwidth=3 rank=source shape=oval]\n\toa -> "icomp INPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech -> "icomp CONTROL" [label="" arrowhead=normal color=blue penwidth=1]\n\t"icomp OUTPUT" -> oc [label="" arrowhead=normal color=black penwidth=1]\n\toc -> ctl_mech [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t"ocomp INPUT" -> oa [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" -> ob [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\toc -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tob -> "ocomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm -> "icomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> "icomp CONTROL" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm -> oa [label="" arrowhead=box color=purple penwidth=1]\n\toc [color=red penwidth=3 rank=max shape=oval]\n\tctl_mech [color=blue penwidth=3 rank=max shape=octagon]\n\tob [color=brown penwidth=3 rank=same shape=oval]\n\tocm [color=purple penwidth=1 rank=min shape=doubleoctagon]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [color=green penwidth=1 rank=same shape=rectangle]\n\t\t"icomp INPUT" -> ia [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [color=purple penwidth=1 rank=same shape=rectangle]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL" -> ia [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [color=red penwidth=1 rank=same shape=rectangle]\n\t\tia -> "icomp OUTPUT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [color=brown penwidth=3 rank=same shape=oval]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_nested=NESTED, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\toa:"OutputPort-RESULT" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\tia:"OutputPort-RESULT" -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=False, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=box color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=box color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=INSET, output_fmt='source')
        assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\ticomp [color=pink penwidth=3 rank=same shape=rectangle]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> icomp [label="" arrowhead=normal color=blue penwidth=1]\n\toa:"OutputPort-RESULT" -> icomp [label="" arrowhead=normal color=black penwidth=1]\n\ticomp -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> icomp [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"icomp INPUT":"OutputPort-INPUT_CIM_ia_InputPort-0" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Parameter_CIM</font></b></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="icomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\tia:"OutputPort-RESULT" -> "icomp OUTPUT":"InputPort-OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'
        gv = ocomp.show_graph(show_node_structure=True, show_cim=True, show_nested=NESTED, output_fmt='source')
        # FIX: NEEDS TO BE CORRECTED ONCE BUG IS FIXED (SEE MESSAGE FOR COMMIT eb61303808ad2a5ba46fdd18d0e583283397915c)
        # assert gv == 'digraph ocomp {\n\tgraph [label=ocomp overlap=False rankdir=BT]\n\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\tedge [fontname=arial fontsize=10]\n\toa [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oa" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oa</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=3 rank=source shape=plaintext]\n\toa:"OutputPort-RESULT" -> "icomp INPUT":"InputPort-INPUT_CIM_ia_InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\tctl_mech:"OutputPort-ia[slope] ControlSignal" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_slope" [label="" arrowhead=normal color=blue penwidth=1]\n\t"icomp OUTPUT":"OutputPort-OUTPUT_CIM_ia_RESULT" -> oc:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\toc:"OutputPort-RESULT" -> ctl_mech:"InputPort-OUTCOME" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_oa_InputPort-0"><b>INPUT_CIM_oa_InputPort-0</b></td><td port="OutputPort-INPUT_CIM_ob_InputPort-0"><b>INPUT_CIM_ob_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Input_CIM</font></b></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_oa_InputPort-0" -> oa:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp INPUT":"OutputPort-INPUT_CIM_ob_InputPort-0" -> ob:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t"ocomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td port="ocomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_oc_RESULT"><b>OUTPUT_CIM_oc_RESULT</b></td><td port="InputPort-OUTPUT_CIM_ob_RESULT"><b>OUTPUT_CIM_ob_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\toc:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_oc_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tob:"OutputPort-RESULT" -> "ocomp OUTPUT":"InputPort-OUTPUT_CIM_ob_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\tocm:"OutputPort-ia[noise] ControlSignal" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_noise" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-ia[intercept] ControlSignal" -> "icomp CONTROL":"InputPort-PARAMETER_CIM_ia_intercept" [label="" arrowhead=normal color=purple penwidth=1]\n\tocm:"OutputPort-oa[slope] ControlSignal" -> oa:"ParameterPort-slope" [label="" arrowhead=box color=purple penwidth=1]\n\toc [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="oc" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >oc</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=3 rank=max shape=plaintext]\n\tctl_mech [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[slope] ControlSignal"><b>ia[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ctl_mech" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ctl_mech</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td></tr></table></td></tr></table></td></tr></table>> color=blue penwidth=3 rank=max shape=plaintext]\n\tob [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ob" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ob</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\tocm [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-ia[noise] ControlSignal"><b>ia[noise] ControlSignal</b></td><td port="OutputPort-ia[intercept] ControlSignal"><b>ia[intercept] ControlSignal</b></td><td port="OutputPort-oa[slope] ControlSignal"><b>oa[slope] ControlSignal</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ocm" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ocm</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTCOME"><b>OUTCOME</b></td><td port="InputPort-OUTCOME-1"><b>OUTCOME-1</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=min shape=plaintext]\n\tsubgraph cluster_icomp {\n\t\tgraph [label=icomp overlap=False rankdir=BT]\n\t\tnode [color=black fontname=arial fontsize=12 penwidth=1 shape=record]\n\t\tedge [fontname=arial fontsize=10]\n\t\t"icomp INPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Input_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Input_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-INPUT_CIM_ia_InputPort-0"><b>INPUT_CIM_ia_InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=green penwidth=1 rank=same shape=plaintext]\n\t\t"icomp INPUT":"OutputPort-INPUT_CIM_ia_InputPort-0" -> ia:"InputPort-InputPort-0" [label="" arrowhead=normal color=black penwidth=1]\n\t\t"icomp CONTROL" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="OutputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="OutputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Parameter_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Parameter_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-PARAMETER_CIM_ia_intercept"><b>PARAMETER_CIM_ia_intercept</b></td><td port="InputPort-PARAMETER_CIM_ia_noise"><b>PARAMETER_CIM_ia_noise</b></td><td port="InputPort-PARAMETER_CIM_ia_slope"><b>PARAMETER_CIM_ia_slope</b></td></tr></table></td></tr></table></td></tr></table>> color=purple penwidth=1 rank=same shape=plaintext]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_intercept" -> ia:"ParameterPort-intercept" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_noise" -> ia:"ParameterPort-noise" [label="" arrowhead=box color=purple penwidth=1]\n\t\t"icomp CONTROL":"OutputPort-PARAMETER_CIM_ia_slope" -> ia:"ParameterPort-slope" [label="" arrowhead=box color=blue penwidth=1]\n\t\t"icomp OUTPUT" [label=<<table border=\'1\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="icomp Output_CIM" colspan="2"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >icomp Output_CIM</font></b></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-OUTPUT_CIM_ia_RESULT"><b>OUTPUT_CIM_ia_RESULT</b></td></tr></table></td></tr></table></td></tr></table>> color=red penwidth=1 rank=same shape=plaintext]\n\t\tia:"OutputPort-RESULT" -> "icomp OUTPUT":"InputPort-OUTPUT_CIM_ia_RESULT" [label="" arrowhead=normal color=black penwidth=1]\n\t\tia [label=<<table border=\'3\' cellborder="0" cellspacing="1" bgcolor="#FFFFF0"><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="OutputPort-RESULT"><b>RESULT</b></td></tr></table></td></tr> <tr><td colspan="1" valign="middle"><b><i>OutputPorts</i></b></td></tr> </table></td></tr><tr><td port="ia" colspan="1"><b><b><i>Mechanism</i></b>:<br/><font point-size="16" >ia</font></b></td><td> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td rowspan="1" valign="middle"><b><i>ParameterPorts</i></b></td> <td> <table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="ParameterPort-intercept"><b>intercept</b></td></tr><tr><td port="ParameterPort-noise"><b>noise</b></td></tr><tr><td port="ParameterPort-offset"><b>offset</b></td></tr><tr><td port="ParameterPort-rate"><b>rate</b></td></tr><tr><td port="ParameterPort-slope"><b>slope</b></td></tr></table></td></tr></table></td></tr><tr><td colspan="2"> <table border="0" cellborder="0" bgcolor="#FAFAD0"> <tr><td colspan="1" valign="middle"><b><i>InputPorts</i></b></td></tr><tr><td><table border="0" cellborder="2" cellspacing="0" color="LIGHTGOLDENRODYELLOW" bgcolor="PALEGOLDENROD"><tr><td port="InputPort-InputPort-0"><b>InputPort-0</b></td></tr></table></td></tr></table></td></tr></table>> color=brown penwidth=3 rank=same shape=plaintext]\n\t\tlabel=icomp\n\t}\n}'
