"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        if (baseProps.deletionProtection) {
            this.setAttribute('deletion_protection.enabled', 'true');
        }
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = ELBV2_ACCOUNTS[region];
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        prefix = prefix || '';
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
// https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html#access-logging-bucket-permissions
const ELBV2_ACCOUNTS = {
    'us-east-1': '127311923021',
    'us-east-2': '033677994240',
    'us-west-1': '027434742980',
    'us-west-2': '797873946194',
    'af-south-1': '098369216593',
    'ca-central-1': '985666609251',
    'eu-central-1': '054676820928',
    'eu-west-1': '156460612806',
    'eu-west-2': '652711504416',
    'eu-west-3': '009996457667',
    'eu-south-1': '635631232127',
    'eu-north-1': '897822967062',
    'ap-east-1': '754344448648',
    'ap-northeast-1': '582318560864',
    'ap-northeast-2': '600734575887',
    'ap-northeast-3': '383597477331',
    'ap-southeast-1': '114774131450',
    'ap-southeast-2': '783225319266',
    'ap-south-1': '718504428378',
    'me-south-1': '076674570225',
    'sa-east-1': '507241528517',
    'us-gov-west-1': '048591011584',
    'us-gov-east-1': '190560391635',
    'cn-north-1': '638102146993',
    'cn-northwest-1': '037604701340',
};
//# sourceMappingURL=data:application/json;base64,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