"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends cdk.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.stringValue({ produce: () => this.targetType }),
            targets: cdk.Lazy.anyValue({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.stringValue({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            this.node.addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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