"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// tslint:disable-next-line:max-line-length
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
//# sourceMappingURL=data:application/json;base64,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