"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.FifoThroughputLimit = exports.DeduplicationScope = exports.QueueEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * What kind of deduplication scope to apply
 */
var DeduplicationScope;
(function (DeduplicationScope) {
    /**
     * Deduplication occurs at the message group level
     */
    DeduplicationScope["MESSAGE_GROUP"] = "messageGroup";
    /**
     * Deduplication occurs at the message queue level
     */
    DeduplicationScope["QUEUE"] = "queue";
})(DeduplicationScope = exports.DeduplicationScope || (exports.DeduplicationScope = {}));
/**
 * Whether the FIFO queue throughput quota applies to the entire queue or per message group
 */
var FifoThroughputLimit;
(function (FifoThroughputLimit) {
    /**
     * Throughput quota applies per queue
     */
    FifoThroughputLimit["PER_QUEUE"] = "perQueue";
    /**
     * Throughput quota applies per message group id
     */
    FifoThroughputLimit["PER_MESSAGE_GROUP_ID"] = "perMessageGroupId";
})(FifoThroughputLimit = exports.FifoThroughputLimit || (exports.FifoThroughputLimit = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sqs_QueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Queue);
            }
            throw error;
        }
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        queue.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.DESTROY);
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        this.deadLetterQueue = props.deadLetterQueue;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing SQS queue provided an ARN
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1)
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sqs_QueueAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromQueueAttributes);
            }
            throw error;
        }
        const stack = core_1.Stack.of(scope);
        const parsedArn = stack.splitArn(attrs.queueArn, core_1.ArnFormat.NO_RESOURCE_NAME);
        const queueName = attrs.queueName || parsedArn.resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${parsedArn.region}.${stack.urlSuffix}/${parsedArn.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = this.determineFifo();
                this.autoCreatePolicy = false;
            }
            /**
             * Determine fifo flag based on queueName and fifo attribute
             */
            determineFifo() {
                if (core_1.Token.isUnresolved(this.queueArn)) {
                    return attrs.fifo || false;
                }
                else {
                    if (typeof attrs.fifo !== 'undefined') {
                        if (attrs.fifo && !queueName.endsWith('.fifo')) {
                            throw new Error("FIFO queue names must end in '.fifo'");
                        }
                        if (!attrs.fifo && queueName.endsWith('.fifo')) {
                            throw new Error("Non-FIFO queue name may not end in '.fifo'");
                        }
                    }
                    return queueName.endsWith('.fifo') ? true : false;
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.deduplicationScope) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.fifoThroughputLimit) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        if (props.deduplicationScope && !fifoQueue) {
            throw new Error('Deduplication scope can only be defined for FIFO queues');
        }
        if (props.fifoThroughputLimit && !fifoQueue) {
            throw new Error('FIFO throughput limit can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            deduplicationScope: props.deduplicationScope,
            fifoThroughputLimit: props.fifoThroughputLimit,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
_a = JSII_RTTI_SYMBOL_1;
Queue[_a] = { fqn: "@aws-cdk/aws-sqs.Queue", version: "1.196.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicXVldWUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJxdWV1ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSx3Q0FBd0M7QUFDeEMsd0NBQWlGO0FBRWpGLDZDQUFrRTtBQUNsRSxtREFBMkM7QUFDM0MscURBQWlEO0FBdUxqRDs7R0FFRztBQUNILElBQVksZUFpQlg7QUFqQkQsV0FBWSxlQUFlO0lBQ3pCOztPQUVHO0lBQ0gsdUNBQW9CLENBQUE7SUFFcEI7O09BRUc7SUFDSCwwQ0FBdUIsQ0FBQTtJQUV2Qjs7OztPQUlHO0lBQ0gsOEJBQVcsQ0FBQTtBQUNiLENBQUMsRUFqQlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUFpQjFCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLGtCQVNYO0FBVEQsV0FBWSxrQkFBa0I7SUFDNUI7O09BRUc7SUFDSCxvREFBOEIsQ0FBQTtJQUM5Qjs7T0FFRztJQUNILHFDQUFlLENBQUE7QUFDakIsQ0FBQyxFQVRXLGtCQUFrQixHQUFsQiwwQkFBa0IsS0FBbEIsMEJBQWtCLFFBUzdCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLG1CQVNYO0FBVEQsV0FBWSxtQkFBbUI7SUFDN0I7O09BRUc7SUFDSCw2Q0FBc0IsQ0FBQTtJQUN0Qjs7T0FFRztJQUNILGlFQUEwQyxDQUFBO0FBQzVDLENBQUMsRUFUVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQVM5QjtBQUVEOztHQUVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsc0JBQVM7SUF3RmxDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBb0IsRUFBRTtRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsU0FBUztTQUM5QixDQUFDLENBQUM7UUFMYyxxQkFBZ0IsR0FBRyxJQUFJLENBQUM7Ozs7OzsrQ0F0RmhDLEtBQUs7Ozs7UUE2RmQsOEJBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVyQixNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsZUFBZTtZQUN6QyxDQUFDLENBQUM7Z0JBQ0EsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsUUFBUTtnQkFDekQsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlLENBQUMsZUFBZTthQUN2RDtZQUNELENBQUMsQ0FBQyxTQUFTLENBQUM7UUFFZCxNQUFNLEVBQUUsbUJBQW1CLEVBQUUsZUFBZSxFQUFFLEdBQUcseUJBQXlCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXRGLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsSUFBSSxHQUFHLFNBQVMsQ0FBQyxTQUFTLElBQUksS0FBSyxDQUFDO1FBRXpDLE1BQU0sS0FBSyxHQUFHLElBQUksd0JBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzNDLFNBQVMsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUM1QixHQUFHLFNBQVM7WUFDWixHQUFHLGVBQWU7WUFDbEIsYUFBYTtZQUNiLFlBQVksRUFBRSxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxhQUFhLENBQUMsU0FBUyxFQUFFO1lBQ3BFLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7WUFDN0Msc0JBQXNCLEVBQUUsS0FBSyxDQUFDLGVBQWUsSUFBSSxLQUFLLENBQUMsZUFBZSxDQUFDLFNBQVMsRUFBRTtZQUNsRiw2QkFBNkIsRUFBRSxLQUFLLENBQUMsc0JBQXNCLElBQUksS0FBSyxDQUFDLHNCQUFzQixDQUFDLFNBQVMsRUFBRTtZQUN2RyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCLElBQUksS0FBSyxDQUFDLGlCQUFpQixDQUFDLFNBQVMsRUFBRTtTQUNsRixDQUFDLENBQUM7UUFDSCxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSxvQkFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRXZFLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUU7WUFDMUQsT0FBTyxFQUFFLEtBQUs7WUFDZCxRQUFRLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDNUIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxtQkFBbUIsQ0FBQztRQUMvQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUM7UUFDMUIsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO1FBRTdDLFNBQVMseUJBQXlCO1lBQ2hDLElBQUksVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksZUFBZSxDQUFDLFdBQVcsQ0FBQztZQUVqRSxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsR0FBRyxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtnQkFDbkUsVUFBVSxHQUFHLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxpREFBaUQ7YUFDcEY7WUFFRCxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsV0FBVyxFQUFFO2dCQUM5QyxPQUFPLEVBQUUsZUFBZSxFQUFFLEVBQUUsRUFBRSxDQUFDO2FBQ2hDO1lBRUQsSUFBSSxVQUFVLEtBQUssZUFBZSxDQUFDLFdBQVcsRUFBRTtnQkFDOUMsT0FBTztvQkFDTCxlQUFlLEVBQUU7d0JBQ2YsY0FBYyxFQUFFLGVBQWU7d0JBQy9CLDRCQUE0QixFQUFFLEtBQUssQ0FBQyxZQUFZLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUU7cUJBQ25GO2lCQUNGLENBQUM7YUFDSDtZQUVELElBQUksVUFBVSxLQUFLLGVBQWUsQ0FBQyxHQUFHLEVBQUU7Z0JBQ3RDLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRTtvQkFDdEUsV0FBVyxFQUFFLGNBQWMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUU7aUJBQzVDLENBQUMsQ0FBQztnQkFFSCxPQUFPO29CQUNMLG1CQUFtQixFQUFFLFNBQVM7b0JBQzlCLGVBQWUsRUFBRTt3QkFDZixjQUFjLEVBQUUsU0FBUyxDQUFDLE1BQU07d0JBQ2hDLDRCQUE0QixFQUFFLEtBQUssQ0FBQyxZQUFZLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUU7cUJBQ25GO2lCQUNGLENBQUM7YUFDSDtZQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0NBQWdDLFVBQVUsRUFBRSxDQUFDLENBQUM7UUFDaEUsQ0FBQztLQUNGO0lBbktEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBZ0I7UUFDdkUsT0FBTyxLQUFLLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7S0FDM0Q7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQjs7Ozs7Ozs7OztRQUNwRixNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxnQkFBUyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDN0UsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsSUFBSSxTQUFTLENBQUMsUUFBUSxDQUFDO1FBQ3hELE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksZUFBZSxTQUFTLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksU0FBUyxDQUFDLE9BQU8sSUFBSSxTQUFTLEVBQUUsQ0FBQztRQUUxSCxNQUFNLE1BQU8sU0FBUSxzQkFBUztZQUE5Qjs7Z0JBQ2tCLGFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsNENBQTRDO2dCQUN2RSxhQUFRLEdBQUcsUUFBUSxDQUFDO2dCQUNwQixjQUFTLEdBQUcsU0FBUyxDQUFDO2dCQUN0Qix3QkFBbUIsR0FBRyxLQUFLLENBQUMsTUFBTTtvQkFDaEQsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQztvQkFDL0MsQ0FBQyxDQUFDLFNBQVMsQ0FBQztnQkFDRSxTQUFJLEdBQVksSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDO2dCQUVsQyxxQkFBZ0IsR0FBRyxLQUFLLENBQUM7WUFvQjlDLENBQUM7WUFsQkM7O2VBRUc7WUFDSyxhQUFhO2dCQUNuQixJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO29CQUNyQyxPQUFPLEtBQUssQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDO2lCQUM1QjtxQkFBTTtvQkFDTCxJQUFJLE9BQU8sS0FBSyxDQUFDLElBQUksS0FBSyxXQUFXLEVBQUU7d0JBQ3JDLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUU7NEJBQzlDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0NBQXNDLENBQUMsQ0FBQzt5QkFDekQ7d0JBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLElBQUksU0FBUyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRTs0QkFDOUMsTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO3lCQUMvRDtxQkFDRjtvQkFDRCxPQUFPLFNBQVMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO2lCQUNuRDtZQUNILENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBaUhEOztPQUVHO0lBQ0ssa0JBQWtCLENBQUMsS0FBaUI7UUFDMUMsMEVBQTBFO1FBQzFFLElBQUksU0FBUyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDM0IsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNsQyxJQUFJLE9BQU8sU0FBUyxLQUFLLFdBQVcsSUFBSSxTQUFTLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFBRSxTQUFTLEdBQUcsSUFBSSxDQUFDO1NBQUU7UUFDekksSUFBSSxPQUFPLFNBQVMsS0FBSyxXQUFXLElBQUksS0FBSyxDQUFDLHlCQUF5QixFQUFFO1lBQUUsU0FBUyxHQUFHLElBQUksQ0FBQztTQUFFO1FBQzlGLElBQUksT0FBTyxTQUFTLEtBQUssV0FBVyxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsRUFBRTtZQUFFLFNBQVMsR0FBRyxJQUFJLENBQUM7U0FBRTtRQUN2RixJQUFJLE9BQU8sU0FBUyxLQUFLLFdBQVcsSUFBSSxLQUFLLENBQUMsbUJBQW1CLEVBQUU7WUFBRSxTQUFTLEdBQUcsSUFBSSxDQUFDO1NBQUU7UUFFeEYsOERBQThEO1FBQzlELElBQUksT0FBTyxTQUFTLEtBQUssUUFBUSxFQUFFO1lBQ2pDLElBQUksU0FBUyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRTtnQkFDN0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO2FBQ3pEO1lBQ0QsSUFBSSxDQUFDLFNBQVMsSUFBSSxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFO2dCQUM3QyxNQUFNLElBQUksS0FBSyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7YUFDL0Q7U0FDRjtRQUVELElBQUksS0FBSyxDQUFDLHlCQUF5QixJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ2pELE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLENBQUMsQ0FBQztTQUNwRjtRQUVELElBQUksS0FBSyxDQUFDLGtCQUFrQixJQUFJLENBQUMsU0FBUyxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMseURBQXlELENBQUMsQ0FBQztTQUM1RTtRQUVELElBQUksS0FBSyxDQUFDLG1CQUFtQixJQUFJLENBQUMsU0FBUyxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELENBQUMsQ0FBQztTQUM5RTtRQUVELE9BQU87WUFDTCx5QkFBeUIsRUFBRSxLQUFLLENBQUMseUJBQXlCO1lBQzFELGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7WUFDNUMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM5QyxTQUFTO1NBQ1YsQ0FBQztLQUNIOztBQS9NSCxzQkFnTkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBrbXMgZnJvbSAnQGF3cy1jZGsvYXdzLWttcyc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgUmVtb3ZhbFBvbGljeSwgU3RhY2ssIFRva2VuLCBBcm5Gb3JtYXQgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgSVF1ZXVlLCBRdWV1ZUF0dHJpYnV0ZXMsIFF1ZXVlQmFzZSB9IGZyb20gJy4vcXVldWUtYmFzZSc7XG5pbXBvcnQgeyBDZm5RdWV1ZSB9IGZyb20gJy4vc3FzLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyB2YWxpZGF0ZVByb3BzIH0gZnJvbSAnLi92YWxpZGF0ZS1wcm9wcyc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgY3JlYXRpbmcgYSBuZXcgUXVldWVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBRdWV1ZVByb3BzIHtcbiAgLyoqXG4gICAqIEEgbmFtZSBmb3IgdGhlIHF1ZXVlLlxuICAgKlxuICAgKiBJZiBzcGVjaWZpZWQgYW5kIHRoaXMgaXMgYSBGSUZPIHF1ZXVlLCBtdXN0IGVuZCBpbiB0aGUgc3RyaW5nICcuZmlmbycuXG4gICAqXG4gICAqIEBkZWZhdWx0IENsb3VkRm9ybWF0aW9uLWdlbmVyYXRlZCBuYW1lXG4gICAqL1xuICByZWFkb25seSBxdWV1ZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2Ygc2Vjb25kcyB0aGF0IEFtYXpvbiBTUVMgcmV0YWlucyBhIG1lc3NhZ2UuXG4gICAqXG4gICAqIFlvdSBjYW4gc3BlY2lmeSBhbiBpbnRlZ2VyIHZhbHVlIGZyb20gNjAgc2Vjb25kcyAoMSBtaW51dGUpIHRvIDEyMDk2MDBcbiAgICogc2Vjb25kcyAoMTQgZGF5cykuIFRoZSBkZWZhdWx0IHZhbHVlIGlzIDM0NTYwMCBzZWNvbmRzICg0IGRheXMpLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5kYXlzKDQpXG4gICAqL1xuICByZWFkb25seSByZXRlbnRpb25QZXJpb2Q/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIHRpbWUgaW4gc2Vjb25kcyB0aGF0IHRoZSBkZWxpdmVyeSBvZiBhbGwgbWVzc2FnZXMgaW4gdGhlIHF1ZXVlIGlzIGRlbGF5ZWQuXG4gICAqXG4gICAqIFlvdSBjYW4gc3BlY2lmeSBhbiBpbnRlZ2VyIHZhbHVlIG9mIDAgdG8gOTAwICgxNSBtaW51dGVzKS4gVGhlIGRlZmF1bHRcbiAgICogdmFsdWUgaXMgMC5cbiAgICpcbiAgICogQGRlZmF1bHQgMFxuICAgKi9cbiAgcmVhZG9ubHkgZGVsaXZlcnlEZWxheT86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgbGltaXQgb2YgaG93IG1hbnkgYnl0ZXMgdGhhdCBhIG1lc3NhZ2UgY2FuIGNvbnRhaW4gYmVmb3JlIEFtYXpvbiBTUVMgcmVqZWN0cyBpdC5cbiAgICpcbiAgICogWW91IGNhbiBzcGVjaWZ5IGFuIGludGVnZXIgdmFsdWUgZnJvbSAxMDI0IGJ5dGVzICgxIEtpQikgdG8gMjYyMTQ0IGJ5dGVzXG4gICAqICgyNTYgS2lCKS4gVGhlIGRlZmF1bHQgdmFsdWUgaXMgMjYyMTQ0ICgyNTYgS2lCKS5cbiAgICpcbiAgICogQGRlZmF1bHQgMjU2S2lCXG4gICAqL1xuICByZWFkb25seSBtYXhNZXNzYWdlU2l6ZUJ5dGVzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBEZWZhdWx0IHdhaXQgdGltZSBmb3IgUmVjZWl2ZU1lc3NhZ2UgY2FsbHMuXG4gICAqXG4gICAqIERvZXMgbm90IHdhaXQgaWYgc2V0IHRvIDAsIG90aGVyd2lzZSB3YWl0cyB0aGlzIGFtb3VudCBvZiBzZWNvbmRzXG4gICAqIGJ5IGRlZmF1bHQgZm9yIG1lc3NhZ2VzIHRvIGFycml2ZS5cbiAgICpcbiAgICogRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBBbWF6b24gU1FTIExvbmcgUG9sbC5cbiAgICpcbiAgICogIEBkZWZhdWx0IDBcbiAgICovXG4gIHJlYWRvbmx5IHJlY2VpdmVNZXNzYWdlV2FpdFRpbWU/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGltZW91dCBvZiBwcm9jZXNzaW5nIGEgc2luZ2xlIG1lc3NhZ2UuXG4gICAqXG4gICAqIEFmdGVyIGRlcXVldWluZywgdGhlIHByb2Nlc3NvciBoYXMgdGhpcyBtdWNoIHRpbWUgdG8gaGFuZGxlIHRoZSBtZXNzYWdlXG4gICAqIGFuZCBkZWxldGUgaXQgZnJvbSB0aGUgcXVldWUgYmVmb3JlIGl0IGJlY29tZXMgdmlzaWJsZSBhZ2FpbiBmb3IgZGVxdWV1ZWluZ1xuICAgKiBieSBhbm90aGVyIHByb2Nlc3Nvci5cbiAgICpcbiAgICogVmFsdWVzIG11c3QgYmUgZnJvbSAwIHRvIDQzMjAwIHNlY29uZHMgKDEyIGhvdXJzKS4gSWYgeW91IGRvbid0IHNwZWNpZnlcbiAgICogYSB2YWx1ZSwgQVdTIENsb3VkRm9ybWF0aW9uIHVzZXMgdGhlIGRlZmF1bHQgdmFsdWUgb2YgMzAgc2Vjb25kcy5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcygzMClcbiAgICovXG4gIHJlYWRvbmx5IHZpc2liaWxpdHlUaW1lb3V0PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFNlbmQgbWVzc2FnZXMgdG8gdGhpcyBxdWV1ZSBpZiB0aGV5IHdlcmUgdW5zdWNjZXNzZnVsbHkgZGVxdWV1ZWQgYSBudW1iZXIgb2YgdGltZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vIGRlYWQtbGV0dGVyIHF1ZXVlXG4gICAqL1xuICByZWFkb25seSBkZWFkTGV0dGVyUXVldWU/OiBEZWFkTGV0dGVyUXVldWU7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIGNvbnRlbnRzIG9mIHRoZSBxdWV1ZSBhcmUgZW5jcnlwdGVkLCBhbmQgYnkgd2hhdCB0eXBlIG9mIGtleS5cbiAgICpcbiAgICogQmUgYXdhcmUgdGhhdCBlbmNyeXB0aW9uIGlzIG5vdCBhdmFpbGFibGUgaW4gYWxsIHJlZ2lvbnMsIHBsZWFzZSBzZWUgdGhlIGRvY3NcbiAgICogZm9yIGN1cnJlbnQgYXZhaWxhYmlsaXR5IGRldGFpbHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFVuZW5jcnlwdGVkXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uPzogUXVldWVFbmNyeXB0aW9uO1xuXG4gIC8qKlxuICAgKiBFeHRlcm5hbCBLTVMgbWFzdGVyIGtleSB0byB1c2UgZm9yIHF1ZXVlIGVuY3J5cHRpb24uXG4gICAqXG4gICAqIEluZGl2aWR1YWwgbWVzc2FnZXMgd2lsbCBiZSBlbmNyeXB0ZWQgdXNpbmcgZGF0YSBrZXlzLiBUaGUgZGF0YSBrZXlzIGluXG4gICAqIHR1cm4gd2lsbCBiZSBlbmNyeXB0ZWQgdXNpbmcgdGhpcyBrZXksIGFuZCByZXVzZWQgZm9yIGEgbWF4aW11bSBvZlxuICAgKiBgZGF0YUtleVJldXNlU2Vjc2Agc2Vjb25kcy5cbiAgICpcbiAgICogSWYgdGhlICdlbmNyeXB0aW9uTWFzdGVyS2V5JyBwcm9wZXJ0eSBpcyBzZXQsICdlbmNyeXB0aW9uJyB0eXBlIHdpbGwgYmVcbiAgICogaW1wbGljaXRseSBzZXQgdG8gXCJLTVNcIi5cbiAgICpcbiAgICogQGRlZmF1bHQgSWYgZW5jcnlwdGlvbiBpcyBzZXQgdG8gS01TIGFuZCBub3Qgc3BlY2lmaWVkLCBhIGtleSB3aWxsIGJlIGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uTWFzdGVyS2V5Pzoga21zLklLZXk7XG5cbiAgLyoqXG4gICAqIFRoZSBsZW5ndGggb2YgdGltZSB0aGF0IEFtYXpvbiBTUVMgcmV1c2VzIGEgZGF0YSBrZXkgYmVmb3JlIGNhbGxpbmcgS01TIGFnYWluLlxuICAgKlxuICAgKiBUaGUgdmFsdWUgbXVzdCBiZSBhbiBpbnRlZ2VyIGJldHdlZW4gNjAgKDEgbWludXRlKSBhbmQgODYsNDAwICgyNFxuICAgKiBob3VycykuIFRoZSBkZWZhdWx0IGlzIDMwMCAoNSBtaW51dGVzKS5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24ubWludXRlcyg1KVxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YUtleVJldXNlPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhpcyBhIGZpcnN0LWluLWZpcnN0LW91dCAoRklGTykgcXVldWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlLCB1bmxlc3MgcXVldWVOYW1lIGVuZHMgaW4gJy5maWZvJyBvciAnY29udGVudEJhc2VkRGVkdXBsaWNhdGlvbicgaXMgdHJ1ZS5cbiAgICovXG4gIHJlYWRvbmx5IGZpZm8/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgd2hldGhlciB0byBlbmFibGUgY29udGVudC1iYXNlZCBkZWR1cGxpY2F0aW9uLlxuICAgKlxuICAgKiBEdXJpbmcgdGhlIGRlZHVwbGljYXRpb24gaW50ZXJ2YWwgKDUgbWludXRlcyksIEFtYXpvbiBTUVMgdHJlYXRzXG4gICAqIG1lc3NhZ2VzIHRoYXQgYXJlIHNlbnQgd2l0aCBpZGVudGljYWwgY29udGVudCAoZXhjbHVkaW5nIGF0dHJpYnV0ZXMpIGFzXG4gICAqIGR1cGxpY2F0ZXMgYW5kIGRlbGl2ZXJzIG9ubHkgb25lIGNvcHkgb2YgdGhlIG1lc3NhZ2UuXG4gICAqXG4gICAqIElmIHlvdSBkb24ndCBlbmFibGUgY29udGVudC1iYXNlZCBkZWR1cGxpY2F0aW9uIGFuZCB5b3Ugd2FudCB0byBkZWR1cGxpY2F0ZVxuICAgKiBtZXNzYWdlcywgcHJvdmlkZSBhbiBleHBsaWNpdCBkZWR1cGxpY2F0aW9uIElEIGluIHlvdXIgU2VuZE1lc3NhZ2UoKSBjYWxsLlxuICAgKlxuICAgKiAoT25seSBhcHBsaWVzIHRvIEZJRk8gcXVldWVzLilcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNvbnRlbnRCYXNlZERlZHVwbGljYXRpb24/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBGb3IgaGlnaCB0aHJvdWdocHV0IGZvciBGSUZPIHF1ZXVlcywgc3BlY2lmaWVzIHdoZXRoZXIgbWVzc2FnZSBkZWR1cGxpY2F0aW9uXG4gICAqIG9jY3VycyBhdCB0aGUgbWVzc2FnZSBncm91cCBvciBxdWV1ZSBsZXZlbC5cbiAgICpcbiAgICogKE9ubHkgYXBwbGllcyB0byBGSUZPIHF1ZXVlcy4pXG4gICAqXG4gICAqIEBkZWZhdWx0IERlZHVwbGljYXRpb25TY29wZS5RVUVVRVxuICAgKi9cbiAgcmVhZG9ubHkgZGVkdXBsaWNhdGlvblNjb3BlPzogRGVkdXBsaWNhdGlvblNjb3BlO1xuXG4gIC8qKlxuICAgKiBGb3IgaGlnaCB0aHJvdWdocHV0IGZvciBGSUZPIHF1ZXVlcywgc3BlY2lmaWVzIHdoZXRoZXIgdGhlIEZJRk8gcXVldWVcbiAgICogdGhyb3VnaHB1dCBxdW90YSBhcHBsaWVzIHRvIHRoZSBlbnRpcmUgcXVldWUgb3IgcGVyIG1lc3NhZ2UgZ3JvdXAuXG4gICAqXG4gICAqIChPbmx5IGFwcGxpZXMgdG8gRklGTyBxdWV1ZXMuKVxuICAgKlxuICAgKiBAZGVmYXVsdCBGaWZvVGhyb3VnaHB1dExpbWl0LlBFUl9RVUVVRVxuICAgKi9cbiAgcmVhZG9ubHkgZmlmb1Rocm91Z2hwdXRMaW1pdD86IEZpZm9UaHJvdWdocHV0TGltaXQ7XG5cbiAgLyoqXG4gICAqIFBvbGljeSB0byBhcHBseSB3aGVuIHRoZSBxdWV1ZSBpcyByZW1vdmVkIGZyb20gdGhlIHN0YWNrXG4gICAqXG4gICAqIEV2ZW4gdGhvdWdoIHF1ZXVlcyBhcmUgdGVjaG5pY2FsbHkgc3RhdGVmdWwsIHRoZWlyIGNvbnRlbnRzIGFyZSB0cmFuc2llbnQgYW5kIGl0XG4gICAqIGlzIGNvbW1vbiB0byBhZGQgYW5kIHJlbW92ZSBRdWV1ZXMgd2hpbGUgcmVhcmNoaXRlY3RpbmcgeW91ciBhcHBsaWNhdGlvbi4gVGhlXG4gICAqIGRlZmF1bHQgaXMgdGhlcmVmb3JlIGBERVNUUk9ZYC4gQ2hhbmdlIGl0IHRvIGBSRVRBSU5gIGlmIHRoZSBtZXNzYWdlcyBhcmUgc29cbiAgICogdmFsdWFibGUgdGhhdCBhY2NpZGVudGFsbHkgbG9zaW5nIHRoZW0gd291bGQgYmUgdW5hY2NlcHRhYmxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBSZW1vdmFsUG9saWN5LkRFU1RST1lcbiAgICovXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5O1xufVxuXG4vKipcbiAqIERlYWQgbGV0dGVyIHF1ZXVlIHNldHRpbmdzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGVhZExldHRlclF1ZXVlIHtcbiAgLyoqXG4gICAqIFRoZSBkZWFkLWxldHRlciBxdWV1ZSB0byB3aGljaCBBbWF6b24gU1FTIG1vdmVzIG1lc3NhZ2VzIGFmdGVyIHRoZSB2YWx1ZSBvZiBtYXhSZWNlaXZlQ291bnQgaXMgZXhjZWVkZWQuXG4gICAqL1xuICByZWFkb25seSBxdWV1ZTogSVF1ZXVlO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIHRpbWVzIGEgbWVzc2FnZSBjYW4gYmUgdW5zdWNjZXNmdWxseSBkZXF1ZXVlZCBiZWZvcmUgYmVpbmcgbW92ZWQgdG8gdGhlIGRlYWQtbGV0dGVyIHF1ZXVlLlxuICAgKi9cbiAgcmVhZG9ubHkgbWF4UmVjZWl2ZUNvdW50OiBudW1iZXI7XG59XG5cbi8qKlxuICogV2hhdCBraW5kIG9mIGVuY3J5cHRpb24gdG8gYXBwbHkgdG8gdGhpcyBxdWV1ZVxuICovXG5leHBvcnQgZW51bSBRdWV1ZUVuY3J5cHRpb24ge1xuICAvKipcbiAgICogTWVzc2FnZXMgaW4gdGhlIHF1ZXVlIGFyZSBub3QgZW5jcnlwdGVkXG4gICAqL1xuICBVTkVOQ1JZUFRFRCA9ICdOT05FJyxcblxuICAvKipcbiAgICogU2VydmVyLXNpZGUgS01TIGVuY3J5cHRpb24gd2l0aCBhIG1hc3RlciBrZXkgbWFuYWdlZCBieSBTUVMuXG4gICAqL1xuICBLTVNfTUFOQUdFRCA9ICdNQU5BR0VEJyxcblxuICAvKipcbiAgICogU2VydmVyLXNpZGUgZW5jcnlwdGlvbiB3aXRoIGEgS01TIGtleSBtYW5hZ2VkIGJ5IHRoZSB1c2VyLlxuICAgKlxuICAgKiBJZiBgZW5jcnlwdGlvbktleWAgaXMgc3BlY2lmaWVkLCB0aGlzIGtleSB3aWxsIGJlIHVzZWQsIG90aGVyd2lzZSwgb25lIHdpbGwgYmUgZGVmaW5lZC5cbiAgICovXG4gIEtNUyA9ICdLTVMnLFxufVxuXG4vKipcbiAqIFdoYXQga2luZCBvZiBkZWR1cGxpY2F0aW9uIHNjb3BlIHRvIGFwcGx5XG4gKi9cbmV4cG9ydCBlbnVtIERlZHVwbGljYXRpb25TY29wZSB7XG4gIC8qKlxuICAgKiBEZWR1cGxpY2F0aW9uIG9jY3VycyBhdCB0aGUgbWVzc2FnZSBncm91cCBsZXZlbFxuICAgKi9cbiAgTUVTU0FHRV9HUk9VUCA9ICdtZXNzYWdlR3JvdXAnLFxuICAvKipcbiAgICogRGVkdXBsaWNhdGlvbiBvY2N1cnMgYXQgdGhlIG1lc3NhZ2UgcXVldWUgbGV2ZWxcbiAgICovXG4gIFFVRVVFID0gJ3F1ZXVlJyxcbn1cblxuLyoqXG4gKiBXaGV0aGVyIHRoZSBGSUZPIHF1ZXVlIHRocm91Z2hwdXQgcXVvdGEgYXBwbGllcyB0byB0aGUgZW50aXJlIHF1ZXVlIG9yIHBlciBtZXNzYWdlIGdyb3VwXG4gKi9cbmV4cG9ydCBlbnVtIEZpZm9UaHJvdWdocHV0TGltaXQge1xuICAvKipcbiAgICogVGhyb3VnaHB1dCBxdW90YSBhcHBsaWVzIHBlciBxdWV1ZVxuICAgKi9cbiAgUEVSX1FVRVVFID0gJ3BlclF1ZXVlJyxcbiAgLyoqXG4gICAqIFRocm91Z2hwdXQgcXVvdGEgYXBwbGllcyBwZXIgbWVzc2FnZSBncm91cCBpZFxuICAgKi9cbiAgUEVSX01FU1NBR0VfR1JPVVBfSUQgPSAncGVyTWVzc2FnZUdyb3VwSWQnLFxufVxuXG4vKipcbiAqIEEgbmV3IEFtYXpvbiBTUVMgcXVldWVcbiAqL1xuZXhwb3J0IGNsYXNzIFF1ZXVlIGV4dGVuZHMgUXVldWVCYXNlIHtcblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIFNRUyBxdWV1ZSBwcm92aWRlZCBhbiBBUk5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0XG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZVxuICAgKiBAcGFyYW0gcXVldWVBcm4gcXVldWUgQVJOIChpLmUuIGFybjphd3M6c3FzOnVzLWVhc3QtMjo0NDQ0NTU1NTY2NjY6cXVldWUxKVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUXVldWVBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcXVldWVBcm46IHN0cmluZyk6IElRdWV1ZSB7XG4gICAgcmV0dXJuIFF1ZXVlLmZyb21RdWV1ZUF0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7IHF1ZXVlQXJuIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyBxdWV1ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUXVldWVBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBRdWV1ZUF0dHJpYnV0ZXMpOiBJUXVldWUge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IHBhcnNlZEFybiA9IHN0YWNrLnNwbGl0QXJuKGF0dHJzLnF1ZXVlQXJuLCBBcm5Gb3JtYXQuTk9fUkVTT1VSQ0VfTkFNRSk7XG4gICAgY29uc3QgcXVldWVOYW1lID0gYXR0cnMucXVldWVOYW1lIHx8IHBhcnNlZEFybi5yZXNvdXJjZTtcbiAgICBjb25zdCBxdWV1ZVVybCA9IGF0dHJzLnF1ZXVlVXJsIHx8IGBodHRwczovL3Nxcy4ke3BhcnNlZEFybi5yZWdpb259LiR7c3RhY2sudXJsU3VmZml4fS8ke3BhcnNlZEFybi5hY2NvdW50fS8ke3F1ZXVlTmFtZX1gO1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUXVldWVCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBxdWV1ZUFybiA9IGF0dHJzLnF1ZXVlQXJuOyAvLyBhcm46YXdzOnNxczp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOnF1ZXVlMVxuICAgICAgcHVibGljIHJlYWRvbmx5IHF1ZXVlVXJsID0gcXVldWVVcmw7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcXVldWVOYW1lID0gcXVldWVOYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGVuY3J5cHRpb25NYXN0ZXJLZXkgPSBhdHRycy5rZXlBcm5cbiAgICAgICAgPyBrbXMuS2V5LmZyb21LZXlBcm4odGhpcywgJ0tleScsIGF0dHJzLmtleUFybilcbiAgICAgICAgOiB1bmRlZmluZWQ7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZmlmbzogYm9vbGVhbiA9IHRoaXMuZGV0ZXJtaW5lRmlmbygpO1xuXG4gICAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgYXV0b0NyZWF0ZVBvbGljeSA9IGZhbHNlO1xuXG4gICAgICAvKipcbiAgICAgICAqIERldGVybWluZSBmaWZvIGZsYWcgYmFzZWQgb24gcXVldWVOYW1lIGFuZCBmaWZvIGF0dHJpYnV0ZVxuICAgICAgICovXG4gICAgICBwcml2YXRlIGRldGVybWluZUZpZm8oKTogYm9vbGVhbiB7XG4gICAgICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQodGhpcy5xdWV1ZUFybikpIHtcbiAgICAgICAgICByZXR1cm4gYXR0cnMuZmlmbyB8fCBmYWxzZTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBpZiAodHlwZW9mIGF0dHJzLmZpZm8gIT09ICd1bmRlZmluZWQnKSB7XG4gICAgICAgICAgICBpZiAoYXR0cnMuZmlmbyAmJiAhcXVldWVOYW1lLmVuZHNXaXRoKCcuZmlmbycpKSB7XG4gICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcIkZJRk8gcXVldWUgbmFtZXMgbXVzdCBlbmQgaW4gJy5maWZvJ1wiKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmICghYXR0cnMuZmlmbyAmJiBxdWV1ZU5hbWUuZW5kc1dpdGgoJy5maWZvJykpIHtcbiAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiTm9uLUZJRk8gcXVldWUgbmFtZSBtYXkgbm90IGVuZCBpbiAnLmZpZm8nXCIpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgICByZXR1cm4gcXVldWVOYW1lLmVuZHNXaXRoKCcuZmlmbycpID8gdHJ1ZSA6IGZhbHNlO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgcXVldWVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBxdWV1ZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGlzIHF1ZXVlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcXVldWVOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBVUkwgb2YgdGhpcyBxdWV1ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHF1ZXVlVXJsOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIElmIHRoaXMgcXVldWUgaXMgZW5jcnlwdGVkLCB0aGlzIGlzIHRoZSBLTVMga2V5LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGVuY3J5cHRpb25NYXN0ZXJLZXk/OiBrbXMuSUtleTtcblxuICAvKipcbiAgICogV2hldGhlciB0aGlzIHF1ZXVlIGlzIGFuIEFtYXpvbiBTUVMgRklGTyBxdWV1ZS4gSWYgZmFsc2UsIHRoaXMgaXMgYSBzdGFuZGFyZCBxdWV1ZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBmaWZvOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJZiB0aGlzIHF1ZXVlIGlzIGNvbmZpZ3VyZWQgd2l0aCBhIGRlYWQtbGV0dGVyIHF1ZXVlLCB0aGlzIGlzIHRoZSBkZWFkLWxldHRlciBxdWV1ZSBzZXR0aW5ncy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWFkTGV0dGVyUXVldWU/OiBEZWFkTGV0dGVyUXVldWU7XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGF1dG9DcmVhdGVQb2xpY3kgPSB0cnVlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBRdWV1ZVByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMucXVldWVOYW1lLFxuICAgIH0pO1xuXG4gICAgdmFsaWRhdGVQcm9wcyhwcm9wcyk7XG5cbiAgICBjb25zdCByZWRyaXZlUG9saWN5ID0gcHJvcHMuZGVhZExldHRlclF1ZXVlXG4gICAgICA/IHtcbiAgICAgICAgZGVhZExldHRlclRhcmdldEFybjogcHJvcHMuZGVhZExldHRlclF1ZXVlLnF1ZXVlLnF1ZXVlQXJuLFxuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IHByb3BzLmRlYWRMZXR0ZXJRdWV1ZS5tYXhSZWNlaXZlQ291bnQsXG4gICAgICB9XG4gICAgICA6IHVuZGVmaW5lZDtcblxuICAgIGNvbnN0IHsgZW5jcnlwdGlvbk1hc3RlcktleSwgZW5jcnlwdGlvblByb3BzIH0gPSBfZGV0ZXJtaW5lRW5jcnlwdGlvblByb3BzLmNhbGwodGhpcyk7XG5cbiAgICBjb25zdCBmaWZvUHJvcHMgPSB0aGlzLmRldGVybWluZUZpZm9Qcm9wcyhwcm9wcyk7XG4gICAgdGhpcy5maWZvID0gZmlmb1Byb3BzLmZpZm9RdWV1ZSB8fCBmYWxzZTtcblxuICAgIGNvbnN0IHF1ZXVlID0gbmV3IENmblF1ZXVlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHF1ZXVlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICAuLi5maWZvUHJvcHMsXG4gICAgICAuLi5lbmNyeXB0aW9uUHJvcHMsXG4gICAgICByZWRyaXZlUG9saWN5LFxuICAgICAgZGVsYXlTZWNvbmRzOiBwcm9wcy5kZWxpdmVyeURlbGF5ICYmIHByb3BzLmRlbGl2ZXJ5RGVsYXkudG9TZWNvbmRzKCksXG4gICAgICBtYXhpbXVtTWVzc2FnZVNpemU6IHByb3BzLm1heE1lc3NhZ2VTaXplQnl0ZXMsXG4gICAgICBtZXNzYWdlUmV0ZW50aW9uUGVyaW9kOiBwcm9wcy5yZXRlbnRpb25QZXJpb2QgJiYgcHJvcHMucmV0ZW50aW9uUGVyaW9kLnRvU2Vjb25kcygpLFxuICAgICAgcmVjZWl2ZU1lc3NhZ2VXYWl0VGltZVNlY29uZHM6IHByb3BzLnJlY2VpdmVNZXNzYWdlV2FpdFRpbWUgJiYgcHJvcHMucmVjZWl2ZU1lc3NhZ2VXYWl0VGltZS50b1NlY29uZHMoKSxcbiAgICAgIHZpc2liaWxpdHlUaW1lb3V0OiBwcm9wcy52aXNpYmlsaXR5VGltZW91dCAmJiBwcm9wcy52aXNpYmlsaXR5VGltZW91dC50b1NlY29uZHMoKSxcbiAgICB9KTtcbiAgICBxdWV1ZS5hcHBseVJlbW92YWxQb2xpY3kocHJvcHMucmVtb3ZhbFBvbGljeSA/PyBSZW1vdmFsUG9saWN5LkRFU1RST1kpO1xuXG4gICAgdGhpcy5xdWV1ZUFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUocXVldWUuYXR0ckFybiwge1xuICAgICAgc2VydmljZTogJ3NxcycsXG4gICAgICByZXNvdXJjZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG4gICAgdGhpcy5xdWV1ZU5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShxdWV1ZS5hdHRyUXVldWVOYW1lKTtcbiAgICB0aGlzLmVuY3J5cHRpb25NYXN0ZXJLZXkgPSBlbmNyeXB0aW9uTWFzdGVyS2V5O1xuICAgIHRoaXMucXVldWVVcmwgPSBxdWV1ZS5yZWY7XG4gICAgdGhpcy5kZWFkTGV0dGVyUXVldWUgPSBwcm9wcy5kZWFkTGV0dGVyUXVldWU7XG5cbiAgICBmdW5jdGlvbiBfZGV0ZXJtaW5lRW5jcnlwdGlvblByb3BzKHRoaXM6IFF1ZXVlKTogeyBlbmNyeXB0aW9uUHJvcHM6IEVuY3J5cHRpb25Qcm9wcywgZW5jcnlwdGlvbk1hc3RlcktleT86IGttcy5JS2V5IH0ge1xuICAgICAgbGV0IGVuY3J5cHRpb24gPSBwcm9wcy5lbmNyeXB0aW9uIHx8IFF1ZXVlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRDtcblxuICAgICAgaWYgKGVuY3J5cHRpb24gIT09IFF1ZXVlRW5jcnlwdGlvbi5LTVMgJiYgcHJvcHMuZW5jcnlwdGlvbk1hc3RlcktleSkge1xuICAgICAgICBlbmNyeXB0aW9uID0gUXVldWVFbmNyeXB0aW9uLktNUzsgLy8gS01TIGlzIGltcGxpZWQgYnkgc3BlY2lmeWluZyBhbiBlbmNyeXB0aW9uIGtleVxuICAgICAgfVxuXG4gICAgICBpZiAoZW5jcnlwdGlvbiA9PT0gUXVldWVFbmNyeXB0aW9uLlVORU5DUllQVEVEKSB7XG4gICAgICAgIHJldHVybiB7IGVuY3J5cHRpb25Qcm9wczoge30gfTtcbiAgICAgIH1cblxuICAgICAgaWYgKGVuY3J5cHRpb24gPT09IFF1ZXVlRW5jcnlwdGlvbi5LTVNfTUFOQUdFRCkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGVuY3J5cHRpb25Qcm9wczoge1xuICAgICAgICAgICAga21zTWFzdGVyS2V5SWQ6ICdhbGlhcy9hd3Mvc3FzJyxcbiAgICAgICAgICAgIGttc0RhdGFLZXlSZXVzZVBlcmlvZFNlY29uZHM6IHByb3BzLmRhdGFLZXlSZXVzZSAmJiBwcm9wcy5kYXRhS2V5UmV1c2UudG9TZWNvbmRzKCksXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgIH1cblxuICAgICAgaWYgKGVuY3J5cHRpb24gPT09IFF1ZXVlRW5jcnlwdGlvbi5LTVMpIHtcbiAgICAgICAgY29uc3QgbWFzdGVyS2V5ID0gcHJvcHMuZW5jcnlwdGlvbk1hc3RlcktleSB8fCBuZXcga21zLktleSh0aGlzLCAnS2V5Jywge1xuICAgICAgICAgIGRlc2NyaXB0aW9uOiBgQ3JlYXRlZCBieSAke3RoaXMubm9kZS5wYXRofWAsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgZW5jcnlwdGlvbk1hc3RlcktleTogbWFzdGVyS2V5LFxuICAgICAgICAgIGVuY3J5cHRpb25Qcm9wczoge1xuICAgICAgICAgICAga21zTWFzdGVyS2V5SWQ6IG1hc3RlcktleS5rZXlBcm4sXG4gICAgICAgICAgICBrbXNEYXRhS2V5UmV1c2VQZXJpb2RTZWNvbmRzOiBwcm9wcy5kYXRhS2V5UmV1c2UgJiYgcHJvcHMuZGF0YUtleVJldXNlLnRvU2Vjb25kcygpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgICB9XG5cbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5leHBlY3RlZCAnZW5jcnlwdGlvblR5cGUnOiAke2VuY3J5cHRpb259YCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIExvb2sgYXQgdGhlIHByb3BzLCBzZWUgaWYgdGhlIEZJRk8gcHJvcHMgYWdyZWUsIGFuZCByZXR1cm4gdGhlIGNvcnJlY3Qgc3Vic2V0IG9mIHByb3BzXG4gICAqL1xuICBwcml2YXRlIGRldGVybWluZUZpZm9Qcm9wcyhwcm9wczogUXVldWVQcm9wcyk6IEZpZm9Qcm9wcyB7XG4gICAgLy8gQ2hlY2sgaWYgYW55IG9mIHRoZSBzaWduYWxzIHRoYXQgd2UgaGF2ZSBzYXkgdGhhdCB0aGlzIGlzIGEgRklGTyBxdWV1ZS5cbiAgICBsZXQgZmlmb1F1ZXVlID0gcHJvcHMuZmlmbztcbiAgICBjb25zdCBxdWV1ZU5hbWUgPSBwcm9wcy5xdWV1ZU5hbWU7XG4gICAgaWYgKHR5cGVvZiBmaWZvUXVldWUgPT09ICd1bmRlZmluZWQnICYmIHF1ZXVlTmFtZSAmJiAhVG9rZW4uaXNVbnJlc29sdmVkKHF1ZXVlTmFtZSkgJiYgcXVldWVOYW1lLmVuZHNXaXRoKCcuZmlmbycpKSB7IGZpZm9RdWV1ZSA9IHRydWU7IH1cbiAgICBpZiAodHlwZW9mIGZpZm9RdWV1ZSA9PT0gJ3VuZGVmaW5lZCcgJiYgcHJvcHMuY29udGVudEJhc2VkRGVkdXBsaWNhdGlvbikgeyBmaWZvUXVldWUgPSB0cnVlOyB9XG4gICAgaWYgKHR5cGVvZiBmaWZvUXVldWUgPT09ICd1bmRlZmluZWQnICYmIHByb3BzLmRlZHVwbGljYXRpb25TY29wZSkgeyBmaWZvUXVldWUgPSB0cnVlOyB9XG4gICAgaWYgKHR5cGVvZiBmaWZvUXVldWUgPT09ICd1bmRlZmluZWQnICYmIHByb3BzLmZpZm9UaHJvdWdocHV0TGltaXQpIHsgZmlmb1F1ZXVlID0gdHJ1ZTsgfVxuXG4gICAgLy8gSWYgd2UgaGF2ZSBhIG5hbWUsIHNlZSB0aGF0IGl0IGFncmVlcyB3aXRoIHRoZSBGSUZPIHNldHRpbmdcbiAgICBpZiAodHlwZW9mIHF1ZXVlTmFtZSA9PT0gJ3N0cmluZycpIHtcbiAgICAgIGlmIChmaWZvUXVldWUgJiYgIXF1ZXVlTmFtZS5lbmRzV2l0aCgnLmZpZm8nKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXCJGSUZPIHF1ZXVlIG5hbWVzIG11c3QgZW5kIGluICcuZmlmbydcIik7XG4gICAgICB9XG4gICAgICBpZiAoIWZpZm9RdWV1ZSAmJiBxdWV1ZU5hbWUuZW5kc1dpdGgoJy5maWZvJykpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiTm9uLUZJRk8gcXVldWUgbmFtZSBtYXkgbm90IGVuZCBpbiAnLmZpZm8nXCIpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChwcm9wcy5jb250ZW50QmFzZWREZWR1cGxpY2F0aW9uICYmICFmaWZvUXVldWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ29udGVudC1iYXNlZCBkZWR1cGxpY2F0aW9uIGNhbiBvbmx5IGJlIGRlZmluZWQgZm9yIEZJRk8gcXVldWVzJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmRlZHVwbGljYXRpb25TY29wZSAmJiAhZmlmb1F1ZXVlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0RlZHVwbGljYXRpb24gc2NvcGUgY2FuIG9ubHkgYmUgZGVmaW5lZCBmb3IgRklGTyBxdWV1ZXMnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuZmlmb1Rocm91Z2hwdXRMaW1pdCAmJiAhZmlmb1F1ZXVlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ZJRk8gdGhyb3VnaHB1dCBsaW1pdCBjYW4gb25seSBiZSBkZWZpbmVkIGZvciBGSUZPIHF1ZXVlcycpO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBjb250ZW50QmFzZWREZWR1cGxpY2F0aW9uOiBwcm9wcy5jb250ZW50QmFzZWREZWR1cGxpY2F0aW9uLFxuICAgICAgZGVkdXBsaWNhdGlvblNjb3BlOiBwcm9wcy5kZWR1cGxpY2F0aW9uU2NvcGUsXG4gICAgICBmaWZvVGhyb3VnaHB1dExpbWl0OiBwcm9wcy5maWZvVGhyb3VnaHB1dExpbWl0LFxuICAgICAgZmlmb1F1ZXVlLFxuICAgIH07XG4gIH1cbn1cblxuaW50ZXJmYWNlIEZpZm9Qcm9wcyB7XG4gIHJlYWRvbmx5IGZpZm9RdWV1ZT86IGJvb2xlYW47XG4gIHJlYWRvbmx5IGNvbnRlbnRCYXNlZERlZHVwbGljYXRpb24/OiBib29sZWFuO1xuICByZWFkb25seSBkZWR1cGxpY2F0aW9uU2NvcGU/OiBEZWR1cGxpY2F0aW9uU2NvcGU7XG4gIHJlYWRvbmx5IGZpZm9UaHJvdWdocHV0TGltaXQ/OiBGaWZvVGhyb3VnaHB1dExpbWl0O1xufVxuXG5pbnRlcmZhY2UgRW5jcnlwdGlvblByb3BzIHtcbiAgcmVhZG9ubHkga21zTWFzdGVyS2V5SWQ/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGttc0RhdGFLZXlSZXVzZVBlcmlvZFNlY29uZHM/OiBudW1iZXI7XG59XG4iXX0=