"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const region_info_1 = require("@aws-cdk/region-info");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_BaseLoadBalancerProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BaseLoadBalancer);
            }
            throw error;
        }
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        this.setAttribute('deletion_protection.enabled', baseProps.deletionProtection ? 'true' : 'false');
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Queries the load balancer context provider for load balancer info.
     * @internal
     */
    static _queryContextProvider(scope, options) {
        if (core_1.Token.isUnresolved(options.userOptions.loadBalancerArn)
            || Object.values(options.userOptions.loadBalancerTags ?? {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to look up a load balancer must be concrete (no Tokens)');
        }
        let cxschemaTags;
        if (options.userOptions.loadBalancerTags) {
            cxschemaTags = util_1.mapTagMapToCxschema(options.userOptions.loadBalancerTags);
        }
        const props = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.LOAD_BALANCER_PROVIDER,
            props: {
                loadBalancerArn: options.userOptions.loadBalancerArn,
                loadBalancerTags: cxschemaTags,
                loadBalancerType: options.loadBalancerType,
            },
            dummyValue: {
                ipAddressType: cxapi.LoadBalancerIpAddressType.DUAL_STACK,
                loadBalancerArn: `arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/${options.loadBalancerType}/my-load-balancer/50dc6c495c0c9188`,
                loadBalancerCanonicalHostedZoneId: 'Z3DZXE0EXAMPLE',
                loadBalancerDnsName: 'my-load-balancer-1234567890.us-west-2.elb.amazonaws.com',
                securityGroupIds: ['sg-1234'],
                vpcId: 'vpc-12345',
            },
        }).value;
        return props;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        prefix = prefix || '';
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = region_info_1.RegionInfo.get(region).elbv2Account;
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.env.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
    validate() {
        const ret = super.validate();
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-elasticloadbalancingv2-loadbalancer.html#cfn-elasticloadbalancingv2-loadbalancer-name
        const loadBalancerName = this.physicalName;
        if (!core_1.Token.isUnresolved(loadBalancerName) && loadBalancerName !== undefined) {
            if (loadBalancerName.length > 32) {
                ret.push(`Load balancer name: "${loadBalancerName}" can have a maximum of 32 characters.`);
            }
            if (loadBalancerName.startsWith('internal-')) {
                ret.push(`Load balancer name: "${loadBalancerName}" must not begin with "internal-".`);
            }
            if (loadBalancerName.startsWith('-') || loadBalancerName.endsWith('-')) {
                ret.push(`Load balancer name: "${loadBalancerName}" must not begin or end with a hyphen.`);
            }
            if (!/^[0-9a-z-]+$/i.test(loadBalancerName)) {
                ret.push(`Load balancer name: "${loadBalancerName}" must contain only alphanumeric characters or hyphens.`);
            }
        }
        return ret;
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
BaseLoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.BaseLoadBalancer", version: "1.198.0" };
//# sourceMappingURL=data:application/json;base64,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