import json
import pickle
import uuid

import pandas
import wizata_dsapi

from .api_dto import ApiDto
from .dataframe_toolkit import DataFrameToolkit
from .ds_dataframe import DSDataFrame
from .mlmodel import MLModel
from .plot import Plot
from .request import Request
from .script import Script


class Execution(ApiDto):
    """
    Execution Context defining an experimentation historical run.

    Execution can be run in the platform or directly within your script.
    Used also as context within a Script function, used the following method:
        - append_model - To store a generated model
        - append_plot - To store a plot
        - result_dataframe - To set the panda Dataframe as output

    :ivar execution_id: UUID of the Execution on Wizata App.
    :ivar experiment_id: UUID of the Experiment on which this Experiment is linked.

    Data

    :ivar request: Request or JSON formatted object to fecth data on Wizata App.
    :ivar input_ds_dataframe: DS Dataframe containing the panda Dataframe input.
    :ivar dataframe: shortcut to Pandas DataFrame used as input.

    Inputs

    :ivar script: Script to execute against the input data.
    :ivar ml_model: Trained ML Model to use against the input data.
    :ivar function: Name of a Wizata built-in function.
    :ivar isAnomalyDetection: True if this execution triggers integrated Anomaly Detection within Wizata.

    Outputs

    :ivar models: Trained Machine Learning Models.
    :ivar plots: Figures generated with Plotly.
    :ivar anomalies: Anomalies detected.
    :ivar output_ds_dataframe: Output dataframe generated by the Script or Model.
    """

    def __init__(self, execution_id=None):

        # Id
        if execution_id is None:
            execution_id = uuid.uuid4()
        self.execution_id = execution_id

        # Experiment information (required only for queries processed from backend)
        self.experiment_id = None

        # Inputs Properties (load)
        self.script = None
        self.request = None
        self.input_ds_dataframe = None
        self.ml_model = None
        self.function = None
        self.isAnomalyDetection = False

        # created/updated
        self.createdById = None
        self.createdDate = None
        self.updatedById = None
        self.updatedDate = None

        # Outputs Properties (generated by Execution)
        self.models = []
        self.anomalies = []
        self.plots = []
        self.output_ds_dataframe = None

    def _get_dataframe(self):
        if self.input_ds_dataframe is None:
            return None
        else:
            return self.input_ds_dataframe.dataframe

    def _set_dataframe(self, value):
        if not isinstance(value, pandas.DataFrame):
            raise ValueError("dataframe must be a panda dataframe.")
        self.input_ds_dataframe = DSDataFrame()
        self.input_ds_dataframe.dataframe = value

    def _del_dataframe(self):
        del self.input_ds_dataframe

    dataframe = property(
        fget=_get_dataframe,
        fset=_set_dataframe,
        fdel=_del_dataframe,
        doc="Input Pandas Dataframe (for id fetch 'input_ds_dataframe.df_id')"
    )

    def _get_result_dataframe(self):
        if self.output_ds_dataframe is None:
            return None
        else:
            return self.output_ds_dataframe.dataframe

    def _set_result_dataframe(self, value):
        if not isinstance(value, pandas.DataFrame):
            raise ValueError("dataframe must be a panda dataframe.")
        self.output_ds_dataframe = DSDataFrame()
        self.output_ds_dataframe.dataframe = value

    def _del_result_dataframe(self):
        del self.output_ds_dataframe

    result_dataframe = property(
        fget=_get_result_dataframe,
        fset=_set_result_dataframe,
        fdel=_del_result_dataframe,
        doc="Output Pandas Dataframe (for id fetch 'output_ds_dataframe.df_id')"
    )

    def append_plot(self, figure, name="Unkwown"):
        """
        Append a plot to the context.

        :param figure: Plotly figure.
        :param name: Name of the plot.
        :return: Plot object prepared.
        """
        plot = Plot()
        plot.name = name
        plot.experiment_id = self.experiment_id
        plot.figure = figure
        self.plots.append(plot)
        return plot

    def append_model(self, trained_model, input_columns, output_columns=None, has_anomalies=False, scaler=None):
        """
        Append a Trained ML Model to the context.

        :param trained_model: Trained Model to be stored as a pickled object.
        :param input_columns: List of str defining input columns to call the model (df.columns)
        :param output_columns: List of output columns - Optional as can be detected automatically during validation.
        :param has_anomalies: False by default, define if the model set anomalies
        :param scaler: Scaler to be stored if necessary.
        :return: ML Model object prepared.
        """
        ml_model = MLModel()

        ml_model.trained_model = trained_model
        ml_model.scaler = scaler

        ml_model.input_columns = input_columns
        ml_model.output_columns = output_columns

        ml_model.has_anomalies = has_anomalies

        self.models.append(ml_model)
        return ml_model

    def api_id(self) -> str:
        """
        Id of the execution (execution_id)

        :return: string formatted UUID of the Execution.
        """
        return str(self.execution_id).upper()

    def endpoint(self) -> str:
        """
        Name of the endpoints used to manipulate execution.
        :return: Endpoint name.
        """
        return "Executions"

    def to_json(self):
        """
        Convert to a json version of Execution definition.
        By default, use DS API format.
        :param result: if set as True, use format for pushing results.
        """
        obj = {
            "id": str(self.execution_id)
        }
        if self.experiment_id is not None:
            obj["experimentId"] = str(self.experiment_id)
        if self.request is not None:
            obj["request"] = json.dumps(self.request.to_json())
        if self.dataframe is not None:
            obj["dataframe"] = DataFrameToolkit.convert_to_json(self.dataframe)
        if self.script is not None:
            obj["scriptId"] = str(self.script.script_id)
        if self.ml_model is not None:
            obj["mlModelId"] = str(self.ml_model.model_id)
        if self.function is not None:
            obj["function"] = self.function
        if self.isAnomalyDetection:
            obj["isAnomalyDetection"] = str(True)
        if self.plots is not None:
            plots_ids = []
            for plot in self.plots:
                plots_ids.append(
                    {
                        "id": str(plot.plot_id)
                    }
                )
            obj["plots"] = plots_ids
        if self.models is not None:
            models_json = []
            for ml_model in self.models:
                models_json.append({"id": str(ml_model.model_id)})
            obj["models"] = models_json
        if self.anomalies is not None:
            obj["anomaliesList"] = self.anomalies
        if self.result_dataframe is not None:
            obj["resultDataframe"] = {
                "id": str(self.output_ds_dataframe.df_id)
            }
        if self.createdById is not None:
            obj["createdById"] = self.createdById
        if self.createdDate is not None:
            obj["createdDate"] = self.createdDate
        if self.updatedById is not None:
            obj["updatedById"] = self.updatedById
        if self.updatedDate is not None:
            obj["updatedDate"] = self.updatedDate
        return obj

    def from_json(self, obj):
        """
        Load an execution from a stored JSON model.
        :param obj: dictionnary representing an execution.
        """
        if "id" in obj.keys():
            self.execution_id = uuid.UUID(obj["id"])
        if "experimentId" in obj.keys() and obj["experimentId"] is not None:
            self.experiment_id = uuid.UUID(obj["experimentId"])
        if "request" in obj.keys() and obj["request"] is not None:
            self.request = Request()
            if isinstance(obj["request"], str):
                self.request.from_json(json.loads(obj["request"]))
            else:
                self.request.from_json(obj["request"])
        if "dataframe" in obj.keys() and obj["dataframe"] is not None:
            if isinstance(obj["dataframe"], str):
                self.request.from_json(json.loads(obj["dataframe"]))
            else:
                self.dataframe = DataFrameToolkit.convert_from_json(obj["dataframe"])
        if "scriptId" in obj.keys() and obj["scriptId"] is not None:
            self.script = Script()
            self.script.script_id = uuid.UUID(obj["scriptId"])
        if "mlModelId" in obj.keys() and obj["mlModelId"] is not None:
            self.ml_model = MLModel()
            self.ml_model.model_id = uuid.UUID(obj["mlModelId"])
        if "function" in obj.keys() and obj["function"] is not None:
            self.function = obj["function"]
        if "isAnomalyDetection" in obj.keys() and obj["isAnomalyDetection"] is not None:
            if isinstance(obj["isAnomalyDetection"], bool):
                self.isAnomalyDetection = obj["isAnomalyDetection"]
            else:
                self.isAnomalyDetection = obj["isAnomalyDetection"] == 'True'
        if "createdById" in obj.keys() and obj["createdById"] is not None:
            self.createdById = obj["createdById"]
        if "createdDate" in obj.keys() and obj["createdDate"] is not None:
            self.createdDate = obj["createdDate"]
        if "updatedById" in obj.keys() and obj["updatedById"] is not None:
            self.updatedById = obj["updatedById"]
        if "updatedDate" in obj.keys() and obj["updatedDate"] is not None:
            self.updatedDate = obj["updatedDate"]

    def to_pickle(self):
        """
        Convert the Execution to a pickle object.
        :return: Pickle object.
        """
        return pickle.dumps(self)
