# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pixy', 'pixy.colours']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pixy',
    'version': '1.0.2',
    'description': 'Add colour and style to terminal text',
    'long_description': 'pixy\n====\n\n[![PyPI version](https://badge.fury.io/py/pixy.svg)](https://pypi.org/project/pixy/)\n![Continuous Integration](https://github.com/kiancross/pixy/workflows/Continuous%20Integration/badge.svg)\n[![codecov](https://codecov.io/gh/kiancross/pixy/branch/master/graph/badge.svg?token=P490PT52VR)](https://codecov.io/gh/kiancross/pixy)\n\npixy is a Python library for adding colour and style to terminal text.\n\n* [Getting Started](#getting-started)\n  * [Installation](#installation)\n  * [Usage](#usage)\n* [Documentation](#documentation)\n* [Examples](#examples)\n* [Tests](#tests)\n* [License](#license)\n\nTerminal output can be styled using [ANSII escape code][ansii_escape_codes]. pixy provides a simple to use yet comprehensive wrapper, abstracting away the complexities.\n\npixy supports:\n  - 3-bit, 4-bit and 8-bit colour;\n  - 24-bit true colour RGB;\n  - changing the foreground and background colour;\n  - text decorators e.g. blink, underline, bold, italic, strike-through;\n  - custom fonts for compliant terminals;\n  - custom ANSII codes for development on non-spec-compliant terminals.\n\n[ansii_escape_codes]: https://en.wikipedia.org/wiki/ANSI_escape_codes\n\nGetting Started\n---------------\n\n### Installation\n\npixy can be installed using `pip`.\n```\n$ python3 -m pip install pixy\n```\n\n### Usage\n\npixy lets you style text straight out of the box.\n\n```python\nimport pixy\n\n# Print "Hello World" in red.\nprint(pixy.red("Hello World"))\n```\n\nIt provides the flexibility to format text as you wish.\n\n```python\nimport pixy\n\n# Create a string "Hello" with bold text and blue background.\ns1 = pixy.pring("Hello", pixy.decorators.bold + pixy.background.blue)\n\n# Concatenate the two strings, giving everything a red background (the\n# red background won\'t be applied to "Hello" because we\'ve already\n# given it a blue background!)\nprint(pixy.pring(s1 + " World. Lorem ipsum dolor...", pixy.background.red))\n```\n\nDocumentation\n-------------\nNot all terminals support all ANSII escape codes. Please check the terminal you are testing on supports the features you are using before opening an issue.\n\n### Helpers \nYou can create a string of a certain colour using the helper functions below:\n\n  - `pixy.black(text)`\n  - `pixy.red(text)`\n  - `pixy.green(text)`\n  - `pixy.yellow(text)`\n  - `pixy.blue(text)`\n  - `pixy.magenta(text)`\n  - `pixy.cyan(text)`\n  - `pixy.white(text)`\n\n#### Example\n```python\nimport pixy\n\n# Print "Hello World" in red.\nprint(pixy.red("Hello World"))\n```\n\n### `pixy.pring(text, style)`\n`pixy.pring` allows you to generate a string with any of the styles documented below. You can use more than one style by adding them together.\n\n  - `text` - the text to apply the style to.\n  - `style` - the style to apply to the text.\n\n#### Example\n```python\nimport pixy\n\n# Print "Hello World" with green text and white background.\nprint(pixy.pring(\n\t"Hello World",\n\tpixy.foreground.green + pixy.background.white\n))\n```\n\n### `pixy.foreground`, `pixy.background`\n`pixy.foreground` and `pixy.background` contain the variables: `black`,\n`red`, `green`, `yellow`, `blue`, `magenta`, `cyan`, `white`. \n\nUsing the `pixy.foreground` variant of the colour will apply the colour\nto the foreground and the `pixy.background` variant will apply it to the\nbackground.\n\nThese colours are known as the 3-bit colours and almost all terminals support\nthem. Most terminals also let you make them bright by adding the\n`pixy.decorators.bold` style.\n\n#### Example\n```python\nimport pixy\n\n# Print "Hello World" with green text.\nprint(pixy.pring("Hello World", pixy.foreground.green))\n```\n\n### `pixy.decorators`\nDecorators let you apply other styles to the text.\n\n| Decorator | Description | Support |\n|--|--|--|\n| `pixy.decorators.bold` | Bold/bright |\n| `pixy.decorators.faint` | Faint/dimmed |\n| `pixy.decorators.italic` | Italicised |\n| `pixy.decorators.underline` | Underlined |\n| `pixy.decorators.slow_blink` | Slow blink (less than 150 per minute) |\n| `pixy.decorators.rapid_blink` | Rapid blink (150+ per minute) | Not widely supported |\n| `pixy.decorators.invert` | Swap foreground and background colours |\n| `pixy.decorators.conceal` | Hide | Not widely supported |\n| `pixy.decorators.strike` | Strike-through |\n| `pixy.decorators.fraktur` | [Fraktur](https://en.wikipedia.org/wiki/Fraktur) | Rarely supported |\n| `pixy.decorators.double_underline` | Double underline |\n| `pixy.decorators.framed` | | Not widely supported |\n| `pixy.decorators.encircled` |  | Not widely supported |\n| `pixy.decorators.overlined` | Overlined |\n\n(Descriptions adapted from [Wikipedia](https://en.wikipedia.org/wiki/ANSI_escape_code#SGR_parameters))\n\n#### Example\n```python\nimport pixy\n\n# "Hello World" underlined.\nprint(pixy.pring("Hello World", pixy.decorators.underline))\n```\n\n### `pixy.ExtendedColour(code, background=False)`\nWith `pixy.ExtendedColour` you can select from a pre-defined selection of 256\ncolours.\n\n  - `code` - corresponds to a colour code found\n     [here](https://en.wikipedia.org/wiki/ANSI_escape_code#8-bit). \n\n  - `background` - boolean value indicating if this is a background or\n     foreground colour.\n\n#### Example\n```python\nimport pixy\n\n# Print "Hello World" in violet.\nprint(pixy.pring("Hello World", pixy.ExtendedColour(99)))\n\n# Print "Hello World" with a violet background.\nprint(pixy.pring("Hello World", pixy.ExtendedColour(99, True)))\n```\n\n### `pixy.TrueColour(red, green, blue, background=False)`\nWith `pixy.TrueColour` you can create an RGB colour.\n\n  - `red` - value between `0` and `255` indicating the intensity of the\n    red component.\n\n  - `green` - value between `0` and `255` indicating the intensity of the\n    green component.\n\n  - `blue` - value between `0` and `255` indicating the intensity of the\n    blue component.\n\n  - `background` - boolean value indicating if this is a background  or\n    foreground colour.\n\n#### Example\n```python\nimport pixy\n\n# Print "Hello World" in violet.\nprint(pixy.pring("Hello World", pixy.TrueColour(238, 130, 238)))\n\n# Print "Hello World" with a violet background.\nprint(pixy.pring("Hello World", pixy.TrueColour(238, 130, 238, True)))\n```\n\n### `pixy.Font(code)`\nAllows you to change the font used. Most terminals do not support this.\n\n  - `code` - a number between `0` and `8` corresponding to a font.\n\n#### Example\n```python\nimport pixy\n\nprint(pixy.pring("Hello World", pixy.Font(3)))\n```\n\n### `pixy.EscapeSequence(...)`\n`pixy.EscapeSequence` can be used in instances that you want to defined your own ANSII escape sequence.\n\n`pixy.EscapeSequence` takes a variable number of arguments - each argument should be an ANSII code.\n\n#### Example\n```python\nimport pixy\n\n# Bright blue using ANSII code 94 supported by some terminals\nprint(pixy.pring("Hello World", pixy.EscapeSequence(94)))\n``` \n## Examples\n\n### Blinking Text\n```python\nimport pixy\n\nprint(pixy.pring("Hello World", pixy.decorators.slow_blink))\n```\n\n### Concatenation\n```python\nimport pixy\n\n# "Hello" in red.\ns1 = pixy.pring("Hello", pixy.foreground.red)\n\n# " World" in blue.\ns2 = pixy.pring(" World", pixy.foreground.blue)\n\n# Concatenate the strings, add a white background and make\n# them bold.\ns3 = pixy.pring(s1 + s2, pixy.background.white + pixy.decorators.bold)\n\nprint(s3)\n```\n\n### Colour shades\n```python\nimport pixy\n\nfor i in range(0, 0xFF, 3): \n    colour = pixy.TrueColour(i, 0, 0, background=True)\n    print(pixy.pring(" ", colour), end="")\n\nprint()\n\nfor i in range(0, 0xFF, 3): \n    colour = pixy.TrueColour(0, i, 0, background=True)\n    print(pixy.pring(" ", colour), end="")\n\nprint()\n\nfor i in range(0, 0xFF, 3): \n    colour = pixy.TrueColour(0, 0, i, background=True)\n    print(pixy.pring(" ", colour), end="")\n\nprint()\n```\n\n### Colour gradient\n```python\nimport pixy\n\ndef gradient(colour_a, colour_b):\n    \n    output = []\n\n    for i in range(0, 100, 2): \n\n        p = i / 100 \n\n        output.append((\n            int(colour_a[0] + p * (colour_b[0] - colour_a[0])),\n            int(colour_a[1] + p * (colour_b[1] - colour_a[1])),\n            int(colour_a[2] + p * (colour_b[2] - colour_a[2]))\n        ))\n\n    return output\n\nred_to_green = gradient((255, 0, 0), (0, 255, 0)) \ngreen_to_blue = gradient((0, 255, 0), (0, 0, 255))\nblue_to_red = gradient((0, 0, 255), (255, 0, 0)) \n\nfor colour in red_to_green:\n    print(pixy.pring(" ", pixy.TrueColour(*colour, background=True)), end="")\n\nprint()\n\nfor colour in green_to_blue:\n    print(pixy.pring(" ", pixy.TrueColour(*colour, background=True)), end="")\n\nprint()\n\nfor colour in blue_to_red:\n    print(pixy.pring(" ", pixy.TrueColour(*colour, background=True)), end="")\n\nprint()\n```\n\nLicense\n-------\n\npixy is licensed under the [MIT license](https://github.com/kiancross/pixy/blob/master/LICENSE).\n',
    'author': 'Kian Cross',
    'author_email': 'kian@kiancross.co.uk',
    'maintainer': 'Kian Cross',
    'maintainer_email': 'kian@kiancross.co.uk',
    'url': 'https://github.com/kiancross/pixy',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
