import logging

from jsm_user_services.support.http_utils import request
from jsm_user_services.support.settings_utils import get_from_settings_or_raise_missing_config

logger = logging.getLogger(__name__)


def perform_recaptcha_validation(g_recaptcha_response: str) -> bool:
    """
    Performs a request to Google in order to validate the reCAPTCHA.
    For more details, check: https://developers.google.com/recaptcha/docs/verify
    """

    google_recaptcha_url = get_from_settings_or_raise_missing_config(
        "GOOGLE_RECAPTCHA_URL", "https://www.google.com/recaptcha/api/siteverify"
    )
    google_recaptcha_secret_key = get_from_settings_or_raise_missing_config("GOOGLE_RECAPTCHA_SECRET_KEY")

    min_score_threshold = float(get_from_settings_or_raise_missing_config("GOOGLE_RECAPTCHA_SCORE_THRESHOLD", "0.8"))

    logger.debug("Performing request to Google to check if recaptcha is valid")


    data = {"response": g_recaptcha_response, "secret": google_recaptcha_secret_key}

    with request() as r:
        resp = r.post(google_recaptcha_url, data=data)
    
    result_json = resp.json()

    logger.info(f"Status code: {resp.status_code}; min_score_threshold: {min_score_threshold}, body: {result_json}")

    if resp.status_code != 200:
        logger.info(f"GoogleRecaptcha response {g_recaptcha_response} status: {resp.status_code} failed: {result_json}")
        return False

    if result_json.get("success") is not True:
        return False

    if result_json.get("score", 0) < min_score_threshold:
        return False

    return True
