"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Starts an AWS Glue job in a Task state
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 */
class GlueStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.getPolicies();
        this.taskMetrics = {
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.props.glueJobName },
        };
    }
    renderTask() {
        var _a, _b;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        return {
            Resource: task_utils_1.integrationResourceArn('glue', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobName: this.props.glueJobName,
                Arguments: (_a = this.props.arguments) === null || _a === void 0 ? void 0 : _a.value,
                Timeout: (_b = this.props.timeout) === null || _b === void 0 ? void 0 : _b.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty,
            }),
            TimeoutSeconds: undefined,
        };
    }
    getPolicies() {
        let iamActions;
        if (this.integrationPattern === sfn.IntegrationPattern.REQUEST_RESPONSE) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun',
            ];
        }
        return [new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'glue',
                        resource: 'job',
                        resourceName: this.props.glueJobName,
                    }),
                ],
                actions: iamActions,
            })];
    }
}
exports.GlueStartJobRun = GlueStartJobRun;
GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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