"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.Data.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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