import autoscaling_api = require('@aws-cdk/aws-autoscaling-api');
import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import events = require('@aws-cdk/aws-events');
import iam = require('@aws-cdk/aws-iam');
import lambda = require('@aws-cdk/aws-lambda');
import s3n = require('@aws-cdk/aws-s3-notifications');
import sqs = require('@aws-cdk/aws-sqs');
import cdk = require('@aws-cdk/cdk');
import { Subscription, SubscriptionProtocol } from './subscription';
export interface ITopic extends cdk.IConstruct, events.IEventRuleTarget, cloudwatch.IAlarmAction, s3n.IBucketNotificationDestination, autoscaling_api.ILifecycleHookTarget {
    readonly topicArn: string;
    readonly topicName: string;
    /**
     * Export this Topic
     */
    export(): TopicImportProps;
    /**
     * Subscribe some endpoint to this topic
     */
    subscribe(name: string, endpoint: string, protocol: SubscriptionProtocol, rawMessageDelivery?: boolean): Subscription;
    /**
     * Defines a subscription from this SNS topic to an SQS queue.
     *
     * The queue resource policy will be updated to allow this SNS topic to send
     * messages to the queue.
     *
     * @param name The subscription name
     * @param queue The target queue
     * @param rawMessageDelivery Enable raw message delivery
     */
    subscribeQueue(queue: sqs.IQueue, rawMessageDelivery?: boolean): Subscription;
    /**
     * Defines a subscription from this SNS Topic to a Lambda function.
     *
     * The Lambda's resource policy will be updated to allow this topic to
     * invoke the function.
     *
     * @param name A name for the subscription
     * @param lambdaFunction The Lambda function to invoke
     */
    subscribeLambda(lambdaFunction: lambda.IFunction): Subscription;
    /**
     * Defines a subscription from this SNS topic to an email address.
     *
     * @param name A name for the subscription
     * @param emailAddress The email address to use.
     * @param jsonFormat True if the email content should be in JSON format (default is false).
     */
    subscribeEmail(name: string, emailAddress: string, options?: EmailSubscriptionOptions): Subscription;
    /**
     * Defines a subscription from this SNS topic to an http:// or https:// URL.
     *
     * @param name A name for the subscription
     * @param url The URL to invoke
     * @param rawMessageDelivery Enable raw message delivery
     */
    subscribeUrl(name: string, url: string, rawMessageDelivery?: boolean): Subscription;
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement: iam.PolicyStatement): void;
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(identity?: iam.IPrincipal): void;
}
/**
 * Either a new or imported Topic
 */
export declare abstract class TopicBase extends cdk.Construct implements ITopic {
    abstract readonly topicArn: string;
    abstract readonly topicName: string;
    /**
     * Controls automatic creation of policy objects.
     *
     * Set by subclasses.
     */
    protected abstract readonly autoCreatePolicy: boolean;
    private policy?;
    /** Buckets permitted to send notifications to this topic */
    private readonly notifyingBuckets;
    /**
     * Indicates if the resource policy that allows CloudWatch events to publish
     * notifications to this topic have been added.
     */
    private eventRuleTargetPolicyAdded;
    /**
     * Export this Topic
     */
    abstract export(): TopicImportProps;
    /**
     * Subscribe some endpoint to this topic
     */
    subscribe(name: string, endpoint: string, protocol: SubscriptionProtocol, rawMessageDelivery?: boolean): Subscription;
    /**
     * Defines a subscription from this SNS topic to an SQS queue.
     *
     * The queue resource policy will be updated to allow this SNS topic to send
     * messages to the queue.
     *
     * @param name The subscription name
     * @param queue The target queue
     * @param rawMessageDelivery Enable raw message delivery
     */
    subscribeQueue(queue: sqs.IQueue, rawMessageDelivery?: boolean): Subscription;
    /**
     * Defines a subscription from this SNS Topic to a Lambda function.
     *
     * The Lambda's resource policy will be updated to allow this topic to
     * invoke the function.
     *
     * @param name A name for the subscription
     * @param lambdaFunction The Lambda function to invoke
     */
    subscribeLambda(lambdaFunction: lambda.IFunction): Subscription;
    /**
     * Defines a subscription from this SNS topic to an email address.
     *
     * @param name A name for the subscription
     * @param emailAddress The email address to use.
     * @param options Options for the email delivery format.
     */
    subscribeEmail(name: string, emailAddress: string, options?: EmailSubscriptionOptions): Subscription;
    /**
     * Defines a subscription from this SNS topic to an http:// or https:// URL.
     *
     * @param name A name for the subscription
     * @param url The URL to invoke
     * @param rawMessageDelivery Enable raw message delivery
     */
    subscribeUrl(name: string, url: string, rawMessageDelivery?: boolean): Subscription;
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement: iam.PolicyStatement): void;
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(identity?: iam.IPrincipal): void;
    /**
     * Returns a RuleTarget that can be used to trigger this SNS topic as a
     * result from a CloudWatch event.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/resource-based-policies-cwe.html#sns-permissions
     */
    asEventRuleTarget(_ruleArn: string, _ruleId: string): events.EventRuleTargetProps;
    /**
     * Allow using SNS topics as lifecycle hook targets
     */
    asLifecycleHookTarget(lifecycleHook: autoscaling_api.ILifecycleHook): autoscaling_api.LifecycleHookTargetProps;
    readonly alarmActionArn: string;
    /**
     * Implements the IBucketNotificationDestination interface, allowing topics to be used
     * as bucket notification destinations.
     *
     * @param bucketArn The ARN of the bucket sending the notifications
     * @param bucketId A unique ID of the bucket
     */
    asBucketNotificationDestination(bucketArn: string, bucketId: string): s3n.BucketNotificationDestinationProps;
}
/**
 * Reference to an external topic.
 */
export interface TopicImportProps {
    readonly topicArn: string;
    readonly topicName: string;
}
/**
 * Options for email subscriptions.
 */
export interface EmailSubscriptionOptions {
    /**
     * Indicates if the full notification JSON should be sent to the email
     * address or just the message text.
     *
     * @default Message text (false)
     */
    readonly json?: boolean;
}
