"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const s3n = require("@aws-cdk/aws-s3-notifications");
const cdk = require("@aws-cdk/cdk");
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends cdk.Construct {
    constructor() {
        super(...arguments);
        /** Buckets permitted to send notifications to this topic */
        this.notifyingBuckets = new Set();
        /**
         * Indicates if the resource policy that allows CloudWatch events to publish
         * notifications to this topic have been added.
         */
        this.eventRuleTargetPolicyAdded = false;
    }
    /**
     * Subscribe some endpoint to this topic
     */
    subscribe(name, endpoint, protocol, rawMessageDelivery) {
        return new subscription_1.Subscription(this, name, {
            topic: this,
            endpoint,
            protocol,
            rawMessageDelivery,
        });
    }
    /**
     * Defines a subscription from this SNS topic to an SQS queue.
     *
     * The queue resource policy will be updated to allow this SNS topic to send
     * messages to the queue.
     *
     * @param name The subscription name
     * @param queue The target queue
     * @param rawMessageDelivery Enable raw message delivery
     */
    subscribeQueue(queue, rawMessageDelivery) {
        if (!cdk.Construct.isConstruct(queue)) {
            throw new Error(`The supplied Queue object must be an instance of Construct`);
        }
        const subscriptionName = this.node.id + 'Subscription';
        if (queue.node.tryFindChild(subscriptionName)) {
            throw new Error(`A subscription between the topic ${this.node.id} and the queue ${queue.node.id} already exists`);
        }
        // we use the queue name as the subscription's. there's no meaning to subscribing
        // the same queue twice on the same topic. Create subscription under *consuming*
        // construct to make sure it ends up in the correct stack in cases of cross-stack subscriptions.
        const sub = new subscription_1.Subscription(queue, subscriptionName, {
            topic: this,
            endpoint: queue.queueArn,
            protocol: subscription_1.SubscriptionProtocol.Sqs,
            rawMessageDelivery,
        });
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        queue.addToResourcePolicy(new iam.PolicyStatement()
            .addResource(queue.queueArn)
            .addAction('sqs:SendMessage')
            .addServicePrincipal('sns.amazonaws.com')
            .setCondition('ArnEquals', { 'aws:SourceArn': this.topicArn }));
        return sub;
    }
    /**
     * Defines a subscription from this SNS Topic to a Lambda function.
     *
     * The Lambda's resource policy will be updated to allow this topic to
     * invoke the function.
     *
     * @param name A name for the subscription
     * @param lambdaFunction The Lambda function to invoke
     */
    subscribeLambda(lambdaFunction) {
        if (!cdk.Construct.isConstruct(lambdaFunction)) {
            throw new Error(`The supplied lambda Function object must be an instance of Construct`);
        }
        const subscriptionName = this.node.id + 'Subscription';
        if (lambdaFunction.node.tryFindChild(subscriptionName)) {
            throw new Error(`A subscription between the topic ${this.node.id} and the lambda ${lambdaFunction.id} already exists`);
        }
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        const sub = new subscription_1.Subscription(lambdaFunction, subscriptionName, {
            topic: this,
            endpoint: lambdaFunction.functionArn,
            protocol: subscription_1.SubscriptionProtocol.Lambda,
        });
        lambdaFunction.addPermission(this.node.id, {
            sourceArn: this.topicArn,
            principal: new iam.ServicePrincipal('sns.amazonaws.com'),
        });
        return sub;
    }
    /**
     * Defines a subscription from this SNS topic to an email address.
     *
     * @param name A name for the subscription
     * @param emailAddress The email address to use.
     * @param options Options for the email delivery format.
     */
    subscribeEmail(name, emailAddress, options) {
        const protocol = (options && options.json ? subscription_1.SubscriptionProtocol.EmailJson : subscription_1.SubscriptionProtocol.Email);
        return new subscription_1.Subscription(this, name, {
            topic: this,
            endpoint: emailAddress,
            protocol
        });
    }
    /**
     * Defines a subscription from this SNS topic to an http:// or https:// URL.
     *
     * @param name A name for the subscription
     * @param url The URL to invoke
     * @param rawMessageDelivery Enable raw message delivery
     */
    subscribeUrl(name, url, rawMessageDelivery) {
        if (!url.startsWith('http://') && !url.startsWith('https://')) {
            throw new Error('URL must start with either http:// or https://');
        }
        const protocol = url.startsWith('https:') ? subscription_1.SubscriptionProtocol.Https : subscription_1.SubscriptionProtocol.Http;
        return new subscription_1.Subscription(this, name, {
            topic: this,
            endpoint: url,
            protocol,
            rawMessageDelivery,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            // statements must be unique, so we use the statement index.
            // potantially SIDs can change as a result of order change, but this should
            // not have an impact on the policy evaluation.
            // https://docs.aws.amazon.com/sns/latest/dg/AccessPolicyLanguage_SpecialInfo.html
            statement.describe(this.policy.document.statementCount.toString());
            this.policy.document.addStatement(statement);
        }
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(identity) {
        if (!identity) {
            return;
        }
        identity.addToPolicy(new iam.PolicyStatement()
            .addResource(this.topicArn)
            .addActions('sns:Publish'));
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SNS topic as a
     * result from a CloudWatch event.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/resource-based-policies-cwe.html#sns-permissions
     */
    asEventRuleTarget(_ruleArn, _ruleId) {
        if (!this.eventRuleTargetPolicyAdded) {
            this.addToResourcePolicy(new iam.PolicyStatement()
                .addAction('sns:Publish')
                .addPrincipal(new iam.ServicePrincipal('events.amazonaws.com'))
                .addResource(this.topicArn));
            this.eventRuleTargetPolicyAdded = true;
        }
        return {
            id: this.node.id,
            arn: this.topicArn,
        };
    }
    /**
     * Allow using SNS topics as lifecycle hook targets
     */
    asLifecycleHookTarget(lifecycleHook) {
        this.grantPublish(lifecycleHook.role);
        return { notificationTargetArn: this.topicArn };
    }
    get alarmActionArn() {
        return this.topicArn;
    }
    /**
     * Implements the IBucketNotificationDestination interface, allowing topics to be used
     * as bucket notification destinations.
     *
     * @param bucketArn The ARN of the bucket sending the notifications
     * @param bucketId A unique ID of the bucket
     */
    asBucketNotificationDestination(bucketArn, bucketId) {
        // allow this bucket to sns:publish to this topic (if it doesn't already have a permission)
        if (!this.notifyingBuckets.has(bucketId)) {
            this.addToResourcePolicy(new iam.PolicyStatement()
                .addServicePrincipal('s3.amazonaws.com')
                .addAction('sns:Publish')
                .addResource(this.topicArn)
                .addCondition('ArnLike', { "aws:SourceArn": bucketArn }));
            this.notifyingBuckets.add(bucketId);
        }
        return {
            arn: this.topicArn,
            type: s3n.BucketNotificationDestinationType.Topic,
            dependencies: [this.policy] // make sure the topic policy resource is created before the notification config
        };
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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