"use strict";
const assert_1 = require("@aws-cdk/assert");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3n = require("@aws-cdk/aws-s3-notifications");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/cdk");
const sns = require("../lib");
module.exports = {
    'topic tests': {
        'all defaults'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic');
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic"
                    }
                }
            });
            test.done();
        },
        'specify topicName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "TopicName": "topicName"
                        }
                    }
                }
            });
            test.done();
        },
        'specify displayName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                displayName: 'displayName'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName"
                        }
                    }
                }
            });
            test.done();
        },
        'specify both'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    }
                }
            });
            test.done();
        },
    },
    'subscription tests': {
        'url subscription'(test) {
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            topic.subscribeUrl('appsubscription', 'https://foobar.com/');
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    },
                    "MyTopicappsubscription00FA69EA": {
                        "Type": "AWS::SNS::Subscription",
                        "Properties": {
                            "Endpoint": "https://foobar.com/",
                            "Protocol": "https",
                            "TopicArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'url subscription (with raw delivery)'(test) {
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            topic.subscribeUrl('appsubscription', 'https://foobar.com/', true);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    },
                    "MyTopicappsubscription00FA69EA": {
                        "Type": "AWS::SNS::Subscription",
                        "Properties": {
                            "Endpoint": "https://foobar.com/",
                            "Protocol": "https",
                            "TopicArn": { "Ref": "MyTopic86869434" },
                            "RawMessageDelivery": true
                        }
                    }
                }
            });
            test.done();
        },
        'queue subscription'(test) {
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            const queue = new sqs.Queue(stack, 'MyQueue');
            topic.subscribeQueue(queue);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    },
                    "MyQueueMyTopicSubscriptionEB66AD1B": {
                        "Type": "AWS::SNS::Subscription",
                        "Properties": {
                            "Endpoint": {
                                "Fn::GetAtt": [
                                    "MyQueueE6CA6235",
                                    "Arn"
                                ]
                            },
                            "Protocol": "sqs",
                            "TopicArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    },
                    "MyQueueE6CA6235": {
                        "Type": "AWS::SQS::Queue"
                    },
                    "MyQueuePolicy6BBEDDAC": {
                        "Type": "AWS::SQS::QueuePolicy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sqs:SendMessage",
                                        "Condition": {
                                            "ArnEquals": {
                                                "aws:SourceArn": {
                                                    "Ref": "MyTopic86869434"
                                                }
                                            }
                                        },
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": "sns.amazonaws.com"
                                        },
                                        "Resource": {
                                            "Fn::GetAtt": [
                                                "MyQueueE6CA6235",
                                                "Arn"
                                            ]
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "Queues": [
                                {
                                    "Ref": "MyQueueE6CA6235"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'queue subscription (with raw delivery)'(test) {
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            const queue = new sqs.Queue(stack, 'MyQueue');
            topic.subscribeQueue(queue, true);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
                "Endpoint": {
                    "Fn::GetAtt": [
                        "MyQueueE6CA6235",
                        "Arn"
                    ]
                },
                "Protocol": "sqs",
                "TopicArn": {
                    "Ref": "MyTopic86869434"
                },
                "RawMessageDelivery": true
            }));
            test.done();
        },
        'lambda subscription'(test) {
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            const fction = new lambda.Function(stack, 'MyFunc', {
                runtime: lambda.Runtime.NodeJS610,
                handler: 'index.handler',
                code: lambda.Code.inline('exports.handler = function(e, c, cb) { return cb() }')
            });
            topic.subscribeLambda(fction);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    },
                    "MyFuncMyTopicSubscription708A6535": {
                        "Type": "AWS::SNS::Subscription",
                        "Properties": {
                            "Endpoint": {
                                "Fn::GetAtt": [
                                    "MyFunc8A243A2C",
                                    "Arn"
                                ]
                            },
                            "Protocol": "lambda",
                            "TopicArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    },
                    "MyFuncServiceRole54065130": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "ManagedPolicyArns": [
                                { "Fn::Join": ["", ["arn:", { "Ref": "AWS::Partition" }, ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"]] }
                            ]
                        }
                    },
                    "MyFunc8A243A2C": {
                        "Type": "AWS::Lambda::Function",
                        "Properties": {
                            "Code": {
                                "ZipFile": "exports.handler = function(e, c, cb) { return cb() }"
                            },
                            "Handler": "index.handler",
                            "Role": {
                                "Fn::GetAtt": [
                                    "MyFuncServiceRole54065130",
                                    "Arn"
                                ]
                            },
                            "Runtime": "nodejs6.10"
                        },
                        "DependsOn": [
                            "MyFuncServiceRole54065130"
                        ]
                    },
                    "MyFuncMyTopicC77D8FAB": {
                        "Type": "AWS::Lambda::Permission",
                        "Properties": {
                            "Action": "lambda:InvokeFunction",
                            "FunctionName": {
                                "Fn::GetAtt": [
                                    "MyFunc8A243A2C",
                                    "Arn"
                                ]
                            },
                            "Principal": "sns.amazonaws.com",
                            "SourceArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'email subscription'(test) {
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            topic.subscribeEmail('emailsub', 'foo@bar.com');
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    },
                    "MyTopicemailsub17B79A3E": {
                        "Type": "AWS::SNS::Subscription",
                        "Properties": {
                            "Endpoint": "foo@bar.com",
                            "Protocol": "email",
                            "TopicArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'multiple subscriptions'(test) {
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            const queue = new sqs.Queue(stack, 'MyQueue');
            const func = new lambda.Function(stack, 'MyFunc', {
                runtime: lambda.Runtime.NodeJS610,
                handler: 'index.handler',
                code: lambda.Code.inline('exports.handler = function(e, c, cb) { return cb() }')
            });
            topic.subscribeQueue(queue);
            topic.subscribeLambda(func);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    },
                    "MyQueueMyTopicSubscriptionEB66AD1B": {
                        "Type": "AWS::SNS::Subscription",
                        "Properties": {
                            "Endpoint": {
                                "Fn::GetAtt": [
                                    "MyQueueE6CA6235",
                                    "Arn"
                                ]
                            },
                            "Protocol": "sqs",
                            "TopicArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    },
                    "MyFuncMyTopicSubscription708A6535": {
                        "Type": "AWS::SNS::Subscription",
                        "Properties": {
                            "Endpoint": {
                                "Fn::GetAtt": [
                                    "MyFunc8A243A2C",
                                    "Arn"
                                ]
                            },
                            "Protocol": "lambda",
                            "TopicArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    },
                    "MyQueueE6CA6235": {
                        "Type": "AWS::SQS::Queue"
                    },
                    "MyQueuePolicy6BBEDDAC": {
                        "Type": "AWS::SQS::QueuePolicy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sqs:SendMessage",
                                        "Condition": {
                                            "ArnEquals": {
                                                "aws:SourceArn": {
                                                    "Ref": "MyTopic86869434"
                                                }
                                            }
                                        },
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": "sns.amazonaws.com"
                                        },
                                        "Resource": {
                                            "Fn::GetAtt": [
                                                "MyQueueE6CA6235",
                                                "Arn"
                                            ]
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "Queues": [
                                {
                                    "Ref": "MyQueueE6CA6235"
                                }
                            ]
                        }
                    },
                    "MyFuncServiceRole54065130": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "ManagedPolicyArns": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition"
                                            },
                                            ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                        ]
                                    ]
                                }
                            ]
                        }
                    },
                    "MyFunc8A243A2C": {
                        "Type": "AWS::Lambda::Function",
                        "Properties": {
                            "Code": {
                                "ZipFile": "exports.handler = function(e, c, cb) { return cb() }"
                            },
                            "Handler": "index.handler",
                            "Role": {
                                "Fn::GetAtt": [
                                    "MyFuncServiceRole54065130",
                                    "Arn"
                                ]
                            },
                            "Runtime": "nodejs6.10"
                        },
                        "DependsOn": [
                            "MyFuncServiceRole54065130"
                        ]
                    },
                    "MyFuncMyTopicC77D8FAB": {
                        "Type": "AWS::Lambda::Permission",
                        "Properties": {
                            "Action": "lambda:InvokeFunction",
                            "FunctionName": {
                                "Fn::GetAtt": [
                                    "MyFunc8A243A2C",
                                    "Arn"
                                ]
                            },
                            "Principal": "sns.amazonaws.com",
                            "SourceArn": {
                                "Ref": "MyTopic86869434"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'invalid use of raw message delivery'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const topic = new sns.Topic(stack, 'Topic');
            // THEN
            test.throws(() => topic.subscribe('Nope', 'endpoint://location', sns.SubscriptionProtocol.Application, true), /Raw message delivery can only be enabled for HTTP\/S and SQS subscriptions/);
            test.done();
        }
    },
    'can add a policy to the topic'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addToResourcePolicy(new iam.PolicyStatement()
            .addAllResources()
            .addActions('sns:*')
            .addPrincipal(new iam.ArnPrincipal('arn')));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        "Sid": "0",
                        "Action": "sns:*",
                        "Effect": "Allow",
                        "Principal": { "AWS": "arn" },
                        "Resource": "*"
                    }]
            }
        }));
        test.done();
    },
    'give publishing permissions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const user = new iam.User(stack, 'User');
        // WHEN
        topic.grantPublish(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                Version: '2012-10-17',
                "Statement": [
                    {
                        "Action": "sns:Publish",
                        "Effect": "Allow",
                        "Resource": stack.node.resolve(topic.topicArn)
                    }
                ],
            }
        }));
        test.done();
    },
    'topics can be used as event rule targets (and then the topic policy will allow that too)'(test) {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'MyTopic');
        const rule = new events.EventRule(stack, 'MyRule', {
            scheduleExpression: 'rate(1 hour)',
        });
        rule.addTarget(topic);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyTopic86869434": {
                    "Type": "AWS::SNS::Topic"
                },
                "MyTopicPolicy12A5EC17": {
                    "Type": "AWS::SNS::TopicPolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Sid": "0",
                                    "Action": "sns:Publish",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["events.", { Ref: "AWS::URLSuffix" }]] }
                                    },
                                    "Resource": {
                                        "Ref": "MyTopic86869434"
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Topics": [
                            {
                                "Ref": "MyTopic86869434"
                            }
                        ]
                    }
                },
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "ScheduleExpression": "rate(1 hour)",
                        "State": "ENABLED",
                        "Targets": [
                            {
                                "Arn": {
                                    "Ref": "MyTopic86869434"
                                },
                                "Id": "MyTopic"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'topic resource policy includes unique SIDs'(test) {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'MyTopic');
        topic.addToResourcePolicy(new iam.PolicyStatement().addAction('statement0'));
        topic.addToResourcePolicy(new iam.PolicyStatement().addAction('statement1'));
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyTopic86869434": {
                    "Type": "AWS::SNS::Topic"
                },
                "MyTopicPolicy12A5EC17": {
                    "Type": "AWS::SNS::TopicPolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "statement0",
                                    "Effect": "Allow",
                                    "Sid": "0"
                                },
                                {
                                    "Action": "statement1",
                                    "Effect": "Allow",
                                    "Sid": "1"
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Topics": [
                            {
                                "Ref": "MyTopic86869434"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'export/import'(test) {
        // GIVEN
        const stack1 = new cdk.Stack();
        const topic = new sns.Topic(stack1, 'Topic');
        const stack2 = new cdk.Stack();
        const queue = new sqs.Queue(stack2, 'Queue');
        // WHEN
        const ref = topic.export();
        const imported = sns.Topic.import(stack2, 'Imported', ref);
        imported.subscribeQueue(queue);
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::SNS::Subscription', {
            "TopicArn": { "Fn::ImportValue": "Stack:TopicTopicArnB66B79C2" },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::SQS::QueuePolicy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        "Action": "sqs:SendMessage",
                        "Condition": {
                            "ArnEquals": {
                                "aws:SourceArn": stack2.node.resolve(imported.topicArn)
                            }
                        },
                        "Principal": { "Service": "sns.amazonaws.com" },
                        "Resource": stack2.node.resolve(queue.queueArn),
                        "Effect": "Allow",
                    }
                ],
            },
        }));
        test.done();
    },
    'asBucketNotificationDestination adds bucket permissions only once for each bucket'(test) {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'MyTopic');
        const bucketArn = 'arn:bucket';
        const bucketId = 'bucketId';
        const dest1 = topic.asBucketNotificationDestination(bucketArn, bucketId);
        test.deepEqual(stack.node.resolve(dest1.arn), stack.node.resolve(topic.topicArn));
        test.deepEqual(dest1.type, s3n.BucketNotificationDestinationType.Topic);
        const dep = dest1.dependencies[0];
        test.deepEqual(stack.node.resolve(dep.node.children[0].logicalId), 'MyTopicPolicy12A5EC17', 'verify topic policy is added as dependency');
        // calling again on the same bucket yields is idempotent
        const dest2 = topic.asBucketNotificationDestination(bucketArn, bucketId);
        test.deepEqual(stack.node.resolve(dest2.arn), stack.node.resolve(topic.topicArn));
        test.deepEqual(dest2.type, s3n.BucketNotificationDestinationType.Topic);
        // another bucket will be added to the topic policy
        const dest3 = topic.asBucketNotificationDestination('bucket2', 'bucket2');
        test.deepEqual(stack.node.resolve(dest3.arn), stack.node.resolve(topic.topicArn));
        test.deepEqual(dest3.type, s3n.BucketNotificationDestinationType.Topic);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyTopic86869434": {
                    "Type": "AWS::SNS::Topic"
                },
                "MyTopicPolicy12A5EC17": {
                    "Type": "AWS::SNS::TopicPolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sns:Publish",
                                    "Condition": {
                                        "ArnLike": {
                                            "aws:SourceArn": "arn:bucket"
                                        }
                                    },
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]] }
                                    },
                                    "Resource": {
                                        "Ref": "MyTopic86869434"
                                    },
                                    "Sid": "0"
                                },
                                {
                                    "Action": "sns:Publish",
                                    "Condition": {
                                        "ArnLike": {
                                            "aws:SourceArn": "bucket2"
                                        }
                                    },
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]] }
                                    },
                                    "Resource": {
                                        "Ref": "MyTopic86869434"
                                    },
                                    "Sid": "1"
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Topics": [
                            {
                                "Ref": "MyTopic86869434"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.deepEqual(stack.node.resolve(topic.metricNumberOfMessagesPublished()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'NumberOfMessagesPublished',
            periodSec: 300,
            statistic: 'Sum'
        });
        test.deepEqual(stack.node.resolve(topic.metricPublishSize()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'PublishSize',
            periodSec: 300,
            statistic: 'Average'
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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