#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
pysilsub.CIE
============

Convenience functions for accessing CIE standards.

Obtained from http://www.cvrl.org/

@author: jtm

"""

from typing import Optional, Union

import numpy as np
import pandas as pd


def get_CIE_2006_10_deg_CMF(
    binwidth: Optional[int] = 1,
    asdf: Optional[bool] = True,
    trim_visible: Optional[bool] = True,
) -> Union[np.ndarray, pd.DataFrame]:
    """Get the CIE 2006 XYZ 10-deg physiologically relevant color matching
    functions.

    Parameters
    ----------
    asdf : bool, optional
        Whether to return the results as a pandas DataFrame. The default is
        False.
    binwidth : int, optional
        Width of the wavelength bins in nanometers (must be `1` or `5`). The
        default is `1`.
    trim_visible : bool, optional
        Whether to trim the CMFs to 380-780 nm. The default is True.

    Returns
    -------
    cmf : numpy.ndarray or pandas.DataFrame
        The CIE 2006 XYZ 10-deg physiologically relevant CMFs.

    """
    # breakpoint()
    colnames = ["Wavelength", "X", "Y", "Z"]

    cmf = np.array(
        [
            390,
            2.95e-03,
            4.08e-04,
            1.32e-02,
            391,
            3.58e-03,
            4.98e-04,
            1.60e-02,
            392,
            4.33e-03,
            6.06e-04,
            1.94e-02,
            393,
            5.24e-03,
            7.37e-04,
            2.34e-02,
            394,
            6.33e-03,
            8.93e-04,
            2.84e-02,
            395,
            7.64e-03,
            1.08e-03,
            3.42e-02,
            396,
            9.20e-03,
            1.30e-03,
            4.13e-02,
            397,
            1.10e-02,
            1.55e-03,
            4.97e-02,
            398,
            1.32e-02,
            1.85e-03,
            5.96e-02,
            399,
            1.58e-02,
            2.20e-03,
            7.13e-02,
            400,
            1.88e-02,
            2.59e-03,
            8.51e-02,
            401,
            2.23e-02,
            3.04e-03,
            1.01e-01,
            402,
            2.63e-02,
            3.54e-03,
            1.20e-01,
            403,
            3.09e-02,
            4.11e-03,
            1.41e-01,
            404,
            3.61e-02,
            4.75e-03,
            1.65e-01,
            405,
            4.20e-02,
            5.47e-03,
            1.93e-01,
            406,
            4.87e-02,
            6.29e-03,
            2.24e-01,
            407,
            5.61e-02,
            7.19e-03,
            2.58e-01,
            408,
            6.43e-02,
            8.18e-03,
            2.96e-01,
            409,
            7.32e-02,
            9.26e-03,
            3.38e-01,
            410,
            8.28e-02,
            1.04e-02,
            3.83e-01,
            411,
            9.30e-02,
            1.16e-02,
            4.32e-01,
            412,
            1.04e-01,
            1.29e-02,
            4.83e-01,
            413,
            1.15e-01,
            1.42e-02,
            5.38e-01,
            414,
            1.27e-01,
            1.56e-02,
            5.96e-01,
            415,
            1.40e-01,
            1.71e-02,
            6.57e-01,
            416,
            1.53e-01,
            1.87e-02,
            7.21e-01,
            417,
            1.66e-01,
            2.04e-02,
            7.88e-01,
            418,
            1.80e-01,
            2.21e-02,
            8.56e-01,
            419,
            1.94e-01,
            2.39e-02,
            9.25e-01,
            420,
            2.08e-01,
            2.58e-02,
            9.93e-01,
            421,
            2.21e-01,
            2.76e-02,
            1.06e00,
            422,
            2.33e-01,
            2.95e-02,
            1.12e00,
            423,
            2.45e-01,
            3.13e-02,
            1.18e00,
            424,
            2.57e-01,
            3.33e-02,
            1.25e00,
            425,
            2.69e-01,
            3.53e-02,
            1.31e00,
            426,
            2.81e-01,
            3.74e-02,
            1.37e00,
            427,
            2.93e-01,
            3.97e-02,
            1.44e00,
            428,
            3.06e-01,
            4.20e-02,
            1.50e00,
            429,
            3.17e-01,
            4.45e-02,
            1.57e00,
            430,
            3.28e-01,
            4.70e-02,
            1.62e00,
            431,
            3.38e-01,
            4.96e-02,
            1.68e00,
            432,
            3.47e-01,
            5.22e-02,
            1.73e00,
            433,
            3.54e-01,
            5.49e-02,
            1.78e00,
            434,
            3.62e-01,
            5.77e-02,
            1.82e00,
            435,
            3.69e-01,
            6.05e-02,
            1.87e00,
            436,
            3.77e-01,
            6.33e-02,
            1.91e00,
            437,
            3.85e-01,
            6.62e-02,
            1.96e00,
            438,
            3.92e-01,
            6.91e-02,
            2.01e00,
            439,
            3.98e-01,
            7.19e-02,
            2.04e00,
            440,
            4.03e-01,
            7.47e-02,
            2.08e00,
            441,
            4.05e-01,
            7.74e-02,
            2.10e00,
            442,
            4.06e-01,
            8.00e-02,
            2.11e00,
            443,
            4.06e-01,
            8.27e-02,
            2.12e00,
            444,
            4.05e-01,
            8.54e-02,
            2.13e00,
            445,
            4.04e-01,
            8.82e-02,
            2.13e00,
            446,
            4.03e-01,
            9.12e-02,
            2.14e00,
            447,
            4.03e-01,
            9.43e-02,
            2.14e00,
            448,
            4.01e-01,
            9.75e-02,
            2.15e00,
            449,
            3.98e-01,
            1.01e-01,
            2.14e00,
            450,
            3.93e-01,
            1.04e-01,
            2.13e00,
            451,
            3.86e-01,
            1.07e-01,
            2.10e00,
            452,
            3.78e-01,
            1.10e-01,
            2.07e00,
            453,
            3.68e-01,
            1.13e-01,
            2.03e00,
            454,
            3.58e-01,
            1.16e-01,
            1.99e00,
            455,
            3.48e-01,
            1.20e-01,
            1.95e00,
            456,
            3.38e-01,
            1.23e-01,
            1.91e00,
            457,
            3.29e-01,
            1.27e-01,
            1.87e00,
            458,
            3.19e-01,
            1.32e-01,
            1.84e00,
            459,
            3.10e-01,
            1.36e-01,
            1.80e00,
            460,
            3.01e-01,
            1.41e-01,
            1.77e00,
            461,
            2.92e-01,
            1.47e-01,
            1.74e00,
            462,
            2.83e-01,
            1.52e-01,
            1.70e00,
            463,
            2.74e-01,
            1.58e-01,
            1.67e00,
            464,
            2.64e-01,
            1.64e-01,
            1.63e00,
            465,
            2.53e-01,
            1.70e-01,
            1.58e00,
            466,
            2.42e-01,
            1.76e-01,
            1.53e00,
            467,
            2.30e-01,
            1.82e-01,
            1.48e00,
            468,
            2.17e-01,
            1.88e-01,
            1.43e00,
            469,
            2.04e-01,
            1.94e-01,
            1.37e00,
            470,
            1.91e-01,
            2.00e-01,
            1.31e00,
            471,
            1.78e-01,
            2.06e-01,
            1.25e00,
            472,
            1.65e-01,
            2.12e-01,
            1.19e00,
            473,
            1.53e-01,
            2.18e-01,
            1.13e00,
            474,
            1.40e-01,
            2.25e-01,
            1.07e00,
            475,
            1.28e-01,
            2.31e-01,
            1.01e00,
            476,
            1.17e-01,
            2.38e-01,
            9.54e-01,
            477,
            1.06e-01,
            2.45e-01,
            9.00e-01,
            478,
            9.51e-02,
            2.53e-01,
            8.47e-01,
            479,
            8.52e-02,
            2.60e-01,
            7.98e-01,
            480,
            7.59e-02,
            2.68e-01,
            7.52e-01,
            481,
            6.73e-02,
            2.77e-01,
            7.08e-01,
            482,
            5.93e-02,
            2.85e-01,
            6.67e-01,
            483,
            5.18e-02,
            2.94e-01,
            6.28e-01,
            484,
            4.49e-02,
            3.02e-01,
            5.91e-01,
            485,
            3.84e-02,
            3.11e-01,
            5.55e-01,
            486,
            3.24e-02,
            3.19e-01,
            5.20e-01,
            487,
            2.69e-02,
            3.28e-01,
            4.86e-01,
            488,
            2.20e-02,
            3.37e-01,
            4.55e-01,
            489,
            1.77e-02,
            3.46e-01,
            4.25e-01,
            490,
            1.40e-02,
            3.55e-01,
            3.98e-01,
            491,
            1.08e-02,
            3.66e-01,
            3.73e-01,
            492,
            8.17e-03,
            3.78e-01,
            3.50e-01,
            493,
            6.04e-03,
            3.90e-01,
            3.29e-01,
            494,
            4.46e-03,
            4.02e-01,
            3.09e-01,
            495,
            3.45e-03,
            4.15e-01,
            2.91e-01,
            496,
            3.01e-03,
            4.27e-01,
            2.73e-01,
            497,
            3.09e-03,
            4.40e-01,
            2.56e-01,
            498,
            3.61e-03,
            4.52e-01,
            2.39e-01,
            499,
            4.49e-03,
            4.65e-01,
            2.23e-01,
            500,
            5.65e-03,
            4.78e-01,
            2.08e-01,
            501,
            7.04e-03,
            4.92e-01,
            1.93e-01,
            502,
            8.67e-03,
            5.05e-01,
            1.79e-01,
            503,
            1.06e-02,
            5.20e-01,
            1.65e-01,
            504,
            1.29e-02,
            5.34e-01,
            1.52e-01,
            505,
            1.56e-02,
            5.49e-01,
            1.39e-01,
            506,
            1.88e-02,
            5.64e-01,
            1.28e-01,
            507,
            2.26e-02,
            5.79e-01,
            1.16e-01,
            508,
            2.69e-02,
            5.94e-01,
            1.06e-01,
            509,
            3.20e-02,
            6.10e-01,
            9.68e-02,
            510,
            3.78e-02,
            6.25e-01,
            8.85e-02,
            511,
            4.43e-02,
            6.40e-01,
            8.12e-02,
            512,
            5.15e-02,
            6.55e-01,
            7.46e-02,
            513,
            5.91e-02,
            6.70e-01,
            6.87e-02,
            514,
            6.71e-02,
            6.86e-01,
            6.33e-02,
            515,
            7.54e-02,
            7.01e-01,
            5.82e-02,
            516,
            8.38e-02,
            7.17e-01,
            5.35e-02,
            517,
            9.23e-02,
            7.33e-01,
            4.91e-02,
            518,
            1.01e-01,
            7.49e-01,
            4.51e-02,
            519,
            1.10e-01,
            7.64e-01,
            4.13e-02,
            520,
            1.20e-01,
            7.79e-01,
            3.78e-02,
            521,
            1.30e-01,
            7.92e-01,
            3.47e-02,
            522,
            1.41e-01,
            8.05e-01,
            3.18e-02,
            523,
            1.52e-01,
            8.16e-01,
            2.91e-02,
            524,
            1.64e-01,
            8.27e-01,
            2.66e-02,
            525,
            1.76e-01,
            8.38e-01,
            2.43e-02,
            526,
            1.88e-01,
            8.47e-01,
            2.22e-02,
            527,
            2.00e-01,
            8.57e-01,
            2.03e-02,
            528,
            2.13e-01,
            8.66e-01,
            1.85e-02,
            529,
            2.25e-01,
            8.75e-01,
            1.69e-02,
            530,
            2.38e-01,
            8.83e-01,
            1.54e-02,
            531,
            2.51e-01,
            8.91e-01,
            1.41e-02,
            532,
            2.64e-01,
            8.99e-01,
            1.28e-02,
            533,
            2.77e-01,
            9.07e-01,
            1.17e-02,
            534,
            2.90e-01,
            9.15e-01,
            1.07e-02,
            535,
            3.05e-01,
            9.23e-01,
            9.75e-03,
            536,
            3.20e-01,
            9.32e-01,
            8.89e-03,
            537,
            3.35e-01,
            9.41e-01,
            8.09e-03,
            538,
            3.51e-01,
            9.50e-01,
            7.36e-03,
            539,
            3.68e-01,
            9.59e-01,
            6.69e-03,
            540,
            3.84e-01,
            9.67e-01,
            6.08e-03,
            541,
            4.01e-01,
            9.73e-01,
            5.53e-03,
            542,
            4.17e-01,
            9.79e-01,
            5.03e-03,
            543,
            4.33e-01,
            9.83e-01,
            4.57e-03,
            544,
            4.48e-01,
            9.87e-01,
            4.15e-03,
            545,
            4.63e-01,
            9.89e-01,
            3.77e-03,
            546,
            4.78e-01,
            9.90e-01,
            3.42e-03,
            547,
            4.93e-01,
            9.90e-01,
            3.11e-03,
            548,
            5.07e-01,
            9.90e-01,
            2.82e-03,
            549,
            5.22e-01,
            9.90e-01,
            2.56e-03,
            550,
            5.37e-01,
            9.91e-01,
            2.32e-03,
            551,
            5.53e-01,
            9.92e-01,
            2.11e-03,
            552,
            5.70e-01,
            9.94e-01,
            1.91e-03,
            553,
            5.87e-01,
            9.97e-01,
            1.73e-03,
            554,
            6.05e-01,
            9.99e-01,
            1.57e-03,
            555,
            6.23e-01,
            1.00e00,
            1.43e-03,
            556,
            6.41e-01,
            1.00e00,
            1.29e-03,
            557,
            6.59e-01,
            9.99e-01,
            1.17e-03,
            558,
            6.77e-01,
            9.98e-01,
            1.07e-03,
            559,
            6.95e-01,
            9.96e-01,
            9.67e-04,
            560,
            7.12e-01,
            9.94e-01,
            8.78e-04,
            561,
            7.30e-01,
            9.93e-01,
            7.97e-04,
            562,
            7.48e-01,
            9.91e-01,
            7.23e-04,
            563,
            7.65e-01,
            9.89e-01,
            6.56e-04,
            564,
            7.83e-01,
            9.87e-01,
            5.96e-04,
            565,
            8.02e-01,
            9.85e-01,
            5.41e-04,
            566,
            8.20e-01,
            9.82e-01,
            4.91e-04,
            567,
            8.39e-01,
            9.78e-01,
            4.46e-04,
            568,
            8.57e-01,
            9.74e-01,
            4.05e-04,
            569,
            8.75e-01,
            9.69e-01,
            3.68e-04,
            570,
            8.93e-01,
            9.64e-01,
            3.34e-04,
            571,
            9.11e-01,
            9.58e-01,
            3.04e-04,
            572,
            9.27e-01,
            9.52e-01,
            2.76e-04,
            573,
            9.43e-01,
            9.45e-01,
            2.51e-04,
            574,
            9.58e-01,
            9.37e-01,
            2.28e-04,
            575,
            9.72e-01,
            9.29e-01,
            2.08e-04,
            576,
            9.85e-01,
            9.19e-01,
            1.89e-04,
            577,
            9.97e-01,
            9.08e-01,
            1.72e-04,
            578,
            1.01e00,
            8.98e-01,
            1.57e-04,
            579,
            1.02e00,
            8.87e-01,
            1.43e-04,
            580,
            1.03e00,
            8.78e-01,
            1.30e-04,
            581,
            1.05e00,
            8.69e-01,
            1.18e-04,
            582,
            1.06e00,
            8.61e-01,
            1.08e-04,
            583,
            1.08e00,
            8.53e-01,
            9.83e-05,
            584,
            1.09e00,
            8.45e-01,
            8.97e-05,
            585,
            1.11e00,
            8.37e-01,
            8.18e-05,
            586,
            1.12e00,
            8.28e-01,
            7.47e-05,
            587,
            1.13e00,
            8.19e-01,
            6.82e-05,
            588,
            1.14e00,
            8.09e-01,
            6.23e-05,
            589,
            1.14e00,
            7.98e-01,
            5.70e-05,
            590,
            1.15e00,
            7.87e-01,
            5.21e-05,
            591,
            1.15e00,
            7.76e-01,
            4.76e-05,
            592,
            1.16e00,
            7.64e-01,
            4.36e-05,
            593,
            1.16e00,
            7.52e-01,
            3.99e-05,
            594,
            1.16e00,
            7.40e-01,
            3.65e-05,
            595,
            1.16e00,
            7.27e-01,
            3.35e-05,
            596,
            1.16e00,
            7.15e-01,
            3.07e-05,
            597,
            1.16e00,
            7.02e-01,
            2.81e-05,
            598,
            1.16e00,
            6.89e-01,
            2.58e-05,
            599,
            1.15e00,
            6.76e-01,
            2.37e-05,
            600,
            1.15e00,
            6.63e-01,
            2.18e-05,
            601,
            1.14e00,
            6.50e-01,
            2.00e-05,
            602,
            1.14e00,
            6.37e-01,
            1.84e-05,
            603,
            1.13e00,
            6.24e-01,
            1.69e-05,
            604,
            1.12e00,
            6.10e-01,
            1.55e-05,
            605,
            1.11e00,
            5.97e-01,
            1.43e-05,
            606,
            1.10e00,
            5.83e-01,
            1.32e-05,
            607,
            1.09e00,
            5.70e-01,
            1.21e-05,
            608,
            1.08e00,
            5.56e-01,
            1.12e-05,
            609,
            1.06e00,
            5.42e-01,
            1.03e-05,
            610,
            1.05e00,
            5.28e-01,
            9.53e-06,
            611,
            1.03e00,
            5.14e-01,
            8.80e-06,
            612,
            1.02e00,
            5.01e-01,
            8.13e-06,
            613,
            9.98e-01,
            4.87e-01,
            7.51e-06,
            614,
            9.80e-01,
            4.74e-01,
            6.95e-06,
            615,
            9.62e-01,
            4.60e-01,
            6.43e-06,
            616,
            9.42e-01,
            4.47e-01,
            0.00e00,
            617,
            9.23e-01,
            4.33e-01,
            0.00e00,
            618,
            9.03e-01,
            4.20e-01,
            0.00e00,
            619,
            8.83e-01,
            4.08e-01,
            0.00e00,
            620,
            8.63e-01,
            3.95e-01,
            0.00e00,
            621,
            8.43e-01,
            3.83e-01,
            0.00e00,
            622,
            8.23e-01,
            3.71e-01,
            0.00e00,
            623,
            8.03e-01,
            3.59e-01,
            0.00e00,
            624,
            7.82e-01,
            3.47e-01,
            0.00e00,
            625,
            7.60e-01,
            3.35e-01,
            0.00e00,
            626,
            7.37e-01,
            3.23e-01,
            0.00e00,
            627,
            7.14e-01,
            3.11e-01,
            0.00e00,
            628,
            6.90e-01,
            2.99e-01,
            0.00e00,
            629,
            6.65e-01,
            2.87e-01,
            0.00e00,
            630,
            6.41e-01,
            2.75e-01,
            0.00e00,
            631,
            6.18e-01,
            2.64e-01,
            0.00e00,
            632,
            5.95e-01,
            2.53e-01,
            0.00e00,
            633,
            5.72e-01,
            2.42e-01,
            0.00e00,
            634,
            5.50e-01,
            2.32e-01,
            0.00e00,
            635,
            5.29e-01,
            2.22e-01,
            0.00e00,
            636,
            5.08e-01,
            2.12e-01,
            0.00e00,
            637,
            4.88e-01,
            2.03e-01,
            0.00e00,
            638,
            4.69e-01,
            1.94e-01,
            0.00e00,
            639,
            4.50e-01,
            1.86e-01,
            0.00e00,
            640,
            4.32e-01,
            1.78e-01,
            0.00e00,
            641,
            4.15e-01,
            1.70e-01,
            0.00e00,
            642,
            3.98e-01,
            1.62e-01,
            0.00e00,
            643,
            3.82e-01,
            1.55e-01,
            0.00e00,
            644,
            3.66e-01,
            1.48e-01,
            0.00e00,
            645,
            3.50e-01,
            1.41e-01,
            0.00e00,
            646,
            3.33e-01,
            1.34e-01,
            0.00e00,
            647,
            3.17e-01,
            1.27e-01,
            0.00e00,
            648,
            3.02e-01,
            1.21e-01,
            0.00e00,
            649,
            2.86e-01,
            1.15e-01,
            0.00e00,
            650,
            2.71e-01,
            1.08e-01,
            0.00e00,
            651,
            2.57e-01,
            1.03e-01,
            0.00e00,
            652,
            2.43e-01,
            9.68e-02,
            0.00e00,
            653,
            2.30e-01,
            9.14e-02,
            0.00e00,
            654,
            2.18e-01,
            8.63e-02,
            0.00e00,
            655,
            2.06e-01,
            8.14e-02,
            0.00e00,
            656,
            1.94e-01,
            7.67e-02,
            0.00e00,
            657,
            1.83e-01,
            7.23e-02,
            0.00e00,
            658,
            1.73e-01,
            6.81e-02,
            0.00e00,
            659,
            1.63e-01,
            6.41e-02,
            0.00e00,
            660,
            1.54e-01,
            6.03e-02,
            0.00e00,
            661,
            1.45e-01,
            5.68e-02,
            0.00e00,
            662,
            1.36e-01,
            5.34e-02,
            0.00e00,
            663,
            1.28e-01,
            5.02e-02,
            0.00e00,
            664,
            1.21e-01,
            4.71e-02,
            0.00e00,
            665,
            1.14e-01,
            4.43e-02,
            0.00e00,
            666,
            1.07e-01,
            4.15e-02,
            0.00e00,
            667,
            1.00e-01,
            3.90e-02,
            0.00e00,
            668,
            9.41e-02,
            3.66e-02,
            0.00e00,
            669,
            8.83e-02,
            3.43e-02,
            0.00e00,
            670,
            8.28e-02,
            3.21e-02,
            0.00e00,
            671,
            7.76e-02,
            3.01e-02,
            0.00e00,
            672,
            7.27e-02,
            2.82e-02,
            0.00e00,
            673,
            6.81e-02,
            2.63e-02,
            0.00e00,
            674,
            6.37e-02,
            2.46e-02,
            0.00e00,
            675,
            5.95e-02,
            2.30e-02,
            0.00e00,
            676,
            5.56e-02,
            2.15e-02,
            0.00e00,
            677,
            5.20e-02,
            2.01e-02,
            0.00e00,
            678,
            4.85e-02,
            1.87e-02,
            0.00e00,
            679,
            4.53e-02,
            1.75e-02,
            0.00e00,
            680,
            4.22e-02,
            1.63e-02,
            0.00e00,
            681,
            3.93e-02,
            1.52e-02,
            0.00e00,
            682,
            3.67e-02,
            1.41e-02,
            0.00e00,
            683,
            3.41e-02,
            1.32e-02,
            0.00e00,
            684,
            3.17e-02,
            1.22e-02,
            0.00e00,
            685,
            2.95e-02,
            1.14e-02,
            0.00e00,
            686,
            2.74e-02,
            1.05e-02,
            0.00e00,
            687,
            2.54e-02,
            9.78e-03,
            0.00e00,
            688,
            2.35e-02,
            9.06e-03,
            0.00e00,
            689,
            2.18e-02,
            8.40e-03,
            0.00e00,
            690,
            2.03e-02,
            7.80e-03,
            0.00e00,
            691,
            1.88e-02,
            7.24e-03,
            0.00e00,
            692,
            1.75e-02,
            6.73e-03,
            0.00e00,
            693,
            1.63e-02,
            6.27e-03,
            0.00e00,
            694,
            1.52e-02,
            5.83e-03,
            0.00e00,
            695,
            1.41e-02,
            5.43e-03,
            0.00e00,
            696,
            1.31e-02,
            5.05e-03,
            0.00e00,
            697,
            1.22e-02,
            4.70e-03,
            0.00e00,
            698,
            1.14e-02,
            4.37e-03,
            0.00e00,
            699,
            1.06e-02,
            4.06e-03,
            0.00e00,
            700,
            9.82e-03,
            3.78e-03,
            0.00e00,
            701,
            9.13e-03,
            3.51e-03,
            0.00e00,
            702,
            8.49e-03,
            3.27e-03,
            0.00e00,
            703,
            7.89e-03,
            3.04e-03,
            0.00e00,
            704,
            7.33e-03,
            2.82e-03,
            0.00e00,
            705,
            6.81e-03,
            2.62e-03,
            0.00e00,
            706,
            6.32e-03,
            2.43e-03,
            0.00e00,
            707,
            5.86e-03,
            2.25e-03,
            0.00e00,
            708,
            5.43e-03,
            2.09e-03,
            0.00e00,
            709,
            5.04e-03,
            1.94e-03,
            0.00e00,
            710,
            4.67e-03,
            1.80e-03,
            0.00e00,
            711,
            4.32e-03,
            1.66e-03,
            0.00e00,
            712,
            4.01e-03,
            1.54e-03,
            0.00e00,
            713,
            3.71e-03,
            1.43e-03,
            0.00e00,
            714,
            3.44e-03,
            1.33e-03,
            0.00e00,
            715,
            3.19e-03,
            1.23e-03,
            0.00e00,
            716,
            2.96e-03,
            1.14e-03,
            0.00e00,
            717,
            2.75e-03,
            1.06e-03,
            0.00e00,
            718,
            2.56e-03,
            9.85e-04,
            0.00e00,
            719,
            2.37e-03,
            9.15e-04,
            0.00e00,
            720,
            2.21e-03,
            8.50e-04,
            0.00e00,
            721,
            2.05e-03,
            7.90e-04,
            0.00e00,
            722,
            1.90e-03,
            7.33e-04,
            0.00e00,
            723,
            1.77e-03,
            6.81e-04,
            0.00e00,
            724,
            1.64e-03,
            6.33e-04,
            0.00e00,
            725,
            1.52e-03,
            5.88e-04,
            0.00e00,
            726,
            1.42e-03,
            5.47e-04,
            0.00e00,
            727,
            1.32e-03,
            5.09e-04,
            0.00e00,
            728,
            1.23e-03,
            4.73e-04,
            0.00e00,
            729,
            1.14e-03,
            4.40e-04,
            0.00e00,
            730,
            1.06e-03,
            4.10e-04,
            0.00e00,
            731,
            9.88e-04,
            3.82e-04,
            0.00e00,
            732,
            9.19e-04,
            3.55e-04,
            0.00e00,
            733,
            8.55e-04,
            3.30e-04,
            0.00e00,
            734,
            7.96e-04,
            3.08e-04,
            0.00e00,
            735,
            7.40e-04,
            2.86e-04,
            0.00e00,
            736,
            6.88e-04,
            2.66e-04,
            0.00e00,
            737,
            6.40e-04,
            2.47e-04,
            0.00e00,
            738,
            5.95e-04,
            2.30e-04,
            0.00e00,
            739,
            5.54e-04,
            2.14e-04,
            0.00e00,
            740,
            5.15e-04,
            1.99e-04,
            0.00e00,
            741,
            4.80e-04,
            1.86e-04,
            0.00e00,
            742,
            4.48e-04,
            1.73e-04,
            0.00e00,
            743,
            4.17e-04,
            1.62e-04,
            0.00e00,
            744,
            3.89e-04,
            1.51e-04,
            0.00e00,
            745,
            3.63e-04,
            1.41e-04,
            0.00e00,
            746,
            3.39e-04,
            1.31e-04,
            0.00e00,
            747,
            3.16e-04,
            1.22e-04,
            0.00e00,
            748,
            2.94e-04,
            1.14e-04,
            0.00e00,
            749,
            2.74e-04,
            1.06e-04,
            0.00e00,
            750,
            2.56e-04,
            9.93e-05,
            0.00e00,
            751,
            2.38e-04,
            9.27e-05,
            0.00e00,
            752,
            2.22e-04,
            8.65e-05,
            0.00e00,
            753,
            2.08e-04,
            8.07e-05,
            0.00e00,
            754,
            1.94e-04,
            7.54e-05,
            0.00e00,
            755,
            1.81e-04,
            7.04e-05,
            0.00e00,
            756,
            1.69e-04,
            6.58e-05,
            0.00e00,
            757,
            1.58e-04,
            6.15e-05,
            0.00e00,
            758,
            1.47e-04,
            5.75e-05,
            0.00e00,
            759,
            1.38e-04,
            5.37e-05,
            0.00e00,
            760,
            1.29e-04,
            5.02e-05,
            0.00e00,
            761,
            1.20e-04,
            4.69e-05,
            0.00e00,
            762,
            1.12e-04,
            4.38e-05,
            0.00e00,
            763,
            1.05e-04,
            4.10e-05,
            0.00e00,
            764,
            9.81e-05,
            3.83e-05,
            0.00e00,
            765,
            9.17e-05,
            3.58e-05,
            0.00e00,
            766,
            8.58e-05,
            3.35e-05,
            0.00e00,
            767,
            8.03e-05,
            3.14e-05,
            0.00e00,
            768,
            7.51e-05,
            2.94e-05,
            0.00e00,
            769,
            7.03e-05,
            2.75e-05,
            0.00e00,
            770,
            6.58e-05,
            2.57e-05,
            0.00e00,
            771,
            6.15e-05,
            2.41e-05,
            0.00e00,
            772,
            5.75e-05,
            2.25e-05,
            0.00e00,
            773,
            5.38e-05,
            2.11e-05,
            0.00e00,
            774,
            5.03e-05,
            1.97e-05,
            0.00e00,
            775,
            4.71e-05,
            1.85e-05,
            0.00e00,
            776,
            4.41e-05,
            1.73e-05,
            0.00e00,
            777,
            4.13e-05,
            1.62e-05,
            0.00e00,
            778,
            3.87e-05,
            1.52e-05,
            0.00e00,
            779,
            3.63e-05,
            1.43e-05,
            0.00e00,
            780,
            3.41e-05,
            1.34e-05,
            0.00e00,
            781,
            3.20e-05,
            1.26e-05,
            0.00e00,
            782,
            3.00e-05,
            1.18e-05,
            0.00e00,
            783,
            2.81e-05,
            1.10e-05,
            0.00e00,
            784,
            2.63e-05,
            1.04e-05,
            0.00e00,
            785,
            2.47e-05,
            9.72e-06,
            0.00e00,
            786,
            2.32e-05,
            9.12e-06,
            0.00e00,
            787,
            2.17e-05,
            8.56e-06,
            0.00e00,
            788,
            2.04e-05,
            8.03e-06,
            0.00e00,
            789,
            1.91e-05,
            7.54e-06,
            0.00e00,
            790,
            1.79e-05,
            7.07e-06,
            0.00e00,
            791,
            1.68e-05,
            6.64e-06,
            0.00e00,
            792,
            1.58e-05,
            6.23e-06,
            0.00e00,
            793,
            1.48e-05,
            5.85e-06,
            0.00e00,
            794,
            1.39e-05,
            5.49e-06,
            0.00e00,
            795,
            1.31e-05,
            5.16e-06,
            0.00e00,
            796,
            1.23e-05,
            4.85e-06,
            0.00e00,
            797,
            1.15e-05,
            4.56e-06,
            0.00e00,
            798,
            1.08e-05,
            4.28e-06,
            0.00e00,
            799,
            1.02e-05,
            4.03e-06,
            0.00e00,
            800,
            9.57e-06,
            3.79e-06,
            0.00e00,
            801,
            9.00e-06,
            3.56e-06,
            0.00e00,
            802,
            8.46e-06,
            3.35e-06,
            0.00e00,
            803,
            7.96e-06,
            3.16e-06,
            0.00e00,
            804,
            7.48e-06,
            2.97e-06,
            0.00e00,
            805,
            7.04e-06,
            2.79e-06,
            0.00e00,
            806,
            6.62e-06,
            2.63e-06,
            0.00e00,
            807,
            6.22e-06,
            2.47e-06,
            0.00e00,
            808,
            5.85e-06,
            2.33e-06,
            0.00e00,
            809,
            5.50e-06,
            2.19e-06,
            0.00e00,
            810,
            5.17e-06,
            2.06e-06,
            0.00e00,
            811,
            4.86e-06,
            1.94e-06,
            0.00e00,
            812,
            4.57e-06,
            1.82e-06,
            0.00e00,
            813,
            4.30e-06,
            1.72e-06,
            0.00e00,
            814,
            4.05e-06,
            1.62e-06,
            0.00e00,
            815,
            3.82e-06,
            1.52e-06,
            0.00e00,
            816,
            3.59e-06,
            1.44e-06,
            0.00e00,
            817,
            3.39e-06,
            1.35e-06,
            0.00e00,
            818,
            3.19e-06,
            1.28e-06,
            0.00e00,
            819,
            3.01e-06,
            1.20e-06,
            0.00e00,
            820,
            2.84e-06,
            1.14e-06,
            0.00e00,
            821,
            2.68e-06,
            1.07e-06,
            0.00e00,
            822,
            2.52e-06,
            1.01e-06,
            0.00e00,
            823,
            2.38e-06,
            9.53e-07,
            0.00e00,
            824,
            2.24e-06,
            8.99e-07,
            0.00e00,
            825,
            2.11e-06,
            8.48e-07,
            0.00e00,
            826,
            1.99e-06,
            8.00e-07,
            0.00e00,
            827,
            1.88e-06,
            7.54e-07,
            0.00e00,
            828,
            1.77e-06,
            7.12e-07,
            0.00e00,
            829,
            1.67e-06,
            6.72e-07,
            0.00e00,
            830,
            1.58e-06,
            6.35e-07,
            0.00e00,
        ]
    )
    cmf = cmf.reshape(441, 4).astype(np.float64).T
    if trim_visible:
        prepend = np.array(
            [[wl, 0.00e00, 0.00e00, 0.00e00] for wl in range(380, 390)]
        )
        cmf = np.vstack((prepend, cmf.T))
        cmf = cmf[cmf[:, 0] < 781]
    cmf = cmf.T[:, ::binwidth].T
    if asdf:
        cmf = pd.DataFrame(data=cmf, columns=colnames)
        cmf.set_index("Wavelength", inplace=True)
        cmf.index = pd.Int64Index(cmf.index)
    return cmf


def get_CIE_CMF(asdf: Optional[bool] = True, binwidth: Optional[int] = 1):
    """Get the CIE 1931 XYZ 2-deg color matching functions.

    Parameters
    ----------
    asdf : bool, optional
        Whether to return the results as a pandas DataFrame. The default is
        False.
    binwidth : int, optional
        Width of the wavelength bins in nanometers (must be `1` or `5`). The
        default is `1`.

    Returns
    -------
    cmf : numpy.ndarray or pandas.DataFrame
        The CIE 1931 XYZ 2-deg CMFs.

    """
    colnames = ["Wavelength", "X", "Y", "Z"]

    cmf = np.array(
        [
            380,
            0.001368000000,
            0.000039000000,
            0.006450001000,
            381,
            0.001502050000,
            0.000042826400,
            0.007083216000,
            382,
            0.001642328000,
            0.000046914600,
            0.007745488000,
            383,
            0.001802382000,
            0.000051589600,
            0.008501152000,
            384,
            0.001995757000,
            0.000057176400,
            0.009414544000,
            385,
            0.002236000000,
            0.000064000000,
            0.010549990000,
            386,
            0.002535385000,
            0.000072344210,
            0.011965800000,
            387,
            0.002892603000,
            0.000082212240,
            0.013655870000,
            388,
            0.003300829000,
            0.000093508160,
            0.015588050000,
            389,
            0.003753236000,
            0.000106136100,
            0.017730150000,
            390,
            0.004243000000,
            0.000120000000,
            0.020050010000,
            391,
            0.004762389000,
            0.000134984000,
            0.022511360000,
            392,
            0.005330048000,
            0.000151492000,
            0.025202880000,
            393,
            0.005978712000,
            0.000170208000,
            0.028279720000,
            394,
            0.006741117000,
            0.000191816000,
            0.031897040000,
            395,
            0.007650000000,
            0.000217000000,
            0.036210000000,
            396,
            0.008751373000,
            0.000246906700,
            0.041437710000,
            397,
            0.010028880000,
            0.000281240000,
            0.047503720000,
            398,
            0.011421700000,
            0.000318520000,
            0.054119880000,
            399,
            0.012869010000,
            0.000357266700,
            0.060998030000,
            400,
            0.014310000000,
            0.000396000000,
            0.067850010000,
            401,
            0.015704430000,
            0.000433714700,
            0.074486320000,
            402,
            0.017147440000,
            0.000473024000,
            0.081361560000,
            403,
            0.018781220000,
            0.000517876000,
            0.089153640000,
            404,
            0.020748010000,
            0.000572218700,
            0.098540480000,
            405,
            0.023190000000,
            0.000640000000,
            0.110200000000,
            406,
            0.026207360000,
            0.000724560000,
            0.124613300000,
            407,
            0.029782480000,
            0.000825500000,
            0.141701700000,
            408,
            0.033880920000,
            0.000941160000,
            0.161303500000,
            409,
            0.038468240000,
            0.001069880000,
            0.183256800000,
            410,
            0.043510000000,
            0.001210000000,
            0.207400000000,
            411,
            0.048995600000,
            0.001362091000,
            0.233692100000,
            412,
            0.055022600000,
            0.001530752000,
            0.262611400000,
            413,
            0.061718800000,
            0.001720368000,
            0.294774600000,
            414,
            0.069212000000,
            0.001935323000,
            0.330798500000,
            415,
            0.077630000000,
            0.002180000000,
            0.371300000000,
            416,
            0.086958110000,
            0.002454800000,
            0.416209100000,
            417,
            0.097176720000,
            0.002764000000,
            0.465464200000,
            418,
            0.108406300000,
            0.003117800000,
            0.519694800000,
            419,
            0.120767200000,
            0.003526400000,
            0.579530300000,
            420,
            0.134380000000,
            0.004000000000,
            0.645600000000,
            421,
            0.149358200000,
            0.004546240000,
            0.718483800000,
            422,
            0.165395700000,
            0.005159320000,
            0.796713300000,
            423,
            0.181983100000,
            0.005829280000,
            0.877845900000,
            424,
            0.198611000000,
            0.006546160000,
            0.959439000000,
            425,
            0.214770000000,
            0.007300000000,
            1.039050100000,
            426,
            0.230186800000,
            0.008086507000,
            1.115367300000,
            427,
            0.244879700000,
            0.008908720000,
            1.188497100000,
            428,
            0.258777300000,
            0.009767680000,
            1.258123300000,
            429,
            0.271807900000,
            0.010664430000,
            1.323929600000,
            430,
            0.283900000000,
            0.011600000000,
            1.385600000000,
            431,
            0.294943800000,
            0.012573170000,
            1.442635200000,
            432,
            0.304896500000,
            0.013582720000,
            1.494803500000,
            433,
            0.313787300000,
            0.014629680000,
            1.542190300000,
            434,
            0.321645400000,
            0.015715090000,
            1.584880700000,
            435,
            0.328500000000,
            0.016840000000,
            1.622960000000,
            436,
            0.334351300000,
            0.018007360000,
            1.656404800000,
            437,
            0.339210100000,
            0.019214480000,
            1.685295900000,
            438,
            0.343121300000,
            0.020453920000,
            1.709874500000,
            439,
            0.346129600000,
            0.021718240000,
            1.730382100000,
            440,
            0.348280000000,
            0.023000000000,
            1.747060000000,
            441,
            0.349599900000,
            0.024294610000,
            1.760044600000,
            442,
            0.350147400000,
            0.025610240000,
            1.769623300000,
            443,
            0.350013000000,
            0.026958570000,
            1.776263700000,
            444,
            0.349287000000,
            0.028351250000,
            1.780433400000,
            445,
            0.348060000000,
            0.029800000000,
            1.782600000000,
            446,
            0.346373300000,
            0.031310830000,
            1.782968200000,
            447,
            0.344262400000,
            0.032883680000,
            1.781699800000,
            448,
            0.341808800000,
            0.034521120000,
            1.779198200000,
            449,
            0.339094100000,
            0.036225710000,
            1.775867100000,
            450,
            0.336200000000,
            0.038000000000,
            1.772110000000,
            451,
            0.333197700000,
            0.039846670000,
            1.768258900000,
            452,
            0.330041100000,
            0.041768000000,
            1.764039000000,
            453,
            0.326635700000,
            0.043766000000,
            1.758943800000,
            454,
            0.322886800000,
            0.045842670000,
            1.752466300000,
            455,
            0.318700000000,
            0.048000000000,
            1.744100000000,
            456,
            0.314025100000,
            0.050243680000,
            1.733559500000,
            457,
            0.308884000000,
            0.052573040000,
            1.720858100000,
            458,
            0.303290400000,
            0.054980560000,
            1.705936900000,
            459,
            0.297257900000,
            0.057458720000,
            1.688737200000,
            460,
            0.290800000000,
            0.060000000000,
            1.669200000000,
            461,
            0.283970100000,
            0.062601970000,
            1.647528700000,
            462,
            0.276721400000,
            0.065277520000,
            1.623412700000,
            463,
            0.268917800000,
            0.068042080000,
            1.596022300000,
            464,
            0.260422700000,
            0.070911090000,
            1.564528000000,
            465,
            0.251100000000,
            0.073900000000,
            1.528100000000,
            466,
            0.240847500000,
            0.077016000000,
            1.486111400000,
            467,
            0.229851200000,
            0.080266400000,
            1.439521500000,
            468,
            0.218407200000,
            0.083666800000,
            1.389879900000,
            469,
            0.206811500000,
            0.087232800000,
            1.338736200000,
            470,
            0.195360000000,
            0.090980000000,
            1.287640000000,
            471,
            0.184213600000,
            0.094917550000,
            1.237422300000,
            472,
            0.173327300000,
            0.099045840000,
            1.187824300000,
            473,
            0.162688100000,
            0.103367400000,
            1.138761100000,
            474,
            0.152283300000,
            0.107884600000,
            1.090148000000,
            475,
            0.142100000000,
            0.112600000000,
            1.041900000000,
            476,
            0.132178600000,
            0.117532000000,
            0.994197600000,
            477,
            0.122569600000,
            0.122674400000,
            0.947347300000,
            478,
            0.113275200000,
            0.127992800000,
            0.901453100000,
            479,
            0.104297900000,
            0.133452800000,
            0.856619300000,
            480,
            0.095640000000,
            0.139020000000,
            0.812950100000,
            481,
            0.087299550000,
            0.144676400000,
            0.770517300000,
            482,
            0.079308040000,
            0.150469300000,
            0.729444800000,
            483,
            0.071717760000,
            0.156461900000,
            0.689913600000,
            484,
            0.064580990000,
            0.162717700000,
            0.652104900000,
            485,
            0.057950010000,
            0.169300000000,
            0.616200000000,
            486,
            0.051862110000,
            0.176243100000,
            0.582328600000,
            487,
            0.046281520000,
            0.183558100000,
            0.550416200000,
            488,
            0.041150880000,
            0.191273500000,
            0.520337600000,
            489,
            0.036412830000,
            0.199418000000,
            0.491967300000,
            490,
            0.032010000000,
            0.208020000000,
            0.465180000000,
            491,
            0.027917200000,
            0.217119900000,
            0.439924600000,
            492,
            0.024144400000,
            0.226734500000,
            0.416183600000,
            493,
            0.020687000000,
            0.236857100000,
            0.393882200000,
            494,
            0.017540400000,
            0.247481200000,
            0.372945900000,
            495,
            0.014700000000,
            0.258600000000,
            0.353300000000,
            496,
            0.012161790000,
            0.270184900000,
            0.334857800000,
            497,
            0.009919960000,
            0.282293900000,
            0.317552100000,
            498,
            0.007967240000,
            0.295050500000,
            0.301337500000,
            499,
            0.006296346000,
            0.308578000000,
            0.286168600000,
            500,
            0.004900000000,
            0.323000000000,
            0.272000000000,
            501,
            0.003777173000,
            0.338402100000,
            0.258817100000,
            502,
            0.002945320000,
            0.354685800000,
            0.246483800000,
            503,
            0.002424880000,
            0.371698600000,
            0.234771800000,
            504,
            0.002236293000,
            0.389287500000,
            0.223453300000,
            505,
            0.002400000000,
            0.407300000000,
            0.212300000000,
            506,
            0.002925520000,
            0.425629900000,
            0.201169200000,
            507,
            0.003836560000,
            0.444309600000,
            0.190119600000,
            508,
            0.005174840000,
            0.463394400000,
            0.179225400000,
            509,
            0.006982080000,
            0.482939500000,
            0.168560800000,
            510,
            0.009300000000,
            0.503000000000,
            0.158200000000,
            511,
            0.012149490000,
            0.523569300000,
            0.148138300000,
            512,
            0.015535880000,
            0.544512000000,
            0.138375800000,
            513,
            0.019477520000,
            0.565690000000,
            0.128994200000,
            514,
            0.023992770000,
            0.586965300000,
            0.120075100000,
            515,
            0.029100000000,
            0.608200000000,
            0.111700000000,
            516,
            0.034814850000,
            0.629345600000,
            0.103904800000,
            517,
            0.041120160000,
            0.650306800000,
            0.096667480000,
            518,
            0.047985040000,
            0.670875200000,
            0.089982720000,
            519,
            0.055378610000,
            0.690842400000,
            0.083845310000,
            520,
            0.063270000000,
            0.710000000000,
            0.078249990000,
            521,
            0.071635010000,
            0.728185200000,
            0.073208990000,
            522,
            0.080462240000,
            0.745463600000,
            0.068678160000,
            523,
            0.089739960000,
            0.761969400000,
            0.064567840000,
            524,
            0.099456450000,
            0.777836800000,
            0.060788350000,
            525,
            0.109600000000,
            0.793200000000,
            0.057250010000,
            526,
            0.120167400000,
            0.808110400000,
            0.053904350000,
            527,
            0.131114500000,
            0.822496200000,
            0.050746640000,
            528,
            0.142367900000,
            0.836306800000,
            0.047752760000,
            529,
            0.153854200000,
            0.849491600000,
            0.044898590000,
            530,
            0.165500000000,
            0.862000000000,
            0.042160000000,
            531,
            0.177257100000,
            0.873810800000,
            0.039507280000,
            532,
            0.189140000000,
            0.884962400000,
            0.036935640000,
            533,
            0.201169400000,
            0.895493600000,
            0.034458360000,
            534,
            0.213365800000,
            0.905443200000,
            0.032088720000,
            535,
            0.225749900000,
            0.914850100000,
            0.029840000000,
            536,
            0.238320900000,
            0.923734800000,
            0.027711810000,
            537,
            0.251066800000,
            0.932092400000,
            0.025694440000,
            538,
            0.263992200000,
            0.939922600000,
            0.023787160000,
            539,
            0.277101700000,
            0.947225200000,
            0.021989250000,
            540,
            0.290400000000,
            0.954000000000,
            0.020300000000,
            541,
            0.303891200000,
            0.960256100000,
            0.018718050000,
            542,
            0.317572600000,
            0.966007400000,
            0.017240360000,
            543,
            0.331438400000,
            0.971260600000,
            0.015863640000,
            544,
            0.345482800000,
            0.976022500000,
            0.014584610000,
            545,
            0.359700000000,
            0.980300000000,
            0.013400000000,
            546,
            0.374083900000,
            0.984092400000,
            0.012307230000,
            547,
            0.388639600000,
            0.987418200000,
            0.011301880000,
            548,
            0.403378400000,
            0.990312800000,
            0.010377920000,
            549,
            0.418311500000,
            0.992811600000,
            0.009529306000,
            550,
            0.433449900000,
            0.994950100000,
            0.008749999000,
            551,
            0.448795300000,
            0.996710800000,
            0.008035200000,
            552,
            0.464336000000,
            0.998098300000,
            0.007381600000,
            553,
            0.480064000000,
            0.999112000000,
            0.006785400000,
            554,
            0.495971300000,
            0.999748200000,
            0.006242800000,
            555,
            0.512050100000,
            1.000000000000,
            0.005749999000,
            556,
            0.528295900000,
            0.999856700000,
            0.005303600000,
            557,
            0.544691600000,
            0.999304600000,
            0.004899800000,
            558,
            0.561209400000,
            0.998325500000,
            0.004534200000,
            559,
            0.577821500000,
            0.996898700000,
            0.004202400000,
            560,
            0.594500000000,
            0.995000000000,
            0.003900000000,
            561,
            0.611220900000,
            0.992600500000,
            0.003623200000,
            562,
            0.627975800000,
            0.989742600000,
            0.003370600000,
            563,
            0.644760200000,
            0.986444400000,
            0.003141400000,
            564,
            0.661569700000,
            0.982724100000,
            0.002934800000,
            565,
            0.678400000000,
            0.978600000000,
            0.002749999000,
            566,
            0.695239200000,
            0.974083700000,
            0.002585200000,
            567,
            0.712058600000,
            0.969171200000,
            0.002438600000,
            568,
            0.728828400000,
            0.963856800000,
            0.002309400000,
            569,
            0.745518800000,
            0.958134900000,
            0.002196800000,
            570,
            0.762100000000,
            0.952000000000,
            0.002100000000,
            571,
            0.778543200000,
            0.945450400000,
            0.002017733000,
            572,
            0.794825600000,
            0.938499200000,
            0.001948200000,
            573,
            0.810926400000,
            0.931162800000,
            0.001889800000,
            574,
            0.826824800000,
            0.923457600000,
            0.001840933000,
            575,
            0.842500000000,
            0.915400000000,
            0.001800000000,
            576,
            0.857932500000,
            0.907006400000,
            0.001766267000,
            577,
            0.873081600000,
            0.898277200000,
            0.001737800000,
            578,
            0.887894400000,
            0.889204800000,
            0.001711200000,
            579,
            0.902318100000,
            0.879781600000,
            0.001683067000,
            580,
            0.916300000000,
            0.870000000000,
            0.001650001000,
            581,
            0.929799500000,
            0.859861300000,
            0.001610133000,
            582,
            0.942798400000,
            0.849392000000,
            0.001564400000,
            583,
            0.955277600000,
            0.838622000000,
            0.001513600000,
            584,
            0.967217900000,
            0.827581300000,
            0.001458533000,
            585,
            0.978600000000,
            0.816300000000,
            0.001400000000,
            586,
            0.989385600000,
            0.804794700000,
            0.001336667000,
            587,
            0.999548800000,
            0.793082000000,
            0.001270000000,
            588,
            1.009089200000,
            0.781192000000,
            0.001205000000,
            589,
            1.018006400000,
            0.769154700000,
            0.001146667000,
            590,
            1.026300000000,
            0.757000000000,
            0.001100000000,
            591,
            1.033982700000,
            0.744754100000,
            0.001068800000,
            592,
            1.040986000000,
            0.732422400000,
            0.001049400000,
            593,
            1.047188000000,
            0.720003600000,
            0.001035600000,
            594,
            1.052466700000,
            0.707496500000,
            0.001021200000,
            595,
            1.056700000000,
            0.694900000000,
            0.001000000000,
            596,
            1.059794400000,
            0.682219200000,
            0.000968640000,
            597,
            1.061799200000,
            0.669471600000,
            0.000929920000,
            598,
            1.062806800000,
            0.656674400000,
            0.000886880000,
            599,
            1.062909600000,
            0.643844800000,
            0.000842560000,
            600,
            1.062200000000,
            0.631000000000,
            0.000800000000,
            601,
            1.060735200000,
            0.618155500000,
            0.000760960000,
            602,
            1.058443600000,
            0.605314400000,
            0.000723680000,
            603,
            1.055224400000,
            0.592475600000,
            0.000685920000,
            604,
            1.050976800000,
            0.579637900000,
            0.000645440000,
            605,
            1.045600000000,
            0.566800000000,
            0.000600000000,
            606,
            1.039036900000,
            0.553961100000,
            0.000547866700,
            607,
            1.031360800000,
            0.541137200000,
            0.000491600000,
            608,
            1.022666200000,
            0.528352800000,
            0.000435400000,
            609,
            1.013047700000,
            0.515632300000,
            0.000383466700,
            610,
            1.002600000000,
            0.503000000000,
            0.000340000000,
            611,
            0.991367500000,
            0.490468800000,
            0.000307253300,
            612,
            0.979331400000,
            0.478030400000,
            0.000283160000,
            613,
            0.966491600000,
            0.465677600000,
            0.000265440000,
            614,
            0.952847900000,
            0.453403200000,
            0.000251813300,
            615,
            0.938400000000,
            0.441200000000,
            0.000240000000,
            616,
            0.923194000000,
            0.429080000000,
            0.000229546700,
            617,
            0.907244000000,
            0.417036000000,
            0.000220640000,
            618,
            0.890502000000,
            0.405032000000,
            0.000211960000,
            619,
            0.872920000000,
            0.393032000000,
            0.000202186700,
            620,
            0.854449900000,
            0.381000000000,
            0.000190000000,
            621,
            0.835084000000,
            0.368918400000,
            0.000174213300,
            622,
            0.814946000000,
            0.356827200000,
            0.000155640000,
            623,
            0.794186000000,
            0.344776800000,
            0.000135960000,
            624,
            0.772954000000,
            0.332817600000,
            0.000116853300,
            625,
            0.751400000000,
            0.321000000000,
            0.000100000000,
            626,
            0.729583600000,
            0.309338100000,
            0.000086133330,
            627,
            0.707588800000,
            0.297850400000,
            0.000074600000,
            628,
            0.685602200000,
            0.286593600000,
            0.000065000000,
            629,
            0.663810400000,
            0.275624500000,
            0.000056933330,
            630,
            0.642400000000,
            0.265000000000,
            0.000049999990,
            631,
            0.621514900000,
            0.254763200000,
            0.000044160000,
            632,
            0.601113800000,
            0.244889600000,
            0.000039480000,
            633,
            0.581105200000,
            0.235334400000,
            0.000035720000,
            634,
            0.561397700000,
            0.226052800000,
            0.000032640000,
            635,
            0.541900000000,
            0.217000000000,
            0.000030000000,
            636,
            0.522599500000,
            0.208161600000,
            0.000027653330,
            637,
            0.503546400000,
            0.199548800000,
            0.000025560000,
            638,
            0.484743600000,
            0.191155200000,
            0.000023640000,
            639,
            0.466193900000,
            0.182974400000,
            0.000021813330,
            640,
            0.447900000000,
            0.175000000000,
            0.000020000000,
            641,
            0.429861300000,
            0.167223500000,
            0.000018133330,
            642,
            0.412098000000,
            0.159646400000,
            0.000016200000,
            643,
            0.394644000000,
            0.152277600000,
            0.000014200000,
            644,
            0.377533300000,
            0.145125900000,
            0.000012133330,
            645,
            0.360800000000,
            0.138200000000,
            0.000010000000,
            646,
            0.344456300000,
            0.131500300000,
            0.000007733333,
            647,
            0.328516800000,
            0.125024800000,
            0.000005400000,
            648,
            0.313019200000,
            0.118779200000,
            0.000003200000,
            649,
            0.298001100000,
            0.112769100000,
            0.000001333333,
            650,
            0.283500000000,
            0.107000000000,
            0.000000000000,
            651,
            0.269544800000,
            0.101476200000,
            0.000000000000,
            652,
            0.256118400000,
            0.096188640000,
            0.000000000000,
            653,
            0.243189600000,
            0.091122960000,
            0.000000000000,
            654,
            0.230727200000,
            0.086264850000,
            0.000000000000,
            655,
            0.218700000000,
            0.081600000000,
            0.000000000000,
            656,
            0.207097100000,
            0.077120640000,
            0.000000000000,
            657,
            0.195923200000,
            0.072825520000,
            0.000000000000,
            658,
            0.185170800000,
            0.068710080000,
            0.000000000000,
            659,
            0.174832300000,
            0.064769760000,
            0.000000000000,
            660,
            0.164900000000,
            0.061000000000,
            0.000000000000,
            661,
            0.155366700000,
            0.057396210000,
            0.000000000000,
            662,
            0.146230000000,
            0.053955040000,
            0.000000000000,
            663,
            0.137490000000,
            0.050673760000,
            0.000000000000,
            664,
            0.129146700000,
            0.047549650000,
            0.000000000000,
            665,
            0.121200000000,
            0.044580000000,
            0.000000000000,
            666,
            0.113639700000,
            0.041758720000,
            0.000000000000,
            667,
            0.106465000000,
            0.039084960000,
            0.000000000000,
            668,
            0.099690440000,
            0.036563840000,
            0.000000000000,
            669,
            0.093330610000,
            0.034200480000,
            0.000000000000,
            670,
            0.087400000000,
            0.032000000000,
            0.000000000000,
            671,
            0.081900960000,
            0.029962610000,
            0.000000000000,
            672,
            0.076804280000,
            0.028076640000,
            0.000000000000,
            673,
            0.072077120000,
            0.026329360000,
            0.000000000000,
            674,
            0.067686640000,
            0.024708050000,
            0.000000000000,
            675,
            0.063600000000,
            0.023200000000,
            0.000000000000,
            676,
            0.059806850000,
            0.021800770000,
            0.000000000000,
            677,
            0.056282160000,
            0.020501120000,
            0.000000000000,
            678,
            0.052971040000,
            0.019281080000,
            0.000000000000,
            679,
            0.049818610000,
            0.018120690000,
            0.000000000000,
            680,
            0.046770000000,
            0.017000000000,
            0.000000000000,
            681,
            0.043784050000,
            0.015903790000,
            0.000000000000,
            682,
            0.040875360000,
            0.014837180000,
            0.000000000000,
            683,
            0.038072640000,
            0.013810680000,
            0.000000000000,
            684,
            0.035404610000,
            0.012834780000,
            0.000000000000,
            685,
            0.032900000000,
            0.011920000000,
            0.000000000000,
            686,
            0.030564190000,
            0.011068310000,
            0.000000000000,
            687,
            0.028380560000,
            0.010273390000,
            0.000000000000,
            688,
            0.026344840000,
            0.009533311000,
            0.000000000000,
            689,
            0.024452750000,
            0.008846157000,
            0.000000000000,
            690,
            0.022700000000,
            0.008210000000,
            0.000000000000,
            691,
            0.021084290000,
            0.007623781000,
            0.000000000000,
            692,
            0.019599880000,
            0.007085424000,
            0.000000000000,
            693,
            0.018237320000,
            0.006591476000,
            0.000000000000,
            694,
            0.016987170000,
            0.006138485000,
            0.000000000000,
            695,
            0.015840000000,
            0.005723000000,
            0.000000000000,
            696,
            0.014790640000,
            0.005343059000,
            0.000000000000,
            697,
            0.013831320000,
            0.004995796000,
            0.000000000000,
            698,
            0.012948680000,
            0.004676404000,
            0.000000000000,
            699,
            0.012129200000,
            0.004380075000,
            0.000000000000,
            700,
            0.011359160000,
            0.004102000000,
            0.000000000000,
            701,
            0.010629350000,
            0.003838453000,
            0.000000000000,
            702,
            0.009938846000,
            0.003589099000,
            0.000000000000,
            703,
            0.009288422000,
            0.003354219000,
            0.000000000000,
            704,
            0.008678854000,
            0.003134093000,
            0.000000000000,
            705,
            0.008110916000,
            0.002929000000,
            0.000000000000,
            706,
            0.007582388000,
            0.002738139000,
            0.000000000000,
            707,
            0.007088746000,
            0.002559876000,
            0.000000000000,
            708,
            0.006627313000,
            0.002393244000,
            0.000000000000,
            709,
            0.006195408000,
            0.002237275000,
            0.000000000000,
            710,
            0.005790346000,
            0.002091000000,
            0.000000000000,
            711,
            0.005409826000,
            0.001953587000,
            0.000000000000,
            712,
            0.005052583000,
            0.001824580000,
            0.000000000000,
            713,
            0.004717512000,
            0.001703580000,
            0.000000000000,
            714,
            0.004403507000,
            0.001590187000,
            0.000000000000,
            715,
            0.004109457000,
            0.001484000000,
            0.000000000000,
            716,
            0.003833913000,
            0.001384496000,
            0.000000000000,
            717,
            0.003575748000,
            0.001291268000,
            0.000000000000,
            718,
            0.003334342000,
            0.001204092000,
            0.000000000000,
            719,
            0.003109075000,
            0.001122744000,
            0.000000000000,
            720,
            0.002899327000,
            0.001047000000,
            0.000000000000,
            721,
            0.002704348000,
            0.000976589600,
            0.000000000000,
            722,
            0.002523020000,
            0.000911108800,
            0.000000000000,
            723,
            0.002354168000,
            0.000850133200,
            0.000000000000,
            724,
            0.002196616000,
            0.000793238400,
            0.000000000000,
            725,
            0.002049190000,
            0.000740000000,
            0.000000000000,
            726,
            0.001910960000,
            0.000690082700,
            0.000000000000,
            727,
            0.001781438000,
            0.000643310000,
            0.000000000000,
            728,
            0.001660110000,
            0.000599496000,
            0.000000000000,
            729,
            0.001546459000,
            0.000558454700,
            0.000000000000,
            730,
            0.001439971000,
            0.000520000000,
            0.000000000000,
            731,
            0.001340042000,
            0.000483913600,
            0.000000000000,
            732,
            0.001246275000,
            0.000450052800,
            0.000000000000,
            733,
            0.001158471000,
            0.000418345200,
            0.000000000000,
            734,
            0.001076430000,
            0.000388718400,
            0.000000000000,
            735,
            0.000999949300,
            0.000361100000,
            0.000000000000,
            736,
            0.000928735800,
            0.000335383500,
            0.000000000000,
            737,
            0.000862433200,
            0.000311440400,
            0.000000000000,
            738,
            0.000800750300,
            0.000289165600,
            0.000000000000,
            739,
            0.000743396000,
            0.000268453900,
            0.000000000000,
            740,
            0.000690078600,
            0.000249200000,
            0.000000000000,
            741,
            0.000640515600,
            0.000231301900,
            0.000000000000,
            742,
            0.000594502100,
            0.000214685600,
            0.000000000000,
            743,
            0.000551864600,
            0.000199288400,
            0.000000000000,
            744,
            0.000512429000,
            0.000185047500,
            0.000000000000,
            745,
            0.000476021300,
            0.000171900000,
            0.000000000000,
            746,
            0.000442453600,
            0.000159778100,
            0.000000000000,
            747,
            0.000411511700,
            0.000148604400,
            0.000000000000,
            748,
            0.000382981400,
            0.000138301600,
            0.000000000000,
            749,
            0.000356649100,
            0.000128792500,
            0.000000000000,
            750,
            0.000332301100,
            0.000120000000,
            0.000000000000,
            751,
            0.000309758600,
            0.000111859500,
            0.000000000000,
            752,
            0.000288887100,
            0.000104322400,
            0.000000000000,
            753,
            0.000269539400,
            0.000097335600,
            0.000000000000,
            754,
            0.000251568200,
            0.000090845870,
            0.000000000000,
            755,
            0.000234826100,
            0.000084800000,
            0.000000000000,
            756,
            0.000219171000,
            0.000079146670,
            0.000000000000,
            757,
            0.000204525800,
            0.000073858000,
            0.000000000000,
            758,
            0.000190840500,
            0.000068916000,
            0.000000000000,
            759,
            0.000178065400,
            0.000064302670,
            0.000000000000,
            760,
            0.000166150500,
            0.000060000000,
            0.000000000000,
            761,
            0.000155023600,
            0.000055981870,
            0.000000000000,
            762,
            0.000144621900,
            0.000052225600,
            0.000000000000,
            763,
            0.000134909800,
            0.000048718400,
            0.000000000000,
            764,
            0.000125852000,
            0.000045447470,
            0.000000000000,
            765,
            0.000117413000,
            0.000042400000,
            0.000000000000,
            766,
            0.000109551500,
            0.000039561040,
            0.000000000000,
            767,
            0.000102224500,
            0.000036915120,
            0.000000000000,
            768,
            0.000095394450,
            0.000034448680,
            0.000000000000,
            769,
            0.000089023900,
            0.000032148160,
            0.000000000000,
            770,
            0.000083075270,
            0.000030000000,
            0.000000000000,
            771,
            0.000077512690,
            0.000027991250,
            0.000000000000,
            772,
            0.000072313040,
            0.000026113560,
            0.000000000000,
            773,
            0.000067457780,
            0.000024360240,
            0.000000000000,
            774,
            0.000062928440,
            0.000022724610,
            0.000000000000,
            775,
            0.000058706520,
            0.000021200000,
            0.000000000000,
            776,
            0.000054770280,
            0.000019778550,
            0.000000000000,
            777,
            0.000051099180,
            0.000018452850,
            0.000000000000,
            778,
            0.000047676540,
            0.000017216870,
            0.000000000000,
            779,
            0.000044485670,
            0.000016064590,
            0.000000000000,
            780,
            0.000041509940,
            0.000014990000,
            0.000000000000,
        ]
    )

    cmf = cmf.reshape(401, 4).astype(np.float64).T
    cmf = cmf[:, ::binwidth]
    if asdf:
        cmf = pd.DataFrame(data=cmf.T, columns=colnames)
        cmf.set_index("Wavelength", inplace=True)
        cmf.index = pd.Int64Index(cmf.index)
    return cmf


def get_CIES026(
    binwidth: Optional[int] = 1, fillna: Optional[bool] = True
) -> pd.DataFrame:
    """Get the CIE026 spectral sensitivities.

    Parameters
    ----------
    binwidth : int, optional
        Width of the wavelength bins in nanometers (must be `1` or `5`). The
        default is `1`.
    fillna : bool, optional
        Whether to replace nan values with zero.

    Returns
    -------
    sss : numpy.ndarray or pandas.DataFrame
        CIES026 spectral sensitivities for s, m, l, rods, and melanopsin.

    """
    if binwidth not in [1, 2, 5]:
        raise ValueError("Must specify 1 or 5 for binwidth")

    colnames = ["Wavelength", "S", "M", "L", "R", "I"]

    sss = np.array(
        [
            380,
            np.nan,
            np.nan,
            np.nan,
            0.000589,
            0.00091816,
            381,
            np.nan,
            np.nan,
            np.nan,
            0.000665,
            0.0010456,
            382,
            np.nan,
            np.nan,
            np.nan,
            0.000752,
            0.0011786,
            383,
            np.nan,
            np.nan,
            np.nan,
            0.000854,
            0.0013228,
            384,
            np.nan,
            np.nan,
            np.nan,
            0.000972,
            0.0014838,
            385,
            np.nan,
            np.nan,
            np.nan,
            0.001108,
            0.0016672,
            386,
            np.nan,
            np.nan,
            np.nan,
            0.001268,
            0.001881,
            387,
            np.nan,
            np.nan,
            np.nan,
            0.001453,
            0.0021299,
            388,
            np.nan,
            np.nan,
            np.nan,
            0.001668,
            0.0024146,
            389,
            np.nan,
            np.nan,
            np.nan,
            0.001918,
            0.0027358,
            390,
            0.0061427,
            0.00035823,
            0.00040762,
            0.002209,
            0.0030944,
            391,
            0.0074428,
            0.00043866,
            0.00049707,
            0.002547,
            0.0035071,
            392,
            0.0090166,
            0.00053623,
            0.00060471,
            0.002939,
            0.0039908,
            393,
            0.010917,
            0.00065406,
            0.00073364,
            0.003394,
            0.0045468,
            394,
            0.013205,
            0.00079565,
            0.00088725,
            0.003921,
            0.0051763,
            395,
            0.015952,
            0.00096483,
            0.0010692,
            0.00453,
            0.0058804,
            396,
            0.019235,
            0.0011657,
            0.0012834,
            0.00524,
            0.0066933,
            397,
            0.023144,
            0.0014026,
            0.0015338,
            0.00605,
            0.007651,
            398,
            0.027775,
            0.0016799,
            0.0018244,
            0.00698,
            0.0087569,
            399,
            0.033234,
            0.0020018,
            0.002159,
            0.00806,
            0.010015,
            400,
            0.039631,
            0.0023721,
            0.0025407,
            0.00929,
            0.011428,
            401,
            0.04708,
            0.0027943,
            0.0029728,
            0.0107,
            0.013077,
            402,
            0.055701,
            0.0032737,
            0.0034599,
            0.01231,
            0.01504,
            403,
            0.065614,
            0.0038166,
            0.0040079,
            0.01413,
            0.017317,
            404,
            0.076932,
            0.0044302,
            0.0046237,
            0.01619,
            0.019907,
            405,
            0.089761,
            0.0051232,
            0.0053155,
            0.01852,
            0.022811,
            406,
            0.10419,
            0.0059046,
            0.0060914,
            0.02113,
            0.026319,
            407,
            0.12027,
            0.0067801,
            0.0069529,
            0.02405,
            0.030596,
            408,
            0.13804,
            0.0077526,
            0.0078963,
            0.0273,
            0.035454,
            409,
            0.15749,
            0.0088229,
            0.008913,
            0.03089,
            0.040703,
            410,
            0.17853,
            0.0099884,
            0.0099884,
            0.03484,
            0.046155,
            411,
            0.20108,
            0.011245,
            0.011105,
            0.03916,
            0.051782,
            412,
            0.22509,
            0.012595,
            0.012261,
            0.0439,
            0.05778,
            413,
            0.25057,
            0.014042,
            0.013458,
            0.049,
            0.064297,
            414,
            0.27751,
            0.015594,
            0.014704,
            0.0545,
            0.07148,
            415,
            0.30594,
            0.01726,
            0.016013,
            0.0604,
            0.079477,
            416,
            0.33586,
            0.019047,
            0.017396,
            0.0668,
            0.089181,
            417,
            0.36698,
            0.020955,
            0.018845,
            0.0736,
            0.10076,
            418,
            0.39888,
            0.022976,
            0.020344,
            0.0808,
            0.11326,
            419,
            0.431,
            0.025102,
            0.02187,
            0.0885,
            0.12573,
            420,
            0.46269,
            0.027316,
            0.023396,
            0.0966,
            0.13724,
            421,
            0.49336,
            0.029606,
            0.024896,
            0.1052,
            0.14745,
            422,
            0.52301,
            0.031975,
            0.026376,
            0.1141,
            0.15701,
            423,
            0.55194,
            0.034433,
            0.027854,
            0.1235,
            0.16646,
            424,
            0.5806,
            0.036998,
            0.029355,
            0.1334,
            0.17632,
            425,
            0.60957,
            0.039693,
            0.03091,
            0.1436,
            0.1871,
            426,
            0.63936,
            0.04254,
            0.03255,
            0.1541,
            0.19921,
            427,
            0.66965,
            0.045547,
            0.034271,
            0.1651,
            0.21241,
            428,
            0.69983,
            0.048716,
            0.036062,
            0.1764,
            0.22623,
            429,
            0.72918,
            0.052047,
            0.037905,
            0.1879,
            0.2402,
            430,
            0.75689,
            0.055538,
            0.039781,
            0.1998,
            0.25387,
            431,
            0.78229,
            0.059188,
            0.04167,
            0.2119,
            0.26702,
            432,
            0.80567,
            0.062982,
            0.043573,
            0.2243,
            0.27998,
            433,
            0.8276,
            0.066903,
            0.045493,
            0.2369,
            0.29303,
            434,
            0.84878,
            0.070929,
            0.047439,
            0.2496,
            0.3065,
            435,
            0.86998,
            0.07503,
            0.049417,
            0.2625,
            0.32068,
            436,
            0.89176,
            0.079177,
            0.051434,
            0.2755,
            0.33602,
            437,
            0.91344,
            0.083346,
            0.053474,
            0.2886,
            0.35236,
            438,
            0.93398,
            0.087516,
            0.05551,
            0.3017,
            0.36913,
            439,
            0.95222,
            0.091662,
            0.057517,
            0.3149,
            0.38573,
            440,
            0.96696,
            0.095761,
            0.059462,
            0.3281,
            0.40159,
            441,
            0.97734,
            0.099798,
            0.061324,
            0.3412,
            0.41647,
            442,
            0.98403,
            0.1038,
            0.063129,
            0.3543,
            0.4308,
            443,
            0.98814,
            0.10783,
            0.064919,
            0.3673,
            0.44492,
            444,
            0.99085,
            0.11195,
            0.066742,
            0.3803,
            0.4592,
            445,
            0.99334,
            0.11622,
            0.068654,
            0.3931,
            0.474,
            446,
            0.99637,
            0.12071,
            0.070696,
            0.406,
            0.48952,
            447,
            0.99904,
            0.12536,
            0.072851,
            0.418,
            0.50552,
            448,
            0.99998,
            0.13011,
            0.075078,
            0.431,
            0.52174,
            449,
            0.99784,
            0.13486,
            0.077332,
            0.443,
            0.5379,
            450,
            0.99133,
            0.13949,
            0.079565,
            0.455,
            0.55371,
            451,
            0.97966,
            0.14394,
            0.081737,
            0.467,
            0.5691,
            452,
            0.96391,
            0.14828,
            0.083883,
            0.479,
            0.58424,
            453,
            0.94557,
            0.15264,
            0.08606,
            0.49,
            0.59928,
            454,
            0.92608,
            0.15716,
            0.088332,
            0.502,
            0.61437,
            455,
            0.90673,
            0.16201,
            0.09077,
            0.513,
            0.62965,
            456,
            0.88851,
            0.16733,
            0.09344,
            0.524,
            0.64519,
            457,
            0.87135,
            0.17314,
            0.096358,
            0.535,
            0.66089,
            458,
            0.855,
            0.17942,
            0.09953,
            0.546,
            0.67666,
            459,
            0.8392,
            0.18612,
            0.10296,
            0.557,
            0.69241,
            460,
            0.82373,
            0.1932,
            0.10666,
            0.567,
            0.70805,
            461,
            0.80831,
            0.20062,
            0.11063,
            0.578,
            0.72359,
            462,
            0.79243,
            0.20832,
            0.11483,
            0.588,
            0.73911,
            463,
            0.77557,
            0.21621,
            0.11922,
            0.599,
            0.75456,
            464,
            0.75724,
            0.22423,
            0.12374,
            0.61,
            0.76994,
            465,
            0.73704,
            0.23228,
            0.12834,
            0.62,
            0.78522,
            466,
            0.71473,
            0.24026,
            0.13295,
            0.631,
            0.80068,
            467,
            0.69056,
            0.24816,
            0.13757,
            0.642,
            0.81635,
            468,
            0.66489,
            0.25599,
            0.14222,
            0.653,
            0.8318,
            469,
            0.63808,
            0.26374,
            0.14691,
            0.664,
            0.84659,
            470,
            0.61046,
            0.27144,
            0.15165,
            0.676,
            0.86029,
            471,
            0.58235,
            0.2791,
            0.15648,
            0.687,
            0.87292,
            472,
            0.55407,
            0.28675,
            0.1614,
            0.699,
            0.88487,
            473,
            0.5259,
            0.29448,
            0.16646,
            0.71,
            0.89624,
            474,
            0.49811,
            0.30232,
            0.17169,
            0.722,
            0.90716,
            475,
            0.47089,
            0.31037,
            0.17712,
            0.734,
            0.91773,
            476,
            0.44445,
            0.31869,
            0.18278,
            0.745,
            0.92834,
            477,
            0.41899,
            0.32731,
            0.18868,
            0.757,
            0.93895,
            478,
            0.3947,
            0.33623,
            0.19484,
            0.769,
            0.94903,
            479,
            0.37171,
            0.34548,
            0.20126,
            0.781,
            0.95809,
            480,
            0.35011,
            0.35507,
            0.20794,
            0.793,
            0.96561,
            481,
            0.3299,
            0.36499,
            0.21488,
            0.805,
            0.97198,
            482,
            0.31086,
            0.37514,
            0.22202,
            0.817,
            0.97783,
            483,
            0.29274,
            0.38541,
            0.22932,
            0.828,
            0.98301,
            484,
            0.27534,
            0.39565,
            0.23668,
            0.84,
            0.98733,
            485,
            0.2585,
            0.40569,
            0.24405,
            0.851,
            0.99062,
            486,
            0.24216,
            0.41543,
            0.25135,
            0.862,
            0.99334,
            487,
            0.2265,
            0.42506,
            0.2587,
            0.873,
            0.99589,
            488,
            0.21173,
            0.43485,
            0.26625,
            0.884,
            0.99801,
            489,
            0.19796,
            0.4451,
            0.2742,
            0.894,
            0.99946,
            490,
            0.1853,
            0.45614,
            0.28275,
            0.904,
            1,
            491,
            0.17375,
            0.46824,
            0.29207,
            0.914,
            0.99956,
            492,
            0.16315,
            0.48125,
            0.30209,
            0.923,
            0.99836,
            493,
            0.15331,
            0.49493,
            0.31267,
            0.932,
            0.99659,
            494,
            0.14409,
            0.50895,
            0.32364,
            0.941,
            0.99442,
            495,
            0.13535,
            0.52297,
            0.33479,
            0.949,
            0.99202,
            496,
            0.12701,
            0.5367,
            0.34594,
            0.957,
            0.98879,
            497,
            0.11902,
            0.55019,
            0.35713,
            0.964,
            0.98422,
            498,
            0.11133,
            0.56362,
            0.36842,
            0.97,
            0.97866,
            499,
            0.10393,
            0.57715,
            0.37991,
            0.976,
            0.97245,
            500,
            0.096799,
            0.591,
            0.39171,
            0.982,
            0.96595,
            501,
            0.089917,
            0.60535,
            0.40391,
            0.986,
            0.95884,
            502,
            0.083288,
            0.62016,
            0.4165,
            0.99,
            0.95072,
            503,
            0.076916,
            0.63534,
            0.42945,
            0.994,
            0.94178,
            504,
            0.070805,
            0.6508,
            0.44272,
            0.997,
            0.93224,
            505,
            0.064961,
            0.6664,
            0.45625,
            0.998,
            0.9223,
            506,
            0.059405,
            0.68205,
            0.47,
            1,
            0.91183,
            507,
            0.054208,
            0.69767,
            0.48393,
            1,
            0.9006,
            508,
            0.049428,
            0.71319,
            0.49801,
            1,
            0.88866,
            509,
            0.045099,
            0.72853,
            0.51223,
            0.998,
            0.87607,
            510,
            0.041234,
            0.74361,
            0.52654,
            0.997,
            0.86289,
            511,
            0.037814,
            0.7584,
            0.54092,
            0.994,
            0.8488,
            512,
            0.034763,
            0.77297,
            0.55541,
            0.99,
            0.83368,
            513,
            0.032003,
            0.78746,
            0.57003,
            0.986,
            0.81783,
            514,
            0.029475,
            0.80202,
            0.58483,
            0.981,
            0.80158,
            515,
            0.02713,
            0.81681,
            0.59987,
            0.975,
            0.78523,
            516,
            0.024938,
            0.83192,
            0.61516,
            0.968,
            0.76872,
            517,
            0.022893,
            0.8471,
            0.63057,
            0.961,
            0.75181,
            518,
            0.020996,
            0.86197,
            0.64588,
            0.953,
            0.73459,
            519,
            0.019243,
            0.87615,
            0.66088,
            0.944,
            0.71717,
            520,
            0.01763,
            0.88921,
            0.67531,
            0.935,
            0.69963,
            521,
            0.01615,
            0.90081,
            0.68898,
            0.925,
            0.68189,
            522,
            0.014791,
            0.91101,
            0.70189,
            0.915,
            0.66388,
            523,
            0.013541,
            0.91997,
            0.71414,
            0.904,
            0.64572,
            524,
            0.012388,
            0.92789,
            0.72584,
            0.892,
            0.62753,
            525,
            0.011325,
            0.93498,
            0.73711,
            0.88,
            0.60942,
            526,
            0.010344,
            0.94141,
            0.74805,
            0.867,
            0.59134,
            527,
            0.0094409,
            0.94728,
            0.75868,
            0.854,
            0.57321,
            528,
            0.0086137,
            0.95262,
            0.76903,
            0.84,
            0.5551,
            529,
            0.0078583,
            0.9575,
            0.7791,
            0.826,
            0.53711,
            530,
            0.0071709,
            0.96196,
            0.7889,
            0.811,
            0.51931,
            531,
            0.0065465,
            0.96608,
            0.79847,
            0.796,
            0.50165,
            532,
            0.0059778,
            0.96997,
            0.80795,
            0.781,
            0.48407,
            533,
            0.0054579,
            0.97374,
            0.81748,
            0.765,
            0.46664,
            534,
            0.0049812,
            0.97754,
            0.82724,
            0.749,
            0.44944,
            535,
            0.0045429,
            0.98148,
            0.8374,
            0.733,
            0.43253,
            536,
            0.0041391,
            0.98563,
            0.84808,
            0.717,
            0.41586,
            537,
            0.0037679,
            0.98975,
            0.85906,
            0.7,
            0.39937,
            538,
            0.0034278,
            0.99355,
            0.87007,
            0.683,
            0.38314,
            539,
            0.003117,
            0.99671,
            0.88078,
            0.667,
            0.36722,
            540,
            0.0028335,
            0.99893,
            0.89087,
            0.65,
            0.35171,
            541,
            0.0025756,
            0.99994,
            0.90006,
            0.633,
            0.33654,
            542,
            0.0023408,
            0.99969,
            0.90825,
            0.616,
            0.32165,
            543,
            0.0021272,
            0.99818,
            0.91543,
            0.599,
            0.30708,
            544,
            0.0019328,
            0.9954,
            0.92158,
            0.581,
            0.2929,
            545,
            0.0017557,
            0.99138,
            0.92666,
            0.564,
            0.27914,
            546,
            0.0015945,
            0.9862,
            0.93074,
            0.548,
            0.26574,
            547,
            0.0014478,
            0.98023,
            0.93416,
            0.531,
            0.25265,
            548,
            0.0013143,
            0.97391,
            0.93732,
            0.514,
            0.23992,
            549,
            0.0011928,
            0.96765,
            0.94063,
            0.497,
            0.22759,
            550,
            0.0010823,
            0.96188,
            0.94453,
            0.481,
            0.21572,
            551,
            0.00098182,
            0.95682,
            0.94929,
            0.465,
            0.20424,
            552,
            0.00089053,
            0.95215,
            0.95468,
            0.448,
            0.19307,
            553,
            0.00080769,
            0.9474,
            0.96031,
            0.433,
            0.18229,
            554,
            0.00073257,
            0.9421,
            0.96579,
            0.417,
            0.17193,
            555,
            0.00066451,
            0.93583,
            0.9707,
            0.402,
            0.16206,
            556,
            0.00060289,
            0.92827,
            0.97476,
            0.3864,
            0.1526,
            557,
            0.00054706,
            0.91967,
            0.97806,
            0.3715,
            0.14349,
            558,
            0.00049646,
            0.91036,
            0.98082,
            0.3569,
            0.13475,
            559,
            0.00045057,
            0.90068,
            0.98327,
            0.3427,
            0.12642,
            560,
            0.00040893,
            0.89095,
            0.98564,
            0.3288,
            0.11853,
            561,
            0.00037114,
            0.88139,
            0.98808,
            0.3151,
            0.11101,
            562,
            0.00033684,
            0.87183,
            0.99056,
            0.3018,
            0.10379,
            563,
            0.00030572,
            0.86206,
            0.99294,
            0.2888,
            0.096921,
            564,
            0.0002775,
            0.85184,
            0.99512,
            0.2762,
            0.090426,
            565,
            0.00025192,
            0.84097,
            0.99698,
            0.2639,
            0.084346,
            566,
            0.00022872,
            0.8293,
            0.99841,
            0.2519,
            0.07862,
            567,
            0.0002077,
            0.81691,
            0.99939,
            0.2403,
            0.073175,
            568,
            0.00018864,
            0.80391,
            0.99991,
            0.2291,
            0.068029,
            569,
            0.00017136,
            0.79041,
            0.99996,
            0.2182,
            0.063198,
            570,
            0.00015569,
            0.77653,
            0.99954,
            0.2076,
            0.058701,
            571,
            0.00014148,
            0.76231,
            0.99862,
            0.1974,
            0.054483,
            572,
            0.0001286,
            0.74767,
            0.99705,
            0.1876,
            0.050489,
            573,
            0.00011691,
            0.73248,
            0.9947,
            0.1782,
            0.046734,
            574,
            0.00010632,
            0.71662,
            0.99142,
            0.169,
            0.043236,
            575,
            9.67e-05,
            0.70001,
            0.98706,
            0.1602,
            0.040009,
            576,
            8.80e-05,
            0.68265,
            0.9816,
            0.1517,
            0.03701,
            577,
            8.01e-05,
            0.66482,
            0.97545,
            0.1436,
            0.03419,
            578,
            7.29e-05,
            0.64686,
            0.96912,
            0.1358,
            0.031556,
            579,
            6.64e-05,
            0.62907,
            0.96309,
            0.1284,
            0.029115,
            580,
            6.05e-05,
            0.61173,
            0.95784,
            0.1212,
            0.026875,
            581,
            5.51e-05,
            0.595,
            0.95366,
            0.1143,
            0.024801,
            582,
            5.02e-05,
            0.57878,
            0.95024,
            0.1078,
            0.02286,
            583,
            4.58e-05,
            0.56293,
            0.94709,
            0.1015,
            0.021053,
            584,
            4.18e-05,
            0.54732,
            0.94375,
            0.0956,
            0.019386,
            585,
            3.81e-05,
            0.53182,
            0.93978,
            0.0899,
            0.017862,
            586,
            3.48e-05,
            0.51635,
            0.93483,
            0.0845,
            0.016458,
            587,
            3.18e-05,
            0.50087,
            0.92892,
            0.0793,
            0.015147,
            588,
            2.90e-05,
            0.48535,
            0.92218,
            0.0745,
            0.013931,
            589,
            2.65e-05,
            0.46978,
            0.91473,
            0.0699,
            0.012812,
            590,
            2.43e-05,
            0.45414,
            0.90669,
            0.0655,
            0.01179,
            591,
            2.22e-05,
            0.43845,
            0.89817,
            0.0613,
            0.010849,
            592,
            2.03e-05,
            0.42278,
            0.88919,
            0.0574,
            0.0099711,
            593,
            1.86e-05,
            0.40719,
            0.87976,
            0.0537,
            0.0091585,
            594,
            1.70e-05,
            0.39175,
            0.86989,
            0.0502,
            0.0084124,
            595,
            1.56e-05,
            0.37653,
            0.8596,
            0.0469,
            0.0077343,
            596,
            1.43e-05,
            0.36156,
            0.84891,
            0.0438,
            0.0071125,
            597,
            1.31e-05,
            0.34686,
            0.83786,
            0.0409,
            0.0065348,
            598,
            1.20e-05,
            0.33242,
            0.82652,
            0.03816,
            0.0060011,
            599,
            1.10e-05,
            0.31826,
            0.81494,
            0.03558,
            0.0055117,
            600,
            1.01e-05,
            0.30438,
            0.80317,
            0.03315,
            0.0050669,
            601,
            9.31e-06,
            0.29078,
            0.79125,
            0.03087,
            0.0046587,
            602,
            8.56e-06,
            0.27751,
            0.77912,
            0.02874,
            0.0042795,
            603,
            7.87e-06,
            0.26458,
            0.76669,
            0.02674,
            0.0039294,
            604,
            7.24e-06,
            0.25201,
            0.7539,
            0.02487,
            0.0036087,
            605,
            6.67e-06,
            0.23984,
            0.74068,
            0.02312,
            0.0033177,
            606,
            6.14e-06,
            0.22806,
            0.72699,
            0.02147,
            0.0030511,
            607,
            5.66e-06,
            0.2167,
            0.71291,
            0.01994,
            0.0028037,
            608,
            5.22e-06,
            0.20575,
            0.69849,
            0.01851,
            0.0025756,
            609,
            4.81e-06,
            0.19522,
            0.68383,
            0.01718,
            0.0023667,
            610,
            4.44e-06,
            0.1851,
            0.66899,
            0.01593,
            0.002177,
            611,
            4.10e-06,
            0.1754,
            0.65404,
            0.01477,
            0.0020032,
            612,
            3.79e-06,
            0.16609,
            0.63898,
            0.01369,
            0.0018419,
            613,
            3.50e-06,
            0.15717,
            0.62382,
            0.01269,
            0.0016932,
            614,
            3.24e-06,
            0.14862,
            0.60858,
            0.01175,
            0.0015569,
            615,
            2.99e-06,
            0.14043,
            0.59325,
            0.01088,
            0.0014331,
            616,
            np.nan,
            0.13259,
            0.57786,
            0.01007,
            0.0013197,
            617,
            np.nan,
            0.12509,
            0.56249,
            0.00932,
            0.0012145,
            618,
            np.nan,
            0.11793,
            0.54725,
            0.00862,
            0.0011174,
            619,
            np.nan,
            0.11109,
            0.53221,
            0.00797,
            0.0010284,
            620,
            np.nan,
            0.10457,
            0.51745,
            0.00737,
            0.00094731,
            621,
            np.nan,
            0.098366,
            0.50299,
            0.00682,
            0.00087281,
            622,
            np.nan,
            0.092468,
            0.48869,
            0.0063,
            0.00080358,
            623,
            np.nan,
            0.086876,
            0.47438,
            0.00582,
            0.00073962,
            624,
            np.nan,
            0.081583,
            0.4599,
            0.00538,
            0.00068097,
            625,
            np.nan,
            0.076584,
            0.44512,
            0.00497,
            0.00062765,
            626,
            np.nan,
            0.071868,
            0.43001,
            0.00459,
            0.00057875,
            627,
            np.nan,
            0.067419,
            0.41469,
            0.00424,
            0.00053336,
            628,
            np.nan,
            0.063218,
            0.39934,
            0.003913,
            0.00049144,
            629,
            np.nan,
            0.059249,
            0.38412,
            0.003613,
            0.00045298,
            630,
            np.nan,
            0.055499,
            0.36917,
            0.003335,
            0.00041796,
            631,
            np.nan,
            0.051955,
            0.35458,
            0.003079,
            0.00038579,
            632,
            np.nan,
            0.04861,
            0.34039,
            0.002842,
            0.00035591,
            633,
            np.nan,
            0.045459,
            0.32661,
            0.002623,
            0.00032829,
            634,
            np.nan,
            0.042494,
            0.31325,
            0.002421,
            0.00030293,
            635,
            np.nan,
            0.03971,
            0.30032,
            0.002235,
            0.0002798,
            636,
            np.nan,
            0.037095,
            0.28782,
            0.002062,
            0.00025854,
            637,
            np.nan,
            0.034635,
            0.27576,
            0.001903,
            0.00023879,
            638,
            np.nan,
            0.032313,
            0.26416,
            0.001757,
            0.00022051,
            639,
            np.nan,
            0.030115,
            0.25301,
            0.001621,
            0.0002037,
            640,
            np.nan,
            0.028031,
            0.24232,
            0.001497,
            0.00018834,
            641,
            np.nan,
            0.026056,
            0.23206,
            0.001382,
            0.00017419,
            642,
            np.nan,
            0.024201,
            0.22216,
            0.001276,
            0.00016102,
            643,
            np.nan,
            0.022476,
            0.21252,
            0.001178,
            0.00014882,
            644,
            np.nan,
            0.020887,
            0.20306,
            0.001088,
            0.00013759,
            645,
            np.nan,
            0.019437,
            0.19373,
            0.001005,
            0.00012734,
            646,
            np.nan,
            0.01812,
            0.18449,
            0.000928,
            0.00011789,
            647,
            np.nan,
            0.016915,
            0.1754,
            0.000857,
            0.0001091,
            648,
            np.nan,
            0.015799,
            0.16651,
            0.000792,
            0.00010095,
            649,
            np.nan,
            0.014754,
            0.15787,
            0.000732,
            9.34e-05,
            650,
            np.nan,
            0.013766,
            0.14951,
            0.000677,
            8.66e-05,
            651,
            np.nan,
            0.012825,
            0.14147,
            0.000626,
            8.02e-05,
            652,
            np.nan,
            0.01193,
            0.13376,
            0.000579,
            7.43e-05,
            653,
            np.nan,
            0.011085,
            0.12638,
            0.000536,
            6.89e-05,
            654,
            np.nan,
            0.010289,
            0.11934,
            0.000496,
            6.38e-05,
            655,
            np.nan,
            0.0095432,
            0.11264,
            0.000459,
            5.92e-05,
            656,
            np.nan,
            0.0088461,
            0.10626,
            0.000425,
            5.49e-05,
            657,
            np.nan,
            0.008196,
            0.10021,
            0.0003935,
            5.09e-05,
            658,
            np.nan,
            0.0075906,
            0.094456,
            0.0003645,
            4.72e-05,
            659,
            np.nan,
            0.0070275,
            0.088993,
            0.0003377,
            4.38e-05,
            660,
            np.nan,
            0.0065045,
            0.083808,
            0.0003129,
            4.07e-05,
            661,
            np.nan,
            0.0060195,
            0.078886,
            0.0002901,
            3.78e-05,
            662,
            np.nan,
            0.0055709,
            0.074219,
            0.0002689,
            3.51e-05,
            663,
            np.nan,
            0.0051573,
            0.069795,
            0.0002493,
            3.26e-05,
            664,
            np.nan,
            0.0047769,
            0.065605,
            0.0002313,
            3.03e-05,
            665,
            np.nan,
            0.0044279,
            0.061638,
            0.0002146,
            2.81e-05,
            666,
            np.nan,
            0.0041083,
            0.057886,
            0.0001991,
            2.62e-05,
            667,
            np.nan,
            0.0038147,
            0.054337,
            0.0001848,
            2.43e-05,
            668,
            np.nan,
            0.0035439,
            0.050981,
            0.0001716,
            2.26e-05,
            669,
            np.nan,
            0.0032933,
            0.04781,
            0.0001593,
            2.10e-05,
            670,
            np.nan,
            0.0030605,
            0.044813,
            0.000148,
            1.96e-05,
            671,
            np.nan,
            0.0028437,
            0.041983,
            0.0001375,
            1.82e-05,
            672,
            np.nan,
            0.0026418,
            0.039311,
            0.0001277,
            1.69e-05,
            673,
            np.nan,
            0.0024537,
            0.036789,
            0.0001187,
            1.57e-05,
            674,
            np.nan,
            0.0022787,
            0.03441,
            0.0001104,
            1.47e-05,
            675,
            np.nan,
            0.002116,
            0.032166,
            0.0001026,
            1.36e-05,
            676,
            np.nan,
            0.0019646,
            0.030051,
            9.54e-05,
            1.27e-05,
            677,
            np.nan,
            0.0018238,
            0.028059,
            8.88e-05,
            1.18e-05,
            678,
            np.nan,
            0.0016929,
            0.026186,
            8.26e-05,
            1.10e-05,
            679,
            np.nan,
            0.0015712,
            0.024426,
            7.69e-05,
            1.03e-05,
            680,
            np.nan,
            0.001458,
            0.022774,
            7.15e-05,
            9.58e-06,
            681,
            np.nan,
            0.0013527,
            0.021224,
            6.66e-05,
            8.93e-06,
            682,
            np.nan,
            0.0012548,
            0.019768,
            6.20e-05,
            8.33e-06,
            683,
            np.nan,
            0.0011634,
            0.018399,
            5.78e-05,
            7.76e-06,
            684,
            np.nan,
            0.0010781,
            0.017109,
            5.38e-05,
            7.24e-06,
            685,
            np.nan,
            0.00099842,
            0.015894,
            5.01e-05,
            6.75e-06,
            686,
            np.nan,
            0.00092396,
            0.014749,
            4.67e-05,
            6.31e-06,
            687,
            np.nan,
            0.00085471,
            0.013677,
            4.36e-05,
            5.88e-06,
            688,
            np.nan,
            0.00079065,
            0.01268,
            4.06e-05,
            5.49e-06,
            689,
            np.nan,
            0.00073168,
            0.011759,
            3.79e-05,
            5.13e-06,
            690,
            np.nan,
            0.00067765,
            0.010912,
            3.53e-05,
            4.79e-06,
            691,
            np.nan,
            0.0006283,
            0.010137,
            3.30e-05,
            4.47e-06,
            692,
            np.nan,
            0.00058311,
            0.0094257,
            3.08e-05,
            4.18e-06,
            693,
            np.nan,
            0.00054158,
            0.0087692,
            2.87e-05,
            3.90e-06,
            694,
            np.nan,
            0.00050329,
            0.0081608,
            2.68e-05,
            3.65e-06,
            695,
            np.nan,
            0.00046787,
            0.0075945,
            2.50e-05,
            3.41e-06,
            696,
            np.nan,
            0.00043501,
            0.0070659,
            2.34e-05,
            3.19e-06,
            697,
            np.nan,
            0.00040449,
            0.0065725,
            2.18e-05,
            2.98e-06,
            698,
            np.nan,
            0.00037614,
            0.0061126,
            2.04e-05,
            2.79e-06,
            699,
            np.nan,
            0.00034978,
            0.0056844,
            1.91e-05,
            2.61e-06,
            700,
            np.nan,
            0.00032528,
            0.0052861,
            1.78e-05,
            2.44e-06,
            701,
            np.nan,
            0.00030248,
            0.0049157,
            1.66e-05,
            2.28e-06,
            702,
            np.nan,
            0.00028124,
            0.0045709,
            1.56e-05,
            2.14e-06,
            703,
            np.nan,
            0.00026143,
            0.0042491,
            1.45e-05,
            2.00e-06,
            704,
            np.nan,
            0.00024293,
            0.0039483,
            1.36e-05,
            1.87e-06,
            705,
            np.nan,
            0.00022564,
            0.0036667,
            1.27e-05,
            1.75e-06,
            706,
            np.nan,
            0.00020948,
            0.003403,
            1.19e-05,
            1.64e-06,
            707,
            np.nan,
            0.0001944,
            0.0031563,
            1.11e-05,
            1.54e-06,
            708,
            np.nan,
            0.00018037,
            0.0029262,
            1.04e-05,
            1.44e-06,
            709,
            np.nan,
            0.00016735,
            0.0027121,
            9.76e-06,
            1.35e-06,
            710,
            np.nan,
            0.00015529,
            0.0025133,
            9.14e-06,
            1.27e-06,
            711,
            np.nan,
            0.00014413,
            0.002329,
            8.56e-06,
            1.19e-06,
            712,
            np.nan,
            0.00013383,
            0.0021584,
            8.02e-06,
            1.11e-06,
            713,
            np.nan,
            0.00012431,
            0.0020007,
            7.51e-06,
            1.04e-06,
            714,
            np.nan,
            0.00011551,
            0.0018552,
            7.04e-06,
            9.78e-07,
            715,
            np.nan,
            0.00010739,
            0.0017211,
            6.60e-06,
            9.18e-07,
            716,
            np.nan,
            9.99e-05,
            0.0015975,
            6.18e-06,
            8.62e-07,
            717,
            np.nan,
            9.29e-05,
            0.0014834,
            5.80e-06,
            8.09e-07,
            718,
            np.nan,
            8.65e-05,
            0.0013779,
            5.44e-06,
            7.59e-07,
            719,
            np.nan,
            8.05e-05,
            0.00128,
            5.10e-06,
            7.12e-07,
            720,
            np.nan,
            7.49e-05,
            0.001189,
            4.78e-06,
            6.69e-07,
            721,
            np.nan,
            6.98e-05,
            0.0011043,
            4.49e-06,
            6.28e-07,
            722,
            np.nan,
            6.49e-05,
            0.0010256,
            4.21e-06,
            5.90e-07,
            723,
            np.nan,
            6.05e-05,
            0.0009526,
            3.95e-06,
            5.54e-07,
            724,
            np.nan,
            5.63e-05,
            0.00088496,
            3.71e-06,
            5.21e-07,
            725,
            np.nan,
            5.25e-05,
            0.0008224,
            3.48e-06,
            4.90e-07,
            726,
            np.nan,
            4.89e-05,
            0.00076459,
            3.27e-06,
            4.60e-07,
            727,
            np.nan,
            4.56e-05,
            0.00071111,
            3.07e-06,
            4.33e-07,
            728,
            np.nan,
            4.25e-05,
            0.00066157,
            2.88e-06,
            4.07e-07,
            729,
            np.nan,
            3.97e-05,
            0.00061561,
            2.71e-06,
            3.82e-07,
            730,
            np.nan,
            3.70e-05,
            0.00057292,
            2.55e-06,
            3.60e-07,
            731,
            np.nan,
            3.46e-05,
            0.00053321,
            2.39e-06,
            3.39e-07,
            732,
            np.nan,
            3.23e-05,
            0.00049623,
            2.25e-06,
            3.18e-07,
            733,
            np.nan,
            3.01e-05,
            0.00046178,
            2.12e-06,
            3.00e-07,
            734,
            np.nan,
            2.81e-05,
            0.00042965,
            1.99e-06,
            2.82e-07,
            735,
            np.nan,
            2.62e-05,
            0.00039967,
            1.87e-06,
            2.65e-07,
            736,
            np.nan,
            2.45e-05,
            0.00037169,
            1.76e-06,
            2.50e-07,
            737,
            np.nan,
            2.28e-05,
            0.00034565,
            1.66e-06,
            2.35e-07,
            738,
            np.nan,
            2.13e-05,
            0.00032149,
            1.56e-06,
            2.22e-07,
            739,
            np.nan,
            1.99e-05,
            0.00029916,
            1.47e-06,
            2.09e-07,
            740,
            np.nan,
            1.86e-05,
            0.00027855,
            1.38e-06,
            1.97e-07,
            741,
            np.nan,
            1.74e-05,
            0.00025958,
            1.30e-06,
            1.85e-07,
            742,
            np.nan,
            1.63e-05,
            0.00024206,
            1.22e-06,
            1.75e-07,
            743,
            np.nan,
            1.53e-05,
            0.00022581,
            1.15e-06,
            1.65e-07,
            744,
            np.nan,
            1.43e-05,
            0.00021067,
            1.08e-06,
            1.55e-07,
            745,
            np.nan,
            1.34e-05,
            0.00019653,
            1.02e-06,
            1.46e-07,
            746,
            np.nan,
            1.25e-05,
            0.00018327,
            9.62e-07,
            1.38e-07,
            747,
            np.nan,
            1.17e-05,
            0.00017087,
            9.07e-07,
            1.30e-07,
            748,
            np.nan,
            1.10e-05,
            0.00015929,
            8.55e-07,
            1.23e-07,
            749,
            np.nan,
            1.03e-05,
            0.00014851,
            8.06e-07,
            1.16e-07,
            750,
            np.nan,
            9.63e-06,
            0.00013848,
            7.60e-07,
            1.09e-07,
            751,
            np.nan,
            9.02e-06,
            0.00012918,
            7.16e-07,
            1.03e-07,
            752,
            np.nan,
            8.45e-06,
            0.00012054,
            6.75e-07,
            9.74e-08,
            753,
            np.nan,
            7.92e-06,
            0.00011252,
            6.37e-07,
            9.19e-08,
            754,
            np.nan,
            7.43e-06,
            0.00010506,
            6.01e-07,
            8.68e-08,
            755,
            np.nan,
            6.97e-06,
            9.81e-05,
            5.67e-07,
            8.20e-08,
            756,
            np.nan,
            6.53e-06,
            9.17e-05,
            5.35e-07,
            7.74e-08,
            757,
            np.nan,
            6.13e-06,
            8.56e-05,
            5.05e-07,
            7.31e-08,
            758,
            np.nan,
            5.75e-06,
            8.00e-05,
            4.77e-07,
            6.91e-08,
            759,
            np.nan,
            5.40e-06,
            7.48e-05,
            4.50e-07,
            6.53e-08,
            760,
            np.nan,
            5.07e-06,
            6.99e-05,
            4.25e-07,
            6.17e-08,
            761,
            np.nan,
            4.75e-06,
            6.53e-05,
            4.01e-07,
            5.83e-08,
            762,
            np.nan,
            4.46e-06,
            6.10e-05,
            3.79e-07,
            5.51e-08,
            763,
            np.nan,
            4.18e-06,
            5.70e-05,
            3.58e-07,
            5.21e-08,
            764,
            np.nan,
            3.93e-06,
            5.33e-05,
            3.38e-07,
            4.93e-08,
            765,
            np.nan,
            3.69e-06,
            4.98e-05,
            3.20e-07,
            4.66e-08,
            766,
            np.nan,
            3.46e-06,
            4.66e-05,
            3.02e-07,
            4.41e-08,
            767,
            np.nan,
            3.25e-06,
            4.36e-05,
            2.86e-07,
            4.17e-08,
            768,
            np.nan,
            3.05e-06,
            4.09e-05,
            2.70e-07,
            3.94e-08,
            769,
            np.nan,
            2.87e-06,
            3.82e-05,
            2.55e-07,
            3.73e-08,
            770,
            np.nan,
            2.70e-06,
            3.58e-05,
            2.41e-07,
            3.53e-08,
            771,
            np.nan,
            2.53e-06,
            3.35e-05,
            2.28e-07,
            3.34e-08,
            772,
            np.nan,
            2.38e-06,
            3.13e-05,
            2.16e-07,
            3.17e-08,
            773,
            np.nan,
            2.23e-06,
            2.93e-05,
            2.04e-07,
            3.00e-08,
            774,
            np.nan,
            2.09e-06,
            2.74e-05,
            1.93e-07,
            2.84e-08,
            775,
            np.nan,
            1.97e-06,
            2.56e-05,
            1.83e-07,
            2.69e-08,
            776,
            np.nan,
            1.85e-06,
            2.40e-05,
            1.73e-07,
            2.55e-08,
            777,
            np.nan,
            1.74e-06,
            2.25e-05,
            1.64e-07,
            2.42e-08,
            778,
            np.nan,
            1.64e-06,
            2.11e-05,
            1.55e-07,
            2.29e-08,
            779,
            np.nan,
            1.55e-06,
            1.98e-05,
            1.47e-07,
            2.17e-08,
            780,
            np.nan,
            1.46e-06,
            1.86e-05,
            1.39e-07,
            2.05e-08,
        ]
    )

    sss = sss.reshape(401, 6).astype(np.float64).T
    sss = sss[:, ::binwidth]
    sss = pd.DataFrame(data=sss.T, columns=colnames)
    sss.set_index("Wavelength", inplace=True)
    sss.columns.name = "Photoreceptor"
    sss.index = pd.Int64Index(sss.index)
    if fillna:
        sss = sss.fillna(0)
    return sss


def get_CIE_1924_photopic_vl(binwidth: Optional[int] = 1) -> pd.DataFrame:
    """Get the CIE1924 photopic luminosity function.

    Parameters
    ----------
    binwidth : int, optional
        Width of the wavelength bins in nanometers (must be `1` or `5`). The
        default is `1`.

    Returns
    -------
    vl : pd.Series
        The CIE1924 photopic luminosity function.

    """
    if binwidth not in [1, 5]:
        raise ValueError("Must specify 1 or 5 for binwidth")

    vl = np.array(
        [
            380,
            0.0000390000000,
            381,
            0.0000428264000,
            382,
            0.0000469146000,
            383,
            0.0000515896000,
            384,
            0.0000571764000,
            385,
            0.0000640000000,
            386,
            0.0000723442100,
            387,
            0.0000822122400,
            388,
            0.0000935081600,
            389,
            0.0001061361000,
            390,
            0.0001200000000,
            391,
            0.0001349840000,
            392,
            0.0001514920000,
            393,
            0.0001702080000,
            394,
            0.0001918160000,
            395,
            0.0002170000000,
            396,
            0.0002469067000,
            397,
            0.0002812400000,
            398,
            0.0003185200000,
            399,
            0.0003572667000,
            400,
            0.0003960000000,
            401,
            0.0004337147000,
            402,
            0.0004730240000,
            403,
            0.0005178760000,
            404,
            0.0005722187000,
            405,
            0.0006400000000,
            406,
            0.0007245600000,
            407,
            0.0008255000000,
            408,
            0.0009411600000,
            409,
            0.0010698800000,
            410,
            0.0012100000000,
            411,
            0.0013620910000,
            412,
            0.0015307520000,
            413,
            0.0017203680000,
            414,
            0.0019353230000,
            415,
            0.0021800000000,
            416,
            0.0024548000000,
            417,
            0.0027640000000,
            418,
            0.0031178000000,
            419,
            0.0035264000000,
            420,
            0.0040000000000,
            421,
            0.0045462400000,
            422,
            0.0051593200000,
            423,
            0.0058292800000,
            424,
            0.0065461600000,
            425,
            0.0073000000000,
            426,
            0.0080865070000,
            427,
            0.0089087200000,
            428,
            0.0097676800000,
            429,
            0.0106644300000,
            430,
            0.0116000000000,
            431,
            0.0125731700000,
            432,
            0.0135827200000,
            433,
            0.0146296800000,
            434,
            0.0157150900000,
            435,
            0.0168400000000,
            436,
            0.0180073600000,
            437,
            0.0192144800000,
            438,
            0.0204539200000,
            439,
            0.0217182400000,
            440,
            0.0230000000000,
            441,
            0.0242946100000,
            442,
            0.0256102400000,
            443,
            0.0269585700000,
            444,
            0.0283512500000,
            445,
            0.0298000000000,
            446,
            0.0313108300000,
            447,
            0.0328836800000,
            448,
            0.0345211200000,
            449,
            0.0362257100000,
            450,
            0.0380000000000,
            451,
            0.0398466700000,
            452,
            0.0417680000000,
            453,
            0.0437660000000,
            454,
            0.0458426700000,
            455,
            0.0480000000000,
            456,
            0.0502436800000,
            457,
            0.0525730400000,
            458,
            0.0549805600000,
            459,
            0.0574587200000,
            460,
            0.0600000000000,
            461,
            0.0626019700000,
            462,
            0.0652775200000,
            463,
            0.0680420800000,
            464,
            0.0709110900000,
            465,
            0.0739000000000,
            466,
            0.0770160000000,
            467,
            0.0802664000000,
            468,
            0.0836668000000,
            469,
            0.0872328000000,
            470,
            0.0909800000000,
            471,
            0.0949175500000,
            472,
            0.0990458400000,
            473,
            0.1033674000000,
            474,
            0.1078846000000,
            475,
            0.1126000000000,
            476,
            0.1175320000000,
            477,
            0.1226744000000,
            478,
            0.1279928000000,
            479,
            0.1334528000000,
            480,
            0.1390200000000,
            481,
            0.1446764000000,
            482,
            0.1504693000000,
            483,
            0.1564619000000,
            484,
            0.1627177000000,
            485,
            0.1693000000000,
            486,
            0.1762431000000,
            487,
            0.1835581000000,
            488,
            0.1912735000000,
            489,
            0.1994180000000,
            490,
            0.2080200000000,
            491,
            0.2171199000000,
            492,
            0.2267345000000,
            493,
            0.2368571000000,
            494,
            0.2474812000000,
            495,
            0.2586000000000,
            496,
            0.2701849000000,
            497,
            0.2822939000000,
            498,
            0.2950505000000,
            499,
            0.3085780000000,
            500,
            0.3230000000000,
            501,
            0.3384021000000,
            502,
            0.3546858000000,
            503,
            0.3716986000000,
            504,
            0.3892875000000,
            505,
            0.4073000000000,
            506,
            0.4256299000000,
            507,
            0.4443096000000,
            508,
            0.4633944000000,
            509,
            0.4829395000000,
            510,
            0.5030000000000,
            511,
            0.5235693000000,
            512,
            0.5445120000000,
            513,
            0.5656900000000,
            514,
            0.5869653000000,
            515,
            0.6082000000000,
            516,
            0.6293456000000,
            517,
            0.6503068000000,
            518,
            0.6708752000000,
            519,
            0.6908424000000,
            520,
            0.7100000000000,
            521,
            0.7281852000000,
            522,
            0.7454636000000,
            523,
            0.7619694000000,
            524,
            0.7778368000000,
            525,
            0.7932000000000,
            526,
            0.8081104000000,
            527,
            0.8224962000000,
            528,
            0.8363068000000,
            529,
            0.8494916000000,
            530,
            0.8620000000000,
            531,
            0.8738108000000,
            532,
            0.8849624000000,
            533,
            0.8954936000000,
            534,
            0.9054432000000,
            535,
            0.9148501000000,
            536,
            0.9237348000000,
            537,
            0.9320924000000,
            538,
            0.9399226000000,
            539,
            0.9472252000000,
            540,
            0.9540000000000,
            541,
            0.9602561000000,
            542,
            0.9660074000000,
            543,
            0.9712606000000,
            544,
            0.9760225000000,
            545,
            0.9803000000000,
            546,
            0.9840924000000,
            547,
            0.9874182000000,
            548,
            0.9903128000000,
            549,
            0.9928116000000,
            550,
            0.9949501000000,
            551,
            0.9967108000000,
            552,
            0.9980983000000,
            553,
            0.9991120000000,
            554,
            0.9997482000000,
            555,
            1.0000000000000,
            556,
            0.9998567000000,
            557,
            0.9993046000000,
            558,
            0.9983255000000,
            559,
            0.9968987000000,
            560,
            0.9950000000000,
            561,
            0.9926005000000,
            562,
            0.9897426000000,
            563,
            0.9864444000000,
            564,
            0.9827241000000,
            565,
            0.9786000000000,
            566,
            0.9740837000000,
            567,
            0.9691712000000,
            568,
            0.9638568000000,
            569,
            0.9581349000000,
            570,
            0.9520000000000,
            571,
            0.9454504000000,
            572,
            0.9384992000000,
            573,
            0.9311628000000,
            574,
            0.9234576000000,
            575,
            0.9154000000000,
            576,
            0.9070064000000,
            577,
            0.8982772000000,
            578,
            0.8892048000000,
            579,
            0.8797816000000,
            580,
            0.8700000000000,
            581,
            0.8598613000000,
            582,
            0.8493920000000,
            583,
            0.8386220000000,
            584,
            0.8275813000000,
            585,
            0.8163000000000,
            586,
            0.8047947000000,
            587,
            0.7930820000000,
            588,
            0.7811920000000,
            589,
            0.7691547000000,
            590,
            0.7570000000000,
            591,
            0.7447541000000,
            592,
            0.7324224000000,
            593,
            0.7200036000000,
            594,
            0.7074965000000,
            595,
            0.6949000000000,
            596,
            0.6822192000000,
            597,
            0.6694716000000,
            598,
            0.6566744000000,
            599,
            0.6438448000000,
            600,
            0.6310000000000,
            601,
            0.6181555000000,
            602,
            0.6053144000000,
            603,
            0.5924756000000,
            604,
            0.5796379000000,
            605,
            0.5668000000000,
            606,
            0.5539611000000,
            607,
            0.5411372000000,
            608,
            0.5283528000000,
            609,
            0.5156323000000,
            610,
            0.5030000000000,
            611,
            0.4904688000000,
            612,
            0.4780304000000,
            613,
            0.4656776000000,
            614,
            0.4534032000000,
            615,
            0.4412000000000,
            616,
            0.4290800000000,
            617,
            0.4170360000000,
            618,
            0.4050320000000,
            619,
            0.3930320000000,
            620,
            0.3810000000000,
            621,
            0.3689184000000,
            622,
            0.3568272000000,
            623,
            0.3447768000000,
            624,
            0.3328176000000,
            625,
            0.3210000000000,
            626,
            0.3093381000000,
            627,
            0.2978504000000,
            628,
            0.2865936000000,
            629,
            0.2756245000000,
            630,
            0.2650000000000,
            631,
            0.2547632000000,
            632,
            0.2448896000000,
            633,
            0.2353344000000,
            634,
            0.2260528000000,
            635,
            0.2170000000000,
            636,
            0.2081616000000,
            637,
            0.1995488000000,
            638,
            0.1911552000000,
            639,
            0.1829744000000,
            640,
            0.1750000000000,
            641,
            0.1672235000000,
            642,
            0.1596464000000,
            643,
            0.1522776000000,
            644,
            0.1451259000000,
            645,
            0.1382000000000,
            646,
            0.1315003000000,
            647,
            0.1250248000000,
            648,
            0.1187792000000,
            649,
            0.1127691000000,
            650,
            0.1070000000000,
            651,
            0.1014762000000,
            652,
            0.0961886400000,
            653,
            0.0911229600000,
            654,
            0.0862648500000,
            655,
            0.0816000000000,
            656,
            0.0771206400000,
            657,
            0.0728255200000,
            658,
            0.0687100800000,
            659,
            0.0647697600000,
            660,
            0.0610000000000,
            661,
            0.0573962100000,
            662,
            0.0539550400000,
            663,
            0.0506737600000,
            664,
            0.0475496500000,
            665,
            0.0445800000000,
            666,
            0.0417587200000,
            667,
            0.0390849600000,
            668,
            0.0365638400000,
            669,
            0.0342004800000,
            670,
            0.0320000000000,
            671,
            0.0299626100000,
            672,
            0.0280766400000,
            673,
            0.0263293600000,
            674,
            0.0247080500000,
            675,
            0.0232000000000,
            676,
            0.0218007700000,
            677,
            0.0205011200000,
            678,
            0.0192810800000,
            679,
            0.0181206900000,
            680,
            0.0170000000000,
            681,
            0.0159037900000,
            682,
            0.0148371800000,
            683,
            0.0138106800000,
            684,
            0.0128347800000,
            685,
            0.0119200000000,
            686,
            0.0110683100000,
            687,
            0.0102733900000,
            688,
            0.0095333110000,
            689,
            0.0088461570000,
            690,
            0.0082100000000,
            691,
            0.0076237810000,
            692,
            0.0070854240000,
            693,
            0.0065914760000,
            694,
            0.0061384850000,
            695,
            0.0057230000000,
            696,
            0.0053430590000,
            697,
            0.0049957960000,
            698,
            0.0046764040000,
            699,
            0.0043800750000,
            700,
            0.0041020000000,
            701,
            0.0038384530000,
            702,
            0.0035890990000,
            703,
            0.0033542190000,
            704,
            0.0031340930000,
            705,
            0.0029290000000,
            706,
            0.0027381390000,
            707,
            0.0025598760000,
            708,
            0.0023932440000,
            709,
            0.0022372750000,
            710,
            0.0020910000000,
            711,
            0.0019535870000,
            712,
            0.0018245800000,
            713,
            0.0017035800000,
            714,
            0.0015901870000,
            715,
            0.0014840000000,
            716,
            0.0013844960000,
            717,
            0.0012912680000,
            718,
            0.0012040920000,
            719,
            0.0011227440000,
            720,
            0.0010470000000,
            721,
            0.0009765896000,
            722,
            0.0009111088000,
            723,
            0.0008501332000,
            724,
            0.0007932384000,
            725,
            0.0007400000000,
            726,
            0.0006900827000,
            727,
            0.0006433100000,
            728,
            0.0005994960000,
            729,
            0.0005584547000,
            730,
            0.0005200000000,
            731,
            0.0004839136000,
            732,
            0.0004500528000,
            733,
            0.0004183452000,
            734,
            0.0003887184000,
            735,
            0.0003611000000,
            736,
            0.0003353835000,
            737,
            0.0003114404000,
            738,
            0.0002891656000,
            739,
            0.0002684539000,
            740,
            0.0002492000000,
            741,
            0.0002313019000,
            742,
            0.0002146856000,
            743,
            0.0001992884000,
            744,
            0.0001850475000,
            745,
            0.0001719000000,
            746,
            0.0001597781000,
            747,
            0.0001486044000,
            748,
            0.0001383016000,
            749,
            0.0001287925000,
            750,
            0.0001200000000,
            751,
            0.0001118595000,
            752,
            0.0001043224000,
            753,
            0.0000973356000,
            754,
            0.0000908458700,
            755,
            0.0000848000000,
            756,
            0.0000791466700,
            757,
            0.0000738580000,
            758,
            0.0000689160000,
            759,
            0.0000643026700,
            760,
            0.0000600000000,
            761,
            0.0000559818700,
            762,
            0.0000522256000,
            763,
            0.0000487184000,
            764,
            0.0000454474700,
            765,
            0.0000424000000,
            766,
            0.0000395610400,
            767,
            0.0000369151200,
            768,
            0.0000344486800,
            769,
            0.0000321481600,
            770,
            0.0000300000000,
            771,
            0.0000279912500,
            772,
            0.0000261135600,
            773,
            0.0000243602400,
            774,
            0.0000227246100,
            775,
            0.0000212000000,
            776,
            0.0000197785500,
            777,
            0.0000184528500,
            778,
            0.0000172168700,
            779,
            0.0000160645900,
            780,
            0.0000149900000,
        ]
    )

    vl = vl.reshape(401, 2).astype(np.float64).T
    vl = vl[:, ::binwidth]
    idx = pd.Int64Index(vl[0], name="Wavelength")
    return pd.Series(data=vl[1], index=idx, name="vl")


def get_matrix_LMStoXYZ():
    """Get LMS to XYZ conversion matrix for 10 degree field size.

    Returns
    -------
    np.ndarray
        The matrix.

    """
    return np.array(
        [
            [1.93986443, -1.34664359, 0.43044935],
            [0.69283932, 0.34967567, 0],
            [0.0, 0.0, 2.14687945],
        ]
    )


def get_CIE170_2_chromaticity_coordinates(
    binwidth: int = 1, connect: bool = True
) -> pd.DataFrame():

    colnames = ["Wavelength", "x", "y", "z"]
    coord = np.array(
        [
            390,
            0.17842,
            0.02464,
            0.79694,
            391,
            0.17838,
            0.02482,
            0.79679,
            392,
            0.17831,
            0.02496,
            0.79673,
            393,
            0.17819,
            0.02505,
            0.79676,
            394,
            0.17803,
            0.02510,
            0.79687,
            395,
            0.17784,
            0.02509,
            0.79706,
            396,
            0.17763,
            0.02504,
            0.79733,
            397,
            0.17738,
            0.02494,
            0.79769,
            398,
            0.17711,
            0.02478,
            0.79811,
            399,
            0.17682,
            0.02458,
            0.79860,
            400,
            0.17652,
            0.02432,
            0.79915,
            401,
            0.17621,
            0.02403,
            0.79976,
            402,
            0.17590,
            0.02371,
            0.80040,
            403,
            0.17559,
            0.02338,
            0.80103,
            404,
            0.17530,
            0.02307,
            0.80163,
            405,
            0.17504,
            0.02279,
            0.80217,
            406,
            0.17480,
            0.02255,
            0.80265,
            407,
            0.17457,
            0.02236,
            0.80308,
            408,
            0.17432,
            0.02218,
            0.80349,
            409,
            0.17405,
            0.02202,
            0.80393,
            410,
            0.17372,
            0.02185,
            0.80442,
            411,
            0.17333,
            0.02168,
            0.80499,
            412,
            0.17290,
            0.02150,
            0.80560,
            413,
            0.17243,
            0.02133,
            0.80623,
            414,
            0.17196,
            0.02118,
            0.80685,
            415,
            0.17151,
            0.02106,
            0.80744,
            416,
            0.17107,
            0.02097,
            0.80796,
            417,
            0.17065,
            0.02092,
            0.80844,
            418,
            0.17023,
            0.02090,
            0.80887,
            419,
            0.16980,
            0.02093,
            0.80927,
            420,
            0.16935,
            0.02100,
            0.80966,
            421,
            0.16886,
            0.02111,
            0.81004,
            422,
            0.16834,
            0.02126,
            0.81040,
            423,
            0.16780,
            0.02144,
            0.81076,
            424,
            0.16726,
            0.02165,
            0.81109,
            425,
            0.16672,
            0.02188,
            0.81140,
            426,
            0.16620,
            0.02213,
            0.81167,
            427,
            0.16569,
            0.02240,
            0.81190,
            428,
            0.16518,
            0.02271,
            0.81211,
            429,
            0.16465,
            0.02307,
            0.81228,
            430,
            0.16408,
            0.02349,
            0.81243,
            431,
            0.16346,
            0.02398,
            0.81255,
            432,
            0.16280,
            0.02453,
            0.81267,
            433,
            0.16212,
            0.02512,
            0.81276,
            434,
            0.16142,
            0.02572,
            0.81285,
            435,
            0.16074,
            0.02632,
            0.81294,
            436,
            0.16010,
            0.02689,
            0.81301,
            437,
            0.15949,
            0.02744,
            0.81306,
            438,
            0.15890,
            0.02800,
            0.81309,
            439,
            0.15831,
            0.02860,
            0.81309,
            440,
            0.15769,
            0.02925,
            0.81306,
            441,
            0.15703,
            0.02998,
            0.81299,
            442,
            0.15632,
            0.03080,
            0.81289,
            443,
            0.15557,
            0.03168,
            0.81275,
            444,
            0.15479,
            0.03262,
            0.81259,
            445,
            0.15400,
            0.03360,
            0.81240,
            446,
            0.15320,
            0.03462,
            0.81218,
            447,
            0.15238,
            0.03570,
            0.81192,
            448,
            0.15154,
            0.03686,
            0.81159,
            449,
            0.15068,
            0.03814,
            0.81118,
            450,
            0.14977,
            0.03958,
            0.81065,
            451,
            0.14882,
            0.04119,
            0.80999,
            452,
            0.14779,
            0.04300,
            0.80921,
            453,
            0.14670,
            0.04499,
            0.80831,
            454,
            0.14551,
            0.04717,
            0.80732,
            455,
            0.14423,
            0.04951,
            0.80626,
            456,
            0.14283,
            0.05202,
            0.80515,
            457,
            0.14132,
            0.05471,
            0.80397,
            458,
            0.13972,
            0.05759,
            0.80269,
            459,
            0.13802,
            0.06067,
            0.80130,
            460,
            0.13627,
            0.06397,
            0.79976,
            461,
            0.13446,
            0.06751,
            0.79803,
            462,
            0.13258,
            0.07132,
            0.79610,
            463,
            0.13062,
            0.07543,
            0.79394,
            464,
            0.12855,
            0.07992,
            0.79153,
            465,
            0.12634,
            0.08482,
            0.78884,
            466,
            0.12395,
            0.09020,
            0.78585,
            467,
            0.12137,
            0.09611,
            0.78252,
            468,
            0.11859,
            0.10260,
            0.77881,
            469,
            0.11563,
            0.10971,
            0.77466,
            470,
            0.11247,
            0.11750,
            0.77003,
            471,
            0.10912,
            0.12603,
            0.76485,
            472,
            0.10557,
            0.13535,
            0.75908,
            473,
            0.10182,
            0.14553,
            0.75265,
            474,
            0.09784,
            0.15663,
            0.74553,
            475,
            0.09363,
            0.16873,
            0.73765,
            476,
            0.08916,
            0.18188,
            0.72896,
            477,
            0.08447,
            0.19609,
            0.71944,
            478,
            0.07957,
            0.21135,
            0.70908,
            479,
            0.07450,
            0.22760,
            0.69790,
            480,
            0.06929,
            0.24478,
            0.68593,
            481,
            0.06400,
            0.26281,
            0.67319,
            482,
            0.05863,
            0.28170,
            0.65966,
            483,
            0.05323,
            0.30150,
            0.64528,
            484,
            0.04781,
            0.32221,
            0.62998,
            485,
            0.04243,
            0.34386,
            0.61371,
            486,
            0.03714,
            0.36644,
            0.59642,
            487,
            0.03201,
            0.38982,
            0.57817,
            488,
            0.02711,
            0.41387,
            0.55902,
            489,
            0.02250,
            0.43840,
            0.53909,
            490,
            0.01826,
            0.46323,
            0.51851,
            491,
            0.01443,
            0.48816,
            0.49741,
            492,
            0.01110,
            0.51301,
            0.47589,
            493,
            0.00834,
            0.53760,
            0.45406,
            494,
            0.00623,
            0.56173,
            0.43204,
            495,
            0.00486,
            0.58520,
            0.40993,
            496,
            0.00428,
            0.60787,
            0.38785,
            497,
            0.00443,
            0.62973,
            0.36584,
            498,
            0.00520,
            0.65087,
            0.34393,
            499,
            0.00648,
            0.67137,
            0.32214,
            500,
            0.00817,
            0.69130,
            0.30052,
            501,
            0.01017,
            0.71070,
            0.27912,
            502,
            0.01251,
            0.72943,
            0.25806,
            503,
            0.01525,
            0.74730,
            0.23744,
            504,
            0.01846,
            0.76414,
            0.21740,
            505,
            0.02218,
            0.77978,
            0.19804,
            506,
            0.02648,
            0.79401,
            0.17951,
            507,
            0.03143,
            0.80653,
            0.16204,
            508,
            0.03704,
            0.81710,
            0.14587,
            509,
            0.04333,
            0.82555,
            0.13111,
            510,
            0.05030,
            0.83185,
            0.11785,
            511,
            0.05788,
            0.83606,
            0.10606,
            512,
            0.06588,
            0.83858,
            0.09554,
            513,
            0.07408,
            0.83984,
            0.08609,
            514,
            0.08228,
            0.84018,
            0.07754,
            515,
            0.09030,
            0.83993,
            0.06977,
            516,
            0.09804,
            0.83930,
            0.06266,
            517,
            0.10558,
            0.83823,
            0.05620,
            518,
            0.11304,
            0.83660,
            0.05035,
            519,
            0.12056,
            0.83434,
            0.04510,
            520,
            0.12825,
            0.83134,
            0.04040,
            521,
            0.13620,
            0.82759,
            0.03621,
            522,
            0.14434,
            0.82318,
            0.03248,
            523,
            0.15262,
            0.81824,
            0.02913,
            524,
            0.16097,
            0.81290,
            0.02613,
            525,
            0.16931,
            0.80726,
            0.02343,
            526,
            0.17760,
            0.80140,
            0.02100,
            527,
            0.18580,
            0.79539,
            0.01881,
            528,
            0.19386,
            0.78929,
            0.01686,
            529,
            0.20176,
            0.78314,
            0.01511,
            530,
            0.20946,
            0.77699,
            0.01355,
            531,
            0.21696,
            0.77088,
            0.01216,
            532,
            0.22432,
            0.76476,
            0.01092,
            533,
            0.23159,
            0.75860,
            0.00980,
            534,
            0.23885,
            0.75235,
            0.00879,
            535,
            0.24615,
            0.74597,
            0.00788,
            536,
            0.25353,
            0.73942,
            0.00705,
            537,
            0.26097,
            0.73274,
            0.00630,
            538,
            0.26841,
            0.72597,
            0.00562,
            539,
            0.27582,
            0.71916,
            0.00502,
            540,
            0.28316,
            0.71236,
            0.00448,
            541,
            0.29038,
            0.70561,
            0.00401,
            542,
            0.29751,
            0.69891,
            0.00359,
            543,
            0.30452,
            0.69226,
            0.00322,
            544,
            0.31144,
            0.68568,
            0.00288,
            545,
            0.31826,
            0.67915,
            0.00259,
            546,
            0.32499,
            0.67269,
            0.00233,
            547,
            0.33163,
            0.66628,
            0.00209,
            548,
            0.33819,
            0.65992,
            0.00188,
            549,
            0.34469,
            0.65361,
            0.00169,
            550,
            0.35114,
            0.64734,
            0.00152,
            551,
            0.35755,
            0.64109,
            0.00136,
            552,
            0.36397,
            0.63481,
            0.00122,
            553,
            0.37043,
            0.62848,
            0.00109,
            554,
            0.37696,
            0.62206,
            0.00098,
            555,
            0.38361,
            0.61552,
            0.00088,
            556,
            0.39036,
            0.60885,
            0.00079,
            557,
            0.39716,
            0.60214,
            0.00071,
            558,
            0.40393,
            0.59543,
            0.00064,
            559,
            0.41062,
            0.58881,
            0.00057,
            560,
            0.41716,
            0.58232,
            0.00051,
            561,
            0.42354,
            0.57600,
            0.00046,
            562,
            0.42980,
            0.56979,
            0.00042,
            563,
            0.43602,
            0.56360,
            0.00037,
            564,
            0.44227,
            0.55740,
            0.00034,
            565,
            0.44859,
            0.55110,
            0.00030,
            566,
            0.45503,
            0.54470,
            0.00027,
            567,
            0.46153,
            0.53823,
            0.00025,
            568,
            0.46803,
            0.53175,
            0.00022,
            569,
            0.47450,
            0.52530,
            0.00020,
            570,
            0.48088,
            0.51894,
            0.00018,
            571,
            0.48714,
            0.51269,
            0.00016,
            572,
            0.49331,
            0.50654,
            0.00015,
            573,
            0.49939,
            0.50048,
            0.00013,
            574,
            0.50541,
            0.49447,
            0.00012,
            575,
            0.51138,
            0.48851,
            0.00011,
            576,
            0.51732,
            0.48258,
            0.00010,
            577,
            0.52323,
            0.47668,
            0.00009,
            578,
            0.52914,
            0.47078,
            0.00008,
            579,
            0.53505,
            0.46488,
            0.00007,
            580,
            0.54097,
            0.45896,
            0.00007,
            581,
            0.54689,
            0.45305,
            0.00006,
            582,
            0.55275,
            0.44719,
            0.00006,
            583,
            0.55849,
            0.44146,
            0.00005,
            584,
            0.56405,
            0.43591,
            0.00005,
            585,
            0.56937,
            0.43059,
            0.00004,
            586,
            0.57444,
            0.42552,
            0.00004,
            587,
            0.57930,
            0.42066,
            0.00004,
            588,
            0.58401,
            0.41596,
            0.00003,
            589,
            0.58860,
            0.41137,
            0.00003,
            590,
            0.59312,
            0.40685,
            0.00003,
            591,
            0.59760,
            0.40238,
            0.00002,
            592,
            0.60201,
            0.39796,
            0.00002,
            593,
            0.60636,
            0.39362,
            0.00002,
            594,
            0.61060,
            0.38938,
            0.00002,
            595,
            0.61474,
            0.38524,
            0.00002,
            596,
            0.61877,
            0.38121,
            0.00002,
            597,
            0.62269,
            0.37729,
            0.00002,
            598,
            0.62653,
            0.37346,
            0.00001,
            599,
            0.63028,
            0.36971,
            0.00001,
            600,
            0.63396,
            0.36602,
            0.00001,
            601,
            0.63758,
            0.36241,
            0.00001,
            602,
            0.64111,
            0.35888,
            0.00001,
            603,
            0.64454,
            0.35545,
            0.00001,
            604,
            0.64785,
            0.35214,
            0.00001,
            605,
            0.65103,
            0.34896,
            0.00001,
            606,
            0.65407,
            0.34593,
            0.00001,
            607,
            0.65697,
            0.34302,
            0.00001,
            608,
            0.65975,
            0.34024,
            0.00001,
            609,
            0.66242,
            0.33758,
            0.00001,
            610,
            0.66498,
            0.33502,
            0.00001,
            611,
            0.66744,
            0.33256,
            0.00001,
            612,
            0.66980,
            0.33019,
            0.00001,
            613,
            0.67208,
            0.32791,
            0.00001,
            614,
            0.67427,
            0.32572,
            0.00000,
            615,
            0.67638,
            0.32361,
            0.00000,
            616,
            0.67841,
            0.32159,
            0.00000,
            617,
            0.68038,
            0.31962,
            0.00000,
            618,
            0.68228,
            0.31772,
            0.00000,
            619,
            0.68414,
            0.31586,
            0.00000,
            620,
            0.68596,
            0.31404,
            0.00000,
            621,
            0.68773,
            0.31227,
            0.00000,
            622,
            0.68945,
            0.31055,
            0.00000,
            623,
            0.69109,
            0.30891,
            0.00000,
            624,
            0.69262,
            0.30738,
            0.00000,
            625,
            0.69405,
            0.30595,
            0.00000,
            626,
            0.69535,
            0.30465,
            0.00000,
            627,
            0.69656,
            0.30344,
            0.00000,
            628,
            0.69768,
            0.30232,
            0.00000,
            629,
            0.69875,
            0.30125,
            0.00000,
            630,
            0.69977,
            0.30023,
            0.00000,
            631,
            0.70077,
            0.29923,
            0.00000,
            632,
            0.70174,
            0.29826,
            0.00000,
            633,
            0.70268,
            0.29732,
            0.00000,
            634,
            0.70359,
            0.29641,
            0.00000,
            635,
            0.70447,
            0.29553,
            0.00000,
            636,
            0.70533,
            0.29467,
            0.00000,
            637,
            0.70618,
            0.29382,
            0.00000,
            638,
            0.70702,
            0.29298,
            0.00000,
            639,
            0.70786,
            0.29214,
            0.00000,
            640,
            0.70871,
            0.29129,
            0.00000,
            641,
            0.70957,
            0.29043,
            0.00000,
            642,
            0.71042,
            0.28958,
            0.00000,
            643,
            0.71122,
            0.28878,
            0.00000,
            644,
            0.71195,
            0.28805,
            0.00000,
            645,
            0.71259,
            0.28741,
            0.00000,
            646,
            0.71312,
            0.28688,
            0.00000,
            647,
            0.71356,
            0.28644,
            0.00000,
            648,
            0.71395,
            0.28605,
            0.00000,
            649,
            0.71431,
            0.28569,
            0.00000,
            650,
            0.71465,
            0.28535,
            0.00000,
            651,
            0.71501,
            0.28499,
            0.00000,
            652,
            0.71537,
            0.28463,
            0.00000,
            653,
            0.71574,
            0.28426,
            0.00000,
            654,
            0.71611,
            0.28389,
            0.00000,
            655,
            0.71648,
            0.28352,
            0.00000,
            656,
            0.71685,
            0.28315,
            0.00000,
            657,
            0.71721,
            0.28279,
            0.00000,
            658,
            0.71756,
            0.28244,
            0.00000,
            659,
            0.71791,
            0.28209,
            0.00000,
            660,
            0.71824,
            0.28176,
            0.00000,
            661,
            0.71857,
            0.28143,
            0.00000,
            662,
            0.71887,
            0.28113,
            0.00000,
            663,
            0.71916,
            0.28084,
            0.00000,
            664,
            0.71943,
            0.28057,
            0.00000,
            665,
            0.71967,
            0.28033,
            0.00000,
            666,
            0.71988,
            0.28012,
            0.00000,
            667,
            0.72007,
            0.27993,
            0.00000,
            668,
            0.72024,
            0.27976,
            0.00000,
            669,
            0.72039,
            0.27961,
            0.00000,
            670,
            0.72054,
            0.27946,
            0.00000,
            671,
            0.72067,
            0.27933,
            0.00000,
            672,
            0.72080,
            0.27920,
            0.00000,
            673,
            0.72093,
            0.27907,
            0.00000,
            674,
            0.72104,
            0.27896,
            0.00000,
            675,
            0.72115,
            0.27885,
            0.00000,
            676,
            0.72125,
            0.27875,
            0.00000,
            677,
            0.72134,
            0.27866,
            0.00000,
            678,
            0.72143,
            0.27857,
            0.00000,
            679,
            0.72151,
            0.27849,
            0.00000,
            680,
            0.72158,
            0.27842,
            0.00000,
            681,
            0.72165,
            0.27835,
            0.00000,
            682,
            0.72171,
            0.27829,
            0.00000,
            683,
            0.72177,
            0.27823,
            0.00000,
            684,
            0.72183,
            0.27817,
            0.00000,
            685,
            0.72187,
            0.27813,
            0.00000,
            686,
            0.72192,
            0.27808,
            0.00000,
            687,
            0.72195,
            0.27805,
            0.00000,
            688,
            0.72199,
            0.27801,
            0.00000,
            689,
            0.72202,
            0.27798,
            0.00000,
            690,
            0.72205,
            0.27795,
            0.00000,
            691,
            0.72208,
            0.27792,
            0.00000,
            692,
            0.72211,
            0.27789,
            0.00000,
            693,
            0.72213,
            0.27787,
            0.00000,
            694,
            0.72215,
            0.27785,
            0.00000,
            695,
            0.72217,
            0.27783,
            0.00000,
            696,
            0.72218,
            0.27782,
            0.00000,
            697,
            0.72218,
            0.27782,
            0.00000,
            698,
            0.72219,
            0.27781,
            0.00000,
            699,
            0.72219,
            0.27781,
            0.00000,
            700,
            0.72219,
            0.27781,
            0.00000,
            701,
            0.72219,
            0.27781,
            0.00000,
            702,
            0.72219,
            0.27781,
            0.00000,
            703,
            0.72219,
            0.27781,
            0.00000,
            704,
            0.72219,
            0.27781,
            0.00000,
            705,
            0.72219,
            0.27781,
            0.00000,
            706,
            0.72218,
            0.27782,
            0.00000,
            707,
            0.72217,
            0.27783,
            0.00000,
            708,
            0.72216,
            0.27784,
            0.00000,
            709,
            0.72215,
            0.27785,
            0.00000,
            710,
            0.72213,
            0.27787,
            0.00000,
            711,
            0.72210,
            0.27790,
            0.00000,
            712,
            0.72207,
            0.27793,
            0.00000,
            713,
            0.72204,
            0.27796,
            0.00000,
            714,
            0.72201,
            0.27799,
            0.00000,
            715,
            0.72198,
            0.27802,
            0.00000,
            716,
            0.72195,
            0.27805,
            0.00000,
            717,
            0.72192,
            0.27808,
            0.00000,
            718,
            0.72189,
            0.27811,
            0.00000,
            719,
            0.72185,
            0.27815,
            0.00000,
            720,
            0.72182,
            0.27818,
            0.00000,
            721,
            0.72179,
            0.27821,
            0.00000,
            722,
            0.72175,
            0.27825,
            0.00000,
            723,
            0.72171,
            0.27829,
            0.00000,
            724,
            0.72167,
            0.27833,
            0.00000,
            725,
            0.72163,
            0.27837,
            0.00000,
            726,
            0.72159,
            0.27841,
            0.00000,
            727,
            0.72155,
            0.27845,
            0.00000,
            728,
            0.72151,
            0.27849,
            0.00000,
            729,
            0.72147,
            0.27853,
            0.00000,
            730,
            0.72142,
            0.27858,
            0.00000,
            731,
            0.72138,
            0.27862,
            0.00000,
            732,
            0.72134,
            0.27866,
            0.00000,
            733,
            0.72130,
            0.27870,
            0.00000,
            734,
            0.72125,
            0.27875,
            0.00000,
            735,
            0.72120,
            0.27880,
            0.00000,
            736,
            0.72115,
            0.27885,
            0.00000,
            737,
            0.72109,
            0.27891,
            0.00000,
            738,
            0.72104,
            0.27896,
            0.00000,
            739,
            0.72097,
            0.27903,
            0.00000,
            740,
            0.72091,
            0.27909,
            0.00000,
            741,
            0.72084,
            0.27916,
            0.00000,
            742,
            0.72078,
            0.27922,
            0.00000,
            743,
            0.72071,
            0.27929,
            0.00000,
            744,
            0.72064,
            0.27936,
            0.00000,
            745,
            0.72057,
            0.27943,
            0.00000,
            746,
            0.72050,
            0.27950,
            0.00000,
            747,
            0.72043,
            0.27957,
            0.00000,
            748,
            0.72036,
            0.27964,
            0.00000,
            749,
            0.72029,
            0.27971,
            0.00000,
            750,
            0.72022,
            0.27978,
            0.00000,
            751,
            0.72015,
            0.27985,
            0.00000,
            752,
            0.72009,
            0.27991,
            0.00000,
            753,
            0.72002,
            0.27998,
            0.00000,
            754,
            0.71995,
            0.28005,
            0.00000,
            755,
            0.71988,
            0.28012,
            0.00000,
            756,
            0.71980,
            0.28020,
            0.00000,
            757,
            0.71973,
            0.28027,
            0.00000,
            758,
            0.71965,
            0.28035,
            0.00000,
            759,
            0.71958,
            0.28042,
            0.00000,
            760,
            0.71950,
            0.28050,
            0.00000,
            761,
            0.71943,
            0.28057,
            0.00000,
            762,
            0.71935,
            0.28065,
            0.00000,
            763,
            0.71928,
            0.28072,
            0.00000,
            764,
            0.71921,
            0.28079,
            0.00000,
            765,
            0.71915,
            0.28085,
            0.00000,
            766,
            0.71908,
            0.28092,
            0.00000,
            767,
            0.71901,
            0.28099,
            0.00000,
            768,
            0.71894,
            0.28106,
            0.00000,
            769,
            0.71888,
            0.28112,
            0.00000,
            770,
            0.71881,
            0.28119,
            0.00000,
            771,
            0.71874,
            0.28126,
            0.00000,
            772,
            0.71867,
            0.28133,
            0.00000,
            773,
            0.71860,
            0.28140,
            0.00000,
            774,
            0.71852,
            0.28148,
            0.00000,
            775,
            0.71845,
            0.28155,
            0.00000,
            776,
            0.71838,
            0.28162,
            0.00000,
            777,
            0.71830,
            0.28170,
            0.00000,
            778,
            0.71822,
            0.28178,
            0.00000,
            779,
            0.71814,
            0.28186,
            0.00000,
            780,
            0.71807,
            0.28193,
            0.00000,
            781,
            0.71799,
            0.28201,
            0.00000,
            782,
            0.71791,
            0.28209,
            0.00000,
            783,
            0.71783,
            0.28217,
            0.00000,
            784,
            0.71775,
            0.28225,
            0.00000,
            785,
            0.71766,
            0.28234,
            0.00000,
            786,
            0.71758,
            0.28242,
            0.00000,
            787,
            0.71750,
            0.28250,
            0.00000,
            788,
            0.71742,
            0.28258,
            0.00000,
            789,
            0.71733,
            0.28267,
            0.00000,
            790,
            0.71725,
            0.28275,
            0.00000,
            791,
            0.71716,
            0.28284,
            0.00000,
            792,
            0.71708,
            0.28292,
            0.00000,
            793,
            0.71699,
            0.28301,
            0.00000,
            794,
            0.71690,
            0.28310,
            0.00000,
            795,
            0.71681,
            0.28319,
            0.00000,
            796,
            0.71671,
            0.28329,
            0.00000,
            797,
            0.71661,
            0.28339,
            0.00000,
            798,
            0.71651,
            0.28349,
            0.00000,
            799,
            0.71641,
            0.28359,
            0.00000,
            800,
            0.71630,
            0.28370,
            0.00000,
            801,
            0.71619,
            0.28381,
            0.00000,
            802,
            0.71609,
            0.28391,
            0.00000,
            803,
            0.71598,
            0.28402,
            0.00000,
            804,
            0.71588,
            0.28412,
            0.00000,
            805,
            0.71577,
            0.28423,
            0.00000,
            806,
            0.71566,
            0.28434,
            0.00000,
            807,
            0.71556,
            0.28444,
            0.00000,
            808,
            0.71545,
            0.28455,
            0.00000,
            809,
            0.71534,
            0.28466,
            0.00000,
            810,
            0.71523,
            0.28477,
            0.00000,
            811,
            0.71513,
            0.28487,
            0.00000,
            812,
            0.71502,
            0.28498,
            0.00000,
            813,
            0.71491,
            0.28509,
            0.00000,
            814,
            0.71480,
            0.28520,
            0.00000,
            815,
            0.71469,
            0.28531,
            0.00000,
            816,
            0.71459,
            0.28541,
            0.00000,
            817,
            0.71449,
            0.28551,
            0.00000,
            818,
            0.71438,
            0.28562,
            0.00000,
            819,
            0.71428,
            0.28572,
            0.00000,
            820,
            0.71418,
            0.28582,
            0.00000,
            821,
            0.71409,
            0.28591,
            0.00000,
            822,
            0.71400,
            0.28600,
            0.00000,
            823,
            0.71391,
            0.28609,
            0.00000,
            824,
            0.71382,
            0.28618,
            0.00000,
            825,
            0.71373,
            0.28627,
            0.00000,
            826,
            0.71365,
            0.28635,
            0.00000,
            827,
            0.71358,
            0.28642,
            0.00000,
            828,
            0.71350,
            0.28650,
            0.00000,
            829,
            0.71343,
            0.28657,
            0.00000,
            830,
            0.71336,
            0.28664,
            0.00000,
        ]
    )

    coord = coord.reshape(441, 4).astype(np.float64).T
    coord = coord[:, ::binwidth]
    coord = pd.DataFrame(data=coord.T, columns=colnames)
    coord.set_index("Wavelength", inplace=True)
    coord.index = pd.Int64Index(coord.index)
    if connect:
        coord = coord.append(coord.iloc[0], ignore_index=True)
    return coord
