"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'Bus'
        }));
        test.done();
    },
    'named event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'myEventBus'
        }));
        test.done();
    },
    'partner event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: 'aws.partner/PartnerName/acct1/repo1'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'aws.partner/PartnerName/acct1/repo1',
            EventSourceName: 'aws.partner/PartnerName/acct1/repo1'
        }));
        test.done();
    },
    'can get bus name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const bus = new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus'
        });
        // WHEN
        new core_1.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                EventBusName: bus.eventBusName
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            EventBusName: { Ref: 'BusEA82B648' }
        }));
        test.done();
    },
    'can get bus arn'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const bus = new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus'
        });
        // WHEN
        new core_1.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                EventBusArn: bus.eventBusArn
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            EventBusArn: { 'Fn::GetAtt': ['BusEA82B648', 'Arn'] }
        }));
        test.done();
    },
    'event bus name cannot be default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'default'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must not be 'default'/);
        test.done();
    },
    'event bus name cannot contain slash'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'my/bus'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must not contain '\/'/);
        test.done();
    },
    'event bus cannot have name and source name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myBus',
            eventSourceName: 'myBus'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' and 'eventSourceName' cannot both be provided/);
        test.done();
    },
    'event bus name cannot be empty string'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: ''
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must satisfy: /);
        test.done();
    },
    'event bus source name must follow pattern'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: 'invalid-partner'
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventSourceName' must satisfy: \/\^aws/);
        test.done();
    },
    'event bus source name cannot be empty string'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: ''
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventSourceName' must satisfy: /);
        test.done();
    },
    'can grant PutEvents'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com')
        });
        // WHEN
        lib_1.EventBus.grantPutEvents(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'events:PutEvents',
                        Effect: 'Allow',
                        Resource: '*'
                    }
                ],
                Version: '2012-10-17'
            },
            Roles: [
                {
                    Ref: 'Role1ABCC5F0'
                }
            ]
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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