"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExecuteApi = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [execute-api](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonapigateway.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ExecuteApi extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [execute-api](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonapigateway.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'execute-api';
        this.actionList = {
            "InvalidateCache": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html",
                "description": "Used to invalidate API cache upon a client request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            },
            "Invoke": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html",
                "description": "Used to invoke an API upon a client request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            },
            "ManageConnections": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html",
                "description": "ManageConnections controls access to the @connections API",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "execute-api-general": {
                "name": "execute-api-general",
                "url": "",
                "arn": "arn:${Partition}:execute-api:${Region}:${Account}:${ApiId}/${Stage}/${Method}/${ApiSpecificResourcePath}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Used to invalidate API cache upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html
     */
    toInvalidateCache() {
        this.add('execute-api:InvalidateCache');
        return this;
    }
    /**
     * Used to invoke an API upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html
     */
    toInvoke() {
        this.add('execute-api:Invoke');
        return this;
    }
    /**
     * ManageConnections controls access to the @connections API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html
     */
    toManageConnections() {
        this.add('execute-api:ManageConnections');
        return this;
    }
    /**
     * Adds a resource of type execute-api-general to the statement
     *
     * @param apiId - Identifier for the apiId.
     * @param stage - Identifier for the stage.
     * @param method - Identifier for the method.
     * @param apiSpecificResourcePath - Identifier for the apiSpecificResourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExecuteApiGeneral(apiId, stage, method, apiSpecificResourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:execute-api:${Region}:${Account}:${ApiId}/${Stage}/${Method}/${ApiSpecificResourcePath}';
        arn = arn.replace('${ApiId}', apiId);
        arn = arn.replace('${Stage}', stage);
        arn = arn.replace('${Method}', method);
        arn = arn.replace('${ApiSpecificResourcePath}', apiSpecificResourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.ExecuteApi = ExecuteApi;
//# sourceMappingURL=data:application/json;base64,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