"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chime = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [chime](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chime extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [chime](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'chime';
        this.actionList = {
            "AcceptDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account",
                "accessLevel": "Write"
            },
            "ActivateUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to activate users in an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AddDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to add a domain to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "AddOrUpdateGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html",
                "description": "Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumberWithUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html",
                "description": "Grants permission to associate a phone number with an Amazon Chime user",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "AssociateSigninDelegateGroupsWithAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html",
                "description": "Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "AuthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "BatchCreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html",
                "description": "Grants permission to create new attendees for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "BatchCreateRoomMembership": {
                "url": "",
                "description": "Grants permission to batch add room members",
                "accessLevel": "Write"
            },
            "BatchDeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html",
                "description": "Grants permission to move up to 50 phone numbers to the deletion queue",
                "accessLevel": "Write"
            },
            "BatchSuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html",
                "description": "Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account",
                "accessLevel": "Write"
            },
            "BatchUnsuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html",
                "description": "Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account",
                "accessLevel": "Write"
            },
            "BatchUpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers",
                "accessLevel": "Write"
            },
            "BatchUpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html",
                "description": "Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "ConnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/active_directory.html",
                "description": "Grants permission to connect an Active Directory to your Amazon Chime Enterprise account",
                "accessLevel": "Write",
                "dependentActions": [
                    "ds:ConnectDirectory"
                ]
            },
            "CreateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html",
                "description": "Grants permission to create an Amazon Chime account under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "CreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html",
                "description": "Grants permission to create a new attendee for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "CreateBot": {
                "url": "",
                "description": "Grants permission to create a bot for an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateBotMembership": {
                "url": "",
                "description": "Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to create a new Call Detail Record S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:CreateBucket",
                    "s3:ListAllMyBuckets"
                ]
            },
            "CreateMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html",
                "description": "Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html",
                "description": "Grants permission to create a phone number order with the Carriers",
                "accessLevel": "Write"
            },
            "CreateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html",
                "description": "Grants permission to create a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "CreateRoom": {
                "url": "",
                "description": "Grants permission to create a room",
                "accessLevel": "Write"
            },
            "CreateRoomMembership": {
                "url": "",
                "description": "Grants permission to add a room member",
                "accessLevel": "Write"
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html",
                "description": "Grants permission to create a user under the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "CreateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "DeleteAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html",
                "description": "Grants permission to delete the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "DeleteAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html",
                "description": "Grants permission to delete the specified attendee from an Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeleteCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:DeleteBucket"
                ]
            },
            "DeleteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete delegated AWS account management from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to delete a domain from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteEventsConfiguration": {
                "url": "",
                "description": "Grants permission to delete an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "DeleteGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "DeleteMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html",
                "description": "Grants permission to delete the specified Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html",
                "description": "Grants permission to move a phone number to the deletion queue",
                "accessLevel": "Write"
            },
            "DeleteProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html",
                "description": "Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteRoom": {
                "url": "",
                "description": "Grants permission to delete a room",
                "accessLevel": "Write"
            },
            "DeleteRoomMembership": {
                "url": "",
                "description": "Grants permission to remove a room member",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html",
                "description": "Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html",
                "description": "Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html",
                "description": "Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumberFromUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html",
                "description": "Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DisassociateSigninDelegateGroupsFromAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html",
                "description": "Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "DisconnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "GetAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html",
                "description": "Grants permission to get details for the specified Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details for the account resource associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html",
                "description": "Grants permission to get account settings for the specified Amazon Chime account ID",
                "accessLevel": "Read"
            },
            "GetAccountWithOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html",
                "description": "Grants permission to get attendee details for a specified meeting ID and attendee ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetBot": {
                "url": "",
                "description": "Grants permission to retrieve details for the specified bot",
                "accessLevel": "Read"
            },
            "GetCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account",
                "accessLevel": "Read",
                "dependentActions": [
                    "s3:GetBucketAcl",
                    "s3:GetBucketLocation",
                    "s3:GetBucketLogging",
                    "s3:GetBucketVersioning",
                    "s3:GetBucketWebsite"
                ]
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to get domain details for a domain associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetEventsConfiguration": {
                "url": "",
                "description": "Grants permission to retrieve details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Read"
            },
            "GetGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html",
                "description": "Grants permission to get global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html",
                "description": "Grants permission to get the meeting record for a specified meeting ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetMeetingDetail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get attendee, connection, and other details for a meeting",
                "accessLevel": "Read"
            },
            "GetPhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html",
                "description": "Grants permission to get details for the specified phone number",
                "accessLevel": "Read"
            },
            "GetPhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html",
                "description": "Grants permission to get details for the specified phone number order",
                "accessLevel": "Read"
            },
            "GetPhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html",
                "description": "Grants permission to get phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html",
                "description": "Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetRetentionSettings": {
                "url": "",
                "description": "Gets the retention settings for the specified Amazon Chime account.",
                "accessLevel": "Read"
            },
            "GetRoom": {
                "url": "",
                "description": "Grants permission to retrieve a room",
                "accessLevel": "Read"
            },
            "GetTelephonyLimits": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to get telephony limits for the AWS account",
                "accessLevel": "Read"
            },
            "GetUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html",
                "description": "Grants permission to get details for the specified user ID",
                "accessLevel": "Read"
            },
            "GetUserActivityReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get a summary of user activity on the user details page",
                "accessLevel": "Read"
            },
            "GetUserByEmail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account",
                "accessLevel": "Read"
            },
            "GetUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html",
                "description": "Grants permission to get user settings related to the specified Amazon Chime user",
                "accessLevel": "Read"
            },
            "GetVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html",
                "description": "Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html",
                "description": "Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html",
                "description": "Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTerminationHealth": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html",
                "description": "Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "InviteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html",
                "description": "Grants permission to invite as many as 50 users to the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsersFromProvider": {
                "url": "",
                "description": "Grants permission to invite users from a third party provider to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ListAccountUsageReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list Amazon Chime account usage reporting data",
                "accessLevel": "List"
            },
            "ListAccounts": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html",
                "description": "Grants permission to list the Amazon Chime accounts under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListApiKeys": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration",
                "accessLevel": "List"
            },
            "ListAttendeeTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK attendee resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListAttendees": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html",
                "description": "Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListBots": {
                "url": "",
                "description": "Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Call Detail Record S3 buckets",
                "accessLevel": "List",
                "dependentActions": [
                    "s3:ListAllMyBuckets",
                    "s3:ListBucket"
                ]
            },
            "ListCallingRegions": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to list the calling regions available for the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListDelegates": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list account delegate information associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListDirectories": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list active Active Directories hosted in the Directory Service of your AWS account",
                "accessLevel": "List"
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to list domains associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListMeetingEvents": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list all events that occurred for a specified meeting",
                "accessLevel": "List"
            },
            "ListMeetingTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListMeetings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html",
                "description": "Grants permission to list up to 100 active Amazon Chime SDK meetings",
                "accessLevel": "Read"
            },
            "ListMeetingsReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list meetings ended during the specified date range",
                "accessLevel": "List"
            },
            "ListPhoneNumberOrders": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html",
                "description": "Grants permission to list the phone number orders under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListPhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html",
                "description": "Grants permission to list the phone numbers under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListProxySessions": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html",
                "description": "Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListRoomMemberships": {
                "url": "",
                "description": "Grants permission to list all room members",
                "accessLevel": "Read"
            },
            "ListRooms": {
                "url": "",
                "description": "Grants permission to list rooms",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html",
                "description": "Grants permission to list the users that belong to the specified Amazon Chime account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html",
                "description": "Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListVoiceConnectors": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html",
                "description": "Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account",
                "accessLevel": "List"
            },
            "LogoutUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html",
                "description": "Grants permission to log out the specified user from all of the devices they are currently logged into",
                "accessLevel": "Write"
            },
            "PutEventsConfiguration": {
                "url": "",
                "description": "Grants permission to update details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "PutRetentionSettings": {
                "url": "",
                "description": "Puts retention settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to add logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write",
                "dependentActions": [
                    "logs:CreateLogDelivery",
                    "logs:CreateLogGroup",
                    "logs:DeleteLogDelivery",
                    "logs:DescribeLogGroups",
                    "logs:GetLogDelivery",
                    "logs:ListLogDeliveries"
                ]
            },
            "PutVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html",
                "description": "Grants permission to update the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html",
                "description": "Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html",
                "description": "Grants permission to update the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "RedactConversationMessage": {
                "url": "",
                "description": "Redacts the specified Chime conversation Message",
                "accessLevel": "Write"
            },
            "RedactRoomMessage": {
                "url": "",
                "description": "Redacts the specified Chime room Message",
                "accessLevel": "Write"
            },
            "RegenerateSecurityToken": {
                "url": "",
                "description": "Grants permission to regenerate the security token for the specified bot",
                "accessLevel": "Write"
            },
            "RenameAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/rename-account.html",
                "description": "Grants permission to modify the account name for your Amazon Chime Enterprise or Team account",
                "accessLevel": "Write"
            },
            "RenewDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to renew the delegation request associated with an Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to reset the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetPersonalPIN": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html",
                "description": "Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account",
                "accessLevel": "Write"
            },
            "RestorePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html",
                "description": "Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory",
                "accessLevel": "Write"
            },
            "RetrieveDataExports": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to download the file containing links to all user attachments returned as part of the \"Request attachments\" action",
                "accessLevel": "List"
            },
            "SearchAvailablePhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html",
                "description": "Grants permission to search phone numbers that can be ordered from the carrier",
                "accessLevel": "Read"
            },
            "StartDataExport": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to submit the \"Request attachments\" request",
                "accessLevel": "Write"
            },
            "SubmitSupportRequest": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html",
                "description": "Grants permission to submit a customer service support request",
                "accessLevel": "Write"
            },
            "SuspendUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to suspend users from an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "TagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "TagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UnauthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "UntagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "UpdateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html",
                "description": "Grants permission to update account details for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to update the OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html",
                "description": "Grants permission to update the settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateBot": {
                "url": "",
                "description": "Grants permission to update the status of the specified bot",
                "accessLevel": "Write"
            },
            "UpdateCDRSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update your Call Detail Record S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:CreateBucket",
                    "s3:DeleteBucket",
                    "s3:ListAllMyBuckets"
                ]
            },
            "UpdateGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html",
                "description": "Grants permission to update the global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details for the specified phone number",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html",
                "description": "Grants permission to update phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html",
                "description": "Grants permission to update a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateRoom": {
                "url": "",
                "description": "Grants permission to update a room",
                "accessLevel": "Write"
            },
            "UpdateRoomMembership": {
                "url": "",
                "description": "Grants permission to update room membership role",
                "accessLevel": "Write"
            },
            "UpdateSupportedLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the supported license tiers available for users in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html",
                "description": "Grants permission to update user details for a specified user ID",
                "accessLevel": "Write"
            },
            "UpdateUserLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the licenses for your Amazon Chime users",
                "accessLevel": "Write"
            },
            "UpdateUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html",
                "description": "Grants permission to update user settings related to the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html",
                "description": "Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html",
                "description": "Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "ValidateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to validate the account resource in your Amazon Chime account",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "meeting": {
                "name": "meeting",
                "url": "",
                "arn": "arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toAcceptDelegate() {
        this.add('chime:AcceptDelegate');
        return this;
    }
    /**
     * Grants permission to activate users in an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toActivateUsers() {
        this.add('chime:ActivateUsers');
        return this;
    }
    /**
     * Grants permission to add a domain to your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toAddDomain() {
        this.add('chime:AddDomain');
        return this;
    }
    /**
     * Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html
     */
    toAddOrUpdateGroups() {
        this.add('chime:AddOrUpdateGroups');
        return this;
    }
    /**
     * Grants permission to associate a phone number with an Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html
     */
    toAssociatePhoneNumberWithUser() {
        this.add('chime:AssociatePhoneNumberWithUser');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html
     */
    toAssociatePhoneNumbersWithVoiceConnector() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnector');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html
     */
    toAssociatePhoneNumbersWithVoiceConnectorGroup() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html
     */
    toAssociateSigninDelegateGroupsWithAccount() {
        this.add('chime:AssociateSigninDelegateGroupsWithAccount');
        return this;
    }
    /**
     * Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toAuthorizeDirectory() {
        this.add('chime:AuthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to create new attendees for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html
     */
    toBatchCreateAttendee() {
        this.add('chime:BatchCreateAttendee');
        return this;
    }
    /**
     * Grants permission to batch add room members
     *
     * Access Level: Write
     */
    toBatchCreateRoomMembership() {
        this.add('chime:BatchCreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to move up to 50 phone numbers to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html
     */
    toBatchDeletePhoneNumber() {
        this.add('chime:BatchDeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html
     */
    toBatchSuspendUser() {
        this.add('chime:BatchSuspendUser');
        return this;
    }
    /**
     * Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html
     */
    toBatchUnsuspendUser() {
        this.add('chime:BatchUnsuspendUser');
        return this;
    }
    /**
     * Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html
     */
    toBatchUpdatePhoneNumber() {
        this.add('chime:BatchUpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html
     */
    toBatchUpdateUser() {
        this.add('chime:BatchUpdateUser');
        return this;
    }
    /**
     * Grants permission to connect an Active Directory to your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:ConnectDirectory
     *
     * https://docs.aws.amazon.com/chime/latest/ag/active_directory.html
     */
    toConnectDirectory() {
        this.add('chime:ConnectDirectory');
        return this;
    }
    /**
     * Grants permission to create an Amazon Chime account under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html
     */
    toCreateAccount() {
        this.add('chime:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toCreateApiKey() {
        this.add('chime:CreateApiKey');
        return this;
    }
    /**
     * Grants permission to create a new attendee for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html
     */
    toCreateAttendee() {
        this.add('chime:CreateAttendee');
        return this;
    }
    /**
     * Grants permission to create a bot for an Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    toCreateBot() {
        this.add('chime:CreateBot');
        return this;
    }
    /**
     * Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    toCreateBotMembership() {
        this.add('chime:CreateBotMembership');
        return this;
    }
    /**
     * Grants permission to create a new Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toCreateCDRBucket() {
        this.add('chime:CreateCDRBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html
     */
    toCreateMeeting() {
        this.add('chime:CreateMeeting');
        return this;
    }
    /**
     * Grants permission to create a phone number order with the Carriers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html
     */
    toCreatePhoneNumberOrder() {
        this.add('chime:CreatePhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to create a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html
     */
    toCreateProxySession() {
        this.add('chime:CreateProxySession');
        return this;
    }
    /**
     * Grants permission to create a room
     *
     * Access Level: Write
     */
    toCreateRoom() {
        this.add('chime:CreateRoom');
        return this;
    }
    /**
     * Grants permission to add a room member
     *
     * Access Level: Write
     */
    toCreateRoomMembership() {
        this.add('chime:CreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to create a user under the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.add('chime:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html
     */
    toCreateVoiceConnector() {
        this.add('chime:CreateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html
     */
    toCreateVoiceConnectorGroup() {
        this.add('chime:CreateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html
     */
    toDeleteAccount() {
        this.add('chime:DeleteAccount');
        return this;
    }
    /**
     * Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toDeleteAccountOpenIdConfig() {
        this.add('chime:DeleteAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toDeleteApiKey() {
        this.add('chime:DeleteApiKey');
        return this;
    }
    /**
     * Grants permission to delete the specified attendee from an Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html
     */
    toDeleteAttendee() {
        this.add('chime:DeleteAttendee');
        return this;
    }
    /**
     * Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteCDRBucket() {
        this.add('chime:DeleteCDRBucket');
        return this;
    }
    /**
     * Grants permission to delete delegated AWS account management from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteDelegate() {
        this.add('chime:DeleteDelegate');
        return this;
    }
    /**
     * Grants permission to delete a domain from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toDeleteDomain() {
        this.add('chime:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    toDeleteEventsConfiguration() {
        this.add('chime:DeleteEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteGroups() {
        this.add('chime:DeleteGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html
     */
    toDeleteMeeting() {
        this.add('chime:DeleteMeeting');
        return this;
    }
    /**
     * Grants permission to move a phone number to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html
     */
    toDeletePhoneNumber() {
        this.add('chime:DeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html
     */
    toDeleteProxySession() {
        this.add('chime:DeleteProxySession');
        return this;
    }
    /**
     * Grants permission to delete a room
     *
     * Access Level: Write
     */
    toDeleteRoom() {
        this.add('chime:DeleteRoom');
        return this;
    }
    /**
     * Grants permission to remove a room member
     *
     * Access Level: Write
     */
    toDeleteRoomMembership() {
        this.add('chime:DeleteRoomMembership');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html
     */
    toDeleteVoiceConnector() {
        this.add('chime:DeleteVoiceConnector');
        return this;
    }
    /**
     * Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html
     */
    toDeleteVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:DeleteVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html
     */
    toDeleteVoiceConnectorGroup() {
        this.add('chime:DeleteVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html
     */
    toDeleteVoiceConnectorOrigination() {
        this.add('chime:DeleteVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html
     */
    toDeleteVoiceConnectorProxy() {
        this.add('chime:DeleteVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html
     */
    toDeleteVoiceConnectorStreamingConfiguration() {
        this.add('chime:DeleteVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html
     */
    toDeleteVoiceConnectorTermination() {
        this.add('chime:DeleteVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html
     */
    toDeleteVoiceConnectorTerminationCredentials() {
        this.add('chime:DeleteVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html
     */
    toDisassociatePhoneNumberFromUser() {
        this.add('chime:DisassociatePhoneNumberFromUser');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html
     */
    toDisassociatePhoneNumbersFromVoiceConnector() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnector');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html
     */
    toDisassociatePhoneNumbersFromVoiceConnectorGroup() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html
     */
    toDisassociateSigninDelegateGroupsFromAccount() {
        this.add('chime:DisassociateSigninDelegateGroupsFromAccount');
        return this;
    }
    /**
     * Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDisconnectDirectory() {
        this.add('chime:DisconnectDirectory');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html
     */
    toGetAccount() {
        this.add('chime:GetAccount');
        return this;
    }
    /**
     * Grants permission to get details for the account resource associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetAccountResource() {
        this.add('chime:GetAccountResource');
        return this;
    }
    /**
     * Grants permission to get account settings for the specified Amazon Chime account ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        this.add('chime:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toGetAccountWithOpenIdConfig() {
        this.add('chime:GetAccountWithOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to get attendee details for a specified meeting ID and attendee ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html
     */
    toGetAttendee() {
        this.add('chime:GetAttendee');
        return this;
    }
    /**
     * Grants permission to retrieve details for the specified bot
     *
     * Access Level: Read
     */
    toGetBot() {
        this.add('chime:GetBot');
        return this;
    }
    /**
     * Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     * - s3:GetBucketLogging
     * - s3:GetBucketVersioning
     * - s3:GetBucketWebsite
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetCDRBucket() {
        this.add('chime:GetCDRBucket');
        return this;
    }
    /**
     * Grants permission to get domain details for a domain associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toGetDomain() {
        this.add('chime:GetDomain');
        return this;
    }
    /**
     * Grants permission to retrieve details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Read
     */
    toGetEventsConfiguration() {
        this.add('chime:GetEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to get global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html
     */
    toGetGlobalSettings() {
        this.add('chime:GetGlobalSettings');
        return this;
    }
    /**
     * Grants permission to get the meeting record for a specified meeting ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html
     */
    toGetMeeting() {
        this.add('chime:GetMeeting');
        return this;
    }
    /**
     * Grants permission to get attendee, connection, and other details for a meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetMeetingDetail() {
        this.add('chime:GetMeetingDetail');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html
     */
    toGetPhoneNumber() {
        this.add('chime:GetPhoneNumber');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number order
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html
     */
    toGetPhoneNumberOrder() {
        this.add('chime:GetPhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to get phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html
     */
    toGetPhoneNumberSettings() {
        this.add('chime:GetPhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html
     */
    toGetProxySession() {
        this.add('chime:GetProxySession');
        return this;
    }
    /**
     * Gets the retention settings for the specified Amazon Chime account.
     *
     * Access Level: Read
     */
    toGetRetentionSettings() {
        this.add('chime:GetRetentionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a room
     *
     * Access Level: Read
     */
    toGetRoom() {
        this.add('chime:GetRoom');
        return this;
    }
    /**
     * Grants permission to get telephony limits for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    toGetTelephonyLimits() {
        this.add('chime:GetTelephonyLimits');
        return this;
    }
    /**
     * Grants permission to get details for the specified user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        this.add('chime:GetUser');
        return this;
    }
    /**
     * Grants permission to get a summary of user activity on the user details page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    toGetUserActivityReportData() {
        this.add('chime:GetUserActivityReportData');
        return this;
    }
    /**
     * Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    toGetUserByEmail() {
        this.add('chime:GetUserByEmail');
        return this;
    }
    /**
     * Grants permission to get user settings related to the specified Amazon Chime user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html
     */
    toGetUserSettings() {
        this.add('chime:GetUserSettings');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html
     */
    toGetVoiceConnector() {
        this.add('chime:GetVoiceConnector');
        return this;
    }
    /**
     * Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html
     */
    toGetVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:GetVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html
     */
    toGetVoiceConnectorGroup() {
        this.add('chime:GetVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html
     */
    toGetVoiceConnectorLoggingConfiguration() {
        this.add('chime:GetVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html
     */
    toGetVoiceConnectorOrigination() {
        this.add('chime:GetVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html
     */
    toGetVoiceConnectorProxy() {
        this.add('chime:GetVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html
     */
    toGetVoiceConnectorStreamingConfiguration() {
        this.add('chime:GetVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html
     */
    toGetVoiceConnectorTermination() {
        this.add('chime:GetVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html
     */
    toGetVoiceConnectorTerminationHealth() {
        this.add('chime:GetVoiceConnectorTerminationHealth');
        return this;
    }
    /**
     * Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toInviteDelegate() {
        this.add('chime:InviteDelegate');
        return this;
    }
    /**
     * Grants permission to invite as many as 50 users to the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html
     */
    toInviteUsers() {
        this.add('chime:InviteUsers');
        return this;
    }
    /**
     * Grants permission to invite users from a third party provider to your Amazon Chime account
     *
     * Access Level: Write
     */
    toInviteUsersFromProvider() {
        this.add('chime:InviteUsersFromProvider');
        return this;
    }
    /**
     * Grants permission to list Amazon Chime account usage reporting data
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListAccountUsageReportData() {
        this.add('chime:ListAccountUsageReportData');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime accounts under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html
     */
    toListAccounts() {
        this.add('chime:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toListApiKeys() {
        this.add('chime:ListApiKeys');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK attendee resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html
     */
    toListAttendeeTags() {
        this.add('chime:ListAttendeeTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html
     */
    toListAttendees() {
        this.add('chime:ListAttendees');
        return this;
    }
    /**
     * Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account
     *
     * Access Level: List
     */
    toListBots() {
        this.add('chime:ListBots');
        return this;
    }
    /**
     * Grants permission to list Call Detail Record S3 buckets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - s3:ListAllMyBuckets
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListCDRBucket() {
        this.add('chime:ListCDRBucket');
        return this;
    }
    /**
     * Grants permission to list the calling regions available for the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    toListCallingRegions() {
        this.add('chime:ListCallingRegions');
        return this;
    }
    /**
     * Grants permission to list account delegate information associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListDelegates() {
        this.add('chime:ListDelegates');
        return this;
    }
    /**
     * Grants permission to list active Active Directories hosted in the Directory Service of your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListDirectories() {
        this.add('chime:ListDirectories');
        return this;
    }
    /**
     * Grants permission to list domains associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toListDomains() {
        this.add('chime:ListDomains');
        return this;
    }
    /**
     * Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListGroups() {
        this.add('chime:ListGroups');
        return this;
    }
    /**
     * Grants permission to list all events that occurred for a specified meeting
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListMeetingEvents() {
        this.add('chime:ListMeetingEvents');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html
     */
    toListMeetingTags() {
        this.add('chime:ListMeetingTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 active Amazon Chime SDK meetings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html
     */
    toListMeetings() {
        this.add('chime:ListMeetings');
        return this;
    }
    /**
     * Grants permission to list meetings ended during the specified date range
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListMeetingsReportData() {
        this.add('chime:ListMeetingsReportData');
        return this;
    }
    /**
     * Grants permission to list the phone number orders under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html
     */
    toListPhoneNumberOrders() {
        this.add('chime:ListPhoneNumberOrders');
        return this;
    }
    /**
     * Grants permission to list the phone numbers under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html
     */
    toListPhoneNumbers() {
        this.add('chime:ListPhoneNumbers');
        return this;
    }
    /**
     * Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html
     */
    toListProxySessions() {
        this.add('chime:ListProxySessions');
        return this;
    }
    /**
     * Grants permission to list all room members
     *
     * Access Level: Read
     */
    toListRoomMemberships() {
        this.add('chime:ListRoomMemberships');
        return this;
    }
    /**
     * Grants permission to list rooms
     *
     * Access Level: Read
     */
    toListRooms() {
        this.add('chime:ListRooms');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('chime:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the users that belong to the specified Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.add('chime:ListUsers');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html
     */
    toListVoiceConnectorGroups() {
        this.add('chime:ListVoiceConnectorGroups');
        return this;
    }
    /**
     * Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html
     */
    toListVoiceConnectorTerminationCredentials() {
        this.add('chime:ListVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html
     */
    toListVoiceConnectors() {
        this.add('chime:ListVoiceConnectors');
        return this;
    }
    /**
     * Grants permission to log out the specified user from all of the devices they are currently logged into
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html
     */
    toLogoutUser() {
        this.add('chime:LogoutUser');
        return this;
    }
    /**
     * Grants permission to update details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    toPutEventsConfiguration() {
        this.add('chime:PutEventsConfiguration');
        return this;
    }
    /**
     * Puts retention settings for the specified Amazon Chime account
     *
     * Access Level: Write
     */
    toPutRetentionSettings() {
        this.add('chime:PutRetentionSettings');
        return this;
    }
    /**
     * Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html
     */
    toPutVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:PutVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to add logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogDelivery
     * - logs:CreateLogGroup
     * - logs:DeleteLogDelivery
     * - logs:DescribeLogGroups
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html
     */
    toPutVoiceConnectorLoggingConfiguration() {
        this.add('chime:PutVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html
     */
    toPutVoiceConnectorOrigination() {
        this.add('chime:PutVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html
     */
    toPutVoiceConnectorProxy() {
        this.add('chime:PutVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html
     */
    toPutVoiceConnectorStreamingConfiguration() {
        this.add('chime:PutVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html
     */
    toPutVoiceConnectorTermination() {
        this.add('chime:PutVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html
     */
    toPutVoiceConnectorTerminationCredentials() {
        this.add('chime:PutVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Redacts the specified Chime conversation Message
     *
     * Access Level: Write
     */
    toRedactConversationMessage() {
        this.add('chime:RedactConversationMessage');
        return this;
    }
    /**
     * Redacts the specified Chime room Message
     *
     * Access Level: Write
     */
    toRedactRoomMessage() {
        this.add('chime:RedactRoomMessage');
        return this;
    }
    /**
     * Grants permission to regenerate the security token for the specified bot
     *
     * Access Level: Write
     */
    toRegenerateSecurityToken() {
        this.add('chime:RegenerateSecurityToken');
        return this;
    }
    /**
     * Grants permission to modify the account name for your Amazon Chime Enterprise or Team account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/rename-account.html
     */
    toRenameAccount() {
        this.add('chime:RenameAccount');
        return this;
    }
    /**
     * Grants permission to renew the delegation request associated with an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toRenewDelegate() {
        this.add('chime:RenewDelegate');
        return this;
    }
    /**
     * Grants permission to reset the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toResetAccountResource() {
        this.add('chime:ResetAccountResource');
        return this;
    }
    /**
     * Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html
     */
    toResetPersonalPIN() {
        this.add('chime:ResetPersonalPIN');
        return this;
    }
    /**
     * Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html
     */
    toRestorePhoneNumber() {
        this.add('chime:RestorePhoneNumber');
        return this;
    }
    /**
     * Grants permission to download the file containing links to all user attachments returned as part of the "Request attachments" action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    toRetrieveDataExports() {
        this.add('chime:RetrieveDataExports');
        return this;
    }
    /**
     * Grants permission to search phone numbers that can be ordered from the carrier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html
     */
    toSearchAvailablePhoneNumbers() {
        this.add('chime:SearchAvailablePhoneNumbers');
        return this;
    }
    /**
     * Grants permission to submit the "Request attachments" request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    toStartDataExport() {
        this.add('chime:StartDataExport');
        return this;
    }
    /**
     * Grants permission to submit a customer service support request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html
     */
    toSubmitSupportRequest() {
        this.add('chime:SubmitSupportRequest');
        return this;
    }
    /**
     * Grants permission to suspend users from an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toSuspendUsers() {
        this.add('chime:SuspendUsers');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html
     */
    toTagAttendee() {
        this.add('chime:TagAttendee');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html
     */
    toTagMeeting() {
        this.add('chime:TagMeeting');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('chime:TagResource');
        return this;
    }
    /**
     * Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUnauthorizeDirectory() {
        this.add('chime:UnauthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html
     */
    toUntagAttendee() {
        this.add('chime:UntagAttendee');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html
     */
    toUntagMeeting() {
        this.add('chime:UntagMeeting');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('chime:UntagResource');
        return this;
    }
    /**
     * Grants permission to update account details for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html
     */
    toUpdateAccount() {
        this.add('chime:UpdateAccount');
        return this;
    }
    /**
     * Grants permission to update the OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toUpdateAccountOpenIdConfig() {
        this.add('chime:UpdateAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to update the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUpdateAccountResource() {
        this.add('chime:UpdateAccountResource');
        return this;
    }
    /**
     * Grants permission to update the settings for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html
     */
    toUpdateAccountSettings() {
        this.add('chime:UpdateAccountSettings');
        return this;
    }
    /**
     * Grants permission to update the status of the specified bot
     *
     * Access Level: Write
     */
    toUpdateBot() {
        this.add('chime:UpdateBot');
        return this;
    }
    /**
     * Grants permission to update your Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:DeleteBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUpdateCDRSettings() {
        this.add('chime:UpdateCDRSettings');
        return this;
    }
    /**
     * Grants permission to update the global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html
     */
    toUpdateGlobalSettings() {
        this.add('chime:UpdateGlobalSettings');
        return this;
    }
    /**
     * Grants permission to update phone number details for the specified phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html
     */
    toUpdatePhoneNumber() {
        this.add('chime:UpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html
     */
    toUpdatePhoneNumberSettings() {
        this.add('chime:UpdatePhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to update a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html
     */
    toUpdateProxySession() {
        this.add('chime:UpdateProxySession');
        return this;
    }
    /**
     * Grants permission to update a room
     *
     * Access Level: Write
     */
    toUpdateRoom() {
        this.add('chime:UpdateRoom');
        return this;
    }
    /**
     * Grants permission to update room membership role
     *
     * Access Level: Write
     */
    toUpdateRoomMembership() {
        this.add('chime:UpdateRoomMembership');
        return this;
    }
    /**
     * Grants permission to update the supported license tiers available for users in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toUpdateSupportedLicenses() {
        this.add('chime:UpdateSupportedLicenses');
        return this;
    }
    /**
     * Grants permission to update user details for a specified user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        this.add('chime:UpdateUser');
        return this;
    }
    /**
     * Grants permission to update the licenses for your Amazon Chime users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toUpdateUserLicenses() {
        this.add('chime:UpdateUserLicenses');
        return this;
    }
    /**
     * Grants permission to update user settings related to the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html
     */
    toUpdateUserSettings() {
        this.add('chime:UpdateUserSettings');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html
     */
    toUpdateVoiceConnector() {
        this.add('chime:UpdateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html
     */
    toUpdateVoiceConnectorGroup() {
        this.add('chime:UpdateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to validate the account resource in your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toValidateAccountResource() {
        this.add('chime:ValidateAccountResource');
        return this;
    }
    /**
     * Adds a resource of type meeting to the statement
     *
     * @param meetingId - Identifier for the meetingId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMeeting(meetingId, accountId, partition) {
        var arn = 'arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}';
        arn = arn.replace('${MeetingId}', meetingId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateMeeting()
     * - .toTagMeeting()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toTagMeeting()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - meeting
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateMeeting()
     * - .toTagMeeting()
     * - .toTagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Chime = Chime;
//# sourceMappingURL=data:application/json;base64,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