"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Clouddirectory = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [clouddirectory](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonclouddirectory.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Clouddirectory extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [clouddirectory](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonclouddirectory.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'clouddirectory';
        this.actionList = {
            "AddFacetToObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AddFacetToObject.html",
                "description": "Adds a new Facet to an object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ApplySchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ApplySchema.html",
                "description": "Copies input published schema into Directory with same name and version as that of published schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "AttachObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachObject.html",
                "description": "Attaches an existing object to another existing object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AttachPolicy": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachPolicy.html",
                "description": "Attaches a policy object to any other object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AttachToIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachToIndex.html",
                "description": "Attaches the specified object to the specified index.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AttachTypedLink": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachTypedLink.html",
                "description": "Attaches a typed link b/w a source & target object reference.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "BatchRead": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchRead.html",
                "description": "Performs all the read operations in a batch. Each individual operation inside BatchRead needs to be granted permissions explicitly.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "BatchWrite": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchWrite.html",
                "description": "Performs all the write operations in a batch. Each individual operation inside BatchWrite needs to be granted permissions explicitly.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateDirectory.html",
                "description": "Creates a Directory by copying the published schema into the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "CreateFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateFacet.html",
                "description": "Creates a new Facet in a schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "CreateIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateIndex.html",
                "description": "Creates an index object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateObject.html",
                "description": "Creates an object in a Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateSchema.html",
                "description": "Creates a new schema in a development state.",
                "accessLevel": "Write"
            },
            "CreateTypedLinkFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateTypedLinkFacet.html",
                "description": "Creates a new Typed Link facet in a schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "DeleteDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteDirectory.html",
                "description": "Deletes a directory. Only disabled directories can be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteFacet.html",
                "description": "Deletes a given Facet. All attributes and Rules associated with the facet will be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteObject.html",
                "description": "Deletes an object and its associated attributes.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteSchema.html",
                "description": "Deletes a given schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "DeleteTypedLinkFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteTypedLinkFacet.html",
                "description": "Deletes a given TypedLink Facet. All attributes and Rules associated with the facet will be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "DetachFromIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachFromIndex.html",
                "description": "Detaches the specified object from the specified index.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DetachObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachObject.html",
                "description": "Detaches a given object from the parent object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DetachPolicy": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachPolicy.html",
                "description": "Detaches a policy from an object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DetachTypedLink": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachTypedLink.html",
                "description": "Detaches a given typed link b/w given source and target object reference.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DisableDirectory.html",
                "description": "Disables the specified directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_EnableDirectory.html",
                "description": "Enables the specified directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetDirectory.html",
                "description": "Retrieves metadata about a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetFacet.html",
                "description": "Gets details of the Facet, such as Facet Name, Attributes, Rules, or ObjectType.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "GetLinkAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetLinkAttributes.html",
                "description": "Retrieves attributes that are associated with a typed link.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetObjectAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectAttributes.html",
                "description": "Retrieves attributes within a facet that are associated with an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetObjectInformation": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectInformation.html",
                "description": "Retrieves metadata about an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetSchemaAsJson": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetSchemaAsJson.html",
                "description": "Retrieves a JSON representation of the schema.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "GetTypedLinkFacetInformation": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetTypedLinkFacetInformation.html",
                "description": "Returns identity attributes order information associated with a given typed link facet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListAppliedSchemaArns": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAppliedSchemaArns.html",
                "description": "Lists schemas applied to a directory.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListAttachedIndices": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAttachedIndices.html",
                "description": "Lists indices attached to an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListDevelopmentSchemaArns": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDevelopmentSchemaArns.html",
                "description": "Retrieves the ARNs of schemas in the development state.",
                "accessLevel": "List"
            },
            "ListDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDirectories.html",
                "description": "Lists directories created within an account.",
                "accessLevel": "List"
            },
            "ListFacetAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetAttributes.html",
                "description": "Retrieves attributes attached to the facet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListFacetNames": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetNames.html",
                "description": "Retrieves the names of facets that exist in a schema.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListIncomingTypedLinks": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIncomingTypedLinks.html",
                "description": "Returns a paginated list of all incoming TypedLinks for a given object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIndex.html",
                "description": "Lists objects attached to the specified index.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectAttributes.html",
                "description": "Lists all attributes associated with an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectChildren": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectChildren.html",
                "description": "Returns a paginated list of child objects associated with a given object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectParentPaths": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParentPaths.html",
                "description": "Retrieves all available parent paths for any object type such as node, leaf node, policy node, and index node objects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectParents": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParents.html",
                "description": "Lists parent objects associated with a given object in pagination fashion.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectPolicies": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectPolicies.html",
                "description": "Returns policies attached to an object in pagination fashion.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListOutgoingTypedLinks": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListOutgoingTypedLinks.html",
                "description": "Returns a paginated list of all outgoing TypedLinks for a given object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListPolicyAttachments": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPolicyAttachments.html",
                "description": "Returns all of the ObjectIdentifiers to which a given policy is attached.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListPublishedSchemaArns": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPublishedSchemaArns.html",
                "description": "Retrieves published schema ARNs.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTagsForResource.html",
                "description": "Returns tags for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListTypedLinkFacetAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetAttributes.html",
                "description": "Returns a paginated list of attributes associated with typed link facet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListTypedLinkFacetNames": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetNames.html",
                "description": "Returns a paginated list of typed link facet names that exist in a schema.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "LookupPolicy": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_LookupPolicy.html",
                "description": "Lists all policies from the root of the Directory to the object specified.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "PublishSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PublishSchema.html",
                "description": "Publishes a development schema with a version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "PutSchemaFromJson": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PutSchemaFromJson.html",
                "description": "Allows a schema to be updated using JSON upload. Only available for development schemas.",
                "accessLevel": "Write"
            },
            "RemoveFacetFromObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_RemoveFacetFromObject.html",
                "description": "Removes the specified facet from the specified object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_TagResource.html",
                "description": "Adds tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UntagResource.html",
                "description": "Removes tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateFacet.html",
                "description": "Adds/Updates/Deletes existing Attributes, Rules, or ObjectType of a Facet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "UpdateLinkAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateLinkAttributes.html",
                "description": "Updates a given typed link’s attributes. Attributes to be updated must not contribute to the typed link’s identity, as defined by its IdentityAttributeOrder.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateObjectAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateObjectAttributes.html",
                "description": "Updates a given object's attributes.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateSchema.html",
                "description": "Updates the schema name with a new name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "UpdateTypedLinkFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateTypedLinkFacet.html",
                "description": "Adds/Updates/Deletes existing Attributes, Rules, identity attribute order of a TypedLink Facet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "appliedSchema": {
                "name": "appliedSchema",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}/schema/${SchemaName}/${Version}",
                "conditionKeys": []
            },
            "developmentSchema": {
                "name": "developmentSchema",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:schema/development/${SchemaName}",
                "conditionKeys": []
            },
            "directory": {
                "name": "directory",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}",
                "conditionKeys": []
            },
            "publishedSchema": {
                "name": "publishedSchema",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:schema/published/${SchemaName}/${Version}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Adds a new Facet to an object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AddFacetToObject.html
     */
    toAddFacetToObject() {
        this.add('clouddirectory:AddFacetToObject');
        return this;
    }
    /**
     * Copies input published schema into Directory with same name and version as that of published schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ApplySchema.html
     */
    toApplySchema() {
        this.add('clouddirectory:ApplySchema');
        return this;
    }
    /**
     * Attaches an existing object to another existing object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachObject.html
     */
    toAttachObject() {
        this.add('clouddirectory:AttachObject');
        return this;
    }
    /**
     * Attaches a policy object to any other object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachPolicy.html
     */
    toAttachPolicy() {
        this.add('clouddirectory:AttachPolicy');
        return this;
    }
    /**
     * Attaches the specified object to the specified index.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachToIndex.html
     */
    toAttachToIndex() {
        this.add('clouddirectory:AttachToIndex');
        return this;
    }
    /**
     * Attaches a typed link b/w a source & target object reference.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachTypedLink.html
     */
    toAttachTypedLink() {
        this.add('clouddirectory:AttachTypedLink');
        return this;
    }
    /**
     * Performs all the read operations in a batch. Each individual operation inside BatchRead needs to be granted permissions explicitly.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchRead.html
     */
    toBatchRead() {
        this.add('clouddirectory:BatchRead');
        return this;
    }
    /**
     * Performs all the write operations in a batch. Each individual operation inside BatchWrite needs to be granted permissions explicitly.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchWrite.html
     */
    toBatchWrite() {
        this.add('clouddirectory:BatchWrite');
        return this;
    }
    /**
     * Creates a Directory by copying the published schema into the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateDirectory.html
     */
    toCreateDirectory() {
        this.add('clouddirectory:CreateDirectory');
        return this;
    }
    /**
     * Creates a new Facet in a schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateFacet.html
     */
    toCreateFacet() {
        this.add('clouddirectory:CreateFacet');
        return this;
    }
    /**
     * Creates an index object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateIndex.html
     */
    toCreateIndex() {
        this.add('clouddirectory:CreateIndex');
        return this;
    }
    /**
     * Creates an object in a Directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateObject.html
     */
    toCreateObject() {
        this.add('clouddirectory:CreateObject');
        return this;
    }
    /**
     * Creates a new schema in a development state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateSchema.html
     */
    toCreateSchema() {
        this.add('clouddirectory:CreateSchema');
        return this;
    }
    /**
     * Creates a new Typed Link facet in a schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateTypedLinkFacet.html
     */
    toCreateTypedLinkFacet() {
        this.add('clouddirectory:CreateTypedLinkFacet');
        return this;
    }
    /**
     * Deletes a directory. Only disabled directories can be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteDirectory.html
     */
    toDeleteDirectory() {
        this.add('clouddirectory:DeleteDirectory');
        return this;
    }
    /**
     * Deletes a given Facet. All attributes and Rules associated with the facet will be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteFacet.html
     */
    toDeleteFacet() {
        this.add('clouddirectory:DeleteFacet');
        return this;
    }
    /**
     * Deletes an object and its associated attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteObject.html
     */
    toDeleteObject() {
        this.add('clouddirectory:DeleteObject');
        return this;
    }
    /**
     * Deletes a given schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteSchema.html
     */
    toDeleteSchema() {
        this.add('clouddirectory:DeleteSchema');
        return this;
    }
    /**
     * Deletes a given TypedLink Facet. All attributes and Rules associated with the facet will be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteTypedLinkFacet.html
     */
    toDeleteTypedLinkFacet() {
        this.add('clouddirectory:DeleteTypedLinkFacet');
        return this;
    }
    /**
     * Detaches the specified object from the specified index.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachFromIndex.html
     */
    toDetachFromIndex() {
        this.add('clouddirectory:DetachFromIndex');
        return this;
    }
    /**
     * Detaches a given object from the parent object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachObject.html
     */
    toDetachObject() {
        this.add('clouddirectory:DetachObject');
        return this;
    }
    /**
     * Detaches a policy from an object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachPolicy.html
     */
    toDetachPolicy() {
        this.add('clouddirectory:DetachPolicy');
        return this;
    }
    /**
     * Detaches a given typed link b/w given source and target object reference.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachTypedLink.html
     */
    toDetachTypedLink() {
        this.add('clouddirectory:DetachTypedLink');
        return this;
    }
    /**
     * Disables the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DisableDirectory.html
     */
    toDisableDirectory() {
        this.add('clouddirectory:DisableDirectory');
        return this;
    }
    /**
     * Enables the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_EnableDirectory.html
     */
    toEnableDirectory() {
        this.add('clouddirectory:EnableDirectory');
        return this;
    }
    /**
     * Retrieves metadata about a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetDirectory.html
     */
    toGetDirectory() {
        this.add('clouddirectory:GetDirectory');
        return this;
    }
    /**
     * Gets details of the Facet, such as Facet Name, Attributes, Rules, or ObjectType.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetFacet.html
     */
    toGetFacet() {
        this.add('clouddirectory:GetFacet');
        return this;
    }
    /**
     * Retrieves attributes that are associated with a typed link.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetLinkAttributes.html
     */
    toGetLinkAttributes() {
        this.add('clouddirectory:GetLinkAttributes');
        return this;
    }
    /**
     * Retrieves attributes within a facet that are associated with an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectAttributes.html
     */
    toGetObjectAttributes() {
        this.add('clouddirectory:GetObjectAttributes');
        return this;
    }
    /**
     * Retrieves metadata about an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectInformation.html
     */
    toGetObjectInformation() {
        this.add('clouddirectory:GetObjectInformation');
        return this;
    }
    /**
     * Retrieves a JSON representation of the schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetSchemaAsJson.html
     */
    toGetSchemaAsJson() {
        this.add('clouddirectory:GetSchemaAsJson');
        return this;
    }
    /**
     * Returns identity attributes order information associated with a given typed link facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetTypedLinkFacetInformation.html
     */
    toGetTypedLinkFacetInformation() {
        this.add('clouddirectory:GetTypedLinkFacetInformation');
        return this;
    }
    /**
     * Lists schemas applied to a directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAppliedSchemaArns.html
     */
    toListAppliedSchemaArns() {
        this.add('clouddirectory:ListAppliedSchemaArns');
        return this;
    }
    /**
     * Lists indices attached to an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAttachedIndices.html
     */
    toListAttachedIndices() {
        this.add('clouddirectory:ListAttachedIndices');
        return this;
    }
    /**
     * Retrieves the ARNs of schemas in the development state.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDevelopmentSchemaArns.html
     */
    toListDevelopmentSchemaArns() {
        this.add('clouddirectory:ListDevelopmentSchemaArns');
        return this;
    }
    /**
     * Lists directories created within an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDirectories.html
     */
    toListDirectories() {
        this.add('clouddirectory:ListDirectories');
        return this;
    }
    /**
     * Retrieves attributes attached to the facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetAttributes.html
     */
    toListFacetAttributes() {
        this.add('clouddirectory:ListFacetAttributes');
        return this;
    }
    /**
     * Retrieves the names of facets that exist in a schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetNames.html
     */
    toListFacetNames() {
        this.add('clouddirectory:ListFacetNames');
        return this;
    }
    /**
     * Returns a paginated list of all incoming TypedLinks for a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIncomingTypedLinks.html
     */
    toListIncomingTypedLinks() {
        this.add('clouddirectory:ListIncomingTypedLinks');
        return this;
    }
    /**
     * Lists objects attached to the specified index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIndex.html
     */
    toListIndex() {
        this.add('clouddirectory:ListIndex');
        return this;
    }
    /**
     * Lists all attributes associated with an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectAttributes.html
     */
    toListObjectAttributes() {
        this.add('clouddirectory:ListObjectAttributes');
        return this;
    }
    /**
     * Returns a paginated list of child objects associated with a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectChildren.html
     */
    toListObjectChildren() {
        this.add('clouddirectory:ListObjectChildren');
        return this;
    }
    /**
     * Retrieves all available parent paths for any object type such as node, leaf node, policy node, and index node objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParentPaths.html
     */
    toListObjectParentPaths() {
        this.add('clouddirectory:ListObjectParentPaths');
        return this;
    }
    /**
     * Lists parent objects associated with a given object in pagination fashion.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParents.html
     */
    toListObjectParents() {
        this.add('clouddirectory:ListObjectParents');
        return this;
    }
    /**
     * Returns policies attached to an object in pagination fashion.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectPolicies.html
     */
    toListObjectPolicies() {
        this.add('clouddirectory:ListObjectPolicies');
        return this;
    }
    /**
     * Returns a paginated list of all outgoing TypedLinks for a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListOutgoingTypedLinks.html
     */
    toListOutgoingTypedLinks() {
        this.add('clouddirectory:ListOutgoingTypedLinks');
        return this;
    }
    /**
     * Returns all of the ObjectIdentifiers to which a given policy is attached.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPolicyAttachments.html
     */
    toListPolicyAttachments() {
        this.add('clouddirectory:ListPolicyAttachments');
        return this;
    }
    /**
     * Retrieves published schema ARNs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPublishedSchemaArns.html
     */
    toListPublishedSchemaArns() {
        this.add('clouddirectory:ListPublishedSchemaArns');
        return this;
    }
    /**
     * Returns tags for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('clouddirectory:ListTagsForResource');
        return this;
    }
    /**
     * Returns a paginated list of attributes associated with typed link facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetAttributes.html
     */
    toListTypedLinkFacetAttributes() {
        this.add('clouddirectory:ListTypedLinkFacetAttributes');
        return this;
    }
    /**
     * Returns a paginated list of typed link facet names that exist in a schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetNames.html
     */
    toListTypedLinkFacetNames() {
        this.add('clouddirectory:ListTypedLinkFacetNames');
        return this;
    }
    /**
     * Lists all policies from the root of the Directory to the object specified.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_LookupPolicy.html
     */
    toLookupPolicy() {
        this.add('clouddirectory:LookupPolicy');
        return this;
    }
    /**
     * Publishes a development schema with a version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PublishSchema.html
     */
    toPublishSchema() {
        this.add('clouddirectory:PublishSchema');
        return this;
    }
    /**
     * Allows a schema to be updated using JSON upload. Only available for development schemas.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PutSchemaFromJson.html
     */
    toPutSchemaFromJson() {
        this.add('clouddirectory:PutSchemaFromJson');
        return this;
    }
    /**
     * Removes the specified facet from the specified object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_RemoveFacetFromObject.html
     */
    toRemoveFacetFromObject() {
        this.add('clouddirectory:RemoveFacetFromObject');
        return this;
    }
    /**
     * Adds tags to a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('clouddirectory:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('clouddirectory:UntagResource');
        return this;
    }
    /**
     * Adds/Updates/Deletes existing Attributes, Rules, or ObjectType of a Facet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateFacet.html
     */
    toUpdateFacet() {
        this.add('clouddirectory:UpdateFacet');
        return this;
    }
    /**
     * Updates a given typed link’s attributes. Attributes to be updated must not contribute to the typed link’s identity, as defined by its IdentityAttributeOrder.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateLinkAttributes.html
     */
    toUpdateLinkAttributes() {
        this.add('clouddirectory:UpdateLinkAttributes');
        return this;
    }
    /**
     * Updates a given object's attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateObjectAttributes.html
     */
    toUpdateObjectAttributes() {
        this.add('clouddirectory:UpdateObjectAttributes');
        return this;
    }
    /**
     * Updates the schema name with a new name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateSchema.html
     */
    toUpdateSchema() {
        this.add('clouddirectory:UpdateSchema');
        return this;
    }
    /**
     * Adds/Updates/Deletes existing Attributes, Rules, identity attribute order of a TypedLink Facet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateTypedLinkFacet.html
     */
    toUpdateTypedLinkFacet() {
        this.add('clouddirectory:UpdateTypedLinkFacet');
        return this;
    }
    /**
     * Adds a resource of type appliedSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param directoryId - Identifier for the directoryId.
     * @param schemaName - Identifier for the schemaName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAppliedSchema(directoryId, schemaName, version, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}/schema/${SchemaName}/${Version}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type developmentSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevelopmentSchema(schemaName, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:schema/development/${SchemaName}';
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDirectory(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type publishedSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param schemaName - Identifier for the schemaName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPublishedSchema(schemaName, version, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:schema/published/${SchemaName}/${Version}';
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Clouddirectory = Clouddirectory;
//# sourceMappingURL=data:application/json;base64,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