"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Logs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [logs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Logs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [logs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'logs';
        this.actionList = {
            "AssociateKmsKey": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_AssociateKmsKey.html",
                "description": "Associates the specified AWS Key Management Service (AWS KMS) customer master key (CMK) with the specified log group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "CancelExportTask": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CancelExportTask.html",
                "description": "Cancels an export task if it is in PENDING or RUNNING state",
                "accessLevel": "Write"
            },
            "CreateExportTask": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateExportTask.html",
                "description": "Creates an ExportTask which allows you to efficiently export data from a Log Group to your Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "CreateLogDelivery": {
                "url": "",
                "description": "Creates the log delivery",
                "accessLevel": "Write"
            },
            "CreateLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogGroup.html",
                "description": "Creates a new log group with the specified name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "CreateLogStream": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogStream.html",
                "description": "Creates a new log stream with the specified name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteDestination": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDestination.html",
                "description": "Deletes the destination with the specified name and eventually disables all the subscription filters that publish to it",
                "accessLevel": "Write"
            },
            "DeleteLogDelivery": {
                "url": "",
                "description": "Deletes the log delivery information for specified log delivery",
                "accessLevel": "Write"
            },
            "DeleteLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogGroup.html",
                "description": "Deletes the log group with the specified name and permanently deletes all the archived log events associated with it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteLogStream": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogStream.html",
                "description": "Deletes a log stream and permanently deletes all the archived log events associated with it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-stream": {
                        "required": true
                    }
                }
            },
            "DeleteMetricFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteMetricFilter.html",
                "description": "Deletes a metric filter associated with the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteResourcePolicy.html",
                "description": "Deletes a resource policy from this account",
                "accessLevel": "Write"
            },
            "DeleteRetentionPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteRetentionPolicy.html",
                "description": "Deletes the retention policy of the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteSubscriptionFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteSubscriptionFilter.html",
                "description": "Deletes a subscription filter associated with the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeDestinations": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDestinations.html",
                "description": "Returns all the destinations that are associated with the AWS account making the request",
                "accessLevel": "List"
            },
            "DescribeExportTasks": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeExportTasks.html",
                "description": "Returns all the export tasks that are associated with the AWS account making the request",
                "accessLevel": "List"
            },
            "DescribeLogGroups": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogGroups.html",
                "description": "Returns all the log groups that are associated with the AWS account making the request",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeLogStreams": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogStreams.html",
                "description": "Returns all the log streams that are associated with the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeMetricFilters": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeMetricFilters.html",
                "description": "Returns all the metrics filters associated with the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeQueries": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeQueries.html",
                "description": "Returns a list of CloudWatch Logs Insights queries that are scheduled, executing, or have been executed recently in this account. You can request all queries, or limit it to queries of a specific log group or queries with a certain status.",
                "accessLevel": "List"
            },
            "DescribeResourcePolicies": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeResourcePolicies.html",
                "description": "Return all the resource policies in this account.",
                "accessLevel": "List"
            },
            "DescribeSubscriptionFilters": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeSubscriptionFilters.html",
                "description": "Returns all the subscription filters associated with the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DisassociateKmsKey": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DisassociateKmsKey.html",
                "description": "Disassociates the associated AWS Key Management Service (AWS KMS) customer master key (CMK) from the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "FilterLogEvents": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_FilterLogEvents.html",
                "description": "Retrieves log events, optionally filtered by a filter pattern from the specified log group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "GetLogDelivery": {
                "url": "",
                "description": "Gets the log delivery information for specified log delivery",
                "accessLevel": "Read"
            },
            "GetLogEvents": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogEvents.html",
                "description": "Retrieves log events from the specified log stream",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-stream": {
                        "required": true
                    }
                }
            },
            "GetLogGroupFields": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogGroupFields.html",
                "description": "Returns a list of the fields that are included in log events in the specified log group, along with the percentage of log events that contain each field. The search is limited to a time period that you specify.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "GetLogRecord": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogRecord.html",
                "description": "Retrieves all the fields and values of a single log event. All fields are retrieved, even if the original query that produced the logRecordPointer retrieved only a subset of fields. Fields are returned as field name/field value pairs.",
                "accessLevel": "Read"
            },
            "GetQueryResults": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetQueryResults.html",
                "description": "Returns the results from the specified query. If the query is in progress, partial results of that current execution are returned. Only the fields requested in the query are returned.",
                "accessLevel": "Read"
            },
            "ListLogDeliveries": {
                "url": "",
                "description": "Lists all the log deliveries for specified account and/or log source",
                "accessLevel": "List"
            },
            "ListTagsLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListTagsLogGroup.html",
                "description": "Lists the tags for the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "PutDestination": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestination.html",
                "description": "Creates or updates a Destination",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ]
            },
            "PutDestinationPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestinationPolicy.html",
                "description": "Creates or updates an access policy associated with an existing Destination",
                "accessLevel": "Write"
            },
            "PutLogEvents": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutLogEvents.html",
                "description": "Uploads a batch of log events to the specified log stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-stream": {
                        "required": true
                    }
                }
            },
            "PutMetricFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutMetricFilter.html",
                "description": "Creates or updates a metric filter and associates it with the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutResourcePolicy.html",
                "description": "Creates or updates a resource policy allowing other AWS services to put log events to this account",
                "accessLevel": "Write"
            },
            "PutRetentionPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutRetentionPolicy.html",
                "description": "Sets the retention of the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "PutSubscriptionFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutSubscriptionFilter.html",
                "description": "Creates or updates a subscription filter and associates it with the specified log group",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "StartQuery": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StartQuery.html",
                "description": "Schedules a query of a log group using CloudWatch Logs Insights. You specify the log group and time range to query, and the query string to use.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "StopQuery": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StopQuery.html",
                "description": "Stops a CloudWatch Logs Insights query that is in progress. If the query has already ended, the operation returns an error indicating that the specified query is not running.",
                "accessLevel": "Read"
            },
            "TagLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TagLogGroup.html",
                "description": "Adds or updates the specified tags for the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "TestMetricFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TestMetricFilter.html",
                "description": "Tests the filter pattern of a metric filter against a sample of log event messages",
                "accessLevel": "Read"
            },
            "UntagLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UntagLogGroup.html",
                "description": "Removes the specified tags from the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "UpdateLogDelivery": {
                "url": "",
                "description": "Updates the log delivery information for specified log delivery",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "log-group": {
                "name": "log-group",
                "url": "",
                "arn": "arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}",
                "conditionKeys": []
            },
            "log-stream": {
                "name": "log-stream",
                "url": "",
                "arn": "arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}:log-stream:${LogStreamName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associates the specified AWS Key Management Service (AWS KMS) customer master key (CMK) with the specified log group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_AssociateKmsKey.html
     */
    toAssociateKmsKey() {
        this.add('logs:AssociateKmsKey');
        return this;
    }
    /**
     * Cancels an export task if it is in PENDING or RUNNING state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask() {
        this.add('logs:CancelExportTask');
        return this;
    }
    /**
     * Creates an ExportTask which allows you to efficiently export data from a Log Group to your Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateExportTask.html
     */
    toCreateExportTask() {
        this.add('logs:CreateExportTask');
        return this;
    }
    /**
     * Creates the log delivery
     *
     * Access Level: Write
     */
    toCreateLogDelivery() {
        this.add('logs:CreateLogDelivery');
        return this;
    }
    /**
     * Creates a new log group with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogGroup.html
     */
    toCreateLogGroup() {
        this.add('logs:CreateLogGroup');
        return this;
    }
    /**
     * Creates a new log stream with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogStream.html
     */
    toCreateLogStream() {
        this.add('logs:CreateLogStream');
        return this;
    }
    /**
     * Deletes the destination with the specified name and eventually disables all the subscription filters that publish to it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDestination.html
     */
    toDeleteDestination() {
        this.add('logs:DeleteDestination');
        return this;
    }
    /**
     * Deletes the log delivery information for specified log delivery
     *
     * Access Level: Write
     */
    toDeleteLogDelivery() {
        this.add('logs:DeleteLogDelivery');
        return this;
    }
    /**
     * Deletes the log group with the specified name and permanently deletes all the archived log events associated with it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogGroup.html
     */
    toDeleteLogGroup() {
        this.add('logs:DeleteLogGroup');
        return this;
    }
    /**
     * Deletes a log stream and permanently deletes all the archived log events associated with it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogStream.html
     */
    toDeleteLogStream() {
        this.add('logs:DeleteLogStream');
        return this;
    }
    /**
     * Deletes a metric filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteMetricFilter.html
     */
    toDeleteMetricFilter() {
        this.add('logs:DeleteMetricFilter');
        return this;
    }
    /**
     * Deletes a resource policy from this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        this.add('logs:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes the retention policy of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    toDeleteRetentionPolicy() {
        this.add('logs:DeleteRetentionPolicy');
        return this;
    }
    /**
     * Deletes a subscription filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteSubscriptionFilter.html
     */
    toDeleteSubscriptionFilter() {
        this.add('logs:DeleteSubscriptionFilter');
        return this;
    }
    /**
     * Returns all the destinations that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDestinations.html
     */
    toDescribeDestinations() {
        this.add('logs:DescribeDestinations');
        return this;
    }
    /**
     * Returns all the export tasks that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        this.add('logs:DescribeExportTasks');
        return this;
    }
    /**
     * Returns all the log groups that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogGroups.html
     */
    toDescribeLogGroups() {
        this.add('logs:DescribeLogGroups');
        return this;
    }
    /**
     * Returns all the log streams that are associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogStreams.html
     */
    toDescribeLogStreams() {
        this.add('logs:DescribeLogStreams');
        return this;
    }
    /**
     * Returns all the metrics filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeMetricFilters.html
     */
    toDescribeMetricFilters() {
        this.add('logs:DescribeMetricFilters');
        return this;
    }
    /**
     * Returns a list of CloudWatch Logs Insights queries that are scheduled, executing, or have been executed recently in this account. You can request all queries, or limit it to queries of a specific log group or queries with a certain status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeQueries.html
     */
    toDescribeQueries() {
        this.add('logs:DescribeQueries');
        return this;
    }
    /**
     * Return all the resource policies in this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeResourcePolicies.html
     */
    toDescribeResourcePolicies() {
        this.add('logs:DescribeResourcePolicies');
        return this;
    }
    /**
     * Returns all the subscription filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeSubscriptionFilters.html
     */
    toDescribeSubscriptionFilters() {
        this.add('logs:DescribeSubscriptionFilters');
        return this;
    }
    /**
     * Disassociates the associated AWS Key Management Service (AWS KMS) customer master key (CMK) from the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DisassociateKmsKey.html
     */
    toDisassociateKmsKey() {
        this.add('logs:DisassociateKmsKey');
        return this;
    }
    /**
     * Retrieves log events, optionally filtered by a filter pattern from the specified log group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_FilterLogEvents.html
     */
    toFilterLogEvents() {
        this.add('logs:FilterLogEvents');
        return this;
    }
    /**
     * Gets the log delivery information for specified log delivery
     *
     * Access Level: Read
     */
    toGetLogDelivery() {
        this.add('logs:GetLogDelivery');
        return this;
    }
    /**
     * Retrieves log events from the specified log stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogEvents.html
     */
    toGetLogEvents() {
        this.add('logs:GetLogEvents');
        return this;
    }
    /**
     * Returns a list of the fields that are included in log events in the specified log group, along with the percentage of log events that contain each field. The search is limited to a time period that you specify.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogGroupFields.html
     */
    toGetLogGroupFields() {
        this.add('logs:GetLogGroupFields');
        return this;
    }
    /**
     * Retrieves all the fields and values of a single log event. All fields are retrieved, even if the original query that produced the logRecordPointer retrieved only a subset of fields. Fields are returned as field name/field value pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogRecord.html
     */
    toGetLogRecord() {
        this.add('logs:GetLogRecord');
        return this;
    }
    /**
     * Returns the results from the specified query. If the query is in progress, partial results of that current execution are returned. Only the fields requested in the query are returned.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetQueryResults.html
     */
    toGetQueryResults() {
        this.add('logs:GetQueryResults');
        return this;
    }
    /**
     * Lists all the log deliveries for specified account and/or log source
     *
     * Access Level: List
     */
    toListLogDeliveries() {
        this.add('logs:ListLogDeliveries');
        return this;
    }
    /**
     * Lists the tags for the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListTagsLogGroup.html
     */
    toListTagsLogGroup() {
        this.add('logs:ListTagsLogGroup');
        return this;
    }
    /**
     * Creates or updates a Destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestination.html
     */
    toPutDestination() {
        this.add('logs:PutDestination');
        return this;
    }
    /**
     * Creates or updates an access policy associated with an existing Destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestinationPolicy.html
     */
    toPutDestinationPolicy() {
        this.add('logs:PutDestinationPolicy');
        return this;
    }
    /**
     * Uploads a batch of log events to the specified log stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutLogEvents.html
     */
    toPutLogEvents() {
        this.add('logs:PutLogEvents');
        return this;
    }
    /**
     * Creates or updates a metric filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutMetricFilter.html
     */
    toPutMetricFilter() {
        this.add('logs:PutMetricFilter');
        return this;
    }
    /**
     * Creates or updates a resource policy allowing other AWS services to put log events to this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        this.add('logs:PutResourcePolicy');
        return this;
    }
    /**
     * Sets the retention of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutRetentionPolicy.html
     */
    toPutRetentionPolicy() {
        this.add('logs:PutRetentionPolicy');
        return this;
    }
    /**
     * Creates or updates a subscription filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutSubscriptionFilter.html
     */
    toPutSubscriptionFilter() {
        this.add('logs:PutSubscriptionFilter');
        return this;
    }
    /**
     * Schedules a query of a log group using CloudWatch Logs Insights. You specify the log group and time range to query, and the query string to use.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StartQuery.html
     */
    toStartQuery() {
        this.add('logs:StartQuery');
        return this;
    }
    /**
     * Stops a CloudWatch Logs Insights query that is in progress. If the query has already ended, the operation returns an error indicating that the specified query is not running.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StopQuery.html
     */
    toStopQuery() {
        this.add('logs:StopQuery');
        return this;
    }
    /**
     * Adds or updates the specified tags for the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TagLogGroup.html
     */
    toTagLogGroup() {
        this.add('logs:TagLogGroup');
        return this;
    }
    /**
     * Tests the filter pattern of a metric filter against a sample of log event messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TestMetricFilter.html
     */
    toTestMetricFilter() {
        this.add('logs:TestMetricFilter');
        return this;
    }
    /**
     * Removes the specified tags from the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UntagLogGroup.html
     */
    toUntagLogGroup() {
        this.add('logs:UntagLogGroup');
        return this;
    }
    /**
     * Updates the log delivery information for specified log delivery
     *
     * Access Level: Write
     */
    toUpdateLogDelivery() {
        this.add('logs:UpdateLogDelivery');
        return this;
    }
    /**
     * Adds a resource of type log-group to the statement
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLogGroup(logGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}';
        arn = arn.replace('${LogGroupName}', logGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type log-stream to the statement
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param logStreamName - Identifier for the logStreamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLogStream(logGroupName, logStreamName, account, region, partition) {
        var arn = 'arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}:log-stream:${LogStreamName}';
        arn = arn.replace('${LogGroupName}', logGroupName);
        arn = arn.replace('${LogStreamName}', logStreamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Logs = Logs;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uY2xvdWR3YXRjaGxvZ3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbWF6b25jbG91ZHdhdGNobG9ncy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBb0U7QUFFcEU7Ozs7R0FJRztBQUNILE1BQWEsSUFBSyxTQUFRLHdCQUFlO0lBa1h2Qzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQXZYTixrQkFBYSxHQUFHLE1BQU0sQ0FBQztRQUNwQixlQUFVLEdBQVk7WUFDOUIsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSx1SEFBdUg7Z0JBQ3RJLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSw2REFBNkQ7Z0JBQzVFLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSw2R0FBNkc7Z0JBQzVILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSwwQkFBMEI7Z0JBQ3pDLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxpREFBaUQ7Z0JBQ2hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSxrREFBa0Q7Z0JBQ2pFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSx5SEFBeUg7Z0JBQ3hJLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxzSEFBc0g7Z0JBQ3JJLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw2RkFBNkY7Z0JBQzVHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSxvR0FBb0c7Z0JBQzNHLGFBQWEsRUFBRSw2Q0FBNkM7Z0JBQzVELGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSx5REFBeUQ7Z0JBQ3hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSx3R0FBd0c7Z0JBQy9HLGFBQWEsRUFBRSx1RUFBdUU7Z0JBQ3RGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSxvR0FBb0c7Z0JBQzNHLGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSx3RkFBd0Y7Z0JBQ3ZHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSw4RUFBOEU7Z0JBQzdGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSxpUEFBaVA7Z0JBQ2hRLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSx3R0FBd0c7Z0JBQy9HLGFBQWEsRUFBRSxtREFBbUQ7Z0JBQ2xFLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsNkJBQTZCLEVBQUU7Z0JBQzdCLEtBQUssRUFBRSwyR0FBMkc7Z0JBQ2xILGFBQWEsRUFBRSw4RUFBOEU7Z0JBQzdGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSwwSEFBMEg7Z0JBQ3pJLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw0RkFBNEY7Z0JBQzNHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSw4REFBOEQ7Z0JBQzdFLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSxvREFBb0Q7Z0JBQ25FLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSxvTkFBb047Z0JBQ25PLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSw0T0FBNE87Z0JBQzNQLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSx5TEFBeUw7Z0JBQ3hNLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSxzRUFBc0U7Z0JBQ3JGLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSw0Q0FBNEM7Z0JBQzNELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxrQ0FBa0M7Z0JBQ2pELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixrQkFBa0IsRUFBRTtvQkFDbEIsY0FBYztpQkFDZjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSxvR0FBb0c7Z0JBQzNHLGFBQWEsRUFBRSw2RUFBNkU7Z0JBQzVGLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSwyREFBMkQ7Z0JBQzFFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSxtRkFBbUY7Z0JBQ2xHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSxvR0FBb0c7Z0JBQ25ILGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSwrQ0FBK0M7Z0JBQzlELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSx5RkFBeUY7Z0JBQ3hHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixrQkFBa0IsRUFBRTtvQkFDbEIsY0FBYztpQkFDZjtnQkFDRCxlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSxrSkFBa0o7Z0JBQ2pLLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSxnTEFBZ0w7Z0JBQy9MLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSxnRUFBZ0U7Z0JBQy9FLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSxvRkFBb0Y7Z0JBQ25HLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSx5REFBeUQ7Z0JBQ3hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1NBQ0YsQ0FBQztRQUNRLGtCQUFhLEdBQWtCO1lBQ3ZDLFdBQVcsRUFBRTtnQkFDWCxNQUFNLEVBQUUsV0FBVztnQkFDbkIsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLHNFQUFzRTtnQkFDN0UsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLEtBQUssRUFBRSxFQUFFO2dCQUNULEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLFVBQVUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLElBQUksR0FBRyxHQUFHLHNFQUFzRSxDQUFDO1FBQ2pGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxXQUFXLENBQUMsWUFBb0IsRUFBRSxhQUFxQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25ILElBQUksR0FBRyxHQUFHLGtHQUFrRyxDQUFDO1FBQzdHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGtCQUFrQixFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3JELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUExNkJELG9CQTA2QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtsb2dzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jbG91ZHdhdGNobG9ncy5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIExvZ3MgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdsb2dzJztcbiAgcHJvdGVjdGVkIGFjdGlvbkxpc3Q6IEFjdGlvbnMgPSB7XG4gICAgXCJBc3NvY2lhdGVLbXNLZXlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzb2NpYXRlS21zS2V5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBc3NvY2lhdGVzIHRoZSBzcGVjaWZpZWQgQVdTIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEFXUyBLTVMpIGN1c3RvbWVyIG1hc3RlciBrZXkgKENNSykgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNhbmNlbEV4cG9ydFRhc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2FuY2VsRXhwb3J0VGFzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ2FuY2VscyBhbiBleHBvcnQgdGFzayBpZiBpdCBpcyBpbiBQRU5ESU5HIG9yIFJVTk5JTkcgc3RhdGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkNyZWF0ZUV4cG9ydFRhc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRXhwb3J0VGFzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhbiBFeHBvcnRUYXNrIHdoaWNoIGFsbG93cyB5b3UgdG8gZWZmaWNpZW50bHkgZXhwb3J0IGRhdGEgZnJvbSBhIExvZyBHcm91cCB0byB5b3VyIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUxvZ0RlbGl2ZXJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyB0aGUgbG9nIGRlbGl2ZXJ5XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJDcmVhdGVMb2dHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMb2dHcm91cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIG5ldyBsb2cgZ3JvdXAgd2l0aCB0aGUgc3BlY2lmaWVkIG5hbWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUxvZ1N0cmVhbVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMb2dTdHJlYW0uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBuZXcgbG9nIHN0cmVhbSB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlRGVzdGluYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRGVzdGluYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIGRlc3RpbmF0aW9uIHdpdGggdGhlIHNwZWNpZmllZCBuYW1lIGFuZCBldmVudHVhbGx5IGRpc2FibGVzIGFsbCB0aGUgc3Vic2NyaXB0aW9uIGZpbHRlcnMgdGhhdCBwdWJsaXNoIHRvIGl0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZWxldGVMb2dEZWxpdmVyeVwiOiB7XG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIGxvZyBkZWxpdmVyeSBpbmZvcm1hdGlvbiBmb3Igc3BlY2lmaWVkIGxvZyBkZWxpdmVyeVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlTG9nR3JvdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTG9nR3JvdXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIGxvZyBncm91cCB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZSBhbmQgcGVybWFuZW50bHkgZGVsZXRlcyBhbGwgdGhlIGFyY2hpdmVkIGxvZyBldmVudHMgYXNzb2NpYXRlZCB3aXRoIGl0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVMb2dTdHJlYW1cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTG9nU3RyZWFtLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgbG9nIHN0cmVhbSBhbmQgcGVybWFuZW50bHkgZGVsZXRlcyBhbGwgdGhlIGFyY2hpdmVkIGxvZyBldmVudHMgYXNzb2NpYXRlZCB3aXRoIGl0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLXN0cmVhbVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlTWV0cmljRmlsdGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZU1ldHJpY0ZpbHRlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIG1ldHJpYyBmaWx0ZXIgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVSZXNvdXJjZVBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVSZXNvdXJjZVBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHJlc291cmNlIHBvbGljeSBmcm9tIHRoaXMgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlUmV0ZW50aW9uUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJldGVudGlvblBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyB0aGUgcmV0ZW50aW9uIHBvbGljeSBvZiB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlU3Vic2NyaXB0aW9uRmlsdGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVN1YnNjcmlwdGlvbkZpbHRlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHN1YnNjcmlwdGlvbiBmaWx0ZXIgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZURlc3RpbmF0aW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURlc3RpbmF0aW9ucy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbGwgdGhlIGRlc3RpbmF0aW9ucyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIEFXUyBhY2NvdW50IG1ha2luZyB0aGUgcmVxdWVzdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZUV4cG9ydFRhc2tzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRXhwb3J0VGFza3MuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYWxsIHRoZSBleHBvcnQgdGFza3MgdGhhdCBhcmUgYXNzb2NpYXRlZCB3aXRoIHRoZSBBV1MgYWNjb3VudCBtYWtpbmcgdGhlIHJlcXVlc3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVMb2dHcm91cHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVMb2dHcm91cHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYWxsIHRoZSBsb2cgZ3JvdXBzIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQVdTIGFjY291bnQgbWFraW5nIHRoZSByZXF1ZXN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlTG9nU3RyZWFtc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUxvZ1N0cmVhbXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYWxsIHRoZSBsb2cgc3RyZWFtcyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVNZXRyaWNGaWx0ZXJzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTWV0cmljRmlsdGVycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbGwgdGhlIG1ldHJpY3MgZmlsdGVycyBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVRdWVyaWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUXVlcmllcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgQ2xvdWRXYXRjaCBMb2dzIEluc2lnaHRzIHF1ZXJpZXMgdGhhdCBhcmUgc2NoZWR1bGVkLCBleGVjdXRpbmcsIG9yIGhhdmUgYmVlbiBleGVjdXRlZCByZWNlbnRseSBpbiB0aGlzIGFjY291bnQuIFlvdSBjYW4gcmVxdWVzdCBhbGwgcXVlcmllcywgb3IgbGltaXQgaXQgdG8gcXVlcmllcyBvZiBhIHNwZWNpZmljIGxvZyBncm91cCBvciBxdWVyaWVzIHdpdGggYSBjZXJ0YWluIHN0YXR1cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVSZXNvdXJjZVBvbGljaWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUmVzb3VyY2VQb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJuIGFsbCB0aGUgcmVzb3VyY2UgcG9saWNpZXMgaW4gdGhpcyBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZVN1YnNjcmlwdGlvbkZpbHRlcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdWJzY3JpcHRpb25GaWx0ZXJzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFsbCB0aGUgc3Vic2NyaXB0aW9uIGZpbHRlcnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRpc2Fzc29jaWF0ZUttc0tleVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVLbXNLZXkuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRpc2Fzc29jaWF0ZXMgdGhlIGFzc29jaWF0ZWQgQVdTIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEFXUyBLTVMpIGN1c3RvbWVyIG1hc3RlciBrZXkgKENNSykgZnJvbSB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRmlsdGVyTG9nRXZlbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0ZpbHRlckxvZ0V2ZW50cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0cmlldmVzIGxvZyBldmVudHMsIG9wdGlvbmFsbHkgZmlsdGVyZWQgYnkgYSBmaWx0ZXIgcGF0dGVybiBmcm9tIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldExvZ0RlbGl2ZXJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyB0aGUgbG9nIGRlbGl2ZXJ5IGluZm9ybWF0aW9uIGZvciBzcGVjaWZpZWQgbG9nIGRlbGl2ZXJ5XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldExvZ0V2ZW50c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRMb2dFdmVudHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBsb2cgZXZlbnRzIGZyb20gdGhlIHNwZWNpZmllZCBsb2cgc3RyZWFtXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctc3RyZWFtXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRMb2dHcm91cEZpZWxkc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRMb2dHcm91cEZpZWxkcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgdGhlIGZpZWxkcyB0aGF0IGFyZSBpbmNsdWRlZCBpbiBsb2cgZXZlbnRzIGluIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwLCBhbG9uZyB3aXRoIHRoZSBwZXJjZW50YWdlIG9mIGxvZyBldmVudHMgdGhhdCBjb250YWluIGVhY2ggZmllbGQuIFRoZSBzZWFyY2ggaXMgbGltaXRlZCB0byBhIHRpbWUgcGVyaW9kIHRoYXQgeW91IHNwZWNpZnkuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldExvZ1JlY29yZFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRMb2dSZWNvcmQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBhbGwgdGhlIGZpZWxkcyBhbmQgdmFsdWVzIG9mIGEgc2luZ2xlIGxvZyBldmVudC4gQWxsIGZpZWxkcyBhcmUgcmV0cmlldmVkLCBldmVuIGlmIHRoZSBvcmlnaW5hbCBxdWVyeSB0aGF0IHByb2R1Y2VkIHRoZSBsb2dSZWNvcmRQb2ludGVyIHJldHJpZXZlZCBvbmx5IGEgc3Vic2V0IG9mIGZpZWxkcy4gRmllbGRzIGFyZSByZXR1cm5lZCBhcyBmaWVsZCBuYW1lL2ZpZWxkIHZhbHVlIHBhaXJzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRRdWVyeVJlc3VsdHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0UXVlcnlSZXN1bHRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSByZXN1bHRzIGZyb20gdGhlIHNwZWNpZmllZCBxdWVyeS4gSWYgdGhlIHF1ZXJ5IGlzIGluIHByb2dyZXNzLCBwYXJ0aWFsIHJlc3VsdHMgb2YgdGhhdCBjdXJyZW50IGV4ZWN1dGlvbiBhcmUgcmV0dXJuZWQuIE9ubHkgdGhlIGZpZWxkcyByZXF1ZXN0ZWQgaW4gdGhlIHF1ZXJ5IGFyZSByZXR1cm5lZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiTGlzdExvZ0RlbGl2ZXJpZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJMaXN0cyBhbGwgdGhlIGxvZyBkZWxpdmVyaWVzIGZvciBzcGVjaWZpZWQgYWNjb3VudCBhbmQvb3IgbG9nIHNvdXJjZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0VGFnc0xvZ0dyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzTG9nR3JvdXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3RzIHRoZSB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJQdXREZXN0aW5hdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXREZXN0aW5hdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvciB1cGRhdGVzIGEgRGVzdGluYXRpb25cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJkZXBlbmRlbnRBY3Rpb25zXCI6IFtcbiAgICAgICAgXCJpYW06UGFzc1JvbGVcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXREZXN0aW5hdGlvblBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXREZXN0aW5hdGlvblBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvciB1cGRhdGVzIGFuIGFjY2VzcyBwb2xpY3kgYXNzb2NpYXRlZCB3aXRoIGFuIGV4aXN0aW5nIERlc3RpbmF0aW9uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRMb2dFdmVudHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0TG9nRXZlbnRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGxvYWRzIGEgYmF0Y2ggb2YgbG9nIGV2ZW50cyB0byB0aGUgc3BlY2lmaWVkIGxvZyBzdHJlYW1cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctc3RyZWFtXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJQdXRNZXRyaWNGaWx0ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0TWV0cmljRmlsdGVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIG9yIHVwZGF0ZXMgYSBtZXRyaWMgZmlsdGVyIGFuZCBhc3NvY2lhdGVzIGl0IHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlB1dFJlc291cmNlUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFJlc291cmNlUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIG9yIHVwZGF0ZXMgYSByZXNvdXJjZSBwb2xpY3kgYWxsb3dpbmcgb3RoZXIgQVdTIHNlcnZpY2VzIHRvIHB1dCBsb2cgZXZlbnRzIHRvIHRoaXMgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiUHV0UmV0ZW50aW9uUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFJldGVudGlvblBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU2V0cyB0aGUgcmV0ZW50aW9uIG9mIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJQdXRTdWJzY3JpcHRpb25GaWx0ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0U3Vic2NyaXB0aW9uRmlsdGVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIG9yIHVwZGF0ZXMgYSBzdWJzY3JpcHRpb24gZmlsdGVyIGFuZCBhc3NvY2lhdGVzIGl0IHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJkZXBlbmRlbnRBY3Rpb25zXCI6IFtcbiAgICAgICAgXCJpYW06UGFzc1JvbGVcIlxuICAgICAgXSxcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTdGFydFF1ZXJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0UXVlcnkuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlNjaGVkdWxlcyBhIHF1ZXJ5IG9mIGEgbG9nIGdyb3VwIHVzaW5nIENsb3VkV2F0Y2ggTG9ncyBJbnNpZ2h0cy4gWW91IHNwZWNpZnkgdGhlIGxvZyBncm91cCBhbmQgdGltZSByYW5nZSB0byBxdWVyeSwgYW5kIHRoZSBxdWVyeSBzdHJpbmcgdG8gdXNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTdG9wUXVlcnlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcFF1ZXJ5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTdG9wcyBhIENsb3VkV2F0Y2ggTG9ncyBJbnNpZ2h0cyBxdWVyeSB0aGF0IGlzIGluIHByb2dyZXNzLiBJZiB0aGUgcXVlcnkgaGFzIGFscmVhZHkgZW5kZWQsIHRoZSBvcGVyYXRpb24gcmV0dXJucyBhbiBlcnJvciBpbmRpY2F0aW5nIHRoYXQgdGhlIHNwZWNpZmllZCBxdWVyeSBpcyBub3QgcnVubmluZy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiVGFnTG9nR3JvdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnTG9nR3JvdXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkFkZHMgb3IgdXBkYXRlcyB0aGUgc3BlY2lmaWVkIHRhZ3MgZm9yIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJUZXN0TWV0cmljRmlsdGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Rlc3RNZXRyaWNGaWx0ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlRlc3RzIHRoZSBmaWx0ZXIgcGF0dGVybiBvZiBhIG1ldHJpYyBmaWx0ZXIgYWdhaW5zdCBhIHNhbXBsZSBvZiBsb2cgZXZlbnQgbWVzc2FnZXNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiVW50YWdMb2dHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ0xvZ0dyb3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZW1vdmVzIHRoZSBzcGVjaWZpZWQgdGFncyBmcm9tIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVMb2dEZWxpdmVyeVwiOiB7XG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgdGhlIGxvZyBkZWxpdmVyeSBpbmZvcm1hdGlvbiBmb3Igc3BlY2lmaWVkIGxvZyBkZWxpdmVyeVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9XG4gIH07XG4gIHByb3RlY3RlZCByZXNvdXJjZVR5cGVzOiBSZXNvdXJjZVR5cGVzID0ge1xuICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImxvZy1ncm91cFwiLFxuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpsb2dzOiR7UmVnaW9ufToke0FjY291bnR9OmxvZy1ncm91cDoke0xvZ0dyb3VwTmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJsb2ctc3RyZWFtXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImxvZy1zdHJlYW1cIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bG9nczoke1JlZ2lvbn06JHtBY2NvdW50fTpsb2ctZ3JvdXA6JHtMb2dHcm91cE5hbWV9OmxvZy1zdHJlYW06JHtMb2dTdHJlYW1OYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2xvZ3NdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmNsb3Vkd2F0Y2hsb2dzLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZXMgdGhlIHNwZWNpZmllZCBBV1MgS2V5IE1hbmFnZW1lbnQgU2VydmljZSAoQVdTIEtNUykgY3VzdG9tZXIgbWFzdGVyIGtleSAoQ01LKSB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVLbXNLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXNzb2NpYXRlS21zS2V5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkFzc29jaWF0ZUttc0tleScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbmNlbHMgYW4gZXhwb3J0IHRhc2sgaWYgaXQgaXMgaW4gUEVORElORyBvciBSVU5OSU5HIHN0YXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NhbmNlbEV4cG9ydFRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2FuY2VsRXhwb3J0VGFzaygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpDYW5jZWxFeHBvcnRUYXNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBFeHBvcnRUYXNrIHdoaWNoIGFsbG93cyB5b3UgdG8gZWZmaWNpZW50bHkgZXhwb3J0IGRhdGEgZnJvbSBhIExvZyBHcm91cCB0byB5b3VyIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRXhwb3J0VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVFeHBvcnRUYXNrKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkNyZWF0ZUV4cG9ydFRhc2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIHRoZSBsb2cgZGVsaXZlcnlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTG9nRGVsaXZlcnkoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6Q3JlYXRlTG9nRGVsaXZlcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbmV3IGxvZyBncm91cCB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMb2dHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVMb2dHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpDcmVhdGVMb2dHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgbG9nIHN0cmVhbSB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMb2dTdHJlYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTG9nU3RyZWFtKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIGRlc3RpbmF0aW9uIHdpdGggdGhlIHNwZWNpZmllZCBuYW1lIGFuZCBldmVudHVhbGx5IGRpc2FibGVzIGFsbCB0aGUgc3Vic2NyaXB0aW9uIGZpbHRlcnMgdGhhdCBwdWJsaXNoIHRvIGl0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURlc3RpbmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURlc3RpbmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlbGV0ZURlc3RpbmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgbG9nIGRlbGl2ZXJ5IGluZm9ybWF0aW9uIGZvciBzcGVjaWZpZWQgbG9nIGRlbGl2ZXJ5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUxvZ0RlbGl2ZXJ5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlbGV0ZUxvZ0RlbGl2ZXJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgbG9nIGdyb3VwIHdpdGggdGhlIHNwZWNpZmllZCBuYW1lIGFuZCBwZXJtYW5lbnRseSBkZWxldGVzIGFsbCB0aGUgYXJjaGl2ZWQgbG9nIGV2ZW50cyBhc3NvY2lhdGVkIHdpdGggaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTG9nR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTG9nR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVsZXRlTG9nR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgbG9nIHN0cmVhbSBhbmQgcGVybWFuZW50bHkgZGVsZXRlcyBhbGwgdGhlIGFyY2hpdmVkIGxvZyBldmVudHMgYXNzb2NpYXRlZCB3aXRoIGl0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUxvZ1N0cmVhbS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVMb2dTdHJlYW0oKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVsZXRlTG9nU3RyZWFtJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIG1ldHJpYyBmaWx0ZXIgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZU1ldHJpY0ZpbHRlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVNZXRyaWNGaWx0ZXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVsZXRlTWV0cmljRmlsdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHJlc291cmNlIHBvbGljeSBmcm9tIHRoaXMgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVSZXNvdXJjZVBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVSZXNvdXJjZVBvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZWxldGVSZXNvdXJjZVBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIHJldGVudGlvbiBwb2xpY3kgb2YgdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUmV0ZW50aW9uUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJldGVudGlvblBvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZWxldGVSZXRlbnRpb25Qb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3Vic2NyaXB0aW9uIGZpbHRlciBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlU3Vic2NyaXB0aW9uRmlsdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVN1YnNjcmlwdGlvbkZpbHRlcigpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZWxldGVTdWJzY3JpcHRpb25GaWx0ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFsbCB0aGUgZGVzdGluYXRpb25zIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQVdTIGFjY291bnQgbWFraW5nIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEZXN0aW5hdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEZXN0aW5hdGlvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVzY3JpYmVEZXN0aW5hdGlvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFsbCB0aGUgZXhwb3J0IHRhc2tzIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQVdTIGFjY291bnQgbWFraW5nIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFeHBvcnRUYXNrcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUV4cG9ydFRhc2tzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlc2NyaWJlRXhwb3J0VGFza3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFsbCB0aGUgbG9nIGdyb3VwcyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIEFXUyBhY2NvdW50IG1ha2luZyB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTG9nR3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTG9nR3JvdXBzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlc2NyaWJlTG9nR3JvdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbGwgdGhlIGxvZyBzdHJlYW1zIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTG9nU3RyZWFtcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUxvZ1N0cmVhbXMoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVzY3JpYmVMb2dTdHJlYW1zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbGwgdGhlIG1ldHJpY3MgZmlsdGVycyBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU1ldHJpY0ZpbHRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVNZXRyaWNGaWx0ZXJzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlc2NyaWJlTWV0cmljRmlsdGVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIENsb3VkV2F0Y2ggTG9ncyBJbnNpZ2h0cyBxdWVyaWVzIHRoYXQgYXJlIHNjaGVkdWxlZCwgZXhlY3V0aW5nLCBvciBoYXZlIGJlZW4gZXhlY3V0ZWQgcmVjZW50bHkgaW4gdGhpcyBhY2NvdW50LiBZb3UgY2FuIHJlcXVlc3QgYWxsIHF1ZXJpZXMsIG9yIGxpbWl0IGl0IHRvIHF1ZXJpZXMgb2YgYSBzcGVjaWZpYyBsb2cgZ3JvdXAgb3IgcXVlcmllcyB3aXRoIGEgY2VydGFpbiBzdGF0dXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVRdWVyaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUXVlcmllcygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZXNjcmliZVF1ZXJpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYWxsIHRoZSByZXNvdXJjZSBwb2xpY2llcyBpbiB0aGlzIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZXNvdXJjZVBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUmVzb3VyY2VQb2xpY2llcygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZXNjcmliZVJlc291cmNlUG9saWNpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFsbCB0aGUgc3Vic2NyaXB0aW9uIGZpbHRlcnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdWJzY3JpcHRpb25GaWx0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU3Vic2NyaXB0aW9uRmlsdGVycygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZXNjcmliZVN1YnNjcmlwdGlvbkZpbHRlcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEaXNhc3NvY2lhdGVzIHRoZSBhc3NvY2lhdGVkIEFXUyBLZXkgTWFuYWdlbWVudCBTZXJ2aWNlIChBV1MgS01TKSBjdXN0b21lciBtYXN0ZXIga2V5IChDTUspIGZyb20gdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGlzYXNzb2NpYXRlS21zS2V5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZUttc0tleSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEaXNhc3NvY2lhdGVLbXNLZXknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgbG9nIGV2ZW50cywgb3B0aW9uYWxseSBmaWx0ZXJlZCBieSBhIGZpbHRlciBwYXR0ZXJuIGZyb20gdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9GaWx0ZXJMb2dFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRmlsdGVyTG9nRXZlbnRzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkZpbHRlckxvZ0V2ZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgdGhlIGxvZyBkZWxpdmVyeSBpbmZvcm1hdGlvbiBmb3Igc3BlY2lmaWVkIGxvZyBkZWxpdmVyeVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICovXG4gIHB1YmxpYyB0b0dldExvZ0RlbGl2ZXJ5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkdldExvZ0RlbGl2ZXJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGxvZyBldmVudHMgZnJvbSB0aGUgc3BlY2lmaWVkIGxvZyBzdHJlYW1cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRMb2dFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TG9nRXZlbnRzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkdldExvZ0V2ZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIHRoZSBmaWVsZHMgdGhhdCBhcmUgaW5jbHVkZWQgaW4gbG9nIGV2ZW50cyBpbiB0aGUgc3BlY2lmaWVkIGxvZyBncm91cCwgYWxvbmcgd2l0aCB0aGUgcGVyY2VudGFnZSBvZiBsb2cgZXZlbnRzIHRoYXQgY29udGFpbiBlYWNoIGZpZWxkLiBUaGUgc2VhcmNoIGlzIGxpbWl0ZWQgdG8gYSB0aW1lIHBlcmlvZCB0aGF0IHlvdSBzcGVjaWZ5LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldExvZ0dyb3VwRmllbGRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExvZ0dyb3VwRmllbGRzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkdldExvZ0dyb3VwRmllbGRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGFsbCB0aGUgZmllbGRzIGFuZCB2YWx1ZXMgb2YgYSBzaW5nbGUgbG9nIGV2ZW50LiBBbGwgZmllbGRzIGFyZSByZXRyaWV2ZWQsIGV2ZW4gaWYgdGhlIG9yaWdpbmFsIHF1ZXJ5IHRoYXQgcHJvZHVjZWQgdGhlIGxvZ1JlY29yZFBvaW50ZXIgcmV0cmlldmVkIG9ubHkgYSBzdWJzZXQgb2YgZmllbGRzLiBGaWVsZHMgYXJlIHJldHVybmVkIGFzIGZpZWxkIG5hbWUvZmllbGQgdmFsdWUgcGFpcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0TG9nUmVjb3JkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExvZ1JlY29yZCgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpHZXRMb2dSZWNvcmQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSByZXN1bHRzIGZyb20gdGhlIHNwZWNpZmllZCBxdWVyeS4gSWYgdGhlIHF1ZXJ5IGlzIGluIHByb2dyZXNzLCBwYXJ0aWFsIHJlc3VsdHMgb2YgdGhhdCBjdXJyZW50IGV4ZWN1dGlvbiBhcmUgcmV0dXJuZWQuIE9ubHkgdGhlIGZpZWxkcyByZXF1ZXN0ZWQgaW4gdGhlIHF1ZXJ5IGFyZSByZXR1cm5lZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRRdWVyeVJlc3VsdHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UXVlcnlSZXN1bHRzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkdldFF1ZXJ5UmVzdWx0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGFsbCB0aGUgbG9nIGRlbGl2ZXJpZXMgZm9yIHNwZWNpZmllZCBhY2NvdW50IGFuZC9vciBsb2cgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKi9cbiAgcHVibGljIHRvTGlzdExvZ0RlbGl2ZXJpZXMoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6TGlzdExvZ0RlbGl2ZXJpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgdGFncyBmb3IgdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0xvZ0dyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzTG9nR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6TGlzdFRhZ3NMb2dHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb3IgdXBkYXRlcyBhIERlc3RpbmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXREZXN0aW5hdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXREZXN0aW5hdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpQdXREZXN0aW5hdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb3IgdXBkYXRlcyBhbiBhY2Nlc3MgcG9saWN5IGFzc29jaWF0ZWQgd2l0aCBhbiBleGlzdGluZyBEZXN0aW5hdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXREZXN0aW5hdGlvblBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXREZXN0aW5hdGlvblBvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpQdXREZXN0aW5hdGlvblBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwbG9hZHMgYSBiYXRjaCBvZiBsb2cgZXZlbnRzIHRvIHRoZSBzcGVjaWZpZWQgbG9nIHN0cmVhbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRMb2dFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0TG9nRXZlbnRzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlB1dExvZ0V2ZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb3IgdXBkYXRlcyBhIG1ldHJpYyBmaWx0ZXIgYW5kIGFzc29jaWF0ZXMgaXQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRNZXRyaWNGaWx0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0TWV0cmljRmlsdGVyKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlB1dE1ldHJpY0ZpbHRlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb3IgdXBkYXRlcyBhIHJlc291cmNlIHBvbGljeSBhbGxvd2luZyBvdGhlciBBV1Mgc2VydmljZXMgdG8gcHV0IGxvZyBldmVudHMgdG8gdGhpcyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFJlc291cmNlUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dFJlc291cmNlUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlB1dFJlc291cmNlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2V0cyB0aGUgcmV0ZW50aW9uIG9mIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFJldGVudGlvblBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRSZXRlbnRpb25Qb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6UHV0UmV0ZW50aW9uUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBvciB1cGRhdGVzIGEgc3Vic2NyaXB0aW9uIGZpbHRlciBhbmQgYXNzb2NpYXRlcyBpdCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRTdWJzY3JpcHRpb25GaWx0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0U3Vic2NyaXB0aW9uRmlsdGVyKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlB1dFN1YnNjcmlwdGlvbkZpbHRlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNjaGVkdWxlcyBhIHF1ZXJ5IG9mIGEgbG9nIGdyb3VwIHVzaW5nIENsb3VkV2F0Y2ggTG9ncyBJbnNpZ2h0cy4gWW91IHNwZWNpZnkgdGhlIGxvZyBncm91cCBhbmQgdGltZSByYW5nZSB0byBxdWVyeSwgYW5kIHRoZSBxdWVyeSBzdHJpbmcgdG8gdXNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0UXVlcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRRdWVyeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpTdGFydFF1ZXJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3RvcHMgYSBDbG91ZFdhdGNoIExvZ3MgSW5zaWdodHMgcXVlcnkgdGhhdCBpcyBpbiBwcm9ncmVzcy4gSWYgdGhlIHF1ZXJ5IGhhcyBhbHJlYWR5IGVuZGVkLCB0aGUgb3BlcmF0aW9uIHJldHVybnMgYW4gZXJyb3IgaW5kaWNhdGluZyB0aGF0IHRoZSBzcGVjaWZpZWQgcXVlcnkgaXMgbm90IHJ1bm5pbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcFF1ZXJ5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BRdWVyeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpTdG9wUXVlcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIG9yIHVwZGF0ZXMgdGhlIHNwZWNpZmllZCB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdMb2dHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdMb2dHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpUYWdMb2dHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRlc3RzIHRoZSBmaWx0ZXIgcGF0dGVybiBvZiBhIG1ldHJpYyBmaWx0ZXIgYWdhaW5zdCBhIHNhbXBsZSBvZiBsb2cgZXZlbnQgbWVzc2FnZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UZXN0TWV0cmljRmlsdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Rlc3RNZXRyaWNGaWx0ZXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6VGVzdE1ldHJpY0ZpbHRlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgdGhlIHNwZWNpZmllZCB0YWdzIGZyb20gdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdMb2dHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ0xvZ0dyb3VwKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlVudGFnTG9nR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBsb2cgZGVsaXZlcnkgaW5mb3JtYXRpb24gZm9yIHNwZWNpZmllZCBsb2cgZGVsaXZlcnlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTG9nRGVsaXZlcnkoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6VXBkYXRlTG9nRGVsaXZlcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBsb2ctZ3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gbG9nR3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGxvZ0dyb3VwTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkxvZ0dyb3VwKGxvZ0dyb3VwTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsb2dzOiR7UmVnaW9ufToke0FjY291bnR9OmxvZy1ncm91cDoke0xvZ0dyb3VwTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0xvZ0dyb3VwTmFtZX0nLCBsb2dHcm91cE5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGxvZy1zdHJlYW0gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gbG9nR3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGxvZ0dyb3VwTmFtZS5cbiAgICogQHBhcmFtIGxvZ1N0cmVhbU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgbG9nU3RyZWFtTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkxvZ1N0cmVhbShsb2dHcm91cE5hbWU6IHN0cmluZywgbG9nU3RyZWFtTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsb2dzOiR7UmVnaW9ufToke0FjY291bnR9OmxvZy1ncm91cDoke0xvZ0dyb3VwTmFtZX06bG9nLXN0cmVhbToke0xvZ1N0cmVhbU5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtMb2dHcm91cE5hbWV9JywgbG9nR3JvdXBOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtMb2dTdHJlYW1OYW1lfScsIGxvZ1N0cmVhbU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG59XG4iXX0=