"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ec2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ec2';
        this.actionList = {
            "AcceptReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html",
                "description": "Grants permission to accept a Convertible Reserved Instance exchange quote",
                "accessLevel": "Write"
            },
            "AcceptTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to accept a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html",
                "description": "Grants permission to accept a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html",
                "description": "Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html",
                "description": "Grants permission to accept a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AdvertiseByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html",
                "description": "Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            },
            "AllocateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html",
                "description": "Grants permission to allocate an Elastic IP address (EIP) to your account",
                "accessLevel": "Write"
            },
            "AllocateHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html",
                "description": "Grants permission to allocate a Dedicated Host to your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dedicated-host": {
                        "required": true
                    }
                }
            },
            "ApplySecurityGroupsToClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html",
                "description": "Grants permission to apply a security group to the association between a Client VPN endpoint and a target network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html",
                "description": "Grants permission to assign one or more IPv6 addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html",
                "description": "Grants permission to assign one or more secondary private IP addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html",
                "description": "Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface",
                "accessLevel": "Write"
            },
            "AssociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html",
                "description": "Grants permission to associate a target network with a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html",
                "description": "Grants permission to associate or disassociate a set of DHCP options with a VPC",
                "accessLevel": "Write"
            },
            "AssociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html",
                "description": "Grants permission to associate an IAM instance profile with a running or stopped instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AssociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html",
                "description": "Grants permission to associate a subnet or gateway with a route table",
                "accessLevel": "Write"
            },
            "AssociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a subnet",
                "accessLevel": "Write"
            },
            "AssociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html",
                "description": "Grants permission to associate an attachment with a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a VPC",
                "accessLevel": "Write"
            },
            "AttachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html",
                "description": "Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AttachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html",
                "description": "Grants permission to attach an internet gateway to a VPC",
                "accessLevel": "Write"
            },
            "AttachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html",
                "description": "Grants permission to attach a network interface to an instance",
                "accessLevel": "Write"
            },
            "AttachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html",
                "description": "Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "AttachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html",
                "description": "Grants permission to attach a virtual private gateway to a VPC",
                "accessLevel": "Write"
            },
            "AuthorizeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html",
                "description": "Grants permission to add an inbound authorization rule to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html",
                "description": "Grants permission to add one or more outbound rules to a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html",
                "description": "Grants permission to add one or more inbound rules to a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "BundleInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html",
                "description": "Grants permission to bundle an instance store-backed Windows instance",
                "accessLevel": "Write"
            },
            "CancelBundleTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html",
                "description": "Grants permission to cancel a bundling operation",
                "accessLevel": "Write"
            },
            "CancelCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html",
                "description": "Grants permission to cancel a Capacity Reservation and release the reserved capacity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CancelConversionTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html",
                "description": "Grants permission to cancel an active conversion task",
                "accessLevel": "Write"
            },
            "CancelExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html",
                "description": "Grants permission to cancel an active export task",
                "accessLevel": "Write"
            },
            "CancelImportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html",
                "description": "Grants permission to cancel an in-process import virtual machine or import snapshot task",
                "accessLevel": "Write"
            },
            "CancelReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html",
                "description": "Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CancelSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html",
                "description": "Grants permission to cancel one or more Spot Fleet requests",
                "accessLevel": "Write"
            },
            "CancelSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html",
                "description": "Grants permission to cancel one or more Spot Instance requests",
                "accessLevel": "Write"
            },
            "ConfirmProductInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html",
                "description": "Grants permission to determine whether an owned product code is associated with an instance",
                "accessLevel": "Write"
            },
            "CopyFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html",
                "description": "Grants permission to copy a source Amazon FPGA image (AFI) to the current Region",
                "accessLevel": "Write"
            },
            "CopyImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html",
                "description": "Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region",
                "accessLevel": "Write"
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html",
                "description": "Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html",
                "description": "Grants permission to create a Capacity Reservation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateCarrierGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCarrierGateway.html",
                "description": "Grants permission to create a carrier gateway and provides CSP connectivity to VPC customers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "carrier-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html",
                "description": "Grants permission to create a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html",
                "description": "Grants permission to add a network route to a Client VPN endpoint's route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html",
                "description": "Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device",
                "accessLevel": "Write"
            },
            "CreateDefaultSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html",
                "description": "Grants permission to create a default subnet in a specified Availability Zone in a default VPC",
                "accessLevel": "Write"
            },
            "CreateDefaultVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html",
                "description": "Grants permission to create a default VPC with a default subnet in each Availability Zone",
                "accessLevel": "Write"
            },
            "CreateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html",
                "description": "Grants permission to create a set of DHCP options for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dhcp-options": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html",
                "description": "Grants permission to create an egress-only internet gateway for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "egress-only-internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html",
                "description": "Grants permission to launch an EC2 Fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html",
                "description": "Grants permission to create one or more flow logs to capture IP traffic for a network interface",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "vpc-flow-log": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "CreateFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html",
                "description": "Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fpga-image": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html",
                "description": "Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance",
                "accessLevel": "Write"
            },
            "CreateInstanceExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html",
                "description": "Grants permission to export a running or stopped instance to an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "export-instance-task": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html",
                "description": "Grants permission to create an internet gateway for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html",
                "description": "Grants permission to create a 2048-bit RSA key pair",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key-pair": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html",
                "description": "Grants permission to create a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateLaunchTemplateVersion": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html",
                "description": "Grants permission to create a new version of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html",
                "description": "Grants permission to create a static route for a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to associate a VPC with a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "CreateManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateManagedPrefixList.html",
                "description": "Grants permission to create a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html",
                "description": "Grants permission to create a NAT gateway in a subnet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "natgateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html",
                "description": "Grants permission to create a network ACL in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html",
                "description": "Grants permission to create a numbered entry (a rule) in a network ACL",
                "accessLevel": "Write"
            },
            "CreateNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html",
                "description": "Grants permission to create a network interface in a subnet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html",
                "description": "Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AuthorizedUser",
                            "ec2:AvailabilityZone",
                            "ec2:Permission",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AuthorizedService"
                        ]
                    }
                }
            },
            "CreatePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html",
                "description": "Grants permission to create a placement group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "placement-group": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html",
                "description": "Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CreateRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html",
                "description": "Grants permission to create a route in a VPC route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "CreateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html",
                "description": "Grants permission to create a route table for a VPC",
                "accessLevel": "Write"
            },
            "CreateSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html",
                "description": "Grants permission to create a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html",
                "description": "Grants permission to create a snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html",
                "description": "Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html",
                "description": "Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs",
                "accessLevel": "Write"
            },
            "CreateSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html",
                "description": "Grants permission to create a subnet in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html",
                "description": "Grants permission to add or overwrite one or more tags for Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "egress-only-internet-gateway": {
                        "required": false
                    },
                    "export-instance-task": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:ImageType",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "natgateway": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:InstanceType",
                            "ec2:Region",
                            "ec2:ReservedInstancesOfferingType",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "ec2:CreateAction"
                ]
            },
            "CreateTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html",
                "description": "Grants permission to create a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html",
                "description": "Grants permission to create a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html",
                "description": "Grants permission to create a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html",
                "description": "Grants permission to create a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html",
                "description": "Grants permission to create a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to create a multicast domain for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html",
                "description": "Grants permission to create a static route for a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html",
                "description": "Grants permission to create a route table for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html",
                "description": "Grants permission to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html",
                "description": "Grants permission to create an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html",
                "description": "Grants permission to create a VPC with a specified CIDR block",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html",
                "description": "Grants permission to create a VPC endpoint for an AWS service",
                "accessLevel": "Write",
                "dependentActions": [
                    "route53:AssociateVPCWithHostedZone"
                ],
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServiceName",
                            "ec2:VpceServiceOwner"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html",
                "description": "Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "CreateVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName"
                        ]
                    }
                }
            },
            "CreateVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html",
                "description": "Grants permission to request a VPC peering connection between two VPCs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc"
                        ]
                    }
                }
            },
            "CreateVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html",
                "description": "Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:GatewayType",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "CreateVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html",
                "description": "Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "CreateVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html",
                "description": "Grants permission to create a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeleteCarrierGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCarrierGateway.html",
                "description": "Grants permission to delete a carrier gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "carrier-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html",
                "description": "Grants permission to delete a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html",
                "description": "Grants permission to delete a route from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html",
                "description": "Grants permission to delete a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "customer-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html",
                "description": "Grants permission to delete a set of DHCP options",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dhcp-options": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html",
                "description": "Grants permission to delete an egress-only internet gateway",
                "accessLevel": "Write"
            },
            "DeleteFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html",
                "description": "Grants permission to delete one or more EC2 Fleets",
                "accessLevel": "Write"
            },
            "DeleteFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html",
                "description": "Grants permission to delete one or more flow logs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-flow-log": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html",
                "description": "Grants permission to delete an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "DeleteInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html",
                "description": "Grants permission to delete an internet gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html",
                "description": "Grants permission to delete a key pair by removing the public key from Amazon EC2",
                "accessLevel": "Write"
            },
            "DeleteLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html",
                "description": "Grants permission to delete a launch template and its associated versions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html",
                "description": "Grants permission to delete one or more versions of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html",
                "description": "Grants permission to delete a route from a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to delete an association between a VPC and local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table-vpc-association": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteManagedPrefixList.html",
                "description": "Grants permission to delete a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html",
                "description": "Grants permission to delete a NAT gateway",
                "accessLevel": "Write"
            },
            "DeleteNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html",
                "description": "Grants permission to delete a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html",
                "description": "Grants permission to delete an inbound or outbound entry (rule) from a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html",
                "description": "Grants permission to delete a detached network interface",
                "accessLevel": "Write"
            },
            "DeleteNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html",
                "description": "Grants permission to delete a permission that is associated with a network interface",
                "accessLevel": "Permissions management"
            },
            "DeletePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html",
                "description": "Grants permission to delete a placement group",
                "accessLevel": "Write"
            },
            "DeleteRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html",
                "description": "Grants permission to delete a route from a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html",
                "description": "Grants permission to delete a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html",
                "description": "Grants permission to delete a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html",
                "description": "Grants permission to delete a snapshot of an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "DeleteSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html",
                "description": "Grants permission to delete a data feed for Spot Instances",
                "accessLevel": "Write"
            },
            "DeleteSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html",
                "description": "Grants permission to delete a subnet",
                "accessLevel": "Write"
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html",
                "description": "Grants permission to delete one or more tags from Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "egress-only-internet-gateway": {
                        "required": false
                    },
                    "export-instance-task": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "natgateway": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html",
                "description": "Grants permission to delete a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html",
                "description": "Grants permission to delete a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html",
                "description": "Grants permission to delete a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html",
                "description": "Grants permission to delete a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html",
                "description": "Grants permission to delete a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html",
                "description": "Grants permissions to delete a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to delete a peering attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html",
                "description": "Grants permission to delete a route from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html",
                "description": "Grants permission to delete a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html",
                "description": "Grants permission to delete a VPC attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html",
                "description": "Grants permission to delete an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DeleteVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html",
                "description": "Grants permission to delete a VPC",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to delete one or more VPC endpoint connection notifications",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to delete one or more VPC endpoint service configurations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html",
                "description": "Grants permission to delete one or more VPC endpoints",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html",
                "description": "Grants permission to delete a VPC peering connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html",
                "description": "Grants permission to delete a VPN connection",
                "accessLevel": "Write"
            },
            "DeleteVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html",
                "description": "Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "DeleteVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html",
                "description": "Grants permission to delete a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeprovisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html",
                "description": "Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool",
                "accessLevel": "Write"
            },
            "DeregisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html",
                "description": "Grants permission to deregister an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "DeregisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "DeregisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeregisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Grants permission to describe the attributes of the AWS account",
                "accessLevel": "List"
            },
            "DescribeAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html",
                "description": "Grants permission to describe one or more Elastic IP addresses",
                "accessLevel": "List"
            },
            "DescribeAggregateIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html",
                "description": "Grants permission to describe the longer ID format settings for all resource types",
                "accessLevel": "List"
            },
            "DescribeAvailabilityZones": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html",
                "description": "Grants permission to describe one or more of the Availability Zones that are available to you",
                "accessLevel": "List"
            },
            "DescribeBundleTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html",
                "description": "Grants permission to describe one or more bundling tasks",
                "accessLevel": "List"
            },
            "DescribeByoipCidrs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html",
                "description": "Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)",
                "accessLevel": "List"
            },
            "DescribeCapacityReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html",
                "description": "Grants permission to describe one or more Capacity Reservations",
                "accessLevel": "List"
            },
            "DescribeCarrierGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCarrierGateways.html",
                "description": "Grants permission to describe one or more Carrier Gateways",
                "accessLevel": "List"
            },
            "DescribeClassicLinkInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html",
                "description": "Grants permission to describe one or more linked EC2-Classic instances",
                "accessLevel": "List"
            },
            "DescribeClientVpnAuthorizationRules": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html",
                "description": "Grants permission to describe the authorization rules for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html",
                "description": "Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html",
                "description": "Grants permission to describe one or more Client VPN endpoints",
                "accessLevel": "List"
            },
            "DescribeClientVpnRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html",
                "description": "Grants permission to describe the routes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnTargetNetworks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html",
                "description": "Grants permission to describe the target networks that are associated with a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeCoipPools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html",
                "description": "Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools",
                "accessLevel": "List"
            },
            "DescribeConversionTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html",
                "description": "Grants permission to describe one or more conversion tasks",
                "accessLevel": "List"
            },
            "DescribeCustomerGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html",
                "description": "Grants permission to describe one or more customer gateways",
                "accessLevel": "List"
            },
            "DescribeDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html",
                "description": "Grants permission to describe one or more DHCP options sets",
                "accessLevel": "List"
            },
            "DescribeEgressOnlyInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html",
                "description": "Grants permission to describe one or more egress-only internet gateways",
                "accessLevel": "List"
            },
            "DescribeElasticGpus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html",
                "description": "Grants permission to describe an Elastic Graphics accelerator that is associated with an instance",
                "accessLevel": "Read"
            },
            "DescribeExportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html",
                "description": "Grants permission to describe one or more export image tasks",
                "accessLevel": "List"
            },
            "DescribeExportTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html",
                "description": "Grants permission to describe one or more export instance tasks",
                "accessLevel": "List"
            },
            "DescribeFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html",
                "description": "Grants permission to describe the state of fast snapshot restores for snapshots",
                "accessLevel": "Read"
            },
            "DescribeFleetHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html",
                "description": "Grants permission to describe the events for an EC2 Fleet during a specified time",
                "accessLevel": "List"
            },
            "DescribeFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html",
                "description": "Grants permission to describe the running instances for an EC2 Fleet",
                "accessLevel": "List"
            },
            "DescribeFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html",
                "description": "Grants permission to describe one or more EC2 Fleets",
                "accessLevel": "List"
            },
            "DescribeFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html",
                "description": "Grants permission to describe one or more flow logs",
                "accessLevel": "List"
            },
            "DescribeFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html",
                "description": "Grants permission to describe the attributes of an Amazon FPGA Image (AFI)",
                "accessLevel": "List"
            },
            "DescribeFpgaImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html",
                "description": "Grants permission to describe one or more Amazon FPGA Images (AFIs)",
                "accessLevel": "List"
            },
            "DescribeHostReservationOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are available to purchase",
                "accessLevel": "List"
            },
            "DescribeHostReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account",
                "accessLevel": "List"
            },
            "DescribeHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html",
                "description": "Grants permission to describe one or more Dedicated Hosts",
                "accessLevel": "List"
            },
            "DescribeIamInstanceProfileAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html",
                "description": "Grants permission to describe the IAM instance profile associations",
                "accessLevel": "List"
            },
            "DescribeIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources",
                "accessLevel": "List"
            },
            "DescribeIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user",
                "accessLevel": "List"
            },
            "DescribeImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html",
                "description": "Grants permission to describe an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "List"
            },
            "DescribeImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html",
                "description": "Grants permission to describe one or more images (AMIs, AKIs, and ARIs)",
                "accessLevel": "List"
            },
            "DescribeImportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html",
                "description": "Grants permission to describe import virtual machine or import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeImportSnapshotTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html",
                "description": "Grants permission to describe import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html",
                "description": "Grants permission to describe the attributes of an instance",
                "accessLevel": "List"
            },
            "DescribeInstanceCreditSpecifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html",
                "description": "Grants permission to describe the credit option for CPU usage of one or more burstable performance instances",
                "accessLevel": "List"
            },
            "DescribeInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html",
                "description": "Grants permission to describe the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "List"
            },
            "DescribeInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html",
                "description": "Grants permission to describe the status of one or more instances",
                "accessLevel": "List"
            },
            "DescribeInstanceTypeOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html",
                "description": "Grants permission to describe the set of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstanceTypes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html",
                "description": "Grants permission to describe the details of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html",
                "description": "Grants permission to describe one or more instances",
                "accessLevel": "List"
            },
            "DescribeInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html",
                "description": "Grants permission to describe one or more internet gateways",
                "accessLevel": "List"
            },
            "DescribeKeyPairs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html",
                "description": "Grants permission to describe one or more key pairs",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html",
                "description": "Grants permission to describe one or more launch template versions",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplates": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html",
                "description": "Grants permission to describe one or more launch templates",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html",
                "description": "Grants permission to describe the associations between virtual interface groups and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVpcAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html",
                "description": "Grants permission to describe an association between VPCs and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html",
                "description": "Grants permission to describe one or more local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaceGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html",
                "description": "Grants permission to describe local gateway virtual interface groups",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html",
                "description": "Grants permission to describe local gateway virtual interfaces",
                "accessLevel": "List"
            },
            "DescribeLocalGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html",
                "description": "Grants permission to describe one or more local gateways",
                "accessLevel": "List"
            },
            "DescribeManagedPrefixLists": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeManagedPrefixLists.html",
                "description": "Grants permission to describe your managed prefix lists and any AWS-managed prefix lists",
                "accessLevel": "List"
            },
            "DescribeMovingAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html",
                "description": "Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform",
                "accessLevel": "List"
            },
            "DescribeNatGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html",
                "description": "Grants permission to describe one or more NAT gateways",
                "accessLevel": "List"
            },
            "DescribeNetworkAcls": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html",
                "description": "Grants permission to describe one or more network ACLs",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html",
                "description": "Grants permission to describe a network interface attribute",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfacePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html",
                "description": "Grants permission to describe the permissions that are associated with a network interface",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html",
                "description": "Grants permission to describe one or more network interfaces",
                "accessLevel": "List"
            },
            "DescribePlacementGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html",
                "description": "Grants permission to describe one or more placement groups",
                "accessLevel": "List"
            },
            "DescribePrefixLists": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html",
                "description": "Grants permission to describe available AWS services in a prefix list format",
                "accessLevel": "List"
            },
            "DescribePrincipalIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html",
                "description": "Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference",
                "accessLevel": "List"
            },
            "DescribePublicIpv4Pools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html",
                "description": "Grants permission to describe one or more IPv4 address pools",
                "accessLevel": "List"
            },
            "DescribeRegions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html",
                "description": "Grants permission to describe one or more AWS Regions that are currently available in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html",
                "description": "Grants permission to describe one or more purchased Reserved Instances in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesListings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html",
                "description": "Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html",
                "description": "Grants permission to describe the modifications made to one or more Reserved Instances",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html",
                "description": "Grants permission to describe the Reserved Instance offerings that are available for purchase",
                "accessLevel": "List"
            },
            "DescribeRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html",
                "description": "Grants permission to describe one or more route tables",
                "accessLevel": "List"
            },
            "DescribeScheduledInstanceAvailability": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html",
                "description": "Grants permission to find available schedules for Scheduled Instances",
                "accessLevel": "Read"
            },
            "DescribeScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html",
                "description": "Grants permission to describe one or more Scheduled Instances in your account",
                "accessLevel": "Read"
            },
            "DescribeSecurityGroupReferences": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html",
                "description": "Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups",
                "accessLevel": "List"
            },
            "DescribeSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html",
                "description": "Grants permission to describe one or more security groups",
                "accessLevel": "List"
            },
            "DescribeSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html",
                "description": "Grants permission to describe an attribute of a snapshot",
                "accessLevel": "List"
            },
            "DescribeSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html",
                "description": "Grants permission to describe one or more EBS snapshots",
                "accessLevel": "List"
            },
            "DescribeSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html",
                "description": "Grants permission to describe the data feed for Spot Instances",
                "accessLevel": "List"
            },
            "DescribeSpotFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html",
                "description": "Grants permission to describe the running instances for a Spot Fleet",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequestHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html",
                "description": "Grants permission to describe the events for a Spot Fleet request during a specified time",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html",
                "description": "Grants permission to describe one or more Spot Fleet requests",
                "accessLevel": "List"
            },
            "DescribeSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html",
                "description": "Grants permission to describe one or more Spot Instance requests",
                "accessLevel": "List"
            },
            "DescribeSpotPriceHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html",
                "description": "Grants permission to describe the Spot Instance price history",
                "accessLevel": "List"
            },
            "DescribeStaleSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html",
                "description": "Grants permission to describe the stale security group rules for security groups in a specified VPC",
                "accessLevel": "List"
            },
            "DescribeSubnets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html",
                "description": "Grants permission to describe one or more subnets",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html",
                "description": "Grants permission to describe one or more tags for an Amazon EC2 resource",
                "accessLevel": "Read"
            },
            "DescribeTrafficMirrorFilters": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html",
                "description": "Grants permission to describe one or more traffic mirror filters",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorSessions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html",
                "description": "Grants permission to describe one or more traffic mirror sessions",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorTargets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html",
                "description": "Grants permission to describe one or more traffic mirror targets",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html",
                "description": "Grants permission to describe one or more attachments between resources and transit gateways",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayMulticastDomains": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html",
                "description": "Grants permission to describe one or more transit gateway multicast domains",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayPeeringAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html",
                "description": "Grants permission to describe one or more transit gateway peering attachments",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html",
                "description": "Grants permission to describe one or more transit gateway route tables",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayVpcAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html",
                "description": "Grants permission to describe one or more VPC attachments on a transit gateway",
                "accessLevel": "List"
            },
            "DescribeTransitGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html",
                "description": "Grants permission to describe one or more transit gateways",
                "accessLevel": "List"
            },
            "DescribeVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html",
                "description": "Grants permission to describe an attribute of an EBS volume",
                "accessLevel": "List"
            },
            "DescribeVolumeStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html",
                "description": "Grants permission to describe the status of one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html",
                "description": "Grants permission to describe one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html",
                "description": "Grants permission to describe the current modification status of one or more EBS volumes",
                "accessLevel": "Read"
            },
            "DescribeVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html",
                "description": "Grants permission to describe an attribute of a VPC",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html",
                "description": "Grants permission to describe the ClassicLink status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to describe the ClassicLink DNS support status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html",
                "description": "Grants permission to describe the VPC endpoint connections to your VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to describe VPC endpoint service configurations (your services)",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html",
                "description": "Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html",
                "description": "Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint",
                "accessLevel": "List"
            },
            "DescribeVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html",
                "description": "Grants permission to describe one or more VPC endpoints",
                "accessLevel": "List"
            },
            "DescribeVpcPeeringConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html",
                "description": "Grants permission to describe one or more VPC peering connections",
                "accessLevel": "List"
            },
            "DescribeVpcs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html",
                "description": "Grants permission to describe one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html",
                "description": "Grants permission to describe one or more VPN connections",
                "accessLevel": "Read"
            },
            "DescribeVpnGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html",
                "description": "Grants permission to describe one or more virtual private gateways",
                "accessLevel": "List"
            },
            "DetachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html",
                "description": "Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DetachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html",
                "description": "Grants permission to detach an internet gateway from a VPC",
                "accessLevel": "Write"
            },
            "DetachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html",
                "description": "Grants permission to detach a network interface from an instance",
                "accessLevel": "Write"
            },
            "DetachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html",
                "description": "Grants permission to detach an EBS volume from an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DetachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html",
                "description": "Grants permission to detach a virtual private gateway from a VPC",
                "accessLevel": "Write"
            },
            "DisableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html",
                "description": "Grants permission to disable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "DisableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html",
                "description": "Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to disable a resource attachment from propagating routes to the specified propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html",
                "description": "Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC",
                "accessLevel": "Write"
            },
            "DisableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html",
                "description": "Grants permission to disable ClassicLink for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to disable ClassicLink DNS support for a VPC",
                "accessLevel": "Write"
            },
            "DisassociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html",
                "description": "Grants permission to disassociate an Elastic IP address from an instance or network interface",
                "accessLevel": "Write"
            },
            "DisassociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html",
                "description": "Grants permission to disassociate a target network from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html",
                "description": "Grants permission to disassociate an IAM instance profile from a running or stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisassociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html",
                "description": "Grants permission to disassociate a subnet from a route table",
                "accessLevel": "Write"
            },
            "DisassociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a subnet",
                "accessLevel": "Write"
            },
            "DisassociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to disassociate one or more subnets from a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html",
                "description": "Grants permission to disassociate a resource attachment from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a VPC",
                "accessLevel": "Write"
            },
            "EnableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html",
                "description": "Grants permission to enable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "EnableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html",
                "description": "Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to enable an attachment to propagate routes to a propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html",
                "description": "Grants permission to enable a virtual private gateway to propagate routes to a VPC route table",
                "accessLevel": "Write"
            },
            "EnableVolumeIO": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html",
                "description": "Grants permission to enable I/O operations for a volume that had I/O operations disabled",
                "accessLevel": "Write"
            },
            "EnableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html",
                "description": "Grants permission to enable a VPC for ClassicLink",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "EnableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink",
                "accessLevel": "Write"
            },
            "ExportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to download the client certificate revocation list for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportClientVpnClientConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html",
                "description": "Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html",
                "description": "Grants permission to export an Amazon Machine Image (AMI) to a VM file",
                "accessLevel": "Write"
            },
            "ExportTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html",
                "description": "Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket",
                "accessLevel": "Write"
            },
            "GetCapacityReservationUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html",
                "description": "Grants permission to get usage information about a Capacity Reservation",
                "accessLevel": "Read"
            },
            "GetCoipPoolUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html",
                "description": "Grants permission to describe the allocations from the specified customer-owned address pool",
                "accessLevel": "Read"
            },
            "GetConsoleOutput": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html",
                "description": "Grants permission to get the console output for an instance",
                "accessLevel": "Read"
            },
            "GetConsoleScreenshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html",
                "description": "Grants permission to retrieve a JPG-format screenshot of a running instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "GetDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html",
                "description": "Grants permission to get the default credit option for CPU usage of a burstable performance instance family",
                "accessLevel": "Read"
            },
            "GetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default",
                "accessLevel": "Read"
            },
            "GetEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html",
                "description": "Grants permission to describe whether EBS encryption by default is enabled for your account",
                "accessLevel": "Read"
            },
            "GetHostReservationPurchasePreview": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html",
                "description": "Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host",
                "accessLevel": "Read"
            },
            "GetLaunchTemplateData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html",
                "description": "Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version",
                "accessLevel": "Read"
            },
            "GetManagedPrefixListAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListAssociations.html",
                "description": "Grants permission to get information about the resources that are associated with the specified managed prefix list",
                "accessLevel": "Read"
            },
            "GetManagedPrefixListEntries": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListEntries.html",
                "description": "Grants permission to get information about the entries for a specified managed prefix list",
                "accessLevel": "Read"
            },
            "GetPasswordData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html",
                "description": "Grants permission to retrieve the encrypted administrator password for a running Windows instance",
                "accessLevel": "Read"
            },
            "GetReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html",
                "description": "Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance",
                "accessLevel": "Read"
            },
            "GetTransitGatewayAttachmentPropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html",
                "description": "Grants permission to list the route tables to which a resource attachment propagates routes",
                "accessLevel": "List"
            },
            "GetTransitGatewayMulticastDomainAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html",
                "description": "Grants permission to get information about the associations for a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTableAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html",
                "description": "Grants permission to get information about associations for a transit gateway route table",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTablePropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html",
                "description": "Grants permission to get information about the route table propagations for a transit gateway route table",
                "accessLevel": "List"
            },
            "ImportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to upload a client certificate revocation list to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ImportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html",
                "description": "Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ImportInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html",
                "description": "Grants permission to create an import instance task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html",
                "description": "Grants permission to import a public key from an RSA key pair that was created with a third-party tool",
                "accessLevel": "Write"
            },
            "ImportSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html",
                "description": "Grants permission to import a disk into an EBS snapshot",
                "accessLevel": "Write"
            },
            "ImportVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html",
                "description": "Grants permission to create an import volume task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ModifyCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html",
                "description": "Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html",
                "description": "Grants permission to modify a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html",
                "description": "Grants permission to change the account level default credit option for CPU usage of burstable performance instances",
                "accessLevel": "Write"
            },
            "ModifyEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html",
                "description": "Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "ModifyFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html",
                "description": "Grants permission to modify an EC2 Fleet",
                "accessLevel": "Write"
            },
            "ModifyFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "ModifyHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html",
                "description": "Grants permission to modify a Dedicated Host",
                "accessLevel": "Write"
            },
            "ModifyIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html",
                "description": "Grants permission to modify the ID format for a resource",
                "accessLevel": "Write"
            },
            "ModifyIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html",
                "description": "Grants permission to modify the ID format of a resource for a specific principal in your account",
                "accessLevel": "Write"
            },
            "ModifyImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ModifyInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html",
                "description": "Grants permission to modify an attribute of an instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCapacityReservationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html",
                "description": "Grants permission to modify the Capacity Reservation settings for a stopped instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html",
                "description": "Grants permission to modify the credit option for CPU usage on an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "ModifyInstanceEventStartTime": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html",
                "description": "Grants permission to modify the start time for a scheduled EC2 instance event",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyInstanceMetadataOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html",
                "description": "Grants permission to modify the metadata options for an instance",
                "accessLevel": "Write"
            },
            "ModifyInstancePlacement": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html",
                "description": "Grants permission to modify the placement attributes for an instance",
                "accessLevel": "Write"
            },
            "ModifyLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html",
                "description": "Grants permission to modify a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyManagedPrefixList.html",
                "description": "Grants permission to modify a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html",
                "description": "Grants permission to modify an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ModifyReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html",
                "description": "Grants permission to modify attributes of one or more Reserved Instances",
                "accessLevel": "Write"
            },
            "ModifySnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html",
                "description": "Grants permission to add or remove permission settings for a snapshot",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "ModifySpotFleetRequest": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html",
                "description": "Grants permission to modify a Spot Fleet request",
                "accessLevel": "Write"
            },
            "ModifySubnetAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html",
                "description": "Grants permission to modify an attribute of a subnet",
                "accessLevel": "Write"
            },
            "ModifyTrafficMirrorFilterNetworkServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html",
                "description": "Grants permission to allow or restrict mirroring network services",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html",
                "description": "Grants permission to modify a traffic mirror rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html",
                "description": "Grants permission to modify a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html",
                "description": "Grants permission to modify a VPC attachment on a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html",
                "description": "Grants permission to modify the parameters of an EBS volume",
                "accessLevel": "Write"
            },
            "ModifyVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html",
                "description": "Grants permission to modify an attribute of a volume",
                "accessLevel": "Write"
            },
            "ModifyVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html",
                "description": "Grants permission to modify an attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html",
                "description": "Grants permission to modify an attribute of a VPC endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html",
                "description": "Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to modify the attributes of a VPC endpoint service configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html",
                "description": "Grants permission to modify the permissions for a VPC endpoint service",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcPeeringConnectionOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html",
                "description": "Grants permission to modify the VPC peering connection options on one side of a VPC peering connection",
                "accessLevel": "Write"
            },
            "ModifyVpcTenancy": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html",
                "description": "Grants permission to modify the instance tenancy attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html",
                "description": "Grants permission to modify the target gateway of a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:GatewayType"
                        ]
                    }
                }
            },
            "ModifyVpnTunnelCertificate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate",
                "description": "Grants permission to modify the certificate for a Site-to-Site VPN connection",
                "accessLevel": "Write"
            },
            "ModifyVpnTunnelOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html",
                "description": "Grants permission to modify the options for a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "MonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html",
                "description": "Grants permission to enable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "MoveAddressToVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html",
                "description": "Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform",
                "accessLevel": "Write"
            },
            "ProvisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html",
                "description": "Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool",
                "accessLevel": "Write"
            },
            "PurchaseHostReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html",
                "description": "Grants permission to purchase a reservation with configurations that match those of a Dedicated Host",
                "accessLevel": "Write"
            },
            "PurchaseReservedInstancesOffering": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html",
                "description": "Grants permission to purchase a Reserved Instance offering",
                "accessLevel": "Write"
            },
            "PurchaseScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html",
                "description": "Grants permission to purchase one or more Scheduled Instances with a specified schedule",
                "accessLevel": "Write"
            },
            "RebootInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html",
                "description": "Grants permission to request a reboot of one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "RegisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html",
                "description": "Grants permission to register an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "RegisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "RegisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RegisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to reject a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html",
                "description": "Grants permission to reject a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html",
                "description": "Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html",
                "description": "Grants permission to reject a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReleaseAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html",
                "description": "Grants permission to release an Elastic IP address",
                "accessLevel": "Write"
            },
            "ReleaseHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html",
                "description": "Grants permission to release one or more On-Demand Dedicated Hosts",
                "accessLevel": "Write"
            },
            "ReplaceIamInstanceProfileAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html",
                "description": "Grants permission to replace an IAM instance profile for an instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "ReplaceNetworkAclAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html",
                "description": "Grants permission to change which network ACL a subnet is associated with",
                "accessLevel": "Write"
            },
            "ReplaceNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html",
                "description": "Grants permission to replace an entry (rule) in a network ACL",
                "accessLevel": "Write"
            },
            "ReplaceRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html",
                "description": "Grants permission to replace a route within a route table in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "ReplaceRouteTableAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html",
                "description": "Grants permission to change the route table that is associated with a subnet",
                "accessLevel": "Write"
            },
            "ReplaceTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html",
                "description": "Grants permission to replace a route in a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReportInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html",
                "description": "Grants permission to submit feedback about the status of an instance",
                "accessLevel": "Write"
            },
            "RequestSpotFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html",
                "description": "Grants permission to create a Spot Fleet request",
                "accessLevel": "Write"
            },
            "RequestSpotInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html",
                "description": "Grants permission to create a Spot Instance request",
                "accessLevel": "Write"
            },
            "ResetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS",
                "accessLevel": "Write"
            },
            "ResetFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value",
                "accessLevel": "Write"
            },
            "ResetImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value",
                "accessLevel": "Write"
            },
            "ResetInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html",
                "description": "Grants permission to reset an attribute of an instance to its default value",
                "accessLevel": "Write"
            },
            "ResetNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html",
                "description": "Grants permission to reset an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ResetSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html",
                "description": "Grants permission to reset permission settings for a snapshot",
                "accessLevel": "Permissions management"
            },
            "RestoreAddressToClassic": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html",
                "description": "Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform",
                "accessLevel": "Write"
            },
            "RestoreManagedPrefixListVersion": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreManagedPrefixListVersion.html",
                "description": "Grants permission to restore the entries from a previous version of a managed prefix list to a new version of the prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RevokeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html",
                "description": "Grants permission to remove an inbound authorization rule from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html",
                "description": "Grants permission to remove one or more outbound rules from a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html",
                "description": "Grants permission to remove one or more inbound rules from a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RunInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html",
                "description": "Grants permission to launch one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:ImageType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy",
                            "ec2:MetadataHttpEndpoint",
                            "ec2:MetadataHttpTokens",
                            "ec2:MetadataHttpPutResponseHopLimit"
                        ]
                    },
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/",
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "elastic-gpu": {
                        "required": false,
                        "conditions": [
                            "ec2:ElasticGpuType"
                        ]
                    },
                    "elastic-inference": {
                        "required": false
                    },
                    "key-pair": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "launch-template": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "placement-group": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroupStrategy",
                            "ec2:Region"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "RunScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html",
                "description": "Grants permission to launch one or more Scheduled Instances",
                "accessLevel": "Write"
            },
            "SearchLocalGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html",
                "description": "Grants permission to search for routes in a local gateway route table",
                "accessLevel": "List"
            },
            "SearchTransitGatewayMulticastGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html",
                "description": "Grants permission to search for groups, sources, and members in a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "SearchTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html",
                "description": "Grants permission to search for routes in a transit gateway route table",
                "accessLevel": "List"
            },
            "SendDiagnosticInterrupt": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html",
                "description": "Grants permission to send a diagnostic interrupt to an Amazon EC2 instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html",
                "description": "Grants permission to start a stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartVpcEndpointServicePrivateDnsVerification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html",
                "description": "Grants permission to start the private DNS verification process for a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "StopInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html",
                "description": "Grants permission to stop an Amazon EBS-backed instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "TerminateClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html",
                "description": "Grants permission to terminate active Client VPN endpoint connections",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "TerminateInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html",
                "description": "Grants permission to shut down one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "UnassignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html",
                "description": "Grants permission to unassign one or more IPv6 addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnassignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html",
                "description": "Grants permission to unassign one or more secondary private IP addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnmonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html",
                "description": "Grants permission to disable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "UpdateSecurityGroupRuleDescriptionsEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html",
                "description": "Grants permission to update descriptions for one or more outbound rules in a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "UpdateSecurityGroupRuleDescriptionsIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html",
                "description": "Grants permission to update descriptions for one or more inbound rules in a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "WithdrawByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html",
                "description": "Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "capacity-reservation": {
                "name": "capacity-reservation",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "carrier-gateway": {
                "name": "carrier-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/Carrier_Gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:carrier-gateway/${CarrierGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "client-vpn-endpoint": {
                "name": "client-vpn-endpoint",
                "url": "https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "customer-gateway": {
                "name": "customer-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dedicated-host": {
                "name": "dedicated-host",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AutoPlacement",
                    "ec2:AvailabilityZone",
                    "ec2:HostRecovery",
                    "ec2:InstanceType",
                    "ec2:Quantity",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dhcp-options": {
                "name": "dhcp-options",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "egress-only-internet-gateway": {
                "name": "egress-only-internet-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/egress-only-internet-gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:egress-only-internet-gateway/${EgressOnlyInternetGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "elastic-gpu": {
                "name": "elastic-gpu",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}",
                "conditionKeys": [
                    "ec2:ElasticGpuType"
                ]
            },
            "elastic-inference": {
                "name": "elastic-inference",
                "url": "https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html",
                "arn": "arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}",
                "conditionKeys": []
            },
            "export-instance-task": {
                "name": "export-instance-task",
                "url": "https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:export-instance-task/${ExportTaskId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:fleet/${FleetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "fpga-image": {
                "name": "fpga-image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format",
                "arn": "arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "image": {
                "name": "image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html",
                "arn": "arn:${Partition}:ec2:${Region}::image/${ImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:ImageType",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType"
                ]
            },
            "instance": {
                "name": "instance",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:EbsOptimized",
                    "ec2:InstanceProfile",
                    "ec2:InstanceType",
                    "ec2:PlacementGroup",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType",
                    "ec2:Tenancy"
                ]
            },
            "internet-gateway": {
                "name": "internet-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "key-pair": {
                "name": "key-pair",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "launch-template": {
                "name": "launch-template",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway": {
                "name": "local-gateway",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table": {
                "name": "local-gateway-route-table",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-virtual-interface-group-association": {
                "name": "local-gateway-route-table-virtual-interface-group-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-vpc-association": {
                "name": "local-gateway-route-table-vpc-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface": {
                "name": "local-gateway-virtual-interface",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface-group": {
                "name": "local-gateway-virtual-interface-group",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "natgateway": {
                "name": "natgateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:natgateway/${NatGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "network-acl": {
                "name": "network-acl",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "network-interface": {
                "name": "network-interface",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AssociatePublicIpAddress",
                    "ec2:AuthorizedService",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Subnet",
                    "ec2:Vpc"
                ]
            },
            "placement-group": {
                "name": "placement-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:PlacementGroupStrategy",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "prefix-list": {
                "name": "prefix-list",
                "url": "",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:prefix-list/${PrefixListId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "reserved-instances": {
                "name": "reserved-instances",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:InstanceType",
                    "ec2:Region",
                    "ec2:ReservedInstancesOfferingType",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "route-table": {
                "name": "route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "security-group": {
                "name": "security-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html",
                "arn": "arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:ParentVolume",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:SnapshotTime",
                    "ec2:VolumeSize"
                ]
            },
            "spot-instance-request": {
                "name": "spot-instance-request",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "subnet": {
                "name": "subnet",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "traffic-mirror-session": {
                "name": "traffic-mirror-session",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-target": {
                "name": "traffic-mirror-target",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter": {
                "name": "traffic-mirror-filter",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter-rule": {
                "name": "traffic-mirror-filter-rule",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}",
                "conditionKeys": [
                    "ec2:Region"
                ]
            },
            "transit-gateway-attachment": {
                "name": "transit-gateway-attachment",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-multicast-domain": {
                "name": "transit-gateway-multicast-domain",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-route-table": {
                "name": "transit-gateway-route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway": {
                "name": "transit-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "volume": {
                "name": "volume",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Encrypted",
                    "ec2:ParentSnapshot",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VolumeIops",
                    "ec2:VolumeSize",
                    "ec2:VolumeType"
                ]
            },
            "vpc": {
                "name": "vpc",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "vpc-endpoint": {
                "name": "vpc-endpoint",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServiceName",
                    "ec2:VpceServiceOwner"
                ]
            },
            "vpc-endpoint-service": {
                "name": "vpc-endpoint-service",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServicePrivateDnsName"
                ]
            },
            "vpc-flow-log": {
                "name": "vpc-flow-log",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpc-peering-connection": {
                "name": "vpc-peering-connection",
                "url": "https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ec2:AccepterVpc",
                    "ec2:Region",
                    "ec2:RequesterVpc",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpn-connection": {
                "name": "vpn-connection",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AuthenticationType",
                    "ec2:DPDTimeoutSeconds",
                    "ec2:GatewayType",
                    "ec2:IKEVersions",
                    "ec2:InsideTunnelCidr",
                    "ec2:Phase1DHGroupNumbers",
                    "ec2:Phase1EncryptionAlgorithms",
                    "ec2:Phase1IntegrityAlgorithms",
                    "ec2:Phase1LifetimeSeconds",
                    "ec2:Phase2DHGroupNumbers",
                    "ec2:Phase2EncryptionAlgorithms",
                    "ec2:Phase2IntegrityAlgorithms",
                    "ec2:Phase2LifetimeSeconds",
                    "ec2:PresharedKeys",
                    "ec2:Region",
                    "ec2:RekeyFuzzPercentage",
                    "ec2:RekeyMarginTimeSeconds",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RoutingType"
                ]
            },
            "vpn-gateway": {
                "name": "vpn-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept a Convertible Reserved Instance exchange quote
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html
     */
    toAcceptReservedInstancesExchangeQuote() {
        this.add('ec2:AcceptReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to accept a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html
     */
    toAcceptTransitGatewayPeeringAttachment() {
        this.add('ec2:AcceptTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to accept a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html
     */
    toAcceptTransitGatewayVpcAttachment() {
        this.add('ec2:AcceptTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html
     */
    toAcceptVpcEndpointConnections() {
        this.add('ec2:AcceptVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to accept a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html
     */
    toAcceptVpcPeeringConnection() {
        this.add('ec2:AcceptVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html
     */
    toAdvertiseByoipCidr() {
        this.add('ec2:AdvertiseByoipCidr');
        return this;
    }
    /**
     * Grants permission to allocate an Elastic IP address (EIP) to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html
     */
    toAllocateAddress() {
        this.add('ec2:AllocateAddress');
        return this;
    }
    /**
     * Grants permission to allocate a Dedicated Host to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html
     */
    toAllocateHosts() {
        this.add('ec2:AllocateHosts');
        return this;
    }
    /**
     * Grants permission to apply a security group to the association between a Client VPN endpoint and a target network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html
     */
    toApplySecurityGroupsToClientVpnTargetNetwork() {
        this.add('ec2:ApplySecurityGroupsToClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to assign one or more IPv6 addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html
     */
    toAssignIpv6Addresses() {
        this.add('ec2:AssignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to assign one or more secondary private IP addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html
     */
    toAssignPrivateIpAddresses() {
        this.add('ec2:AssignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html
     */
    toAssociateAddress() {
        this.add('ec2:AssociateAddress');
        return this;
    }
    /**
     * Grants permission to associate a target network with a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html
     */
    toAssociateClientVpnTargetNetwork() {
        this.add('ec2:AssociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to associate or disassociate a set of DHCP options with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html
     */
    toAssociateDhcpOptions() {
        this.add('ec2:AssociateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to associate an IAM instance profile with a running or stopped instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html
     */
    toAssociateIamInstanceProfile() {
        this.add('ec2:AssociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to associate a subnet or gateway with a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html
     */
    toAssociateRouteTable() {
        this.add('ec2:AssociateRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html
     */
    toAssociateSubnetCidrBlock() {
        this.add('ec2:AssociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html
     */
    toAssociateTransitGatewayMulticastDomain() {
        this.add('ec2:AssociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to associate an attachment with a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html
     */
    toAssociateTransitGatewayRouteTable() {
        this.add('ec2:AssociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html
     */
    toAssociateVpcCidrBlock() {
        this.add('ec2:AssociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html
     */
    toAttachClassicLinkVpc() {
        this.add('ec2:AttachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to attach an internet gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html
     */
    toAttachInternetGateway() {
        this.add('ec2:AttachInternetGateway');
        return this;
    }
    /**
     * Grants permission to attach a network interface to an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html
     */
    toAttachNetworkInterface() {
        this.add('ec2:AttachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html
     */
    toAttachVolume() {
        this.add('ec2:AttachVolume');
        return this;
    }
    /**
     * Grants permission to attach a virtual private gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html
     */
    toAttachVpnGateway() {
        this.add('ec2:AttachVpnGateway');
        return this;
    }
    /**
     * Grants permission to add an inbound authorization rule to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html
     */
    toAuthorizeClientVpnIngress() {
        this.add('ec2:AuthorizeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to add one or more outbound rules to a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html
     */
    toAuthorizeSecurityGroupEgress() {
        this.add('ec2:AuthorizeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to add one or more inbound rules to a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html
     */
    toAuthorizeSecurityGroupIngress() {
        this.add('ec2:AuthorizeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to bundle an instance store-backed Windows instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html
     */
    toBundleInstance() {
        this.add('ec2:BundleInstance');
        return this;
    }
    /**
     * Grants permission to cancel a bundling operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html
     */
    toCancelBundleTask() {
        this.add('ec2:CancelBundleTask');
        return this;
    }
    /**
     * Grants permission to cancel a Capacity Reservation and release the reserved capacity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html
     */
    toCancelCapacityReservation() {
        this.add('ec2:CancelCapacityReservation');
        return this;
    }
    /**
     * Grants permission to cancel an active conversion task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html
     */
    toCancelConversionTask() {
        this.add('ec2:CancelConversionTask');
        return this;
    }
    /**
     * Grants permission to cancel an active export task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask() {
        this.add('ec2:CancelExportTask');
        return this;
    }
    /**
     * Grants permission to cancel an in-process import virtual machine or import snapshot task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html
     */
    toCancelImportTask() {
        this.add('ec2:CancelImportTask');
        return this;
    }
    /**
     * Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html
     */
    toCancelReservedInstancesListing() {
        this.add('ec2:CancelReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Fleet requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html
     */
    toCancelSpotFleetRequests() {
        this.add('ec2:CancelSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Instance requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html
     */
    toCancelSpotInstanceRequests() {
        this.add('ec2:CancelSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to determine whether an owned product code is associated with an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html
     */
    toConfirmProductInstance() {
        this.add('ec2:ConfirmProductInstance');
        return this;
    }
    /**
     * Grants permission to copy a source Amazon FPGA image (AFI) to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html
     */
    toCopyFpgaImage() {
        this.add('ec2:CopyFpgaImage');
        return this;
    }
    /**
     * Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html
     */
    toCopyImage() {
        this.add('ec2:CopyImage');
        return this;
    }
    /**
     * Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.add('ec2:CopySnapshot');
        return this;
    }
    /**
     * Grants permission to create a Capacity Reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html
     */
    toCreateCapacityReservation() {
        this.add('ec2:CreateCapacityReservation');
        return this;
    }
    /**
     * Grants permission to create a carrier gateway and provides CSP connectivity to VPC customers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCarrierGateway.html
     */
    toCreateCarrierGateway() {
        this.add('ec2:CreateCarrierGateway');
        return this;
    }
    /**
     * Grants permission to create a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html
     */
    toCreateClientVpnEndpoint() {
        this.add('ec2:CreateClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to add a network route to a Client VPN endpoint's route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html
     */
    toCreateClientVpnRoute() {
        this.add('ec2:CreateClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html
     */
    toCreateCustomerGateway() {
        this.add('ec2:CreateCustomerGateway');
        return this;
    }
    /**
     * Grants permission to create a default subnet in a specified Availability Zone in a default VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html
     */
    toCreateDefaultSubnet() {
        this.add('ec2:CreateDefaultSubnet');
        return this;
    }
    /**
     * Grants permission to create a default VPC with a default subnet in each Availability Zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html
     */
    toCreateDefaultVpc() {
        this.add('ec2:CreateDefaultVpc');
        return this;
    }
    /**
     * Grants permission to create a set of DHCP options for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html
     */
    toCreateDhcpOptions() {
        this.add('ec2:CreateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to create an egress-only internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html
     */
    toCreateEgressOnlyInternetGateway() {
        this.add('ec2:CreateEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to launch an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        this.add('ec2:CreateFleet');
        return this;
    }
    /**
     * Grants permission to create one or more flow logs to capture IP traffic for a network interface
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html
     */
    toCreateFlowLogs() {
        this.add('ec2:CreateFlowLogs');
        return this;
    }
    /**
     * Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html
     */
    toCreateFpgaImage() {
        this.add('ec2:CreateFpgaImage');
        return this;
    }
    /**
     * Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html
     */
    toCreateImage() {
        this.add('ec2:CreateImage');
        return this;
    }
    /**
     * Grants permission to export a running or stopped instance to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html
     */
    toCreateInstanceExportTask() {
        this.add('ec2:CreateInstanceExportTask');
        return this;
    }
    /**
     * Grants permission to create an internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html
     */
    toCreateInternetGateway() {
        this.add('ec2:CreateInternetGateway');
        return this;
    }
    /**
     * Grants permission to create a 2048-bit RSA key pair
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html
     */
    toCreateKeyPair() {
        this.add('ec2:CreateKeyPair');
        return this;
    }
    /**
     * Grants permission to create a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html
     */
    toCreateLaunchTemplate() {
        this.add('ec2:CreateLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to create a new version of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html
     */
    toCreateLaunchTemplateVersion() {
        this.add('ec2:CreateLaunchTemplateVersion');
        return this;
    }
    /**
     * Grants permission to create a static route for a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html
     */
    toCreateLocalGatewayRoute() {
        this.add('ec2:CreateLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to associate a VPC with a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html
     */
    toCreateLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:CreateLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to create a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateManagedPrefixList.html
     */
    toCreateManagedPrefixList() {
        this.add('ec2:CreateManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to create a NAT gateway in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html
     */
    toCreateNatGateway() {
        this.add('ec2:CreateNatGateway');
        return this;
    }
    /**
     * Grants permission to create a network ACL in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html
     */
    toCreateNetworkAcl() {
        this.add('ec2:CreateNetworkAcl');
        return this;
    }
    /**
     * Grants permission to create a numbered entry (a rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html
     */
    toCreateNetworkAclEntry() {
        this.add('ec2:CreateNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to create a network interface in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html
     */
    toCreateNetworkInterface() {
        this.add('ec2:CreateNetworkInterface');
        return this;
    }
    /**
     * Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html
     */
    toCreateNetworkInterfacePermission() {
        this.add('ec2:CreateNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to create a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html
     */
    toCreatePlacementGroup() {
        this.add('ec2:CreatePlacementGroup');
        return this;
    }
    /**
     * Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html
     */
    toCreateReservedInstancesListing() {
        this.add('ec2:CreateReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to create a route in a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        this.add('ec2:CreateRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html
     */
    toCreateRouteTable() {
        this.add('ec2:CreateRouteTable');
        return this;
    }
    /**
     * Grants permission to create a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html
     */
    toCreateSecurityGroup() {
        this.add('ec2:CreateSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.add('ec2:CreateSnapshot');
        return this;
    }
    /**
     * Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html
     */
    toCreateSnapshots() {
        this.add('ec2:CreateSnapshots');
        return this;
    }
    /**
     * Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html
     */
    toCreateSpotDatafeedSubscription() {
        this.add('ec2:CreateSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to create a subnet in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html
     */
    toCreateSubnet() {
        this.add('ec2:CreateSubnet');
        return this;
    }
    /**
     * Grants permission to add or overwrite one or more tags for Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html
     */
    toCreateTags() {
        this.add('ec2:CreateTags');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html
     */
    toCreateTrafficMirrorFilter() {
        this.add('ec2:CreateTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html
     */
    toCreateTrafficMirrorFilterRule() {
        this.add('ec2:CreateTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html
     */
    toCreateTrafficMirrorSession() {
        this.add('ec2:CreateTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html
     */
    toCreateTrafficMirrorTarget() {
        this.add('ec2:CreateTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to create a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html
     */
    toCreateTransitGateway() {
        this.add('ec2:CreateTransitGateway');
        return this;
    }
    /**
     * Grants permission to create a multicast domain for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html
     */
    toCreateTransitGatewayMulticastDomain() {
        this.add('ec2:CreateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html
     */
    toCreateTransitGatewayPeeringAttachment() {
        this.add('ec2:CreateTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to create a static route for a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html
     */
    toCreateTransitGatewayRoute() {
        this.add('ec2:CreateTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html
     */
    toCreateTransitGatewayRouteTable() {
        this.add('ec2:CreateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html
     */
    toCreateTransitGatewayVpcAttachment() {
        this.add('ec2:CreateTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to create an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html
     */
    toCreateVolume() {
        this.add('ec2:CreateVolume');
        return this;
    }
    /**
     * Grants permission to create a VPC with a specified CIDR block
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html
     */
    toCreateVpc() {
        this.add('ec2:CreateVpc');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint for an AWS service
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html
     */
    toCreateVpcEndpoint() {
        this.add('ec2:CreateVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html
     */
    toCreateVpcEndpointConnectionNotification() {
        this.add('ec2:CreateVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html
     */
    toCreateVpcEndpointServiceConfiguration() {
        this.add('ec2:CreateVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to request a VPC peering connection between two VPCs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html
     */
    toCreateVpcPeeringConnection() {
        this.add('ec2:CreateVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html
     */
    toCreateVpnConnection() {
        this.add('ec2:CreateVpnConnection');
        return this;
    }
    /**
     * Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html
     */
    toCreateVpnConnectionRoute() {
        this.add('ec2:CreateVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to create a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html
     */
    toCreateVpnGateway() {
        this.add('ec2:CreateVpnGateway');
        return this;
    }
    /**
     * Grants permission to delete a carrier gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCarrierGateway.html
     */
    toDeleteCarrierGateway() {
        this.add('ec2:DeleteCarrierGateway');
        return this;
    }
    /**
     * Grants permission to delete a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html
     */
    toDeleteClientVpnEndpoint() {
        this.add('ec2:DeleteClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a route from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html
     */
    toDeleteClientVpnRoute() {
        this.add('ec2:DeleteClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to delete a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html
     */
    toDeleteCustomerGateway() {
        this.add('ec2:DeleteCustomerGateway');
        return this;
    }
    /**
     * Grants permission to delete a set of DHCP options
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html
     */
    toDeleteDhcpOptions() {
        this.add('ec2:DeleteDhcpOptions');
        return this;
    }
    /**
     * Grants permission to delete an egress-only internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html
     */
    toDeleteEgressOnlyInternetGateway() {
        this.add('ec2:DeleteEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete one or more EC2 Fleets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html
     */
    toDeleteFleets() {
        this.add('ec2:DeleteFleets');
        return this;
    }
    /**
     * Grants permission to delete one or more flow logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html
     */
    toDeleteFlowLogs() {
        this.add('ec2:DeleteFlowLogs');
        return this;
    }
    /**
     * Grants permission to delete an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html
     */
    toDeleteFpgaImage() {
        this.add('ec2:DeleteFpgaImage');
        return this;
    }
    /**
     * Grants permission to delete an internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html
     */
    toDeleteInternetGateway() {
        this.add('ec2:DeleteInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete a key pair by removing the public key from Amazon EC2
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html
     */
    toDeleteKeyPair() {
        this.add('ec2:DeleteKeyPair');
        return this;
    }
    /**
     * Grants permission to delete a launch template and its associated versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html
     */
    toDeleteLaunchTemplate() {
        this.add('ec2:DeleteLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html
     */
    toDeleteLaunchTemplateVersions() {
        this.add('ec2:DeleteLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to delete a route from a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html
     */
    toDeleteLocalGatewayRoute() {
        this.add('ec2:DeleteLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete an association between a VPC and local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html
     */
    toDeleteLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:DeleteLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to delete a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteManagedPrefixList.html
     */
    toDeleteManagedPrefixList() {
        this.add('ec2:DeleteManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to delete a NAT gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html
     */
    toDeleteNatGateway() {
        this.add('ec2:DeleteNatGateway');
        return this;
    }
    /**
     * Grants permission to delete a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html
     */
    toDeleteNetworkAcl() {
        this.add('ec2:DeleteNetworkAcl');
        return this;
    }
    /**
     * Grants permission to delete an inbound or outbound entry (rule) from a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html
     */
    toDeleteNetworkAclEntry() {
        this.add('ec2:DeleteNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to delete a detached network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html
     */
    toDeleteNetworkInterface() {
        this.add('ec2:DeleteNetworkInterface');
        return this;
    }
    /**
     * Grants permission to delete a permission that is associated with a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html
     */
    toDeleteNetworkInterfacePermission() {
        this.add('ec2:DeleteNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to delete a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html
     */
    toDeletePlacementGroup() {
        this.add('ec2:DeletePlacementGroup');
        return this;
    }
    /**
     * Grants permission to delete a route from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        this.add('ec2:DeleteRoute');
        return this;
    }
    /**
     * Grants permission to delete a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html
     */
    toDeleteRouteTable() {
        this.add('ec2:DeleteRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html
     */
    toDeleteSecurityGroup() {
        this.add('ec2:DeleteSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a snapshot of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        this.add('ec2:DeleteSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a data feed for Spot Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html
     */
    toDeleteSpotDatafeedSubscription() {
        this.add('ec2:DeleteSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to delete a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html
     */
    toDeleteSubnet() {
        this.add('ec2:DeleteSubnet');
        return this;
    }
    /**
     * Grants permission to delete one or more tags from Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        this.add('ec2:DeleteTags');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html
     */
    toDeleteTrafficMirrorFilter() {
        this.add('ec2:DeleteTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html
     */
    toDeleteTrafficMirrorFilterRule() {
        this.add('ec2:DeleteTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html
     */
    toDeleteTrafficMirrorSession() {
        this.add('ec2:DeleteTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html
     */
    toDeleteTrafficMirrorTarget() {
        this.add('ec2:DeleteTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html
     */
    toDeleteTransitGateway() {
        this.add('ec2:DeleteTransitGateway');
        return this;
    }
    /**
     * Grants permissions to delete a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html
     */
    toDeleteTransitGatewayMulticastDomain() {
        this.add('ec2:DeleteTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to delete a peering attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html
     */
    toDeleteTransitGatewayPeeringAttachment() {
        this.add('ec2:DeleteTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to delete a route from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html
     */
    toDeleteTransitGatewayRoute() {
        this.add('ec2:DeleteTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html
     */
    toDeleteTransitGatewayRouteTable() {
        this.add('ec2:DeleteTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a VPC attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html
     */
    toDeleteTransitGatewayVpcAttachment() {
        this.add('ec2:DeleteTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to delete an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html
     */
    toDeleteVolume() {
        this.add('ec2:DeleteVolume');
        return this;
    }
    /**
     * Grants permission to delete a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html
     */
    toDeleteVpc() {
        this.add('ec2:DeleteVpc');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint connection notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html
     */
    toDeleteVpcEndpointConnectionNotifications() {
        this.add('ec2:DeleteVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint service configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html
     */
    toDeleteVpcEndpointServiceConfigurations() {
        this.add('ec2:DeleteVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoints
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html
     */
    toDeleteVpcEndpoints() {
        this.add('ec2:DeleteVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to delete a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html
     */
    toDeleteVpcPeeringConnection() {
        this.add('ec2:DeleteVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to delete a VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html
     */
    toDeleteVpnConnection() {
        this.add('ec2:DeleteVpnConnection');
        return this;
    }
    /**
     * Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html
     */
    toDeleteVpnConnectionRoute() {
        this.add('ec2:DeleteVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to delete a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html
     */
    toDeleteVpnGateway() {
        this.add('ec2:DeleteVpnGateway');
        return this;
    }
    /**
     * Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html
     */
    toDeprovisionByoipCidr() {
        this.add('ec2:DeprovisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to deregister an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html
     */
    toDeregisterImage() {
        this.add('ec2:DeregisterImage');
        return this;
    }
    /**
     * Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html
     */
    toDeregisterInstanceEventNotificationAttributes() {
        this.add('ec2:DeregisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html
     */
    toDeregisterTransitGatewayMulticastGroupMembers() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html
     */
    toDeregisterTransitGatewayMulticastGroupSources() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to describe the attributes of the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        this.add('ec2:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to describe one or more Elastic IP addresses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html
     */
    toDescribeAddresses() {
        this.add('ec2:DescribeAddresses');
        return this;
    }
    /**
     * Grants permission to describe the longer ID format settings for all resource types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html
     */
    toDescribeAggregateIdFormat() {
        this.add('ec2:DescribeAggregateIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more of the Availability Zones that are available to you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html
     */
    toDescribeAvailabilityZones() {
        this.add('ec2:DescribeAvailabilityZones');
        return this;
    }
    /**
     * Grants permission to describe one or more bundling tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html
     */
    toDescribeBundleTasks() {
        this.add('ec2:DescribeBundleTasks');
        return this;
    }
    /**
     * Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html
     */
    toDescribeByoipCidrs() {
        this.add('ec2:DescribeByoipCidrs');
        return this;
    }
    /**
     * Grants permission to describe one or more Capacity Reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html
     */
    toDescribeCapacityReservations() {
        this.add('ec2:DescribeCapacityReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more Carrier Gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCarrierGateways.html
     */
    toDescribeCarrierGateways() {
        this.add('ec2:DescribeCarrierGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more linked EC2-Classic instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html
     */
    toDescribeClassicLinkInstances() {
        this.add('ec2:DescribeClassicLinkInstances');
        return this;
    }
    /**
     * Grants permission to describe the authorization rules for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html
     */
    toDescribeClientVpnAuthorizationRules() {
        this.add('ec2:DescribeClientVpnAuthorizationRules');
        return this;
    }
    /**
     * Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html
     */
    toDescribeClientVpnConnections() {
        this.add('ec2:DescribeClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more Client VPN endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html
     */
    toDescribeClientVpnEndpoints() {
        this.add('ec2:DescribeClientVpnEndpoints');
        return this;
    }
    /**
     * Grants permission to describe the routes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html
     */
    toDescribeClientVpnRoutes() {
        this.add('ec2:DescribeClientVpnRoutes');
        return this;
    }
    /**
     * Grants permission to describe the target networks that are associated with a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html
     */
    toDescribeClientVpnTargetNetworks() {
        this.add('ec2:DescribeClientVpnTargetNetworks');
        return this;
    }
    /**
     * Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html
     */
    toDescribeCoipPools() {
        this.add('ec2:DescribeCoipPools');
        return this;
    }
    /**
     * Grants permission to describe one or more conversion tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html
     */
    toDescribeConversionTasks() {
        this.add('ec2:DescribeConversionTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more customer gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html
     */
    toDescribeCustomerGateways() {
        this.add('ec2:DescribeCustomerGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more DHCP options sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html
     */
    toDescribeDhcpOptions() {
        this.add('ec2:DescribeDhcpOptions');
        return this;
    }
    /**
     * Grants permission to describe one or more egress-only internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html
     */
    toDescribeEgressOnlyInternetGateways() {
        this.add('ec2:DescribeEgressOnlyInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe an Elastic Graphics accelerator that is associated with an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html
     */
    toDescribeElasticGpus() {
        this.add('ec2:DescribeElasticGpus');
        return this;
    }
    /**
     * Grants permission to describe one or more export image tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html
     */
    toDescribeExportImageTasks() {
        this.add('ec2:DescribeExportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more export instance tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        this.add('ec2:DescribeExportTasks');
        return this;
    }
    /**
     * Grants permission to describe the state of fast snapshot restores for snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html
     */
    toDescribeFastSnapshotRestores() {
        this.add('ec2:DescribeFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to describe the events for an EC2 Fleet during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html
     */
    toDescribeFleetHistory() {
        this.add('ec2:DescribeFleetHistory');
        return this;
    }
    /**
     * Grants permission to describe the running instances for an EC2 Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html
     */
    toDescribeFleetInstances() {
        this.add('ec2:DescribeFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more EC2 Fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html
     */
    toDescribeFleets() {
        this.add('ec2:DescribeFleets');
        return this;
    }
    /**
     * Grants permission to describe one or more flow logs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html
     */
    toDescribeFlowLogs() {
        this.add('ec2:DescribeFlowLogs');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an Amazon FPGA Image (AFI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html
     */
    toDescribeFpgaImageAttribute() {
        this.add('ec2:DescribeFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more Amazon FPGA Images (AFIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html
     */
    toDescribeFpgaImages() {
        this.add('ec2:DescribeFpgaImages');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are available to purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html
     */
    toDescribeHostReservationOfferings() {
        this.add('ec2:DescribeHostReservationOfferings');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html
     */
    toDescribeHostReservations() {
        this.add('ec2:DescribeHostReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more Dedicated Hosts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html
     */
    toDescribeHosts() {
        this.add('ec2:DescribeHosts');
        return this;
    }
    /**
     * Grants permission to describe the IAM instance profile associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html
     */
    toDescribeIamInstanceProfileAssociations() {
        this.add('ec2:DescribeIamInstanceProfileAssociations');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html
     */
    toDescribeIdFormat() {
        this.add('ec2:DescribeIdFormat');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html
     */
    toDescribeIdentityIdFormat() {
        this.add('ec2:DescribeIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html
     */
    toDescribeImageAttribute() {
        this.add('ec2:DescribeImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more images (AMIs, AKIs, and ARIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html
     */
    toDescribeImages() {
        this.add('ec2:DescribeImages');
        return this;
    }
    /**
     * Grants permission to describe import virtual machine or import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html
     */
    toDescribeImportImageTasks() {
        this.add('ec2:DescribeImportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html
     */
    toDescribeImportSnapshotTasks() {
        this.add('ec2:DescribeImportSnapshotTasks');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html
     */
    toDescribeInstanceAttribute() {
        this.add('ec2:DescribeInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the credit option for CPU usage of one or more burstable performance instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html
     */
    toDescribeInstanceCreditSpecifications() {
        this.add('ec2:DescribeInstanceCreditSpecifications');
        return this;
    }
    /**
     * Grants permission to describe the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html
     */
    toDescribeInstanceEventNotificationAttributes() {
        this.add('ec2:DescribeInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html
     */
    toDescribeInstanceStatus() {
        this.add('ec2:DescribeInstanceStatus');
        return this;
    }
    /**
     * Grants permission to describe the set of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html
     */
    toDescribeInstanceTypeOfferings() {
        this.add('ec2:DescribeInstanceTypeOfferings');
        return this;
    }
    /**
     * Grants permission to describe the details of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html
     */
    toDescribeInstanceTypes() {
        this.add('ec2:DescribeInstanceTypes');
        return this;
    }
    /**
     * Grants permission to describe one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html
     */
    toDescribeInstances() {
        this.add('ec2:DescribeInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html
     */
    toDescribeInternetGateways() {
        this.add('ec2:DescribeInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more key pairs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html
     */
    toDescribeKeyPairs() {
        this.add('ec2:DescribeKeyPairs');
        return this;
    }
    /**
     * Grants permission to describe one or more launch template versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html
     */
    toDescribeLaunchTemplateVersions() {
        this.add('ec2:DescribeLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to describe one or more launch templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html
     */
    toDescribeLaunchTemplates() {
        this.add('ec2:DescribeLaunchTemplates');
        return this;
    }
    /**
     * Grants permission to describe the associations between virtual interface groups and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html
     */
    toDescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations');
        return this;
    }
    /**
     * Grants permission to describe an association between VPCs and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html
     */
    toDescribeLocalGatewayRouteTableVpcAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVpcAssociations');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html
     */
    toDescribeLocalGatewayRouteTables() {
        this.add('ec2:DescribeLocalGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interface groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html
     */
    toDescribeLocalGatewayVirtualInterfaceGroups() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaceGroups');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html
     */
    toDescribeLocalGatewayVirtualInterfaces() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html
     */
    toDescribeLocalGateways() {
        this.add('ec2:DescribeLocalGateways');
        return this;
    }
    /**
     * Grants permission to describe your managed prefix lists and any AWS-managed prefix lists
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeManagedPrefixLists.html
     */
    toDescribeManagedPrefixLists() {
        this.add('ec2:DescribeManagedPrefixLists');
        return this;
    }
    /**
     * Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html
     */
    toDescribeMovingAddresses() {
        this.add('ec2:DescribeMovingAddresses');
        return this;
    }
    /**
     * Grants permission to describe one or more NAT gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html
     */
    toDescribeNatGateways() {
        this.add('ec2:DescribeNatGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more network ACLs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html
     */
    toDescribeNetworkAcls() {
        this.add('ec2:DescribeNetworkAcls');
        return this;
    }
    /**
     * Grants permission to describe a network interface attribute
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html
     */
    toDescribeNetworkInterfaceAttribute() {
        this.add('ec2:DescribeNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the permissions that are associated with a network interface
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html
     */
    toDescribeNetworkInterfacePermissions() {
        this.add('ec2:DescribeNetworkInterfacePermissions');
        return this;
    }
    /**
     * Grants permission to describe one or more network interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html
     */
    toDescribeNetworkInterfaces() {
        this.add('ec2:DescribeNetworkInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more placement groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html
     */
    toDescribePlacementGroups() {
        this.add('ec2:DescribePlacementGroups');
        return this;
    }
    /**
     * Grants permission to describe available AWS services in a prefix list format
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html
     */
    toDescribePrefixLists() {
        this.add('ec2:DescribePrefixLists');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html
     */
    toDescribePrincipalIdFormat() {
        this.add('ec2:DescribePrincipalIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more IPv4 address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html
     */
    toDescribePublicIpv4Pools() {
        this.add('ec2:DescribePublicIpv4Pools');
        return this;
    }
    /**
     * Grants permission to describe one or more AWS Regions that are currently available in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html
     */
    toDescribeRegions() {
        this.add('ec2:DescribeRegions');
        return this;
    }
    /**
     * Grants permission to describe one or more purchased Reserved Instances in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html
     */
    toDescribeReservedInstances() {
        this.add('ec2:DescribeReservedInstances');
        return this;
    }
    /**
     * Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html
     */
    toDescribeReservedInstancesListings() {
        this.add('ec2:DescribeReservedInstancesListings');
        return this;
    }
    /**
     * Grants permission to describe the modifications made to one or more Reserved Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html
     */
    toDescribeReservedInstancesModifications() {
        this.add('ec2:DescribeReservedInstancesModifications');
        return this;
    }
    /**
     * Grants permission to describe the Reserved Instance offerings that are available for purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html
     */
    toDescribeReservedInstancesOfferings() {
        this.add('ec2:DescribeReservedInstancesOfferings');
        return this;
    }
    /**
     * Grants permission to describe one or more route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html
     */
    toDescribeRouteTables() {
        this.add('ec2:DescribeRouteTables');
        return this;
    }
    /**
     * Grants permission to find available schedules for Scheduled Instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html
     */
    toDescribeScheduledInstanceAvailability() {
        this.add('ec2:DescribeScheduledInstanceAvailability');
        return this;
    }
    /**
     * Grants permission to describe one or more Scheduled Instances in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html
     */
    toDescribeScheduledInstances() {
        this.add('ec2:DescribeScheduledInstances');
        return this;
    }
    /**
     * Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html
     */
    toDescribeSecurityGroupReferences() {
        this.add('ec2:DescribeSecurityGroupReferences');
        return this;
    }
    /**
     * Grants permission to describe one or more security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html
     */
    toDescribeSecurityGroups() {
        this.add('ec2:DescribeSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html
     */
    toDescribeSnapshotAttribute() {
        this.add('ec2:DescribeSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        this.add('ec2:DescribeSnapshots');
        return this;
    }
    /**
     * Grants permission to describe the data feed for Spot Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html
     */
    toDescribeSpotDatafeedSubscription() {
        this.add('ec2:DescribeSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to describe the running instances for a Spot Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html
     */
    toDescribeSpotFleetInstances() {
        this.add('ec2:DescribeSpotFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe the events for a Spot Fleet request during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html
     */
    toDescribeSpotFleetRequestHistory() {
        this.add('ec2:DescribeSpotFleetRequestHistory');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Fleet requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html
     */
    toDescribeSpotFleetRequests() {
        this.add('ec2:DescribeSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Instance requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html
     */
    toDescribeSpotInstanceRequests() {
        this.add('ec2:DescribeSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to describe the Spot Instance price history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html
     */
    toDescribeSpotPriceHistory() {
        this.add('ec2:DescribeSpotPriceHistory');
        return this;
    }
    /**
     * Grants permission to describe the stale security group rules for security groups in a specified VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html
     */
    toDescribeStaleSecurityGroups() {
        this.add('ec2:DescribeStaleSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe one or more subnets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html
     */
    toDescribeSubnets() {
        this.add('ec2:DescribeSubnets');
        return this;
    }
    /**
     * Grants permission to describe one or more tags for an Amazon EC2 resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.add('ec2:DescribeTags');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html
     */
    toDescribeTrafficMirrorFilters() {
        this.add('ec2:DescribeTrafficMirrorFilters');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html
     */
    toDescribeTrafficMirrorSessions() {
        this.add('ec2:DescribeTrafficMirrorSessions');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror targets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html
     */
    toDescribeTrafficMirrorTargets() {
        this.add('ec2:DescribeTrafficMirrorTargets');
        return this;
    }
    /**
     * Grants permission to describe one or more attachments between resources and transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html
     */
    toDescribeTransitGatewayAttachments() {
        this.add('ec2:DescribeTransitGatewayAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway multicast domains
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html
     */
    toDescribeTransitGatewayMulticastDomains() {
        this.add('ec2:DescribeTransitGatewayMulticastDomains');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway peering attachments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html
     */
    toDescribeTransitGatewayPeeringAttachments() {
        this.add('ec2:DescribeTransitGatewayPeeringAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html
     */
    toDescribeTransitGatewayRouteTables() {
        this.add('ec2:DescribeTransitGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC attachments on a transit gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html
     */
    toDescribeTransitGatewayVpcAttachments() {
        this.add('ec2:DescribeTransitGatewayVpcAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html
     */
    toDescribeTransitGateways() {
        this.add('ec2:DescribeTransitGateways');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an EBS volume
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html
     */
    toDescribeVolumeAttribute() {
        this.add('ec2:DescribeVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html
     */
    toDescribeVolumeStatus() {
        this.add('ec2:DescribeVolumeStatus');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html
     */
    toDescribeVolumes() {
        this.add('ec2:DescribeVolumes');
        return this;
    }
    /**
     * Grants permission to describe the current modification status of one or more EBS volumes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html
     */
    toDescribeVolumesModifications() {
        this.add('ec2:DescribeVolumesModifications');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html
     */
    toDescribeVpcAttribute() {
        this.add('ec2:DescribeVpcAttribute');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html
     */
    toDescribeVpcClassicLink() {
        this.add('ec2:DescribeVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink DNS support status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html
     */
    toDescribeVpcClassicLinkDnsSupport() {
        this.add('ec2:DescribeVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html
     */
    toDescribeVpcEndpointConnectionNotifications() {
        this.add('ec2:DescribeVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to describe the VPC endpoint connections to your VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html
     */
    toDescribeVpcEndpointConnections() {
        this.add('ec2:DescribeVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to describe VPC endpoint service configurations (your services)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html
     */
    toDescribeVpcEndpointServiceConfigurations() {
        this.add('ec2:DescribeVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html
     */
    toDescribeVpcEndpointServicePermissions() {
        this.add('ec2:DescribeVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html
     */
    toDescribeVpcEndpointServices() {
        this.add('ec2:DescribeVpcEndpointServices');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html
     */
    toDescribeVpcEndpoints() {
        this.add('ec2:DescribeVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC peering connections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html
     */
    toDescribeVpcPeeringConnections() {
        this.add('ec2:DescribeVpcPeeringConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html
     */
    toDescribeVpcs() {
        this.add('ec2:DescribeVpcs');
        return this;
    }
    /**
     * Grants permission to describe one or more VPN connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html
     */
    toDescribeVpnConnections() {
        this.add('ec2:DescribeVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more virtual private gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html
     */
    toDescribeVpnGateways() {
        this.add('ec2:DescribeVpnGateways');
        return this;
    }
    /**
     * Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html
     */
    toDetachClassicLinkVpc() {
        this.add('ec2:DetachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to detach an internet gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html
     */
    toDetachInternetGateway() {
        this.add('ec2:DetachInternetGateway');
        return this;
    }
    /**
     * Grants permission to detach a network interface from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html
     */
    toDetachNetworkInterface() {
        this.add('ec2:DetachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to detach an EBS volume from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html
     */
    toDetachVolume() {
        this.add('ec2:DetachVolume');
        return this;
    }
    /**
     * Grants permission to detach a virtual private gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html
     */
    toDetachVpnGateway() {
        this.add('ec2:DetachVpnGateway');
        return this;
    }
    /**
     * Grants permission to disable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html
     */
    toDisableEbsEncryptionByDefault() {
        this.add('ec2:DisableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html
     */
    toDisableFastSnapshotRestores() {
        this.add('ec2:DisableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to disable a resource attachment from propagating routes to the specified propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html
     */
    toDisableTransitGatewayRouteTablePropagation() {
        this.add('ec2:DisableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html
     */
    toDisableVgwRoutePropagation() {
        this.add('ec2:DisableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html
     */
    toDisableVpcClassicLink() {
        this.add('ec2:DisableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink DNS support for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html
     */
    toDisableVpcClassicLinkDnsSupport() {
        this.add('ec2:DisableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to disassociate an Elastic IP address from an instance or network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html
     */
    toDisassociateAddress() {
        this.add('ec2:DisassociateAddress');
        return this;
    }
    /**
     * Grants permission to disassociate a target network from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html
     */
    toDisassociateClientVpnTargetNetwork() {
        this.add('ec2:DisassociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to disassociate an IAM instance profile from a running or stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html
     */
    toDisassociateIamInstanceProfile() {
        this.add('ec2:DisassociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to disassociate a subnet from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html
     */
    toDisassociateRouteTable() {
        this.add('ec2:DisassociateRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html
     */
    toDisassociateSubnetCidrBlock() {
        this.add('ec2:DisassociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to disassociate one or more subnets from a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html
     */
    toDisassociateTransitGatewayMulticastDomain() {
        this.add('ec2:DisassociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to disassociate a resource attachment from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html
     */
    toDisassociateTransitGatewayRouteTable() {
        this.add('ec2:DisassociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html
     */
    toDisassociateVpcCidrBlock() {
        this.add('ec2:DisassociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to enable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html
     */
    toEnableEbsEncryptionByDefault() {
        this.add('ec2:EnableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html
     */
    toEnableFastSnapshotRestores() {
        this.add('ec2:EnableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to enable an attachment to propagate routes to a propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html
     */
    toEnableTransitGatewayRouteTablePropagation() {
        this.add('ec2:EnableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to enable a virtual private gateway to propagate routes to a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html
     */
    toEnableVgwRoutePropagation() {
        this.add('ec2:EnableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to enable I/O operations for a volume that had I/O operations disabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html
     */
    toEnableVolumeIO() {
        this.add('ec2:EnableVolumeIO');
        return this;
    }
    /**
     * Grants permission to enable a VPC for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html
     */
    toEnableVpcClassicLink() {
        this.add('ec2:EnableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html
     */
    toEnableVpcClassicLinkDnsSupport() {
        this.add('ec2:EnableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to download the client certificate revocation list for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html
     */
    toExportClientVpnClientCertificateRevocationList() {
        this.add('ec2:ExportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html
     */
    toExportClientVpnClientConfiguration() {
        this.add('ec2:ExportClientVpnClientConfiguration');
        return this;
    }
    /**
     * Grants permission to export an Amazon Machine Image (AMI) to a VM file
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html
     */
    toExportImage() {
        this.add('ec2:ExportImage');
        return this;
    }
    /**
     * Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html
     */
    toExportTransitGatewayRoutes() {
        this.add('ec2:ExportTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to get usage information about a Capacity Reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html
     */
    toGetCapacityReservationUsage() {
        this.add('ec2:GetCapacityReservationUsage');
        return this;
    }
    /**
     * Grants permission to describe the allocations from the specified customer-owned address pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html
     */
    toGetCoipPoolUsage() {
        this.add('ec2:GetCoipPoolUsage');
        return this;
    }
    /**
     * Grants permission to get the console output for an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html
     */
    toGetConsoleOutput() {
        this.add('ec2:GetConsoleOutput');
        return this;
    }
    /**
     * Grants permission to retrieve a JPG-format screenshot of a running instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html
     */
    toGetConsoleScreenshot() {
        this.add('ec2:GetConsoleScreenshot');
        return this;
    }
    /**
     * Grants permission to get the default credit option for CPU usage of a burstable performance instance family
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html
     */
    toGetDefaultCreditSpecification() {
        this.add('ec2:GetDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html
     */
    toGetEbsDefaultKmsKeyId() {
        this.add('ec2:GetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to describe whether EBS encryption by default is enabled for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html
     */
    toGetEbsEncryptionByDefault() {
        this.add('ec2:GetEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html
     */
    toGetHostReservationPurchasePreview() {
        this.add('ec2:GetHostReservationPurchasePreview');
        return this;
    }
    /**
     * Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html
     */
    toGetLaunchTemplateData() {
        this.add('ec2:GetLaunchTemplateData');
        return this;
    }
    /**
     * Grants permission to get information about the resources that are associated with the specified managed prefix list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListAssociations.html
     */
    toGetManagedPrefixListAssociations() {
        this.add('ec2:GetManagedPrefixListAssociations');
        return this;
    }
    /**
     * Grants permission to get information about the entries for a specified managed prefix list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListEntries.html
     */
    toGetManagedPrefixListEntries() {
        this.add('ec2:GetManagedPrefixListEntries');
        return this;
    }
    /**
     * Grants permission to retrieve the encrypted administrator password for a running Windows instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html
     */
    toGetPasswordData() {
        this.add('ec2:GetPasswordData');
        return this;
    }
    /**
     * Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html
     */
    toGetReservedInstancesExchangeQuote() {
        this.add('ec2:GetReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to list the route tables to which a resource attachment propagates routes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html
     */
    toGetTransitGatewayAttachmentPropagations() {
        this.add('ec2:GetTransitGatewayAttachmentPropagations');
        return this;
    }
    /**
     * Grants permission to get information about the associations for a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html
     */
    toGetTransitGatewayMulticastDomainAssociations() {
        this.add('ec2:GetTransitGatewayMulticastDomainAssociations');
        return this;
    }
    /**
     * Grants permission to get information about associations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html
     */
    toGetTransitGatewayRouteTableAssociations() {
        this.add('ec2:GetTransitGatewayRouteTableAssociations');
        return this;
    }
    /**
     * Grants permission to get information about the route table propagations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html
     */
    toGetTransitGatewayRouteTablePropagations() {
        this.add('ec2:GetTransitGatewayRouteTablePropagations');
        return this;
    }
    /**
     * Grants permission to upload a client certificate revocation list to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html
     */
    toImportClientVpnClientCertificateRevocationList() {
        this.add('ec2:ImportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html
     */
    toImportImage() {
        this.add('ec2:ImportImage');
        return this;
    }
    /**
     * Grants permission to create an import instance task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html
     */
    toImportInstance() {
        this.add('ec2:ImportInstance');
        return this;
    }
    /**
     * Grants permission to import a public key from an RSA key pair that was created with a third-party tool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html
     */
    toImportKeyPair() {
        this.add('ec2:ImportKeyPair');
        return this;
    }
    /**
     * Grants permission to import a disk into an EBS snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html
     */
    toImportSnapshot() {
        this.add('ec2:ImportSnapshot');
        return this;
    }
    /**
     * Grants permission to create an import volume task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html
     */
    toImportVolume() {
        this.add('ec2:ImportVolume');
        return this;
    }
    /**
     * Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html
     */
    toModifyCapacityReservation() {
        this.add('ec2:ModifyCapacityReservation');
        return this;
    }
    /**
     * Grants permission to modify a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html
     */
    toModifyClientVpnEndpoint() {
        this.add('ec2:ModifyClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to change the account level default credit option for CPU usage of burstable performance instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html
     */
    toModifyDefaultCreditSpecification() {
        this.add('ec2:ModifyDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html
     */
    toModifyEbsDefaultKmsKeyId() {
        this.add('ec2:ModifyEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to modify an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html
     */
    toModifyFleet() {
        this.add('ec2:ModifyFleet');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html
     */
    toModifyFpgaImageAttribute() {
        this.add('ec2:ModifyFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html
     */
    toModifyHosts() {
        this.add('ec2:ModifyHosts');
        return this;
    }
    /**
     * Grants permission to modify the ID format for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html
     */
    toModifyIdFormat() {
        this.add('ec2:ModifyIdFormat');
        return this;
    }
    /**
     * Grants permission to modify the ID format of a resource for a specific principal in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html
     */
    toModifyIdentityIdFormat() {
        this.add('ec2:ModifyIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html
     */
    toModifyImageAttribute() {
        this.add('ec2:ModifyImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html
     */
    toModifyInstanceAttribute() {
        this.add('ec2:ModifyInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to modify the Capacity Reservation settings for a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html
     */
    toModifyInstanceCapacityReservationAttributes() {
        this.add('ec2:ModifyInstanceCapacityReservationAttributes');
        return this;
    }
    /**
     * Grants permission to modify the credit option for CPU usage on an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html
     */
    toModifyInstanceCreditSpecification() {
        this.add('ec2:ModifyInstanceCreditSpecification');
        return this;
    }
    /**
     * Grants permission to modify the start time for a scheduled EC2 instance event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html
     */
    toModifyInstanceEventStartTime() {
        this.add('ec2:ModifyInstanceEventStartTime');
        return this;
    }
    /**
     * Grants permission to modify the metadata options for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html
     */
    toModifyInstanceMetadataOptions() {
        this.add('ec2:ModifyInstanceMetadataOptions');
        return this;
    }
    /**
     * Grants permission to modify the placement attributes for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html
     */
    toModifyInstancePlacement() {
        this.add('ec2:ModifyInstancePlacement');
        return this;
    }
    /**
     * Grants permission to modify a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html
     */
    toModifyLaunchTemplate() {
        this.add('ec2:ModifyLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to modify a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyManagedPrefixList.html
     */
    toModifyManagedPrefixList() {
        this.add('ec2:ModifyManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html
     */
    toModifyNetworkInterfaceAttribute() {
        this.add('ec2:ModifyNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to modify attributes of one or more Reserved Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html
     */
    toModifyReservedInstances() {
        this.add('ec2:ModifyReservedInstances');
        return this;
    }
    /**
     * Grants permission to add or remove permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html
     */
    toModifySnapshotAttribute() {
        this.add('ec2:ModifySnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html
     */
    toModifySpotFleetRequest() {
        this.add('ec2:ModifySpotFleetRequest');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html
     */
    toModifySubnetAttribute() {
        this.add('ec2:ModifySubnetAttribute');
        return this;
    }
    /**
     * Grants permission to allow or restrict mirroring network services
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html
     */
    toModifyTrafficMirrorFilterNetworkServices() {
        this.add('ec2:ModifyTrafficMirrorFilterNetworkServices');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html
     */
    toModifyTrafficMirrorFilterRule() {
        this.add('ec2:ModifyTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html
     */
    toModifyTrafficMirrorSession() {
        this.add('ec2:ModifyTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to modify a VPC attachment on a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html
     */
    toModifyTransitGatewayVpcAttachment() {
        this.add('ec2:ModifyTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to modify the parameters of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html
     */
    toModifyVolume() {
        this.add('ec2:ModifyVolume');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html
     */
    toModifyVolumeAttribute() {
        this.add('ec2:ModifyVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html
     */
    toModifyVpcAttribute() {
        this.add('ec2:ModifyVpcAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html
     */
    toModifyVpcEndpoint() {
        this.add('ec2:ModifyVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html
     */
    toModifyVpcEndpointConnectionNotification() {
        this.add('ec2:ModifyVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to modify the attributes of a VPC endpoint service configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html
     */
    toModifyVpcEndpointServiceConfiguration() {
        this.add('ec2:ModifyVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to modify the permissions for a VPC endpoint service
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html
     */
    toModifyVpcEndpointServicePermissions() {
        this.add('ec2:ModifyVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to modify the VPC peering connection options on one side of a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html
     */
    toModifyVpcPeeringConnectionOptions() {
        this.add('ec2:ModifyVpcPeeringConnectionOptions');
        return this;
    }
    /**
     * Grants permission to modify the instance tenancy attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html
     */
    toModifyVpcTenancy() {
        this.add('ec2:ModifyVpcTenancy');
        return this;
    }
    /**
     * Grants permission to modify the target gateway of a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html
     */
    toModifyVpnConnection() {
        this.add('ec2:ModifyVpnConnection');
        return this;
    }
    /**
     * Grants permission to modify the certificate for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate
     */
    toModifyVpnTunnelCertificate() {
        this.add('ec2:ModifyVpnTunnelCertificate');
        return this;
    }
    /**
     * Grants permission to modify the options for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html
     */
    toModifyVpnTunnelOptions() {
        this.add('ec2:ModifyVpnTunnelOptions');
        return this;
    }
    /**
     * Grants permission to enable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html
     */
    toMonitorInstances() {
        this.add('ec2:MonitorInstances');
        return this;
    }
    /**
     * Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html
     */
    toMoveAddressToVpc() {
        this.add('ec2:MoveAddressToVpc');
        return this;
    }
    /**
     * Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html
     */
    toProvisionByoipCidr() {
        this.add('ec2:ProvisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to purchase a reservation with configurations that match those of a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html
     */
    toPurchaseHostReservation() {
        this.add('ec2:PurchaseHostReservation');
        return this;
    }
    /**
     * Grants permission to purchase a Reserved Instance offering
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html
     */
    toPurchaseReservedInstancesOffering() {
        this.add('ec2:PurchaseReservedInstancesOffering');
        return this;
    }
    /**
     * Grants permission to purchase one or more Scheduled Instances with a specified schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html
     */
    toPurchaseScheduledInstances() {
        this.add('ec2:PurchaseScheduledInstances');
        return this;
    }
    /**
     * Grants permission to request a reboot of one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html
     */
    toRebootInstances() {
        this.add('ec2:RebootInstances');
        return this;
    }
    /**
     * Grants permission to register an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html
     */
    toRegisterImage() {
        this.add('ec2:RegisterImage');
        return this;
    }
    /**
     * Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html
     */
    toRegisterInstanceEventNotificationAttributes() {
        this.add('ec2:RegisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html
     */
    toRegisterTransitGatewayMulticastGroupMembers() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html
     */
    toRegisterTransitGatewayMulticastGroupSources() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to reject a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html
     */
    toRejectTransitGatewayPeeringAttachment() {
        this.add('ec2:RejectTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to reject a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html
     */
    toRejectTransitGatewayVpcAttachment() {
        this.add('ec2:RejectTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html
     */
    toRejectVpcEndpointConnections() {
        this.add('ec2:RejectVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to reject a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html
     */
    toRejectVpcPeeringConnection() {
        this.add('ec2:RejectVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to release an Elastic IP address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html
     */
    toReleaseAddress() {
        this.add('ec2:ReleaseAddress');
        return this;
    }
    /**
     * Grants permission to release one or more On-Demand Dedicated Hosts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html
     */
    toReleaseHosts() {
        this.add('ec2:ReleaseHosts');
        return this;
    }
    /**
     * Grants permission to replace an IAM instance profile for an instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html
     */
    toReplaceIamInstanceProfileAssociation() {
        this.add('ec2:ReplaceIamInstanceProfileAssociation');
        return this;
    }
    /**
     * Grants permission to change which network ACL a subnet is associated with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html
     */
    toReplaceNetworkAclAssociation() {
        this.add('ec2:ReplaceNetworkAclAssociation');
        return this;
    }
    /**
     * Grants permission to replace an entry (rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html
     */
    toReplaceNetworkAclEntry() {
        this.add('ec2:ReplaceNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to replace a route within a route table in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html
     */
    toReplaceRoute() {
        this.add('ec2:ReplaceRoute');
        return this;
    }
    /**
     * Grants permission to change the route table that is associated with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html
     */
    toReplaceRouteTableAssociation() {
        this.add('ec2:ReplaceRouteTableAssociation');
        return this;
    }
    /**
     * Grants permission to replace a route in a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html
     */
    toReplaceTransitGatewayRoute() {
        this.add('ec2:ReplaceTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to submit feedback about the status of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html
     */
    toReportInstanceStatus() {
        this.add('ec2:ReportInstanceStatus');
        return this;
    }
    /**
     * Grants permission to create a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html
     */
    toRequestSpotFleet() {
        this.add('ec2:RequestSpotFleet');
        return this;
    }
    /**
     * Grants permission to create a Spot Instance request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html
     */
    toRequestSpotInstances() {
        this.add('ec2:RequestSpotInstances');
        return this;
    }
    /**
     * Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html
     */
    toResetEbsDefaultKmsKeyId() {
        this.add('ec2:ResetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html
     */
    toResetFpgaImageAttribute() {
        this.add('ec2:ResetFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html
     */
    toResetImageAttribute() {
        this.add('ec2:ResetImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an instance to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html
     */
    toResetInstanceAttribute() {
        this.add('ec2:ResetInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html
     */
    toResetNetworkInterfaceAttribute() {
        this.add('ec2:ResetNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to reset permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html
     */
    toResetSnapshotAttribute() {
        this.add('ec2:ResetSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html
     */
    toRestoreAddressToClassic() {
        this.add('ec2:RestoreAddressToClassic');
        return this;
    }
    /**
     * Grants permission to restore the entries from a previous version of a managed prefix list to a new version of the prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreManagedPrefixListVersion.html
     */
    toRestoreManagedPrefixListVersion() {
        this.add('ec2:RestoreManagedPrefixListVersion');
        return this;
    }
    /**
     * Grants permission to remove an inbound authorization rule from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html
     */
    toRevokeClientVpnIngress() {
        this.add('ec2:RevokeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to remove one or more outbound rules from a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html
     */
    toRevokeSecurityGroupEgress() {
        this.add('ec2:RevokeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to remove one or more inbound rules from a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html
     */
    toRevokeSecurityGroupIngress() {
        this.add('ec2:RevokeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to launch one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html
     */
    toRunInstances() {
        this.add('ec2:RunInstances');
        return this;
    }
    /**
     * Grants permission to launch one or more Scheduled Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html
     */
    toRunScheduledInstances() {
        this.add('ec2:RunScheduledInstances');
        return this;
    }
    /**
     * Grants permission to search for routes in a local gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html
     */
    toSearchLocalGatewayRoutes() {
        this.add('ec2:SearchLocalGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to search for groups, sources, and members in a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html
     */
    toSearchTransitGatewayMulticastGroups() {
        this.add('ec2:SearchTransitGatewayMulticastGroups');
        return this;
    }
    /**
     * Grants permission to search for routes in a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html
     */
    toSearchTransitGatewayRoutes() {
        this.add('ec2:SearchTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to send a diagnostic interrupt to an Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html
     */
    toSendDiagnosticInterrupt() {
        this.add('ec2:SendDiagnosticInterrupt');
        return this;
    }
    /**
     * Grants permission to start a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html
     */
    toStartInstances() {
        this.add('ec2:StartInstances');
        return this;
    }
    /**
     * Grants permission to start the private DNS verification process for a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html
     */
    toStartVpcEndpointServicePrivateDnsVerification() {
        this.add('ec2:StartVpcEndpointServicePrivateDnsVerification');
        return this;
    }
    /**
     * Grants permission to stop an Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html
     */
    toStopInstances() {
        this.add('ec2:StopInstances');
        return this;
    }
    /**
     * Grants permission to terminate active Client VPN endpoint connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html
     */
    toTerminateClientVpnConnections() {
        this.add('ec2:TerminateClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to shut down one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html
     */
    toTerminateInstances() {
        this.add('ec2:TerminateInstances');
        return this;
    }
    /**
     * Grants permission to unassign one or more IPv6 addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html
     */
    toUnassignIpv6Addresses() {
        this.add('ec2:UnassignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to unassign one or more secondary private IP addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html
     */
    toUnassignPrivateIpAddresses() {
        this.add('ec2:UnassignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to disable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html
     */
    toUnmonitorInstances() {
        this.add('ec2:UnmonitorInstances');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more outbound rules in a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html
     */
    toUpdateSecurityGroupRuleDescriptionsEgress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsEgress');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more inbound rules in a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html
     */
    toUpdateSecurityGroupRuleDescriptionsIngress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsIngress');
        return this;
    }
    /**
     * Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html
     */
    toWithdrawByoipCidr() {
        this.add('ec2:WithdrawByoipCidr');
        return this;
    }
    /**
     * Adds a resource of type capacity-reservation to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html
     *
     * @param capacityReservationId - Identifier for the capacityReservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCapacityReservation(capacityReservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}';
        arn = arn.replace('${CapacityReservationId}', capacityReservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type carrier-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/Carrier_Gateway.html
     *
     * @param carrierGatewayId - Identifier for the carrierGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCarrierGateway(carrierGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:carrier-gateway/${CarrierGatewayId}';
        arn = arn.replace('${CarrierGatewayId}', carrierGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type client-vpn-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html
     *
     * @param clientVpnEndpointId - Identifier for the clientVpnEndpointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onClientVpnEndpoint(clientVpnEndpointId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}';
        arn = arn.replace('${ClientVpnEndpointId}', clientVpnEndpointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type customer-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param customerGatewayId - Identifier for the customerGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCustomerGateway(customerGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}';
        arn = arn.replace('${CustomerGatewayId}', customerGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dedicated-host to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html
     *
     * @param hostId - Identifier for the hostId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAutoPlacement()
     * - .ifAvailabilityZone()
     * - .ifHostRecovery()
     * - .ifInstanceType()
     * - .ifQuantity()
     * - .ifResourceTag()
     */
    onDedicatedHost(hostId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}';
        arn = arn.replace('${HostId}', hostId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dhcp-options to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html
     *
     * @param dhcpOptionsId - Identifier for the dhcpOptionsId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onDhcpOptions(dhcpOptionsId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}';
        arn = arn.replace('${DhcpOptionsId}', dhcpOptionsId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type egress-only-internet-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/egress-only-internet-gateway.html
     *
     * @param egressOnlyInternetGatewayId - Identifier for the egressOnlyInternetGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onEgressOnlyInternetGateway(egressOnlyInternetGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:egress-only-internet-gateway/${EgressOnlyInternetGatewayId}';
        arn = arn.replace('${EgressOnlyInternetGatewayId}', egressOnlyInternetGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-gpu to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html
     *
     * @param elasticGpuId - Identifier for the elasticGpuId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifElasticGpuType()
     */
    onElasticGpu(elasticGpuId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}';
        arn = arn.replace('${ElasticGpuId}', elasticGpuId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-inference to the statement
     *
     * https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html
     *
     * @param elasticInferenceAcceleratorId - Identifier for the elasticInferenceAcceleratorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onElasticInference(elasticInferenceAcceleratorId, account, region, partition) {
        var arn = 'arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}';
        arn = arn.replace('${ElasticInferenceAcceleratorId}', elasticInferenceAcceleratorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type export-instance-task to the statement
     *
     * https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport.html
     *
     * @param exportTaskId - Identifier for the exportTaskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onExportInstanceTask(exportTaskId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:export-instance-task/${ExportTaskId}';
        arn = arn.replace('${ExportTaskId}', exportTaskId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:fleet/${FleetId}';
        arn = arn.replace('${FleetId}', fleetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fpga-image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param fpgaImageId - Identifier for the fpgaImageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifOwner()
     * - .ifPublic()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onFpgaImage(fpgaImageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}';
        arn = arn.replace('${FpgaImageId}', fpgaImageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html
     *
     * @param imageId - Identifier for the imageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifImageType()
     * - .ifOwner()
     * - .ifPublic()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifRootDeviceType()
     */
    onImage(imageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::image/${ImageId}';
        arn = arn.replace('${ImageId}', imageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifEbsOptimized()
     * - .ifInstanceProfile()
     * - .ifInstanceType()
     * - .ifPlacementGroup()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifRootDeviceType()
     * - .ifTenancy()
     */
    onInstance(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type internet-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html
     *
     * @param internetGatewayId - Identifier for the internetGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onInternetGateway(internetGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}';
        arn = arn.replace('${InternetGatewayId}', internetGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key-pair to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html
     *
     * @param keyPairName - Identifier for the keyPairName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onKeyPair(keyPairName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}';
        arn = arn.replace('${KeyPairName}', keyPairName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type launch-template to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html
     *
     * @param launchTemplateId - Identifier for the launchTemplateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLaunchTemplate(launchTemplateId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}';
        arn = arn.replace('${LaunchTemplateId}', launchTemplateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayId - Identifier for the localGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGateway(localGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}';
        arn = arn.replace('${LocalGatewayId}', localGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableId - Identifier for the localGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTable(localGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}';
        arn = arn.replace('${LocalGatewayRouteTableId}', localGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-virtual-interface-group-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVirtualInterfaceGroupAssociationId - Identifier for the localGatewayRouteTableVirtualInterfaceGroupAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTableVirtualInterfaceGroupAssociation(localGatewayRouteTableVirtualInterfaceGroupAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}', localGatewayRouteTableVirtualInterfaceGroupAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-vpc-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVpcAssociationId - Identifier for the localGatewayRouteTableVpcAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTableVpcAssociation(localGatewayRouteTableVpcAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVpcAssociationId}', localGatewayRouteTableVpcAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceId - Identifier for the localGatewayVirtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayVirtualInterface(localGatewayVirtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceId}', localGatewayVirtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface-group to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceGroupId - Identifier for the localGatewayVirtualInterfaceGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayVirtualInterfaceGroup(localGatewayVirtualInterfaceGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceGroupId}', localGatewayVirtualInterfaceGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type natgateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     *
     * @param natGatewayId - Identifier for the natGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onNatgateway(natGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:natgateway/${NatGatewayId}';
        arn = arn.replace('${NatGatewayId}', natGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-acl to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html
     *
     * @param naclId - Identifier for the naclId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onNetworkAcl(naclId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}';
        arn = arn.replace('${NaclId}', naclId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-interface to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html
     *
     * @param networkInterfaceId - Identifier for the networkInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAssociatePublicIpAddress()
     * - .ifAuthorizedService()
     * - .ifAvailabilityZone()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifSubnet()
     * - .ifVpc()
     */
    onNetworkInterface(networkInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}';
        arn = arn.replace('${NetworkInterfaceId}', networkInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type placement-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html
     *
     * @param placementGroupName - Identifier for the placementGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifPlacementGroupStrategy()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onPlacementGroup(placementGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}';
        arn = arn.replace('${PlacementGroupName}', placementGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type prefix-list to the statement
     *
     * @param prefixListId - Identifier for the prefixListId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onPrefixList(prefixListId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:prefix-list/${PrefixListId}';
        arn = arn.replace('${PrefixListId}', prefixListId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reserved-instances to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html
     *
     * @param reservationId - Identifier for the reservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifInstanceType()
     * - .ifRegion()
     * - .ifReservedInstancesOfferingType()
     * - .ifResourceTag()
     * - .ifTenancy()
     */
    onReservedInstances(reservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}';
        arn = arn.replace('${ReservationId}', reservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html
     *
     * @param routeTableId - Identifier for the routeTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onRouteTable(routeTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}';
        arn = arn.replace('${RouteTableId}', routeTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type security-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html
     *
     * @param securityGroupId - Identifier for the securityGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onSecurityGroup(securityGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}';
        arn = arn.replace('${SecurityGroupId}', securityGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifOwner()
     * - .ifParentVolume()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifSnapshotTime()
     * - .ifVolumeSize()
     */
    onSnapshot(snapshotId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}';
        arn = arn.replace('${SnapshotId}', snapshotId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type spot-instance-request to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html
     *
     * @param spotInstanceRequestId - Identifier for the spotInstanceRequestId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onSpotInstanceRequest(spotInstanceRequestId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}';
        arn = arn.replace('${SpotInstanceRequestId}', spotInstanceRequestId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnet to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param subnetId - Identifier for the subnetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onSubnet(subnetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}';
        arn = arn.replace('${SubnetId}', subnetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-session to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html
     *
     * @param trafficMirrorSessionId - Identifier for the trafficMirrorSessionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorSession(trafficMirrorSessionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}';
        arn = arn.replace('${TrafficMirrorSessionId}', trafficMirrorSessionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-target to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html
     *
     * @param trafficMirrorTargetId - Identifier for the trafficMirrorTargetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorTarget(trafficMirrorTargetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}';
        arn = arn.replace('${TrafficMirrorTargetId}', trafficMirrorTargetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterId - Identifier for the trafficMirrorFilterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorFilter(trafficMirrorFilterId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}';
        arn = arn.replace('${TrafficMirrorFilterId}', trafficMirrorFilterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter-rule to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterRuleId - Identifier for the trafficMirrorFilterRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifRegion()
     */
    onTrafficMirrorFilterRule(trafficMirrorFilterRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}';
        arn = arn.replace('${TrafficMirrorFilterRuleId}', trafficMirrorFilterRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-attachment to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayAttachmentId - Identifier for the transitGatewayAttachmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayAttachment(transitGatewayAttachmentId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}';
        arn = arn.replace('${TransitGatewayAttachmentId}', transitGatewayAttachmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-multicast-domain to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html
     *
     * @param transitGatewayMulticastDomainId - Identifier for the transitGatewayMulticastDomainId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayMulticastDomain(transitGatewayMulticastDomainId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}';
        arn = arn.replace('${TransitGatewayMulticastDomainId}', transitGatewayMulticastDomainId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayRouteTableId - Identifier for the transitGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayRouteTable(transitGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}';
        arn = arn.replace('${TransitGatewayRouteTableId}', transitGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayId - Identifier for the transitGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGateway(transitGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}';
        arn = arn.replace('${TransitGatewayId}', transitGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html
     *
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifEncrypted()
     * - .ifParentSnapshot()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVolumeIops()
     * - .ifVolumeSize()
     * - .ifVolumeType()
     */
    onVolume(volumeId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}';
        arn = arn.replace('${VolumeId}', volumeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param vpcId - Identifier for the vpcId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifTenancy()
     */
    onVpc(vpcId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}';
        arn = arn.replace('${VpcId}', vpcId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceId - Identifier for the vpceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpceServiceName()
     * - .ifVpceServiceOwner()
     */
    onVpcEndpoint(vpceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}';
        arn = arn.replace('${VpceId}', vpceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint-service to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceServiceId - Identifier for the vpceServiceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpceServicePrivateDnsName()
     */
    onVpcEndpointService(vpceServiceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}';
        arn = arn.replace('${VpceServiceId}', vpceServiceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-flow-log to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html
     *
     * @param vpcFlowLogId - Identifier for the vpcFlowLogId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onVpcFlowLog(vpcFlowLogId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}';
        arn = arn.replace('${VpcFlowLogId}', vpcFlowLogId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-peering-connection to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html
     *
     * @param vpcPeeringConnectionId - Identifier for the vpcPeeringConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAccepterVpc()
     * - .ifRegion()
     * - .ifRequesterVpc()
     * - .ifResourceTag()
     */
    onVpcPeeringConnection(vpcPeeringConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}';
        arn = arn.replace('${VpcPeeringConnectionId}', vpcPeeringConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-connection to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnConnectionId - Identifier for the vpnConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAuthenticationType()
     * - .ifDPDTimeoutSeconds()
     * - .ifGatewayType()
     * - .ifIKEVersions()
     * - .ifInsideTunnelCidr()
     * - .ifPhase1DHGroupNumbers()
     * - .ifPhase1EncryptionAlgorithms()
     * - .ifPhase1IntegrityAlgorithms()
     * - .ifPhase1LifetimeSeconds()
     * - .ifPhase2DHGroupNumbers()
     * - .ifPhase2EncryptionAlgorithms()
     * - .ifPhase2IntegrityAlgorithms()
     * - .ifPhase2LifetimeSeconds()
     * - .ifPresharedKeys()
     * - .ifRegion()
     * - .ifRekeyFuzzPercentage()
     * - .ifRekeyMarginTimeSeconds()
     * - .ifResourceTag()
     * - .ifRoutingType()
     */
    onVpnConnection(vpnConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}';
        arn = arn.replace('${VpnConnectionId}', vpnConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnGatewayId - Identifier for the vpnGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVpnGateway(vpnGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}';
        arn = arn.replace('${VpnGatewayId}', vpnGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html#control-tagging
     *
     * Applies to actions:
     * - .toCopySnapshot()
     * - .toCreateCapacityReservation()
     * - .toCreateCarrierGateway()
     * - .toCreateClientVpnEndpoint()
     * - .toCreateDhcpOptions()
     * - .toCreateEgressOnlyInternetGateway()
     * - .toCreateFleet()
     * - .toCreateFlowLogs()
     * - .toCreateFpgaImage()
     * - .toCreateInstanceExportTask()
     * - .toCreateInternetGateway()
     * - .toCreateKeyPair()
     * - .toCreateLaunchTemplate()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateManagedPrefixList()
     * - .toCreateNatGateway()
     * - .toCreateNetworkAcl()
     * - .toCreateNetworkInterface()
     * - .toCreatePlacementGroup()
     * - .toCreateSecurityGroup()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateSubnet()
     * - .toCreateTags()
     * - .toCreateTrafficMirrorFilter()
     * - .toCreateTrafficMirrorSession()
     * - .toCreateTrafficMirrorTarget()
     * - .toCreateTransitGateway()
     * - .toCreateTransitGatewayMulticastDomain()
     * - .toCreateTransitGatewayPeeringAttachment()
     * - .toCreateTransitGatewayRouteTable()
     * - .toCreateTransitGatewayVpcAttachment()
     * - .toCreateVolume()
     * - .toCreateVpc()
     * - .toCreateVpcEndpoint()
     * - .toCreateVpcEndpointServiceConfiguration()
     * - .toDeleteTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - carrier-gateway
     * - client-vpn-endpoint
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpn-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTagExists(value, operator) {
        return this.if(`aws:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .toAcceptTransitGatewayPeeringAttachment()
     * - .toAcceptTransitGatewayVpcAttachment()
     * - .toAcceptVpcEndpointConnections()
     * - .toAcceptVpcPeeringConnection()
     * - .toApplySecurityGroupsToClientVpnTargetNetwork()
     * - .toAssociateClientVpnTargetNetwork()
     * - .toAssociateIamInstanceProfile()
     * - .toAssociateTransitGatewayMulticastDomain()
     * - .toAssociateTransitGatewayRouteTable()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toAuthorizeClientVpnIngress()
     * - .toAuthorizeSecurityGroupEgress()
     * - .toAuthorizeSecurityGroupIngress()
     * - .toCancelCapacityReservation()
     * - .toCreateClientVpnRoute()
     * - .toCreateFlowLogs()
     * - .toCreateLaunchTemplateVersion()
     * - .toCreateLocalGatewayRoute()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateRoute()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateTrafficMirrorFilterRule()
     * - .toCreateTrafficMirrorSession()
     * - .toCreateTrafficMirrorTarget()
     * - .toCreateTransitGatewayMulticastDomain()
     * - .toCreateTransitGatewayPeeringAttachment()
     * - .toCreateTransitGatewayRoute()
     * - .toCreateTransitGatewayRouteTable()
     * - .toCreateTransitGatewayVpcAttachment()
     * - .toCreateVpcEndpoint()
     * - .toCreateVpcPeeringConnection()
     * - .toDeleteCarrierGateway()
     * - .toDeleteClientVpnEndpoint()
     * - .toDeleteClientVpnRoute()
     * - .toDeleteCustomerGateway()
     * - .toDeleteDhcpOptions()
     * - .toDeleteFlowLogs()
     * - .toDeleteInternetGateway()
     * - .toDeleteLaunchTemplate()
     * - .toDeleteLaunchTemplateVersions()
     * - .toDeleteLocalGatewayRoute()
     * - .toDeleteLocalGatewayRouteTableVpcAssociation()
     * - .toDeleteManagedPrefixList()
     * - .toDeleteNetworkAcl()
     * - .toDeleteNetworkAclEntry()
     * - .toDeleteRoute()
     * - .toDeleteRouteTable()
     * - .toDeleteSecurityGroup()
     * - .toDeleteSnapshot()
     * - .toDeleteTags()
     * - .toDeleteTrafficMirrorFilter()
     * - .toDeleteTrafficMirrorFilterRule()
     * - .toDeleteTrafficMirrorSession()
     * - .toDeleteTrafficMirrorTarget()
     * - .toDeleteTransitGateway()
     * - .toDeleteTransitGatewayMulticastDomain()
     * - .toDeleteTransitGatewayPeeringAttachment()
     * - .toDeleteTransitGatewayRoute()
     * - .toDeleteTransitGatewayRouteTable()
     * - .toDeleteTransitGatewayVpcAttachment()
     * - .toDeleteVolume()
     * - .toDeleteVpcEndpointServiceConfigurations()
     * - .toDeleteVpcEndpoints()
     * - .toDeleteVpcPeeringConnection()
     * - .toDeregisterTransitGatewayMulticastGroupMembers()
     * - .toDeregisterTransitGatewayMulticastGroupSources()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toDisableTransitGatewayRouteTablePropagation()
     * - .toDisableVpcClassicLink()
     * - .toDisassociateClientVpnTargetNetwork()
     * - .toDisassociateIamInstanceProfile()
     * - .toDisassociateTransitGatewayMulticastDomain()
     * - .toDisassociateTransitGatewayRouteTable()
     * - .toEnableFastSnapshotRestores()
     * - .toEnableTransitGatewayRouteTablePropagation()
     * - .toEnableVpcClassicLink()
     * - .toGetConsoleScreenshot()
     * - .toImportClientVpnClientCertificateRevocationList()
     * - .toModifyCapacityReservation()
     * - .toModifyClientVpnEndpoint()
     * - .toModifyLaunchTemplate()
     * - .toModifyManagedPrefixList()
     * - .toModifySnapshotAttribute()
     * - .toModifyTrafficMirrorFilterNetworkServices()
     * - .toModifyTrafficMirrorFilterRule()
     * - .toModifyTrafficMirrorSession()
     * - .toModifyTransitGatewayVpcAttachment()
     * - .toModifyVpcEndpoint()
     * - .toModifyVpcEndpointServiceConfiguration()
     * - .toModifyVpcEndpointServicePermissions()
     * - .toModifyVpnConnection()
     * - .toModifyVpnTunnelOptions()
     * - .toRebootInstances()
     * - .toRegisterTransitGatewayMulticastGroupMembers()
     * - .toRegisterTransitGatewayMulticastGroupSources()
     * - .toRejectTransitGatewayPeeringAttachment()
     * - .toRejectTransitGatewayVpcAttachment()
     * - .toRejectVpcEndpointConnections()
     * - .toRejectVpcPeeringConnection()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toReplaceRoute()
     * - .toReplaceTransitGatewayRoute()
     * - .toRestoreManagedPrefixListVersion()
     * - .toRevokeClientVpnIngress()
     * - .toRevokeSecurityGroupEgress()
     * - .toRevokeSecurityGroupIngress()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStartVpcEndpointServicePrivateDnsVerification()
     * - .toStopInstances()
     * - .toTerminateClientVpnConnections()
     * - .toTerminateInstances()
     * - .toUpdateSecurityGroupRuleDescriptionsEgress()
     * - .toUpdateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - carrier-gateway
     * - client-vpn-endpoint
     * - customer-gateway
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html#control-tagging
     *
     * Applies to actions:
     * - .toCopySnapshot()
     * - .toCreateCapacityReservation()
     * - .toCreateCarrierGateway()
     * - .toCreateClientVpnEndpoint()
     * - .toCreateDhcpOptions()
     * - .toCreateEgressOnlyInternetGateway()
     * - .toCreateFleet()
     * - .toCreateFlowLogs()
     * - .toCreateFpgaImage()
     * - .toCreateInstanceExportTask()
     * - .toCreateInternetGateway()
     * - .toCreateKeyPair()
     * - .toCreateLaunchTemplate()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateManagedPrefixList()
     * - .toCreateNatGateway()
     * - .toCreateNetworkAcl()
     * - .toCreateNetworkInterface()
     * - .toCreatePlacementGroup()
     * - .toCreateSecurityGroup()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateSubnet()
     * - .toCreateTags()
     * - .toCreateTrafficMirrorFilter()
     * - .toCreateTrafficMirrorSession()
     * - .toCreateTrafficMirrorTarget()
     * - .toCreateTransitGateway()
     * - .toCreateTransitGatewayMulticastDomain()
     * - .toCreateTransitGatewayPeeringAttachment()
     * - .toCreateTransitGatewayRouteTable()
     * - .toCreateTransitGatewayVpcAttachment()
     * - .toCreateVolume()
     * - .toCreateVpc()
     * - .toCreateVpcEndpoint()
     * - .toCreateVpcEndpointServiceConfiguration()
     * - .toDeleteTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - carrier-gateway
     * - client-vpn-endpoint
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an accepter VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * Applies to actions:
     * - .toAcceptVpcPeeringConnection()
     * - .toCreateVpcPeeringConnection()
     * - .toDeleteVpcPeeringConnection()
     * - .toRejectVpcPeeringConnection()
     *
     * Applies to resource types:
     * - vpc-peering-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAccepterVpc(value, operator) {
        return this.if(`ec2:AccepterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the user wants to associate a public IP address with the instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAssociatePublicIpAddress(value) {
        return this.if(`ec2:AssociatePublicIpAddress`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the authentication type for the VPN tunnel endpoints
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthenticationType(value, operator) {
        return this.if(`ec2:AuthenticationType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS service that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateNetworkInterfacePermission()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedService(value, operator) {
        return this.if(`ec2:AuthorizedService`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an IAM principal that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateNetworkInterfacePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedUser(value, operator) {
        return this.if(`ec2:AuthorizedUser`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Auto Placement properties of a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAutoPlacement(value, operator) {
        return this.if(`ec2:AutoPlacement`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of an Availability Zone in an AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateFlowLogs()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toDisassociateIamInstanceProfile()
     * - .toEnableFastSnapshotRestores()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - dedicated-host
     * - instance
     * - network-interface
     * - reserved-instances
     * - subnet
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAvailabilityZone(value, operator) {
        return this.if(`ec2:AvailabilityZone`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of a resource-creating API action
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html
     *
     * Applies to actions:
     * - .toCreateTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`ec2:CreateAction`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the duration after which DPD timeout occurs on a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifDPDTimeoutSeconds(value, operator) {
        return this.if(`ec2:DPDTimeoutSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether the instance is enabled for EBS optimization
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEbsOptimized(value) {
        return this.if(`ec2:EbsOptimized`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the type of Elastic Graphics accelerator
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - elastic-gpu
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifElasticGpuType(value, operator) {
        return this.if(`ec2:ElasticGpuType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the EBS volume is encrypted
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEncrypted(value) {
        return this.if(`ec2:Encrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the gateway type for a VPN endpoint on the AWS side of a VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnConnection()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGatewayType(value, operator) {
        return this.if(`ec2:GatewayType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether host recovery is enabled for a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifHostRecovery(value, operator) {
        return this.if(`ec2:HostRecovery`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the internet key exchange (IKE) versions that are permitted for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIKEVersions(value, operator) {
        return this.if(`ec2:IKEVersions`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of image (machine, aki, or ari)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - image
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageType(value, operator) {
        return this.if(`ec2:ImageType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the range of inside IP addresses for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInsideTunnelCidr(value, operator) {
        return this.if(`ec2:InsideTunnelCidr`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the market or purchasing option of an instance (on-demand or spot)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceMarketType(value, operator) {
        return this.if(`ec2:InstanceMarketType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an instance profile
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifInstanceProfile(value, operator) {
        return this.if(`ec2:InstanceProfile`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - dedicated-host
     * - instance
     * - reserved-instances
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`ec2:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether users are able to override resources that are specified in the launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsLaunchTemplateResource(value) {
        return this.if(`ec2:IsLaunchTemplateResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of a launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifLaunchTemplate(value, operator) {
        return this.if(`ec2:LaunchTemplate`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the HTTP endpoint is enabled for the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpEndpoint(value, operator) {
        return this.if(`ec2:MetadataHttpEndpoint`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the allowed number of hops when calling the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMetadataHttpPutResponseHopLimit(value, operator) {
        return this.if(`ec2:MetadataHttpPutResponseHopLimit`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether tokens are required when calling the instance metadata service (optional or required)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpTokens(value, operator) {
        return this.if(`ec2:MetadataHttpTokens`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the owner of the resource (amazon, aws-marketplace, or an AWS account ID)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toDeleteSnapshot()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - fpga-image
     * - image
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwner(value, operator) {
        return this.if(`ec2:Owner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the parent snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentSnapshot(value, operator) {
        return this.if(`ec2:ParentSnapshot`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the parent volume from which the snapshot was created
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDeleteSnapshot()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentVolume(value, operator) {
        return this.if(`ec2:ParentVolume`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of permission for a resource (INSTANCE-ATTACH or EIP-ASSOCIATE)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateNetworkInterfacePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermission(value, operator) {
        return this.if(`ec2:Permission`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase1DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.htmls
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase1EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase1IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 1 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase1LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase2DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase2EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase2IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 2 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase2LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of the placement group
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPlacementGroup(value, operator) {
        return this.if(`ec2:PlacementGroup`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the instance placement strategy used by the placement group (cluster, spread, or partition)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - placement-group
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPlacementGroupStrategy(value, operator) {
        return this.if(`ec2:PlacementGroupStrategy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the pre-shared key (PSK) used to establish the initial IKE security association between a virtual private gateway and a customer gateway
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPresharedKeys(value, operator) {
        return this.if(`ec2:PresharedKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the product code that is associated with the AMI
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProductCode(value, operator) {
        return this.if(`ec2:ProductCode`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the image has public launch permissions
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - fpga-image
     * - image
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPublic(value) {
        return this.if(`ec2:Public`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the number of Dedicated Hosts in a request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifQuantity(value, operator) {
        return this.if(`ec2:Quantity`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the name of the AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAcceptTransitGatewayPeeringAttachment()
     * - .toAcceptTransitGatewayVpcAttachment()
     * - .toAcceptVpcEndpointConnections()
     * - .toAcceptVpcPeeringConnection()
     * - .toApplySecurityGroupsToClientVpnTargetNetwork()
     * - .toAssociateClientVpnTargetNetwork()
     * - .toAssociateIamInstanceProfile()
     * - .toAssociateTransitGatewayMulticastDomain()
     * - .toAssociateTransitGatewayRouteTable()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toAuthorizeClientVpnIngress()
     * - .toAuthorizeSecurityGroupEgress()
     * - .toAuthorizeSecurityGroupIngress()
     * - .toCancelCapacityReservation()
     * - .toCopySnapshot()
     * - .toCreateCapacityReservation()
     * - .toCreateCarrierGateway()
     * - .toCreateClientVpnEndpoint()
     * - .toCreateClientVpnRoute()
     * - .toCreateDhcpOptions()
     * - .toCreateEgressOnlyInternetGateway()
     * - .toCreateFleet()
     * - .toCreateFlowLogs()
     * - .toCreateFpgaImage()
     * - .toCreateInstanceExportTask()
     * - .toCreateInternetGateway()
     * - .toCreateKeyPair()
     * - .toCreateLaunchTemplate()
     * - .toCreateLaunchTemplateVersion()
     * - .toCreateLocalGatewayRoute()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateManagedPrefixList()
     * - .toCreateNatGateway()
     * - .toCreateNetworkAcl()
     * - .toCreateNetworkInterface()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreatePlacementGroup()
     * - .toCreateRoute()
     * - .toCreateSecurityGroup()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateSubnet()
     * - .toCreateTags()
     * - .toCreateTrafficMirrorFilter()
     * - .toCreateTrafficMirrorFilterRule()
     * - .toCreateTrafficMirrorSession()
     * - .toCreateTrafficMirrorTarget()
     * - .toCreateTransitGateway()
     * - .toCreateTransitGatewayMulticastDomain()
     * - .toCreateTransitGatewayPeeringAttachment()
     * - .toCreateTransitGatewayRoute()
     * - .toCreateTransitGatewayRouteTable()
     * - .toCreateTransitGatewayVpcAttachment()
     * - .toCreateVolume()
     * - .toCreateVpc()
     * - .toCreateVpcEndpoint()
     * - .toCreateVpcEndpointServiceConfiguration()
     * - .toCreateVpcPeeringConnection()
     * - .toCreateVpnConnection()
     * - .toDeleteCarrierGateway()
     * - .toDeleteClientVpnEndpoint()
     * - .toDeleteClientVpnRoute()
     * - .toDeleteCustomerGateway()
     * - .toDeleteDhcpOptions()
     * - .toDeleteFlowLogs()
     * - .toDeleteInternetGateway()
     * - .toDeleteLaunchTemplate()
     * - .toDeleteLaunchTemplateVersions()
     * - .toDeleteLocalGatewayRoute()
     * - .toDeleteLocalGatewayRouteTableVpcAssociation()
     * - .toDeleteManagedPrefixList()
     * - .toDeleteNetworkAcl()
     * - .toDeleteNetworkAclEntry()
     * - .toDeleteRoute()
     * - .toDeleteRouteTable()
     * - .toDeleteSecurityGroup()
     * - .toDeleteSnapshot()
     * - .toDeleteTags()
     * - .toDeleteTrafficMirrorFilter()
     * - .toDeleteTrafficMirrorFilterRule()
     * - .toDeleteTrafficMirrorSession()
     * - .toDeleteTrafficMirrorTarget()
     * - .toDeleteTransitGateway()
     * - .toDeleteTransitGatewayMulticastDomain()
     * - .toDeleteTransitGatewayPeeringAttachment()
     * - .toDeleteTransitGatewayRoute()
     * - .toDeleteTransitGatewayRouteTable()
     * - .toDeleteTransitGatewayVpcAttachment()
     * - .toDeleteVolume()
     * - .toDeleteVpcEndpointServiceConfigurations()
     * - .toDeleteVpcEndpoints()
     * - .toDeleteVpcPeeringConnection()
     * - .toDeregisterTransitGatewayMulticastGroupMembers()
     * - .toDeregisterTransitGatewayMulticastGroupSources()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toDisableTransitGatewayRouteTablePropagation()
     * - .toDisableVpcClassicLink()
     * - .toDisassociateClientVpnTargetNetwork()
     * - .toDisassociateIamInstanceProfile()
     * - .toDisassociateTransitGatewayMulticastDomain()
     * - .toDisassociateTransitGatewayRouteTable()
     * - .toEnableFastSnapshotRestores()
     * - .toEnableTransitGatewayRouteTablePropagation()
     * - .toEnableVpcClassicLink()
     * - .toGetConsoleScreenshot()
     * - .toImportClientVpnClientCertificateRevocationList()
     * - .toModifyCapacityReservation()
     * - .toModifyClientVpnEndpoint()
     * - .toModifyInstanceEventStartTime()
     * - .toModifyLaunchTemplate()
     * - .toModifyManagedPrefixList()
     * - .toModifySnapshotAttribute()
     * - .toModifyTrafficMirrorFilterNetworkServices()
     * - .toModifyTrafficMirrorFilterRule()
     * - .toModifyTrafficMirrorSession()
     * - .toModifyTransitGatewayVpcAttachment()
     * - .toModifyVpcEndpoint()
     * - .toModifyVpcEndpointServiceConfiguration()
     * - .toModifyVpcEndpointServicePermissions()
     * - .toModifyVpnConnection()
     * - .toModifyVpnTunnelOptions()
     * - .toRebootInstances()
     * - .toRegisterTransitGatewayMulticastGroupMembers()
     * - .toRegisterTransitGatewayMulticastGroupSources()
     * - .toRejectTransitGatewayPeeringAttachment()
     * - .toRejectTransitGatewayVpcAttachment()
     * - .toRejectVpcEndpointConnections()
     * - .toRejectVpcPeeringConnection()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toReplaceRoute()
     * - .toReplaceTransitGatewayRoute()
     * - .toRestoreManagedPrefixListVersion()
     * - .toRevokeClientVpnIngress()
     * - .toRevokeSecurityGroupEgress()
     * - .toRevokeSecurityGroupIngress()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStartVpcEndpointServicePrivateDnsVerification()
     * - .toStopInstances()
     * - .toTerminateClientVpnConnections()
     * - .toTerminateInstances()
     * - .toUpdateSecurityGroupRuleDescriptionsEgress()
     * - .toUpdateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - carrier-gateway
     * - client-vpn-endpoint
     * - customer-gateway
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - traffic-mirror-filter-rule
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRegion(value, operator) {
        return this.if(`ec2:Region`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the percentage of increase of the rekey window (determined by the rekey margin time) within which the rekey time is randomly selected for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyFuzzPercentage(value, operator) {
        return this.if(`ec2:RekeyFuzzPercentage`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the margin time before the phase 2 lifetime expires for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyMarginTimeSeconds(value, operator) {
        return this.if(`ec2:RekeyMarginTimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of a requester VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * Applies to actions:
     * - .toAcceptVpcPeeringConnection()
     * - .toCreateVpcPeeringConnection()
     * - .toDeleteVpcPeeringConnection()
     * - .toRejectVpcPeeringConnection()
     *
     * Applies to resource types:
     * - vpc-peering-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRequesterVpc(value, operator) {
        return this.if(`ec2:RequesterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the payment option of the Reserved Instance offering (No Upfront, Partial Upfront, or All Upfront)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html#ri-payment-options
     *
     * Applies to actions:
     * - .toCreateTags()
     *
     * Applies to resource types:
     * - reserved-instances
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReservedInstancesOfferingType(value, operator) {
        return this.if(`ec2:ReservedInstancesOfferingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`ec2:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .toAcceptTransitGatewayPeeringAttachment()
     * - .toAcceptTransitGatewayVpcAttachment()
     * - .toAcceptVpcEndpointConnections()
     * - .toAcceptVpcPeeringConnection()
     * - .toApplySecurityGroupsToClientVpnTargetNetwork()
     * - .toAssociateClientVpnTargetNetwork()
     * - .toAssociateIamInstanceProfile()
     * - .toAssociateTransitGatewayMulticastDomain()
     * - .toAssociateTransitGatewayRouteTable()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toAuthorizeClientVpnIngress()
     * - .toAuthorizeSecurityGroupEgress()
     * - .toAuthorizeSecurityGroupIngress()
     * - .toCancelCapacityReservation()
     * - .toCreateClientVpnRoute()
     * - .toCreateFlowLogs()
     * - .toCreateLaunchTemplateVersion()
     * - .toCreateLocalGatewayRoute()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateRoute()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateTrafficMirrorFilterRule()
     * - .toCreateTrafficMirrorSession()
     * - .toCreateTrafficMirrorTarget()
     * - .toCreateTransitGatewayMulticastDomain()
     * - .toCreateTransitGatewayPeeringAttachment()
     * - .toCreateTransitGatewayRoute()
     * - .toCreateTransitGatewayRouteTable()
     * - .toCreateTransitGatewayVpcAttachment()
     * - .toCreateVpcEndpoint()
     * - .toCreateVpcPeeringConnection()
     * - .toDeleteCarrierGateway()
     * - .toDeleteClientVpnEndpoint()
     * - .toDeleteClientVpnRoute()
     * - .toDeleteCustomerGateway()
     * - .toDeleteDhcpOptions()
     * - .toDeleteFlowLogs()
     * - .toDeleteInternetGateway()
     * - .toDeleteLaunchTemplate()
     * - .toDeleteLaunchTemplateVersions()
     * - .toDeleteLocalGatewayRoute()
     * - .toDeleteLocalGatewayRouteTableVpcAssociation()
     * - .toDeleteManagedPrefixList()
     * - .toDeleteNetworkAcl()
     * - .toDeleteNetworkAclEntry()
     * - .toDeleteRoute()
     * - .toDeleteRouteTable()
     * - .toDeleteSecurityGroup()
     * - .toDeleteSnapshot()
     * - .toDeleteTags()
     * - .toDeleteTrafficMirrorFilter()
     * - .toDeleteTrafficMirrorFilterRule()
     * - .toDeleteTrafficMirrorSession()
     * - .toDeleteTrafficMirrorTarget()
     * - .toDeleteTransitGateway()
     * - .toDeleteTransitGatewayMulticastDomain()
     * - .toDeleteTransitGatewayPeeringAttachment()
     * - .toDeleteTransitGatewayRoute()
     * - .toDeleteTransitGatewayRouteTable()
     * - .toDeleteTransitGatewayVpcAttachment()
     * - .toDeleteVolume()
     * - .toDeleteVpcEndpointServiceConfigurations()
     * - .toDeleteVpcEndpoints()
     * - .toDeleteVpcPeeringConnection()
     * - .toDeregisterTransitGatewayMulticastGroupMembers()
     * - .toDeregisterTransitGatewayMulticastGroupSources()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toDisableTransitGatewayRouteTablePropagation()
     * - .toDisableVpcClassicLink()
     * - .toDisassociateClientVpnTargetNetwork()
     * - .toDisassociateIamInstanceProfile()
     * - .toDisassociateTransitGatewayMulticastDomain()
     * - .toDisassociateTransitGatewayRouteTable()
     * - .toEnableFastSnapshotRestores()
     * - .toEnableTransitGatewayRouteTablePropagation()
     * - .toEnableVpcClassicLink()
     * - .toGetConsoleScreenshot()
     * - .toImportClientVpnClientCertificateRevocationList()
     * - .toModifyCapacityReservation()
     * - .toModifyClientVpnEndpoint()
     * - .toModifyLaunchTemplate()
     * - .toModifyManagedPrefixList()
     * - .toModifySnapshotAttribute()
     * - .toModifyTrafficMirrorFilterNetworkServices()
     * - .toModifyTrafficMirrorFilterRule()
     * - .toModifyTrafficMirrorSession()
     * - .toModifyTransitGatewayVpcAttachment()
     * - .toModifyVpcEndpoint()
     * - .toModifyVpcEndpointServiceConfiguration()
     * - .toModifyVpcEndpointServicePermissions()
     * - .toModifyVpnConnection()
     * - .toModifyVpnTunnelOptions()
     * - .toRebootInstances()
     * - .toRegisterTransitGatewayMulticastGroupMembers()
     * - .toRegisterTransitGatewayMulticastGroupSources()
     * - .toRejectTransitGatewayPeeringAttachment()
     * - .toRejectTransitGatewayVpcAttachment()
     * - .toRejectVpcEndpointConnections()
     * - .toRejectVpcPeeringConnection()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toReplaceRoute()
     * - .toReplaceTransitGatewayRoute()
     * - .toRestoreManagedPrefixListVersion()
     * - .toRevokeClientVpnIngress()
     * - .toRevokeSecurityGroupEgress()
     * - .toRevokeSecurityGroupIngress()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStartVpcEndpointServicePrivateDnsVerification()
     * - .toStopInstances()
     * - .toTerminateClientVpnConnections()
     * - .toTerminateInstances()
     * - .toUpdateSecurityGroupRuleDescriptionsEgress()
     * - .toUpdateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - carrier-gateway
     * - client-vpn-endpoint
     * - customer-gateway
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ec2:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the version of the instance metadata service for retrieving IAM role credentials for EC2
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRoleDelivery(value, operator) {
        return this.if(`ec2:RoleDelivery`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the root device type of the instance (ebs or instance-store)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - image
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRootDeviceType(value, operator) {
        return this.if(`ec2:RootDeviceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the routing type for the VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoutingType(value, operator) {
        return this.if(`ec2:RoutingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the initiation time of a snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toDeleteSnapshot()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifSnapshotTime(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`ec2:SnapshotTime`, value, operator || 'DateEquals');
    }
    /**
     * Filters access by the ARN of the instance from which the request originated
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSourceInstanceARN(value, operator) {
        return this.if(`ec2:SourceInstanceARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the subnet
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateFlowLogs()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSubnet(value, operator) {
        return this.if(`ec2:Subnet`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the tenancy of the VPC or instance (default, dedicated, or host)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAcceptVpcPeeringConnection()
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateFlowLogs()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVpcPeeringConnection()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableVpcClassicLink()
     * - .toDisassociateIamInstanceProfile()
     * - .toEnableVpcClassicLink()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     * - reserved-instances
     * - vpc
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTenancy(value, operator) {
        return this.if(`ec2:Tenancy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the the number of input/output operations per second (IOPS) provisioned for the volume
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeIops(value, operator) {
        return this.if(`ec2:VolumeIops`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the size of the volume, in GiB.
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteSnapshot()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - snapshot
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeSize(value, operator) {
        return this.if(`ec2:VolumeSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the type of volume (gp2, io1, st1, sc1, or standard)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVolumeType(value, operator) {
        return this.if(`ec2:VolumeType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the VPC
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachClassicLinkVpc()
     * - .toAuthorizeSecurityGroupEgress()
     * - .toAuthorizeSecurityGroupIngress()
     * - .toCreateFlowLogs()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateRoute()
     * - .toCreateTags()
     * - .toDeleteNetworkAcl()
     * - .toDeleteNetworkAclEntry()
     * - .toDeleteRoute()
     * - .toDeleteRouteTable()
     * - .toDeleteSecurityGroup()
     * - .toReplaceRoute()
     * - .toRevokeSecurityGroupEgress()
     * - .toRevokeSecurityGroupIngress()
     * - .toRunInstances()
     * - .toUpdateSecurityGroupRuleDescriptionsEgress()
     * - .toUpdateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - network-acl
     * - network-interface
     * - route-table
     * - security-group
     * - subnet
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifVpc(value, operator) {
        return this.if(`ec2:Vpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the name of the VPC endpoint service
     *
     * Applies to actions:
     * - .toCreateVpcEndpoint()
     *
     * Applies to resource types:
     * - vpc-endpoint
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceName(value, operator) {
        return this.if(`ec2:VpceServiceName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the service owner of the VPC endpoint service (amazon, aws-marketplace, or an AWS account ID)
     *
     * Applies to actions:
     * - .toCreateVpcEndpoint()
     *
     * Applies to resource types:
     * - vpc-endpoint
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceOwner(value, operator) {
        return this.if(`ec2:VpceServiceOwner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the private DNS name of the VPC endpoint service
     *
     * Applies to actions:
     * - .toCreateVpcEndpointServiceConfiguration()
     * - .toModifyVpcEndpointServiceConfiguration()
     *
     * Applies to resource types:
     * - vpc-endpoint-service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServicePrivateDnsName(value, operator) {
        return this.if(`ec2:VpceServicePrivateDnsName`, value, operator || 'StringLike');
    }
}
exports.Ec2 = Ec2;
//# sourceMappingURL=data:application/json;base64,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