"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticfilesystem = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticfilesystem](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticfilesystem extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticfilesystem](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticfilesystem';
        this.actionList = {
            "Backup": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html",
                "description": "Starts a backup job for an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ClientMount": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing read-access to a file system.",
                "accessLevel": "Read",
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "ClientRootAccess": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing root-access to a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "ClientWrite": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing write-access to a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "CreateAccessPoint": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateAccessPoint.html",
                "description": "Creates an access point for the specified file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "CreateFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateFileSystem.html",
                "description": "Creates a new, empty file system.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMountTarget": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateMountTarget.html",
                "description": "Creates a mount target for a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateTags.html",
                "description": "Creates or overwrites tags associated with a file system.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAccessPoint": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteAccessPoint.html",
                "description": "Deletes the specified access point.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "access-point": {
                        "required": true
                    }
                }
            },
            "DeleteFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystem.html",
                "description": "Deletes a file system, permanently severing access to its contents.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystemPolicy.html",
                "description": "Clears the resource-level policy for a given file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteMountTarget": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteMountTarget.html",
                "description": "Deletes the specified mount target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteTags.html",
                "description": "Deletes the specified tags from a file system.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "DescribeAccessPoints": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeAccessPoints.html",
                "description": "Returns the descriptions of Amazon EFS access points.",
                "accessLevel": "List",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeBackupPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeBackupPolicy.html",
                "description": "Returns the current BackupPolicy object for the specified Amazon EFS file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystemPolicy.html",
                "description": "Returns the current resource-level policy for a given file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": false
                    }
                }
            },
            "DescribeFileSystems": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystems.html",
                "description": "Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided; otherwise, returns descriptions of all file systems owned by the caller's AWS account in the AWS region of the endpoint that you're calling.",
                "accessLevel": "List",
                "resourceTypes": {
                    "file-system": {
                        "required": false
                    }
                }
            },
            "DescribeLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeLifecycleConfiguration.html",
                "description": "Returns the current LifecycleConfiguration object for the specified Amazon EFS file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeMountTargetSecurityGroups": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargetSecurityGroups.html",
                "description": "Returns the security groups currently in effect for a mount target.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeMountTargets": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargets.html",
                "description": "Returns the descriptions of all the current mount targets, or a specific mount target, for a file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeTags.html",
                "description": "Returns the tags associated with a file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_ListTagsForResource.html",
                "description": "Returns the tags associated with the specified Amazon EFS resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ModifyMountTargetSecurityGroups": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_ModifyMountTargetSecurityGroups.html",
                "description": "Modifies the set of security groups in effect for a mount target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "PutBackupPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_PutBackupPolicy.html",
                "description": "Enables automatic backups with AWS Backup by creating a new BackupPolicy object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "PutFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_PutFileSystemPolicy.html",
                "description": "Apply a resource-level policy granting and/or restricting actions from given actors for the specified file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "PutLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_PutLifecycleConfiguration.html",
                "description": "Enables lifecycle management by creating a new LifecycleConfiguration object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "Restore": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html",
                "description": "Starts a restore job for an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_TagResource.html",
                "description": "Creates or overwrites tags associated with the specified Amazon EFS resource.",
                "accessLevel": "Tagging"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_UntagResource.html",
                "description": "Deletes the specified tags from a specified Amazon EFS resource.",
                "accessLevel": "Tagging"
            },
            "UpdateFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_UpdateFileSystem.html",
                "description": "Updates the throughput mode or the amount of provisioned throughput of an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "file-system": {
                "name": "file-system",
                "url": "https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:elasticfilesystem:${Region}:${Account}:file-system/${FileSystemId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "access-point": {
                "name": "access-point",
                "url": "https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:elasticfilesystem:${Region}:${Account}:access-point/${AccessPointId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Starts a backup job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    toBackup() {
        this.add('elasticfilesystem:Backup');
        return this;
    }
    /**
     * Permission for allowing read-access to a file system.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccessPointArn()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    toClientMount() {
        this.add('elasticfilesystem:ClientMount');
        return this;
    }
    /**
     * Permission for allowing root-access to a file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccessPointArn()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    toClientRootAccess() {
        this.add('elasticfilesystem:ClientRootAccess');
        return this;
    }
    /**
     * Permission for allowing write-access to a file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccessPointArn()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    toClientWrite() {
        this.add('elasticfilesystem:ClientWrite');
        return this;
    }
    /**
     * Creates an access point for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        this.add('elasticfilesystem:CreateAccessPoint');
        return this;
    }
    /**
     * Creates a new, empty file system.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateFileSystem.html
     */
    toCreateFileSystem() {
        this.add('elasticfilesystem:CreateFileSystem');
        return this;
    }
    /**
     * Creates a mount target for a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateMountTarget.html
     */
    toCreateMountTarget() {
        this.add('elasticfilesystem:CreateMountTarget');
        return this;
    }
    /**
     * Creates or overwrites tags associated with a file system.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateTags.html
     */
    toCreateTags() {
        this.add('elasticfilesystem:CreateTags');
        return this;
    }
    /**
     * Deletes the specified access point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        this.add('elasticfilesystem:DeleteAccessPoint');
        return this;
    }
    /**
     * Deletes a file system, permanently severing access to its contents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystem.html
     */
    toDeleteFileSystem() {
        this.add('elasticfilesystem:DeleteFileSystem');
        return this;
    }
    /**
     * Clears the resource-level policy for a given file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystemPolicy.html
     */
    toDeleteFileSystemPolicy() {
        this.add('elasticfilesystem:DeleteFileSystemPolicy');
        return this;
    }
    /**
     * Deletes the specified mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteMountTarget.html
     */
    toDeleteMountTarget() {
        this.add('elasticfilesystem:DeleteMountTarget');
        return this;
    }
    /**
     * Deletes the specified tags from a file system.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteTags.html
     */
    toDeleteTags() {
        this.add('elasticfilesystem:DeleteTags');
        return this;
    }
    /**
     * Returns the descriptions of Amazon EFS access points.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeAccessPoints.html
     */
    toDescribeAccessPoints() {
        this.add('elasticfilesystem:DescribeAccessPoints');
        return this;
    }
    /**
     * Returns the current BackupPolicy object for the specified Amazon EFS file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeBackupPolicy.html
     */
    toDescribeBackupPolicy() {
        this.add('elasticfilesystem:DescribeBackupPolicy');
        return this;
    }
    /**
     * Returns the current resource-level policy for a given file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystemPolicy.html
     */
    toDescribeFileSystemPolicy() {
        this.add('elasticfilesystem:DescribeFileSystemPolicy');
        return this;
    }
    /**
     * Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided; otherwise, returns descriptions of all file systems owned by the caller's AWS account in the AWS region of the endpoint that you're calling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystems.html
     */
    toDescribeFileSystems() {
        this.add('elasticfilesystem:DescribeFileSystems');
        return this;
    }
    /**
     * Returns the current LifecycleConfiguration object for the specified Amazon EFS file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeLifecycleConfiguration.html
     */
    toDescribeLifecycleConfiguration() {
        this.add('elasticfilesystem:DescribeLifecycleConfiguration');
        return this;
    }
    /**
     * Returns the security groups currently in effect for a mount target.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargetSecurityGroups.html
     */
    toDescribeMountTargetSecurityGroups() {
        this.add('elasticfilesystem:DescribeMountTargetSecurityGroups');
        return this;
    }
    /**
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargets.html
     */
    toDescribeMountTargets() {
        this.add('elasticfilesystem:DescribeMountTargets');
        return this;
    }
    /**
     * Returns the tags associated with a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeTags.html
     */
    toDescribeTags() {
        this.add('elasticfilesystem:DescribeTags');
        return this;
    }
    /**
     * Returns the tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('elasticfilesystem:ListTagsForResource');
        return this;
    }
    /**
     * Modifies the set of security groups in effect for a mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ModifyMountTargetSecurityGroups.html
     */
    toModifyMountTargetSecurityGroups() {
        this.add('elasticfilesystem:ModifyMountTargetSecurityGroups');
        return this;
    }
    /**
     * Enables automatic backups with AWS Backup by creating a new BackupPolicy object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutBackupPolicy.html
     */
    toPutBackupPolicy() {
        this.add('elasticfilesystem:PutBackupPolicy');
        return this;
    }
    /**
     * Apply a resource-level policy granting and/or restricting actions from given actors for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutFileSystemPolicy.html
     */
    toPutFileSystemPolicy() {
        this.add('elasticfilesystem:PutFileSystemPolicy');
        return this;
    }
    /**
     * Enables lifecycle management by creating a new LifecycleConfiguration object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        this.add('elasticfilesystem:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Starts a restore job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    toRestore() {
        this.add('elasticfilesystem:Restore');
        return this;
    }
    /**
     * Creates or overwrites tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_TagResource.html
     */
    toTagResource() {
        this.add('elasticfilesystem:TagResource');
        return this;
    }
    /**
     * Deletes the specified tags from a specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UntagResource.html
     */
    toUntagResource() {
        this.add('elasticfilesystem:UntagResource');
        return this;
    }
    /**
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UpdateFileSystem.html
     */
    toUpdateFileSystem() {
        this.add('elasticfilesystem:UpdateFileSystem');
        return this;
    }
    /**
     * Adds a resource of type file-system to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFileSystem(fileSystemId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:file-system/${FileSystemId}';
        arn = arn.replace('${FileSystemId}', fileSystemId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type access-point to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param accessPointId - Identifier for the accessPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccessPoint(accessPointId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:access-point/${AccessPointId}';
        arn = arn.replace('${AccessPointId}', accessPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFileSystem()
     * - .toCreateTags()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - file-system
     * - access-point
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFileSystem()
     * - .toCreateTags()
     * - .toDeleteTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * The ARN of the access point used to mount the file system
     *
     * Applies to actions:
     * - .toClientMount()
     * - .toClientRootAccess()
     * - .toClientWrite()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointArn(value, operator) {
        return this.if(`elasticfilesystem:AccessPointArn`, value, operator || 'StringLike');
    }
}
exports.Elasticfilesystem = Elasticfilesystem;
//# sourceMappingURL=data:application/json;base64,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