"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Es = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [es](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Es extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [es](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'es';
        this.actionList = {
            "AcceptInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-acceptinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to accept an inbound cross-cluster search connection request",
                "accessLevel": "Write"
            },
            "AddTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-addtags",
                "description": "Grants permission to attach resource tags to an Amazon ES domain.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchdomain",
                "description": "Grants permission to create an Amazon ES domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "CreateElasticsearchServiceRole": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchservicerole",
                "description": "Grants permission to create the service-linked role required for Amazon ES domains that use VPC access.",
                "accessLevel": "Write"
            },
            "CreateOutboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createoutboundcrossclustersearchconnection",
                "description": "Creates a new cross-cluster search connection from a source domain to a destination domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchdomain",
                "description": "Grants permission to delete an Amazon ES domain and all of its data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteElasticsearchServiceRole": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchservicerole",
                "description": "Grants permission to delete the service-linked role required for Amazon ES domains that use VPC access.",
                "accessLevel": "Write"
            },
            "DeleteInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to delete an existing inbound cross-cluster search connection",
                "accessLevel": "Write"
            },
            "DeleteOutboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteoutboundcrossclustersearchconnection",
                "description": "Allows the source domain owner to delete an existing outbound cross-cluster search connection",
                "accessLevel": "Write"
            },
            "DescribeElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain",
                "description": "Grants permission to view a description of the domain configuration for the specified Amazon ES domain, including the domain ID, domain service endpoint, and domain ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchDomainConfig": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomainconfig",
                "description": "Grants permission to view a description of the configuration options and status of an Amazon ES domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchDomains": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain",
                "description": "Grants permission to view a description of the domain configuration for up to five specified Amazon ES domains.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchInstanceTypeLimits": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinstancetypelimits",
                "description": "Grants permission to view the instance count, storage, and master node limits for a given Elasticsearch version and instance type.",
                "accessLevel": "List"
            },
            "DescribeInboundCrossClusterSearchConnections": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinboundcrossclustersearchconnections",
                "description": "Lists all the inbound cross-cluster search connections for a destination domain",
                "accessLevel": "List"
            },
            "DescribeOutboundCrossClusterSearchConnections": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeoutboundcrossclustersearchconnections",
                "description": "Lists all the outbound cross-cluster search connections for a source domain",
                "accessLevel": "List"
            },
            "DescribeReservedElasticsearchInstanceOfferings": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstanceofferings",
                "description": "Grants permission to fetch reserved instance offerings for ES",
                "accessLevel": "List"
            },
            "DescribeReservedElasticsearchInstances": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstances",
                "description": "Grants permission to fetch ES reserved instances already purchased by customer",
                "accessLevel": "List"
            },
            "ESCrossClusterGet": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send cross-cluster requests to a destination domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpDelete": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP DELETE requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpGet": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP GET requests to the Elasticsearch APIs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpHead": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP HEAD requests to the Elasticsearch APIs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPatch": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP PATCH requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPost": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP POST requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPut": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP PUT requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "GetCompatibleElasticsearchVersions": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getcompatibleelasticsearchversions",
                "description": "Grants permission to fetch list of compatible elastic search versions to which Amazon ES domain can be upgraded",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetUpgradeHistory": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradehistory",
                "description": "Grants permission to fetch upgrade history for given ES domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetUpgradeStatus": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradestatus",
                "description": "Grants permission to fetch upgrade status for given ES domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListDomainNames": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listdomainnames",
                "description": "Grants permission to display the names of all Amazon ES domains that the current user owns.",
                "accessLevel": "List"
            },
            "ListElasticsearchInstanceTypeDetails": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypedetails",
                "description": "Grants permission to list all instance types and available features for a given Elasticsearch version.",
                "accessLevel": "List"
            },
            "ListElasticsearchInstanceTypes": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypes",
                "description": "Grants permission to list all Elasticsearch instance types that are supported for a given Elasticsearch version.",
                "accessLevel": "List"
            },
            "ListElasticsearchVersions": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchversions",
                "description": "Grants permission to list all supported Elasticsearch versions on Amazon ES.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags",
                "description": "Grants permission to display all of the tags for an Amazon ES domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PurchaseReservedElasticsearchInstanceOffering": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-purchasereservedelasticsearchinstance",
                "description": "Grants permission to purchase ES reserved instances",
                "accessLevel": "Write"
            },
            "RejectInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-rejectinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to reject an inbound cross-cluster search connection request",
                "accessLevel": "Write"
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags",
                "description": "Grants permission to remove tags from Amazon ES domains.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateElasticsearchDomainConfig": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-updateelasticsearchdomainconfig",
                "description": "Grants permission to modify the configuration of an Amazon ES domain, such as the instance type or number of instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpgradeElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-upgradeelasticsearchdomain",
                "description": "Grants permission to initiate upgrade of elastic search domain to given version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html",
                "arn": "arn:${Partition}:es:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Allows the destination domain owner to accept an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-acceptinboundcrossclustersearchconnection
     */
    toAcceptInboundCrossClusterSearchConnection() {
        this.add('es:AcceptInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to attach resource tags to an Amazon ES domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-addtags
     */
    toAddTags() {
        this.add('es:AddTags');
        return this;
    }
    /**
     * Grants permission to create an Amazon ES domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchdomain
     */
    toCreateElasticsearchDomain() {
        this.add('es:CreateElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to create the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchservicerole
     */
    toCreateElasticsearchServiceRole() {
        this.add('es:CreateElasticsearchServiceRole');
        return this;
    }
    /**
     * Creates a new cross-cluster search connection from a source domain to a destination domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createoutboundcrossclustersearchconnection
     */
    toCreateOutboundCrossClusterSearchConnection() {
        this.add('es:CreateOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to delete an Amazon ES domain and all of its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchdomain
     */
    toDeleteElasticsearchDomain() {
        this.add('es:DeleteElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to delete the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchservicerole
     */
    toDeleteElasticsearchServiceRole() {
        this.add('es:DeleteElasticsearchServiceRole');
        return this;
    }
    /**
     * Allows the destination domain owner to delete an existing inbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteinboundcrossclustersearchconnection
     */
    toDeleteInboundCrossClusterSearchConnection() {
        this.add('es:DeleteInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Allows the source domain owner to delete an existing outbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteoutboundcrossclustersearchconnection
     */
    toDeleteOutboundCrossClusterSearchConnection() {
        this.add('es:DeleteOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for the specified Amazon ES domain, including the domain ID, domain service endpoint, and domain ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    toDescribeElasticsearchDomain() {
        this.add('es:DescribeElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to view a description of the configuration options and status of an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomainconfig
     */
    toDescribeElasticsearchDomainConfig() {
        this.add('es:DescribeElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for up to five specified Amazon ES domains.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    toDescribeElasticsearchDomains() {
        this.add('es:DescribeElasticsearchDomains');
        return this;
    }
    /**
     * Grants permission to view the instance count, storage, and master node limits for a given Elasticsearch version and instance type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinstancetypelimits
     */
    toDescribeElasticsearchInstanceTypeLimits() {
        this.add('es:DescribeElasticsearchInstanceTypeLimits');
        return this;
    }
    /**
     * Lists all the inbound cross-cluster search connections for a destination domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinboundcrossclustersearchconnections
     */
    toDescribeInboundCrossClusterSearchConnections() {
        this.add('es:DescribeInboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Lists all the outbound cross-cluster search connections for a source domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeoutboundcrossclustersearchconnections
     */
    toDescribeOutboundCrossClusterSearchConnections() {
        this.add('es:DescribeOutboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Grants permission to fetch reserved instance offerings for ES
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstanceofferings
     */
    toDescribeReservedElasticsearchInstanceOfferings() {
        this.add('es:DescribeReservedElasticsearchInstanceOfferings');
        return this;
    }
    /**
     * Grants permission to fetch ES reserved instances already purchased by customer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstances
     */
    toDescribeReservedElasticsearchInstances() {
        this.add('es:DescribeReservedElasticsearchInstances');
        return this;
    }
    /**
     * Grants permission to send cross-cluster requests to a destination domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESCrossClusterGet() {
        this.add('es:ESCrossClusterGet');
        return this;
    }
    /**
     * Grants permission to send HTTP DELETE requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpDelete() {
        this.add('es:ESHttpDelete');
        return this;
    }
    /**
     * Grants permission to send HTTP GET requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpGet() {
        this.add('es:ESHttpGet');
        return this;
    }
    /**
     * Grants permission to send HTTP HEAD requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpHead() {
        this.add('es:ESHttpHead');
        return this;
    }
    /**
     * Grants permission to send HTTP PATCH requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpPatch() {
        this.add('es:ESHttpPatch');
        return this;
    }
    /**
     * Grants permission to send HTTP POST requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpPost() {
        this.add('es:ESHttpPost');
        return this;
    }
    /**
     * Grants permission to send HTTP PUT requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpPut() {
        this.add('es:ESHttpPut');
        return this;
    }
    /**
     * Grants permission to fetch list of compatible elastic search versions to which Amazon ES domain can be upgraded
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getcompatibleelasticsearchversions
     */
    toGetCompatibleElasticsearchVersions() {
        this.add('es:GetCompatibleElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to fetch upgrade history for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradehistory
     */
    toGetUpgradeHistory() {
        this.add('es:GetUpgradeHistory');
        return this;
    }
    /**
     * Grants permission to fetch upgrade status for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradestatus
     */
    toGetUpgradeStatus() {
        this.add('es:GetUpgradeStatus');
        return this;
    }
    /**
     * Grants permission to display the names of all Amazon ES domains that the current user owns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listdomainnames
     */
    toListDomainNames() {
        this.add('es:ListDomainNames');
        return this;
    }
    /**
     * Grants permission to list all instance types and available features for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypedetails
     */
    toListElasticsearchInstanceTypeDetails() {
        this.add('es:ListElasticsearchInstanceTypeDetails');
        return this;
    }
    /**
     * Grants permission to list all Elasticsearch instance types that are supported for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypes
     */
    toListElasticsearchInstanceTypes() {
        this.add('es:ListElasticsearchInstanceTypes');
        return this;
    }
    /**
     * Grants permission to list all supported Elasticsearch versions on Amazon ES.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchversions
     */
    toListElasticsearchVersions() {
        this.add('es:ListElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to display all of the tags for an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    toListTags() {
        this.add('es:ListTags');
        return this;
    }
    /**
     * Grants permission to purchase ES reserved instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-purchasereservedelasticsearchinstance
     */
    toPurchaseReservedElasticsearchInstanceOffering() {
        this.add('es:PurchaseReservedElasticsearchInstanceOffering');
        return this;
    }
    /**
     * Allows the destination domain owner to reject an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-rejectinboundcrossclustersearchconnection
     */
    toRejectInboundCrossClusterSearchConnection() {
        this.add('es:RejectInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to remove tags from Amazon ES domains.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    toRemoveTags() {
        this.add('es:RemoveTags');
        return this;
    }
    /**
     * Grants permission to modify the configuration of an Amazon ES domain, such as the instance type or number of instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-updateelasticsearchdomainconfig
     */
    toUpdateElasticsearchDomainConfig() {
        this.add('es:UpdateElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to initiate upgrade of elastic search domain to given version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-upgradeelasticsearchdomain
     */
    toUpgradeElasticsearchDomain() {
        this.add('es:UpgradeElasticsearchDomain');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:es:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Es = Es;
//# sourceMappingURL=data:application/json;base64,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