"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elastictranscoder = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elastictranscoder](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelastictranscoder.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elastictranscoder extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elastictranscoder](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelastictranscoder.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elastictranscoder';
        this.actionList = {
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/cancel-job.html",
                "description": "Cancel a job that Elastic Transcoder has not begun to process",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-job.html",
                "description": "Create a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    },
                    "preset": {
                        "required": true
                    }
                }
            },
            "CreatePipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-pipeline.html",
                "description": "Create a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "CreatePreset": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-preset.html",
                "description": "Create a preset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "preset": {
                        "required": true
                    }
                }
            },
            "DeletePipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-pipeline.html",
                "description": "Delete a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "DeletePreset": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-preset.html",
                "description": "Delete a preset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "preset": {
                        "required": true
                    }
                }
            },
            "ListJobsByPipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-pipeline.html",
                "description": "Get a list of the jobs that you assigned to a pipeline",
                "accessLevel": "List",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "ListJobsByStatus": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-status.html",
                "description": "Get information about all of the jobs associated with the current AWS account that have a specified status",
                "accessLevel": "List"
            },
            "ListPipelines": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-pipelines.html",
                "description": "Get a list of the pipelines associated with the current AWS account",
                "accessLevel": "List"
            },
            "ListPresets": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-presets.html",
                "description": "Get a list of all presets associated with the current AWS account.",
                "accessLevel": "List"
            },
            "ReadJob": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-job.html",
                "description": "Get detailed information about a job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "ReadPipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-pipeline.html",
                "description": "Get detailed information about a pipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "ReadPreset": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-preset.html",
                "description": "Get detailed information about a preset.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "preset": {
                        "required": true
                    }
                }
            },
            "TestRole": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/test-pipeline-role.html",
                "description": "Test the settings for a pipeline to ensure that Elastic Transcoder can create and process jobs",
                "accessLevel": "Write"
            },
            "UpdatePipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline.html",
                "description": "Update settings for a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "UpdatePipelineNotifications": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-notifications.html",
                "description": "Update only Amazon Simple Notification Service (Amazon SNS) notifications for a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "UpdatePipelineStatus": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-status.html",
                "description": "Pause or reactivate a pipeline, so the pipeline stops or restarts processing jobs, update the status for the pipeline.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-jobs.html",
                "arn": "arn:${Partition}:elastictranscoder:${Region}:${Account}:job/${JobId}",
                "conditionKeys": []
            },
            "pipeline": {
                "name": "pipeline",
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-pipelines.html",
                "arn": "arn:${Partition}:elastictranscoder:${Region}:${Account}:pipeline/${PipelineId}",
                "conditionKeys": []
            },
            "preset": {
                "name": "preset",
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-presets.html",
                "arn": "arn:${Partition}:elastictranscoder:${Region}:${Account}:preset/${PresetId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancel a job that Elastic Transcoder has not begun to process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/cancel-job.html
     */
    toCancelJob() {
        this.add('elastictranscoder:CancelJob');
        return this;
    }
    /**
     * Create a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-job.html
     */
    toCreateJob() {
        this.add('elastictranscoder:CreateJob');
        return this;
    }
    /**
     * Create a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-pipeline.html
     */
    toCreatePipeline() {
        this.add('elastictranscoder:CreatePipeline');
        return this;
    }
    /**
     * Create a preset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-preset.html
     */
    toCreatePreset() {
        this.add('elastictranscoder:CreatePreset');
        return this;
    }
    /**
     * Delete a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-pipeline.html
     */
    toDeletePipeline() {
        this.add('elastictranscoder:DeletePipeline');
        return this;
    }
    /**
     * Delete a preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-preset.html
     */
    toDeletePreset() {
        this.add('elastictranscoder:DeletePreset');
        return this;
    }
    /**
     * Get a list of the jobs that you assigned to a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-pipeline.html
     */
    toListJobsByPipeline() {
        this.add('elastictranscoder:ListJobsByPipeline');
        return this;
    }
    /**
     * Get information about all of the jobs associated with the current AWS account that have a specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-status.html
     */
    toListJobsByStatus() {
        this.add('elastictranscoder:ListJobsByStatus');
        return this;
    }
    /**
     * Get a list of the pipelines associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-pipelines.html
     */
    toListPipelines() {
        this.add('elastictranscoder:ListPipelines');
        return this;
    }
    /**
     * Get a list of all presets associated with the current AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-presets.html
     */
    toListPresets() {
        this.add('elastictranscoder:ListPresets');
        return this;
    }
    /**
     * Get detailed information about a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-job.html
     */
    toReadJob() {
        this.add('elastictranscoder:ReadJob');
        return this;
    }
    /**
     * Get detailed information about a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-pipeline.html
     */
    toReadPipeline() {
        this.add('elastictranscoder:ReadPipeline');
        return this;
    }
    /**
     * Get detailed information about a preset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-preset.html
     */
    toReadPreset() {
        this.add('elastictranscoder:ReadPreset');
        return this;
    }
    /**
     * Test the settings for a pipeline to ensure that Elastic Transcoder can create and process jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/test-pipeline-role.html
     */
    toTestRole() {
        this.add('elastictranscoder:TestRole');
        return this;
    }
    /**
     * Update settings for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline.html
     */
    toUpdatePipeline() {
        this.add('elastictranscoder:UpdatePipeline');
        return this;
    }
    /**
     * Update only Amazon Simple Notification Service (Amazon SNS) notifications for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-notifications.html
     */
    toUpdatePipelineNotifications() {
        this.add('elastictranscoder:UpdatePipelineNotifications');
        return this;
    }
    /**
     * Pause or reactivate a pipeline, so the pipeline stops or restarts processing jobs, update the status for the pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-status.html
     */
    toUpdatePipelineStatus() {
        this.add('elastictranscoder:UpdatePipelineStatus');
        return this;
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-pipelines.html
     *
     * @param pipelineId - Identifier for the pipelineId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPipeline(pipelineId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:pipeline/${PipelineId}';
        arn = arn.replace('${PipelineId}', pipelineId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type preset to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-presets.html
     *
     * @param presetId - Identifier for the presetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPreset(presetId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:preset/${PresetId}';
        arn = arn.replace('${PresetId}', presetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Elastictranscoder = Elastictranscoder;
//# sourceMappingURL=data:application/json;base64,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