"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Forecast = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [forecast](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonforecast.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Forecast extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [forecast](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonforecast.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'forecast';
        this.actionList = {
            "CreateDataset": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateDataset.html",
                "description": "Creates a dataset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDatasetGroup": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateDatasetGroup.html",
                "description": "Creates a dataset group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datasetGroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDatasetImportJob": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateDatasetImportJob.html",
                "description": "Creates a dataset import job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datasetImportJob": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateForecast": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateForecast.html",
                "description": "Creates a forecast",
                "accessLevel": "Write",
                "resourceTypes": {
                    "predictor": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateForecastExportJob": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateForecastExportJob.html",
                "description": "Creates a forecast export job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "forecast": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePredictor": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreatePredictor.html",
                "description": "Creates a predictor",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datasetGroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDataset": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteDataset.html",
                "description": "Deletes a dataset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DeleteDatasetGroup": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteDatasetGroup.html",
                "description": "Deletes a dataset group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datasetGroup": {
                        "required": true
                    }
                }
            },
            "DeleteDatasetImportJob": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteDatasetImportJob.html",
                "description": "Deletes a dataset import job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datasetImportJob": {
                        "required": true
                    }
                }
            },
            "DeleteForecast": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteForecast.html",
                "description": "Deletes a forecast",
                "accessLevel": "Write",
                "resourceTypes": {
                    "forecast": {
                        "required": true
                    }
                }
            },
            "DeleteForecastExportJob": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteForecastExportJob.html",
                "description": "Deletes a forecast export job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "forecastExport": {
                        "required": true
                    }
                }
            },
            "DeletePredictor": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeletePredictor.html",
                "description": "Deletes a predictor",
                "accessLevel": "Write",
                "resourceTypes": {
                    "predictor": {
                        "required": true
                    }
                }
            },
            "DescribeDataset": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeDataset.html",
                "description": "Describes a dataset",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeDatasetGroup": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeDatasetGroup.html",
                "description": "Describes a dataset group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "datasetGroup": {
                        "required": true
                    }
                }
            },
            "DescribeDatasetImportJob": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeDatasetImportJob.html",
                "description": "Describes a dataset import job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "datasetImportJob": {
                        "required": true
                    }
                }
            },
            "DescribeForecast": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeForecast.html",
                "description": "Describes a forecast",
                "accessLevel": "Read",
                "resourceTypes": {
                    "forecast": {
                        "required": true
                    }
                }
            },
            "DescribeForecastExportJob": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeForecastExportJob.html",
                "description": "Describes a forecast export job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "forecastExport": {
                        "required": true
                    }
                }
            },
            "DescribePredictor": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribePredictor.html",
                "description": "Describes a predictor",
                "accessLevel": "Read",
                "resourceTypes": {
                    "predictor": {
                        "required": true
                    }
                }
            },
            "GetAccuracyMetrics": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_GetAccuracyMetrics.html",
                "description": "Gets Accuracy Metrics for a predictor",
                "accessLevel": "Read",
                "resourceTypes": {
                    "predictor": {
                        "required": true
                    }
                }
            },
            "ListDatasetGroups": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListDatasetGroups.html",
                "description": "Lists dataset groups",
                "accessLevel": "List"
            },
            "ListDatasetImportJobs": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListDatasetImportJobs.html",
                "description": "Lists dataset import jobs",
                "accessLevel": "List"
            },
            "ListDatasets": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListDatasets.html",
                "description": "Lists datasets",
                "accessLevel": "List"
            },
            "ListForecastExportJobs": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListForecastExportJobs.html",
                "description": "Lists forecast export jobs",
                "accessLevel": "List"
            },
            "ListForecasts": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListForecasts.html",
                "description": "Lists forecasts",
                "accessLevel": "List"
            },
            "ListPredictors": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListPredictors.html",
                "description": "Lists predictors",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListTagsForResource.html",
                "description": "Lists the tags for an Amazon Forecast resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "dataset": {
                        "required": false
                    },
                    "datasetGroup": {
                        "required": false
                    },
                    "datasetImportJob": {
                        "required": false
                    },
                    "forecast": {
                        "required": false
                    },
                    "forecastExport": {
                        "required": false
                    },
                    "predictor": {
                        "required": false
                    }
                }
            },
            "QueryForecast": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_forecastquery_QueryForecast.html",
                "description": "Retrieves a forecast for a single item",
                "accessLevel": "Read",
                "resourceTypes": {
                    "forecast": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_TagResource.html",
                "description": "Associates the specified tags to a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dataset": {
                        "required": false
                    },
                    "datasetGroup": {
                        "required": false
                    },
                    "datasetImportJob": {
                        "required": false
                    },
                    "forecast": {
                        "required": false
                    },
                    "forecastExport": {
                        "required": false
                    },
                    "predictor": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_UntagResource.html",
                "description": "Deletes the specified tags for a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dataset": {
                        "required": false
                    },
                    "datasetGroup": {
                        "required": false
                    },
                    "datasetImportJob": {
                        "required": false
                    },
                    "forecast": {
                        "required": false
                    },
                    "forecastExport": {
                        "required": false
                    },
                    "predictor": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDatasetGroup": {
                "url": "https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_UpdateDatasetGroup.html",
                "description": "Updates a dataset group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    },
                    "datasetGroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "dataset": {
                "name": "dataset",
                "url": "",
                "arn": "arn:${Partition}:forecast:${Region}:${Account}:dataset/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "datasetGroup": {
                "name": "datasetGroup",
                "url": "",
                "arn": "arn:${Partition}:forecast:${Region}:${Account}:dataset-group/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "datasetImportJob": {
                "name": "datasetImportJob",
                "url": "",
                "arn": "arn:${Partition}:forecast:${Region}:${Account}:dataset-import-job/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "algorithm": {
                "name": "algorithm",
                "url": "",
                "arn": "arn:${Partition}:forecast:::algorithm/${ResourceId}",
                "conditionKeys": []
            },
            "predictor": {
                "name": "predictor",
                "url": "",
                "arn": "arn:${Partition}:forecast:${Region}:${Account}:predictor/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "forecast": {
                "name": "forecast",
                "url": "",
                "arn": "arn:${Partition}:forecast:${Region}:${Account}:forecast/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "forecastExport": {
                "name": "forecastExport",
                "url": "",
                "arn": "arn:${Partition}:forecast:${Region}:${Account}:forecast-export-job/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateDataset.html
     */
    toCreateDataset() {
        this.add('forecast:CreateDataset');
        return this;
    }
    /**
     * Creates a dataset group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateDatasetGroup.html
     */
    toCreateDatasetGroup() {
        this.add('forecast:CreateDatasetGroup');
        return this;
    }
    /**
     * Creates a dataset import job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateDatasetImportJob.html
     */
    toCreateDatasetImportJob() {
        this.add('forecast:CreateDatasetImportJob');
        return this;
    }
    /**
     * Creates a forecast
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateForecast.html
     */
    toCreateForecast() {
        this.add('forecast:CreateForecast');
        return this;
    }
    /**
     * Creates a forecast export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreateForecastExportJob.html
     */
    toCreateForecastExportJob() {
        this.add('forecast:CreateForecastExportJob');
        return this;
    }
    /**
     * Creates a predictor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_CreatePredictor.html
     */
    toCreatePredictor() {
        this.add('forecast:CreatePredictor');
        return this;
    }
    /**
     * Deletes a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteDataset.html
     */
    toDeleteDataset() {
        this.add('forecast:DeleteDataset');
        return this;
    }
    /**
     * Deletes a dataset group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteDatasetGroup.html
     */
    toDeleteDatasetGroup() {
        this.add('forecast:DeleteDatasetGroup');
        return this;
    }
    /**
     * Deletes a dataset import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteDatasetImportJob.html
     */
    toDeleteDatasetImportJob() {
        this.add('forecast:DeleteDatasetImportJob');
        return this;
    }
    /**
     * Deletes a forecast
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteForecast.html
     */
    toDeleteForecast() {
        this.add('forecast:DeleteForecast');
        return this;
    }
    /**
     * Deletes a forecast export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeleteForecastExportJob.html
     */
    toDeleteForecastExportJob() {
        this.add('forecast:DeleteForecastExportJob');
        return this;
    }
    /**
     * Deletes a predictor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DeletePredictor.html
     */
    toDeletePredictor() {
        this.add('forecast:DeletePredictor');
        return this;
    }
    /**
     * Describes a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeDataset.html
     */
    toDescribeDataset() {
        this.add('forecast:DescribeDataset');
        return this;
    }
    /**
     * Describes a dataset group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeDatasetGroup.html
     */
    toDescribeDatasetGroup() {
        this.add('forecast:DescribeDatasetGroup');
        return this;
    }
    /**
     * Describes a dataset import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeDatasetImportJob.html
     */
    toDescribeDatasetImportJob() {
        this.add('forecast:DescribeDatasetImportJob');
        return this;
    }
    /**
     * Describes a forecast
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeForecast.html
     */
    toDescribeForecast() {
        this.add('forecast:DescribeForecast');
        return this;
    }
    /**
     * Describes a forecast export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribeForecastExportJob.html
     */
    toDescribeForecastExportJob() {
        this.add('forecast:DescribeForecastExportJob');
        return this;
    }
    /**
     * Describes a predictor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_DescribePredictor.html
     */
    toDescribePredictor() {
        this.add('forecast:DescribePredictor');
        return this;
    }
    /**
     * Gets Accuracy Metrics for a predictor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_GetAccuracyMetrics.html
     */
    toGetAccuracyMetrics() {
        this.add('forecast:GetAccuracyMetrics');
        return this;
    }
    /**
     * Lists dataset groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListDatasetGroups.html
     */
    toListDatasetGroups() {
        this.add('forecast:ListDatasetGroups');
        return this;
    }
    /**
     * Lists dataset import jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListDatasetImportJobs.html
     */
    toListDatasetImportJobs() {
        this.add('forecast:ListDatasetImportJobs');
        return this;
    }
    /**
     * Lists datasets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListDatasets.html
     */
    toListDatasets() {
        this.add('forecast:ListDatasets');
        return this;
    }
    /**
     * Lists forecast export jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListForecastExportJobs.html
     */
    toListForecastExportJobs() {
        this.add('forecast:ListForecastExportJobs');
        return this;
    }
    /**
     * Lists forecasts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListForecasts.html
     */
    toListForecasts() {
        this.add('forecast:ListForecasts');
        return this;
    }
    /**
     * Lists predictors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListPredictors.html
     */
    toListPredictors() {
        this.add('forecast:ListPredictors');
        return this;
    }
    /**
     * Lists the tags for an Amazon Forecast resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('forecast:ListTagsForResource');
        return this;
    }
    /**
     * Retrieves a forecast for a single item
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_forecastquery_QueryForecast.html
     */
    toQueryForecast() {
        this.add('forecast:QueryForecast');
        return this;
    }
    /**
     * Associates the specified tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_TagResource.html
     */
    toTagResource() {
        this.add('forecast:TagResource');
        return this;
    }
    /**
     * Deletes the specified tags for a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_UntagResource.html
     */
    toUntagResource() {
        this.add('forecast:UntagResource');
        return this;
    }
    /**
     * Updates a dataset group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_Operations.htmlAPI_UpdateDatasetGroup.html
     */
    toUpdateDatasetGroup() {
        this.add('forecast:UpdateDatasetGroup');
        return this;
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:forecast:${Region}:${Account}:dataset/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type datasetGroup to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasetGroup(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:forecast:${Region}:${Account}:dataset-group/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type datasetImportJob to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasetImportJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:forecast:${Region}:${Account}:dataset-import-job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type algorithm to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAlgorithm(resourceId, partition) {
        var arn = 'arn:${Partition}:forecast:::algorithm/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type predictor to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPredictor(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:forecast:${Region}:${Account}:predictor/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type forecast to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onForecast(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:forecast:${Region}:${Account}:forecast/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type forecastExport to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onForecastExport(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:forecast:${Region}:${Account}:forecast-export-job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDataset()
     * - .toCreateDatasetGroup()
     * - .toCreateDatasetImportJob()
     * - .toCreateForecast()
     * - .toCreateForecastExportJob()
     * - .toCreatePredictor()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - dataset
     * - datasetGroup
     * - datasetImportJob
     * - predictor
     * - forecast
     * - forecastExport
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDataset()
     * - .toCreateDatasetGroup()
     * - .toCreateDatasetImportJob()
     * - .toCreateForecast()
     * - .toCreateForecastExportJob()
     * - .toCreatePredictor()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Forecast = Forecast;
//# sourceMappingURL=data:application/json;base64,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