"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Freertos = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [freertos](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfreertos.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Freertos extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [freertos](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfreertos.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'freertos';
        this.actionList = {
            "CreateSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Creates a software configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Deletes the software configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                }
            },
            "DescribeHardwarePlatform": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Describes the hardware platform.",
                "accessLevel": "Read"
            },
            "DescribeSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Describes the software configuration.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                }
            },
            "GetSoftwareURL": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Get the URL for Amazon FreeRTOS software download.",
                "accessLevel": "Read"
            },
            "GetSoftwareURLForConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Get the URL for Amazon FreeRTOS software download based on the configuration.",
                "accessLevel": "Read"
            },
            "ListFreeRTOSVersions": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists versions of AmazonFreeRTOS.",
                "accessLevel": "List"
            },
            "ListHardwarePlatforms": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists the hardware platforms.",
                "accessLevel": "List"
            },
            "ListHardwareVendors": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists the hardware vendors.",
                "accessLevel": "List"
            },
            "ListSoftwareConfigurations": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists the software configurations.",
                "accessLevel": "List"
            },
            "UpdateSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Updates the software configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "configuration": {
                "name": "configuration",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "arn": "arn:${Partition}:freertos:${Region}:${Account}:configuration/${ConfigurationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a software configuration.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toCreateSoftwareConfiguration() {
        this.add('freertos:CreateSoftwareConfiguration');
        return this;
    }
    /**
     * Deletes the software configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDeleteSoftwareConfiguration() {
        this.add('freertos:DeleteSoftwareConfiguration');
        return this;
    }
    /**
     * Describes the hardware platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeHardwarePlatform() {
        this.add('freertos:DescribeHardwarePlatform');
        return this;
    }
    /**
     * Describes the software configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeSoftwareConfiguration() {
        this.add('freertos:DescribeSoftwareConfiguration');
        return this;
    }
    /**
     * Get the URL for Amazon FreeRTOS software download.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURL() {
        this.add('freertos:GetSoftwareURL');
        return this;
    }
    /**
     * Get the URL for Amazon FreeRTOS software download based on the configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURLForConfiguration() {
        this.add('freertos:GetSoftwareURLForConfiguration');
        return this;
    }
    /**
     * Lists versions of AmazonFreeRTOS.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListFreeRTOSVersions() {
        this.add('freertos:ListFreeRTOSVersions');
        return this;
    }
    /**
     * Lists the hardware platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwarePlatforms() {
        this.add('freertos:ListHardwarePlatforms');
        return this;
    }
    /**
     * Lists the hardware vendors.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwareVendors() {
        this.add('freertos:ListHardwareVendors');
        return this;
    }
    /**
     * Lists the software configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListSoftwareConfigurations() {
        this.add('freertos:ListSoftwareConfigurations');
        return this;
    }
    /**
     * Updates the software configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toUpdateSoftwareConfiguration() {
        this.add('freertos:UpdateSoftwareConfiguration');
        return this;
    }
    /**
     * Adds a resource of type configuration to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     *
     * @param configurationName - Identifier for the configurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguration(configurationName, account, region, partition) {
        var arn = 'arn:${Partition}:freertos:${Region}:${Account}:configuration/${ConfigurationName}';
        arn = arn.replace('${ConfigurationName}', configurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * A tag key that is present in the request that the user makes to Amazon FreeRTOS.
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/console-tagging-iam.html/
     *
     * Applies to actions:
     * - .toCreateSoftwareConfiguration()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The tag key component of a tag attached to an Amazon FreeRTOS resource.
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/console-tagging-iam.html/
     *
     * Applies to resource types:
     * - configuration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The list of all the tag key names associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/console-tagging-iam.html/
     *
     * Applies to actions:
     * - .toCreateSoftwareConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Freertos = Freertos;
//# sourceMappingURL=data:application/json;base64,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