"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Guardduty = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [guardduty](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonguardduty.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Guardduty extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [guardduty](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonguardduty.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'guardduty';
        this.actionList = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptInvitation.html",
                "description": "Grants permission to accept invitations to become a GuardDuty member account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ArchiveFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ArchiveFindings.html",
                "description": "Grants permission to archive GuardDuty findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreateDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateDetector.html",
                "description": "Grants permission to create a detector.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateFilter.html",
                "description": "Grants permission to create GuardDuty filters. A filters defines finding attributes and conditions used to filter findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateIPSet.html",
                "description": "Grants permission to create an IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMembers.html",
                "description": "Grants permission to create GuardDuty member accounts. The account used to create a member becomes the GuardDuty master account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreatePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreatePublishingDestination.html",
                "description": "Grants permission to create a publishing destination.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject",
                    "s3:ListBucket"
                ],
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreateSampleFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateSampleFindings.html",
                "description": "Grants permission to create sample findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreateThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatIntelSet.html",
                "description": "Grants permission to create GuardDuty ThreatIntelSets. A ThreatIntelSet consists of known malicious IP addresses used by GuardDuty to generate findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeclineInvitations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeclineInvitations.html",
                "description": "Grants permission to decline invitations to become a GuardDuty member account.",
                "accessLevel": "Write"
            },
            "DeleteDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteDetector.html",
                "description": "Grants permission to delete GuardDuty detectors.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DeleteFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteFilter.html",
                "description": "Grants permission to delete GuardDuty filters.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "filter": {
                        "required": true
                    }
                }
            },
            "DeleteIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteIPSet.html",
                "description": "Grants permission to delete GuardDuty IPSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "ipset": {
                        "required": true
                    }
                }
            },
            "DeleteInvitations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteInvitations.html",
                "description": "Grants permission to delete invitations to become a GuardDuty member account.",
                "accessLevel": "Write"
            },
            "DeleteMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMembers.html",
                "description": "Grants permission to delete GuardDuty member accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DeletePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeletePublishingDestination.html",
                "description": "Grants permission to delete a publishing destination.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "publishingdestination": {
                        "required": true
                    }
                }
            },
            "DeleteThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatIntelSet.html",
                "description": "Grants permission to delete GuardDuty ThreatIntelSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "threatintelset": {
                        "required": true
                    }
                }
            },
            "DescribeOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeOrganizationConfiguration.html",
                "description": "Grants permission to retrieve details about the delegated administrator associated with a GuardDuty detector.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DescribePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribePublishingDestination.html",
                "description": "Grants permission to retrieve details about a publishing destination.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "publishingdestination": {
                        "required": true
                    }
                }
            },
            "DisableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisableOrganizationAdminAccount.html",
                "description": "Grants permission to disable the organization delegated administrator for GuardDuty.",
                "accessLevel": "Write"
            },
            "DisassociateFromMasterAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromMasterAccount.html",
                "description": "Grants permission to disassociate a GuardDuty member account from its GuardDuty master account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DisassociateMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateMembers.html",
                "description": "Grants permission to disassociate GuardDuty member accounts from their master GuardDuty account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "EnableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_EnableOrganizationAdminAccount.html",
                "description": "Grants permission to enable an organization delegated administrator for GuardDuty.",
                "accessLevel": "Write"
            },
            "GetDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetDetector.html",
                "description": "Grants permission to retrieve GuardDuty detectors.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFilter.html",
                "description": "Grants permission to retrieve GuardDuty filters.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "filter": {
                        "required": true
                    }
                }
            },
            "GetFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindings.html",
                "description": "Grants permission to retrieve GuardDuty findings.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetFindingsStatistics": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindingsStatistics.html",
                "description": "Grants permission to retrieve a list of GuardDuty finding statistics.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetIPSet.html",
                "description": "Grants permsission to retrieve GuardDuty IPSets.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "ipset": {
                        "required": true
                    }
                }
            },
            "GetInvitationsCount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetInvitationsCount.html",
                "description": "Grants permission to retrieve the count of all GuardDuty invitations sent to a specified account. The count does not include an accepted invitation.",
                "accessLevel": "Read"
            },
            "GetMasterAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMasterAccount.html",
                "description": "Grants permission to retrieve details of the GuardDuty master account associated with a member account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMembers.html",
                "description": "Grants permission to retrieve the member accounts associated with a master account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatIntelSet.html",
                "description": "Grants permission to retrieve GuardDuty ThreatIntelSets.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "threatintelset": {
                        "required": true
                    }
                }
            },
            "InviteMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html",
                "description": "Grants permission to invite other AWS accounts to enable GuardDuty and become GuardDuty member accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListDetectors": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html",
                "description": "Grants permission to retrieve a list of GuardDuty detectors.",
                "accessLevel": "List"
            },
            "ListFilters": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFilters.html",
                "description": "Grants permission to retrieve a list of GuardDuty filters.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFindings.html",
                "description": "Grants permission to retrieve a list of GuardDuty findings.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListInvitations.html",
                "description": "Grants permission to retrieve a lists of all of the GuardDuty membership invitations that were sent to an AWS account.",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMembers.html",
                "description": "Grants permission to retrierve a lsit of GuardDuty member accounts associated with a master account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListOrganizationAdminAccounts": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListOrganizationAdminAccounts.html",
                "description": "Grants permission to list details about the organization delegated administrator for GuardDuty.",
                "accessLevel": "List"
            },
            "ListPublishingDestinations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListPublishingDestinations.html",
                "description": "Grants permission to retrieve a list of publishing destinations.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to retrieve a list of tags associated with a GuardDuty resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "filter": {
                        "required": false
                    },
                    "ipset": {
                        "required": false
                    },
                    "threatintelset": {
                        "required": false
                    }
                }
            },
            "ListThreatIntelSets": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatIntelSets.html",
                "description": "Grants permission to retrieve a list of GuardDuty ThreatIntelSets.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "StartMonitoringMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMonitoringMembers.html",
                "description": "Grants permission to a master account to monitor findings from GuardDuty member accounts. Use this after disabling monitoring of member accounts using the StopMonitoringMembers operation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "StopMonitoringMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StopMonitoringMembers.html",
                "description": "Grants permission to disable monitoring findings from member accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to add tags to a GuardDuty resource. There is a limit of 50 tags per resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "filter": {
                        "required": false
                    },
                    "ipset": {
                        "required": false
                    },
                    "threatintelset": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UnarchiveFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UnarchiveFindings.html",
                "description": "Grants permission to unarchive GuardDuty findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove tags from a GuardDuty resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "filter": {
                        "required": false
                    },
                    "ipset": {
                        "required": false
                    },
                    "threatintelset": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateDetector.html",
                "description": "Grants permission to update GuardDuty detectors.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdateFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFilter.html",
                "description": "Grants permission to updates GuardDuty filters.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "filter": {
                        "required": true
                    }
                }
            },
            "UpdateFindingsFeedback": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFindingsFeedback.html",
                "description": "Grants permission to update findings feedback to mark GuardDuty findings as useful or not useful.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdateIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateIPSet.html",
                "description": "Grants permission to update GuardDuty IPSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "ipset": {
                        "required": true
                    }
                }
            },
            "UpdateOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateOrganizationConfiguration.html",
                "description": "Grants permission to update the delegated administrator configuration associated with a GuardDuty detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdatePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdatePublishingDestination.html",
                "description": "Grants permission to update a publishing destination.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject",
                    "s3:ListBucket"
                ],
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "publishingdestination": {
                        "required": true
                    }
                }
            },
            "UpdateThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatIntelSet.html",
                "description": "Grants permission to updates the GuardDuty ThreatIntelSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "threatintelset": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "detector": {
                "name": "detector",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "filter": {
                "name": "filter",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/filter/${FilterName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ipset": {
                "name": "ipset",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/ipset/${IPSetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "threatintelset": {
                "name": "threatintelset",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${ThreatIntelSetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "publishingdestination": {
                "name": "publishingdestination",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${PublishingDestinationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept invitations to become a GuardDuty member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        this.add('guardduty:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to archive GuardDuty findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ArchiveFindings.html
     */
    toArchiveFindings() {
        this.add('guardduty:ArchiveFindings');
        return this;
    }
    /**
     * Grants permission to create a detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateDetector.html
     */
    toCreateDetector() {
        this.add('guardduty:CreateDetector');
        return this;
    }
    /**
     * Grants permission to create GuardDuty filters. A filters defines finding attributes and conditions used to filter findings.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateFilter.html
     */
    toCreateFilter() {
        this.add('guardduty:CreateFilter');
        return this;
    }
    /**
     * Grants permission to create an IPSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        this.add('guardduty:CreateIPSet');
        return this;
    }
    /**
     * Grants permission to create GuardDuty member accounts. The account used to create a member becomes the GuardDuty master account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        this.add('guardduty:CreateMembers');
        return this;
    }
    /**
     * Grants permission to create a publishing destination.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreatePublishingDestination.html
     */
    toCreatePublishingDestination() {
        this.add('guardduty:CreatePublishingDestination');
        return this;
    }
    /**
     * Grants permission to create sample findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateSampleFindings.html
     */
    toCreateSampleFindings() {
        this.add('guardduty:CreateSampleFindings');
        return this;
    }
    /**
     * Grants permission to create GuardDuty ThreatIntelSets. A ThreatIntelSet consists of known malicious IP addresses used by GuardDuty to generate findings.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatIntelSet.html
     */
    toCreateThreatIntelSet() {
        this.add('guardduty:CreateThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to decline invitations to become a GuardDuty member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        this.add('guardduty:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty detectors.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteDetector.html
     */
    toDeleteDetector() {
        this.add('guardduty:DeleteDetector');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty filters.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteFilter.html
     */
    toDeleteFilter() {
        this.add('guardduty:DeleteFilter');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty IPSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        this.add('guardduty:DeleteIPSet');
        return this;
    }
    /**
     * Grants permission to delete invitations to become a GuardDuty member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        this.add('guardduty:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty member accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        this.add('guardduty:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to delete a publishing destination.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeletePublishingDestination.html
     */
    toDeletePublishingDestination() {
        this.add('guardduty:DeletePublishingDestination');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty ThreatIntelSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatIntelSet.html
     */
    toDeleteThreatIntelSet() {
        this.add('guardduty:DeleteThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to retrieve details about the delegated administrator associated with a GuardDuty detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        this.add('guardduty:DescribeOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve details about a publishing destination.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribePublishingDestination.html
     */
    toDescribePublishingDestination() {
        this.add('guardduty:DescribePublishingDestination');
        return this;
    }
    /**
     * Grants permission to disable the organization delegated administrator for GuardDuty.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisableOrganizationAdminAccount.html
     */
    toDisableOrganizationAdminAccount() {
        this.add('guardduty:DisableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to disassociate a GuardDuty member account from its GuardDuty master account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        this.add('guardduty:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants permission to disassociate GuardDuty member accounts from their master GuardDuty account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        this.add('guardduty:DisassociateMembers');
        return this;
    }
    /**
     * Grants permission to enable an organization delegated administrator for GuardDuty.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_EnableOrganizationAdminAccount.html
     */
    toEnableOrganizationAdminAccount() {
        this.add('guardduty:EnableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty detectors.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetDetector.html
     */
    toGetDetector() {
        this.add('guardduty:GetDetector');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty filters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFilter.html
     */
    toGetFilter() {
        this.add('guardduty:GetFilter');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindings.html
     */
    toGetFindings() {
        this.add('guardduty:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty finding statistics.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindingsStatistics.html
     */
    toGetFindingsStatistics() {
        this.add('guardduty:GetFindingsStatistics');
        return this;
    }
    /**
     * Grants permsission to retrieve GuardDuty IPSets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        this.add('guardduty:GetIPSet');
        return this;
    }
    /**
     * Grants permission to retrieve the count of all GuardDuty invitations sent to a specified account. The count does not include an accepted invitation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        this.add('guardduty:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve details of the GuardDuty master account associated with a member account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        this.add('guardduty:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the member accounts associated with a master account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        this.add('guardduty:GetMembers');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty ThreatIntelSets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatIntelSet.html
     */
    toGetThreatIntelSet() {
        this.add('guardduty:GetThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to invite other AWS accounts to enable GuardDuty and become GuardDuty member accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        this.add('guardduty:InviteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty detectors.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html
     */
    toListDetectors() {
        this.add('guardduty:ListDetectors');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFilters.html
     */
    toListFilters() {
        this.add('guardduty:ListFilters');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty findings.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFindings.html
     */
    toListFindings() {
        this.add('guardduty:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a lists of all of the GuardDuty membership invitations that were sent to an AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        this.add('guardduty:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrierve a lsit of GuardDuty member accounts associated with a master account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        this.add('guardduty:ListMembers');
        return this;
    }
    /**
     * Grants permission to list details about the organization delegated administrator for GuardDuty.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListOrganizationAdminAccounts.html
     */
    toListOrganizationAdminAccounts() {
        this.add('guardduty:ListOrganizationAdminAccounts');
        return this;
    }
    /**
     * Grants permission to retrieve a list of publishing destinations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListPublishingDestinations.html
     */
    toListPublishingDestinations() {
        this.add('guardduty:ListPublishingDestinations');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags associated with a GuardDuty resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('guardduty:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty ThreatIntelSets.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatIntelSets.html
     */
    toListThreatIntelSets() {
        this.add('guardduty:ListThreatIntelSets');
        return this;
    }
    /**
     * Grants permission to a master account to monitor findings from GuardDuty member accounts. Use this after disabling monitoring of member accounts using the StopMonitoringMembers operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMonitoringMembers.html
     */
    toStartMonitoringMembers() {
        this.add('guardduty:StartMonitoringMembers');
        return this;
    }
    /**
     * Grants permission to disable monitoring findings from member accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StopMonitoringMembers.html
     */
    toStopMonitoringMembers() {
        this.add('guardduty:StopMonitoringMembers');
        return this;
    }
    /**
     * Grants permission to add tags to a GuardDuty resource. There is a limit of 50 tags per resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('guardduty:TagResource');
        return this;
    }
    /**
     * Grants permission to unarchive GuardDuty findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UnarchiveFindings.html
     */
    toUnarchiveFindings() {
        this.add('guardduty:UnarchiveFindings');
        return this;
    }
    /**
     * Grants permission to remove tags from a GuardDuty resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('guardduty:UntagResource');
        return this;
    }
    /**
     * Grants permission to update GuardDuty detectors.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateDetector.html
     */
    toUpdateDetector() {
        this.add('guardduty:UpdateDetector');
        return this;
    }
    /**
     * Grants permission to updates GuardDuty filters.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFilter.html
     */
    toUpdateFilter() {
        this.add('guardduty:UpdateFilter');
        return this;
    }
    /**
     * Grants permission to update findings feedback to mark GuardDuty findings as useful or not useful.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFindingsFeedback.html
     */
    toUpdateFindingsFeedback() {
        this.add('guardduty:UpdateFindingsFeedback');
        return this;
    }
    /**
     * Grants permission to update GuardDuty IPSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        this.add('guardduty:UpdateIPSet');
        return this;
    }
    /**
     * Grants permission to update the delegated administrator configuration associated with a GuardDuty detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        this.add('guardduty:UpdateOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to update a publishing destination.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdatePublishingDestination.html
     */
    toUpdatePublishingDestination() {
        this.add('guardduty:UpdatePublishingDestination');
        return this;
    }
    /**
     * Grants permission to updates the GuardDuty ThreatIntelSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatIntelSet.html
     */
    toUpdateThreatIntelSet() {
        this.add('guardduty:UpdateThreatIntelSet');
        return this;
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(detectorId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type filter to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param filterName - Identifier for the filterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFilter(detectorId, filterName, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/filter/${FilterName}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${FilterName}', filterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param iPSetId - Identifier for the iPSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(detectorId, iPSetId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/ipset/${IPSetId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${IPSetId}', iPSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type threatintelset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param threatIntelSetId - Identifier for the threatIntelSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThreatintelset(detectorId, threatIntelSetId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${ThreatIntelSetId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${ThreatIntelSetId}', threatIntelSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type publishingdestination to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param publishingDestinationId - Identifier for the publishingDestinationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPublishingdestination(detectorId, publishingDestinationId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${PublishingDestinationId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${PublishingDestinationId}', publishingDestinationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateFilter()
     * - .toCreateIPSet()
     * - .toCreateThreatIntelSet()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - detector
     * - filter
     * - ipset
     * - threatintelset
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateFilter()
     * - .toCreateIPSet()
     * - .toCreateThreatIntelSet()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Guardduty = Guardduty;
//# sourceMappingURL=data:application/json;base64,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