"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ivs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ivs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoninteractivevideoservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ivs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ivs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoninteractivevideoservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ivs';
        this.actionList = {
            "BatchGetChannel": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_BatchGetChannel.html",
                "description": "Grants permission to get multiple channels simultaneously by channel ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            },
            "BatchGetStreamKey": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_BatchGetStreamKey.html",
                "description": "Grants permission to get multiple stream keys simultaneously by stream key ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Stream-Key": {
                        "required": true
                    }
                }
            },
            "CreateChannel": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_CreateChannel.html",
                "description": "Grants permission to create a new channel and an associated stream key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    },
                    "Stream-Key": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateStreamKey": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_CreateStreamKey.html",
                "description": "Grants permission to create a stream key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Stream-Key": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteChannel": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeleteChannel.html",
                "description": "Grants permission to delete a channel and channel's stream keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    },
                    "Stream-Key": {
                        "required": true
                    }
                }
            },
            "DeletePlaybackKeyPair": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeletePlaybackKeyPair.html",
                "description": "Grants permission to delete the playback key pair for a specified ARN",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Playback-Key-Pair": {
                        "required": true
                    }
                }
            },
            "DeleteStreamKey": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeleteStreamKey.html",
                "description": "Grants permission to delete the stream key for a specified ARN",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Stream-Key": {
                        "required": true
                    }
                }
            },
            "GetChannel": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetChannel.html",
                "description": "Grants permission to get the channel configuration for a specified channel ARN",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            },
            "GetPlaybackKeyPair": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetPlaybackKeyPair.html",
                "description": "Grants permission to get the playback keypair information for a specified ARN",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Playback-Key-Pair": {
                        "required": true
                    }
                }
            },
            "GetStream": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetStream.html",
                "description": "Grants permission to get information about the active (live) stream on a specified channel",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            },
            "GetStreamKey": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetStreamKey.html",
                "description": "Grants permission to get stream-key information for a specified ARN",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Stream-Key": {
                        "required": true
                    }
                }
            },
            "ImportPlaybackKeyPair": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_ImportPlaybackKeyPair.html",
                "description": "Grants permission to import the public key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Playback-Key-Pair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "ListChannels": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListChannels.html",
                "description": "Grants permission to get summary information about channels",
                "accessLevel": "List",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            },
            "ListPlaybackKeyPairs": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListPlaybackKeyPairs.html",
                "description": "Grants permission to get summary information about playback key pairs",
                "accessLevel": "List",
                "resourceTypes": {
                    "Playback-Key-Pair": {
                        "required": true
                    }
                }
            },
            "ListStreamKeys": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListStreamKeys.html",
                "description": "Grants permission to get summary information about stream keys",
                "accessLevel": "List",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    },
                    "Stream-Key": {
                        "required": true
                    }
                }
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListStreams.html",
                "description": "Grants permission to get summary information about live streams",
                "accessLevel": "List",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to get information about the tags for a specified ARN",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Channel": {
                        "required": false
                    },
                    "Playback-Key-Pair": {
                        "required": false
                    },
                    "Stream-Key": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutMetadata": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_PutMetadata.html",
                "description": "Grants permission to insert metadata into an RTMP stream for a specified channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            },
            "StopStream": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_StopStream.html",
                "description": "Grants permission to disconnect a streamer on a specified channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to add or update tags for a resource with a specified ARN",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Channel": {
                        "required": false
                    },
                    "Playback-Key-Pair": {
                        "required": false
                    },
                    "Stream-Key": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove tags for a resource with a specified ARN",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Channel": {
                        "required": false
                    },
                    "Playback-Key-Pair": {
                        "required": false
                    },
                    "Stream-Key": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateChannel": {
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_UpdateChannel.html",
                "description": "Grants permission to update a channel's configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Channel": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "Channel": {
                "name": "Channel",
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_Channel.html",
                "arn": "arn:${Partition}:ivs::${Account}:channel/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Stream-Key": {
                "name": "Stream-Key",
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_StreamKey.html",
                "arn": "arn:${Partition}:ivs::${Account}:stream-key/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Playback-Key-Pair": {
                "name": "Playback-Key-Pair",
                "url": "https://docs.aws.amazon.com/ivs/latest/APIReference/API_PlaybackKeyPair.html",
                "arn": "arn:${Partition}:ivs::${Account}:playback-key/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to get multiple channels simultaneously by channel ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_BatchGetChannel.html
     */
    toBatchGetChannel() {
        this.add('ivs:BatchGetChannel');
        return this;
    }
    /**
     * Grants permission to get multiple stream keys simultaneously by stream key ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_BatchGetStreamKey.html
     */
    toBatchGetStreamKey() {
        this.add('ivs:BatchGetStreamKey');
        return this;
    }
    /**
     * Grants permission to create a new channel and an associated stream key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        this.add('ivs:CreateChannel');
        return this;
    }
    /**
     * Grants permission to create a stream key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_CreateStreamKey.html
     */
    toCreateStreamKey() {
        this.add('ivs:CreateStreamKey');
        return this;
    }
    /**
     * Grants permission to delete a channel and channel's stream keys.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeleteChannel.html
     */
    toDeleteChannel() {
        this.add('ivs:DeleteChannel');
        return this;
    }
    /**
     * Grants permission to delete the playback key pair for a specified ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeletePlaybackKeyPair.html
     */
    toDeletePlaybackKeyPair() {
        this.add('ivs:DeletePlaybackKeyPair');
        return this;
    }
    /**
     * Grants permission to delete the stream key for a specified ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_DeleteStreamKey.html
     */
    toDeleteStreamKey() {
        this.add('ivs:DeleteStreamKey');
        return this;
    }
    /**
     * Grants permission to get the channel configuration for a specified channel ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetChannel.html
     */
    toGetChannel() {
        this.add('ivs:GetChannel');
        return this;
    }
    /**
     * Grants permission to get the playback keypair information for a specified ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetPlaybackKeyPair.html
     */
    toGetPlaybackKeyPair() {
        this.add('ivs:GetPlaybackKeyPair');
        return this;
    }
    /**
     * Grants permission to get information about the active (live) stream on a specified channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetStream.html
     */
    toGetStream() {
        this.add('ivs:GetStream');
        return this;
    }
    /**
     * Grants permission to get stream-key information for a specified ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_GetStreamKey.html
     */
    toGetStreamKey() {
        this.add('ivs:GetStreamKey');
        return this;
    }
    /**
     * Grants permission to import the public key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ImportPlaybackKeyPair.html
     */
    toImportPlaybackKeyPair() {
        this.add('ivs:ImportPlaybackKeyPair');
        return this;
    }
    /**
     * Grants permission to get summary information about channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        this.add('ivs:ListChannels');
        return this;
    }
    /**
     * Grants permission to get summary information about playback key pairs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListPlaybackKeyPairs.html
     */
    toListPlaybackKeyPairs() {
        this.add('ivs:ListPlaybackKeyPairs');
        return this;
    }
    /**
     * Grants permission to get summary information about stream keys
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListStreamKeys.html
     */
    toListStreamKeys() {
        this.add('ivs:ListStreamKeys');
        return this;
    }
    /**
     * Grants permission to get summary information about live streams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListStreams.html
     */
    toListStreams() {
        this.add('ivs:ListStreams');
        return this;
    }
    /**
     * Grants permission to get information about the tags for a specified ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('ivs:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to insert metadata into an RTMP stream for a specified channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_PutMetadata.html
     */
    toPutMetadata() {
        this.add('ivs:PutMetadata');
        return this;
    }
    /**
     * Grants permission to disconnect a streamer on a specified channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_StopStream.html
     */
    toStopStream() {
        this.add('ivs:StopStream');
        return this;
    }
    /**
     * Grants permission to add or update tags for a resource with a specified ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('ivs:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags for a resource with a specified ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('ivs:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a channel's configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        this.add('ivs:UpdateChannel');
        return this;
    }
    /**
     * Adds a resource of type Channel to the statement
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_Channel.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(resourceId, account, partition) {
        var arn = 'arn:${Partition}:ivs::${Account}:channel/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Stream-Key to the statement
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_StreamKey.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamKey(resourceId, account, partition) {
        var arn = 'arn:${Partition}:ivs::${Account}:stream-key/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Playback-Key-Pair to the statement
     *
     * https://docs.aws.amazon.com/ivs/latest/APIReference/API_PlaybackKeyPair.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPlaybackKeyPair(resourceId, account, partition) {
        var arn = 'arn:${Partition}:ivs::${Account}:playback-key/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags associated with the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateStreamKey()
     * - .toImportPlaybackKeyPair()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Channel
     * - Stream-Key
     * - Playback-Key-Pair
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateStreamKey()
     * - .toImportPlaybackKeyPair()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Ivs = Ivs;
//# sourceMappingURL=data:application/json;base64,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